/*
   ode::integratingFactorsOrder1_1(eq,y,x)
  
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 79, pp. 322

   DETAILS: 
   
   ode::int_factors(eq,y,x) tries to solve the ODE eq with respect to y(x)
   using the method of integrating factors from [1]. The implementation 
   only works for first order equations.

   Output: either FAIL or a set of solutions.

   EXAMPLES: 

    > ode::integratingFactorsOrder1_1(diff(y(x),x)+y(x)/x-x^2,y,x,{},{});
    > ode::integratingFactorsOrder1_1(y(x)*(y(x)^2-x)+x^2*diff(y(x),x),y,x,{},{});
*/

// originally called 'ode::int_factors'

ode::integratingFactorsOrder1_1:=
proc(eq,y,x,solveOptions,odeOptions)
  local N,M,_f,g,u,eq0,yp,uu,sing,solu,s, eq1,intOptions,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  yp:= genident("yp");
  eq0:= eq;
  userinfo(2,"trying to find an integrating factor");
  eq:= subs(eq,diff(y(x),x)=yp,y(x)=y);
  if not testtype(eq,Type::PolyExpr(yp)) then
    return(FAIL);
  end_if; 
  eq:= poly(eq,[yp]);
  if degree(eq)=1 then
    M:= coeff(eq,0);
    N:= coeff(eq,1); // M*dx+N*dy=0
    eq:= diff(M,y)-diff(N,x);
    if not has((_f:=ode::normal(eq/N)),y) then // Case (1) from p. 323
      u:= int(_f, x, intOptions);
      // if type(u) = "int" then
      if hastype(u,"int") then 
        return(FAIL)
      end_if;
    elif not has((g:=ode::normal(-eq/M)),x) then // Case (2) from p. 323
      // There is a typo in [1] for the sign of g
      u:= int(g,y,intOptions);
      if hastype(u,"int") then 
        return(FAIL);
      end_if;
      u:= subs(u,y=y(x));
    else
      return(FAIL);
    end_if;
    // We have to take care of the following fact: when multiplying by u
    // in the next ode::exact.... then we may loose some solution when 
    // the denominator of u vanishes. This is the case for the example:
    // y(x)^2 + (x*y(x)-1)*y(x)' =0 
    u:= expand(exp(u),optIgnoreAnalyticConstraints);
    sing:= {};
    if not has((uu:=subs(u,y(x)=y)),x) then
      if (solu:=solvelib::discreteSolve(1/uu,y,op(solveOptions)))<>FAIL then
        for s in solu do
          if traperror((eq1:= evalAt(eq0,y(x)=s))) = 0 and
            testeq(eq1,0,Steps = 5) = TRUE then
            sing:= sing union {s}
          end_if;
        end_for;
      end_if;
    end_if;
    userinfo(1, "found integrating factor",u);
    u:= ode::exact_first_order(u*eq0,y,x,solveOptions,odeOptions);
    if not has(u, FAIL) then
      userinfo(1, "integrating factor method worked");
      return(sing union u);
    end_if;
    
  end_if;
  return(FAIL);
end_proc:



/* ---------------------------------------------------------------------------------------
   'ode::integratingFactorsOrder1_2' -- computing integrating factors 
                                                 mu = 1/(Q(x)-F(x)*Phi(x,y)) 
                                        of 1st order ODEs y' = Phi(x,y)                                       
   ---------------------------------------------------------------------------------------
 
   -----------
   PARAMETERS. 
   -----------

             eq -- expression encoding a 2nd order ODE
              y -- dependent variable
              x -- indepenent variable 
   solveOptions -- options for 'solve'
     odeOptions -- options for 'ode' 

  REFERENCE: Theorem 2.9. in "Symbolic algorithms for the computation of 
             integrating factors", internal document 

  EXAMPLES: (1) >> Phi:= A(x) + B(x)*exp(y(x)/C):
                >> eq:= diff(y(x),x) - Phi:
                >> ode::integratingFactorsOrder1_2(eq,y,x,{},{})
                    -1/((A(x) + exp(y(x)/C)*B(x))/(exp(int(A(x)/C, x))*B(x)) - 
                    A(x)/(exp(int(A(x)/C, x))*B(x)))

            (2) >> Phi:= 1 - (3*x)/(exp(2*x + y(x)/2 + 1)*sin(x)) - exp(x^2):
                >> eq:= diff(y(x),x) - Phi:
                >> ode::integratingFactorsOrder1_2(eq,y,x,{},{})
                    1/((exp(2*x + 1)*exp(x/2 + (PI^(1/2)*erf(x*I)*I)/4)*sin(x)*
                    (exp(x^2) - 1))/(3*x) - (exp(2*x + 1)*exp(x/2 + (PI^(1/2)*
                    erf(x*I)*I)/4)*sin(x)*(exp(x^2) + (3*x)/(exp(2*x + y(x)/2 + 
                    1)*sin(x)) - 1))/(3*x))

  --------------------------------------------------------------------------------------- */


ode::integratingFactorsOrder1_2:=
proc(eq,y,x,solveOptions,odeOptions)
  local intOptions,optIgnoreAnalyticConstraints,Phi,A, B, C, F, Q, mu, tmp, tmp1, tmp2, 
        J, K, U, lcf;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  tmp:= ode::quasiLinearForm(eq,y,x,solveOptions,odeOptions);  
  if tmp = FAIL then 
    return(FAIL);
  end_if; 
  // Now tmp = [y'-Phi(x,y),Phi(x,y),lcf,y,x,solveOptions,odeOptions]. 
  Phi:= tmp[2];
  lcf:= tmp[3];
  if has(Phi,diff(y(x),x)) or lcf = FAIL then 
    return(FAIL) // not a quasi-linear ODE 
  end_if;
  mu:= FAIL; 
  Phi:= subs(Phi,y(x)=y);
  if hastype(Phi,"exp") then 
    C:= ode::normal(diff(Phi,y)/diff(Phi,y,y));
    if not has(C,x) and not has(C,y) then 
      A:= ode::normal(Phi-C*diff(Phi,y));  
      if not has(A,y) and not iszero(C) then 
        B:= ode::normal((Phi-A)/exp(y/C));
        if has(B,y) then  
          B:= ode::normal(combine((Phi-A)/exp(y/C),exp));
        end_if;        
        if not has(B,y) then 
          F:= 1/B*exp(-int(A/C,x));
          Q:= A*F;
          if not iszero(Q-F*Phi) then 
            mu:= 1/lcf*subs(1/(Q-F*Phi),y=y(x));
          end_if;  
        end_if;
      end_if;  
    end_if;  
  end_if;  
  // NOTE: If 'diff(Phi,y,y) = 0', then the ODE is a 1st order linear ODE 
  //       solvable by quadrature. For the case that 'Phi' corresponds to 
  //       some inverse ODE, i.e. we dealing with an equation whose inverse 
  //       ODE is 1st order linear, then the original ODE will be tackled 
  //       by the methods in 'ode::interchange'. These methods compute the 
  //       inverse ODE, solve it and then construct the original ODE solution
  //       from ther solution of the inverse ODE by interchanging 'x' and 'y' 
  //       and solving for 'y'. 
  if mu = FAIL and 
     traperror((K:= diff(Phi,y)/diff(Phi,y,y))) = 0 and 
     traperror((U:= diff(K,x)/diff(K,y))) = 0 and 
     length(U) < 2000 and // using 'MaxSteps' is not helpful here
     traperror((J:= (U*diff(Phi,y)-diff(U,x)-diff(Phi,x))/(Phi+U))) = 0 and 
     ode::odeIszero((tmp1:= diff(U,y))) and ode::odeIszero((tmp2:= diff(J,y))) and 
     iszero(ode::normal(tmp1)) and iszero(ode::normal(tmp2)) then 
    F:= exp(int(J,x));
    Q:= -F*U;
    mu:= 1/lcf*subs(1/(Q-F*Phi),y=y(x));
  end_if;  

  if mu <> FAIL then 
    if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
      return(mu);
    else 
      eq:= int(simplify(mu*eq),x,intOptions);
      eq:= ode::checkIntegration(eq,y,x,1);
      if eq <> FAIL then 
        return(solve(subs(eq,y(x)=y)=genident("C"),y,op(solveOptions)));
      end_if;
    end_if;  
  end_if;

  return(FAIL);
end_proc:  
  
  

/* ---------------------------------------------------------------------------------------
   'ode::integratingFactorsOrder1_3' -- computing integrating factors 
                                                 mu = 1/(F(x)*G(y)*Phi(x,y)) 
                                        of 1st order ODEs y' = Phi(x,y)                                       
   ---------------------------------------------------------------------------------------
 
   -----------
   PARAMETERS. 
   -----------

             eq -- expression encoding a 2nd order ODE
              y -- dependent variable
              x -- indepenent variable 
   solveOptions -- options for 'solve'
     odeOptions -- options for 'ode' 

  REFERENCE: Theorem 2.1. in "Symbolic algorithms for the computation of 
             integrating factors", internal document 

  EXAMPLES: (1) >> Phi:= -(y+b)^2/((x+a)*(1+(y+b)^2*(x+a)*sin(y))):
                >> eq:= diff(y(x),x)-subs(Phi,y=y(x)): 
                >> ode::integratingFactorsOrder1_3(eq,y,x,{},{})  
                    (exp(1/(b + y(x)))*(a + x)*(sin(y(x))*(a + x)*
                    (b + y(x))^2 + 1))/((b + y(x))^2*(a^2 + 2*a*x + x^2))

  --------------------------------------------------------------------------------------- */

ode::integratingFactorsOrder1_3:=
proc(eq,y,x,solveOptions,odeOptions)
  local intOptions,optIgnoreAnalyticConstraints,Phi,SBP,F,G,integrand,mu,tmp,lcf;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  tmp:= ode::quasiLinearForm(eq,y,x,solveOptions,odeOptions);  
  if tmp = FAIL then 
    return(FAIL);
  end_if; 
  // Now tmp = [y'-Phi(x,y),Phi(x,y),lcf,y,x,solveOptions,odeOptions]. 
  Phi:= tmp[2];
  lcf:= tmp[3];
  mu:= FAIL;
  if has(Phi,diff(y(x),x)) or lcf = FAIL then 
    return(FAIL) // not a quasi-linear ODE 
  end_if;  
  Phi:= subs(Phi,y(x)=y);
  if iszero(Phi) then 
    return(FAIL);
  end_if;  
  SBP:= ode::separateByProduct(1/Phi^2*diff(diff(ln(Phi),y),x),y,x);
  if SBP <> FAIL then 
    F:= SBP[1]; // those factors only depenending on 'x' 
    if traperror((integrand:= ode::normal(F*diff(1/(F*Phi),x)))) = 0 and 
       not has(integrand,x) then 
      G:= exp(int(integrand,y));
      if not iszero(F*G*Phi) then 
        mu:= 1/lcf*subs(1/(F*G*Phi),y=y(x));
      end_if;
    end_if;   
  end_if;  

  if mu <> FAIL then 
    if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
      return(mu);
    // else // will be needed when the methods are integrated into the actual solving process
    //  eq:= int(simplify(mu*eq),x,intOptions);
    //  eq:= ode::checkIntegration(eq,y,x,1);
    //  if eq <> FAIL then 
    //    return(solve(subs(eq,y(x)=y)=genident("C"),y,op(solveOptions)));
    //  end_if;
    end_if;  
  end_if;

  
  return(FAIL);
end_proc:


/* ---------------------------------------------------------------------------------------
   'ode::integratingFactorsOrder1_4' -- computing integrating factors 
                                           mu = 1/(F(x)*G(y)), mu = 1/(F(x)+G(y))   
                                        of 1st order ODEs y' = Phi(x,y)                                       
   ---------------------------------------------------------------------------------------
 
   -----------
   PARAMETERS. 
   -----------

             eq -- expression encoding a 2nd order ODE
              y -- dependent variable
              x -- indepenent variable 
   solveOptions -- options for 'solve'
     odeOptions -- options for 'ode' 

  REFERENCE: Theorem 2.2. in "Symbolic algorithms for the computation 
             of integrating factors", internal document 

  EXAMPLES: (1) >> Phi:= -((a + y)*(sin(x)*(b + x)^2*(a + y) + 1))/(b + x)^2:
                >> eq:= diff(y(x),x)-subs(Phi,y=y(x)):
                >> ode::integratingFactorsOrder1_4(eq,y,x,{},{})
                      -exp(1/(b + x))/(a^2 + 2*a*y(x) + y(x)^2)

            (2) >> Phi:= 3*(1+x^2/y^2)*arctan(y/x)+(1-2*y)/x+(1-3*y)*x/y^2:
                >> Phi:= subs(Phi,y=y(x)): 
                >> eq:= diff(y(x),x) - Phi:
                >> ode::integratingFactorsOrder1_6(eq,y,x,{},{})
                      -(2*x^2*y(x)^2)/(x^2 + y(x)^2)

  --------------------------------------------------------------------------------------- */

ode::integratingFactorsOrder1_4:=
proc(eq,y,x,solveOptions,odeOptions)
  local intOptions,optIgnoreAnalyticConstraints,Phi,mu,tmp,invPhi,mu_tilde,xx,yy,lcf;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  tmp:= ode::quasiLinearForm(eq,y,x,solveOptions,odeOptions);  
  if tmp = FAIL then 
    return(FAIL);
  end_if; 
  // Now tmp = [y'-Phi(x,y),Phi(x,y),lcf,y,x,solveOptions,odeOptions]. 
  Phi:= tmp[2];
  lcf:= tmp[3];
  if has(Phi,diff(y(x),x)) or lcf = FAIL then 
    return(FAIL) // not a quasi-linear ODE 
  end_if;  
  yy:= genident();
  xx:= genident();
  invPhi:= 1/subs(Phi,[y(x)=xx,x=yy(xx)],EvalChanges);
  mu:= FAIL;
  mu_tilde:= FAIL; 
  if mu_tilde = FAIL then 
    // Theorem 2.2.
    mu_tilde:= ode::integratingFactorsOrder1_3(diff(yy(xx),xx)-invPhi,
                                               yy,xx,solveOptions,
                                               odeOptions union {"ReturnIntegratingFactorOnly"});
  end_if;
  if mu_tilde = FAIL then 
    // Theorem 2.4.
    mu_tilde:= ode::integratingFactorsOrder1_5(diff(yy(xx),xx)-invPhi,
                                               yy,xx,solveOptions,
                                               odeOptions union {"ReturnIntegratingFactorOnly"});
  end_if;
  if mu_tilde = FAIL then 
    // Theorem 2.9.
    mu_tilde:= ode::integratingFactorsOrder1_2(diff(yy(xx),xx)-invPhi,
                                               yy,xx,solveOptions,
                                               odeOptions union {"ReturnIntegratingFactorOnly"});
  end_if;
  if mu_tilde <> FAIL and not iszero(mu_tilde) then  
    mu:= 1/lcf*subs(ode::normal(invPhi*mu_tilde),[yy(xx)=x,xx=y(x)]);
  end_if;  

  if mu <> FAIL then 
    if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
      return(mu);
    else 
      eq:= int(simplify(mu*eq),x,intOptions);
      eq:= ode::checkIntegration(eq,y,x,1);
      if eq <> FAIL then 
        return(solve(subs(eq,y(x)=y)=genident("C"),y,op(solveOptions)));
      end_if;
    end_if;  
  end_if;
  
   
  return(FAIL);
end_proc:



/* ---------------------------------------------------------------------------------------
   'ode::integratingFactorsOrder1_5' -- computing integrating factors 
                                                 mu = 1/((F(x)+G(y)*Phi(x,y)) 
                                        of 1st order ODEs y' = Phi(x,y)                                       
   ---------------------------------------------------------------------------------------
 
   -----------
   PARAMETERS. 
   -----------

             eq -- expression encoding a 2nd order ODE
              y -- dependent variable
              x -- indepenent variable 
   solveOptions -- options for 'solve'
     odeOptions -- options for 'ode' 

  REFERENCE: Theorem 2.4. in "Symbolic algorithms for the computation of 
             integrating factors", internal document 

  EXAMPLES: (1) >> Phi:= y*x^2/(3*y*(x^2+y^2)*arctan(x/y)+y^2*(1-3*x)+x^2*(1-2*x)):
                >> Phi:= subs(Phi,y=y(x)):
                >> eq:= diff(y(x),x) - Phi:
                >> ode::integratingFactorsOrder1_5(eq,y,x,{},{}))
                      (2*y(x)*(x^2*(2*x - 1) + y(x)^2*(3*x - 1) - 
                       3*arctan(x/y(x))*y(x)*(x^2 + y(x)^2)))/(x^2 + y(x)^2)

  --------------------------------------------------------------------------------------- */

ode::integratingFactorsOrder1_5:=
proc(eq,y,x,solveOptions,odeOptions)
  local intOptions,optIgnoreAnalyticConstraints,Phi,SBP,mu,tmp,tmp2,F_plus_G,F_xx,lcf;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  tmp:= ode::quasiLinearForm(eq,y,x,solveOptions,odeOptions);  
  if tmp = FAIL then 
    return(FAIL);
  end_if; 
  // Now tmp = [y'-Phi(x,y),Phi(x,y),lcf,y,x,solveOptions,odeOptions]. 
  Phi:= tmp[2];
  lcf:= tmp[3];
  mu:= FAIL;
  if has(Phi,diff(y(x),x)) or lcf = FAIL then 
    return(FAIL); // not a quasi-linear ODE 
  end_if;  
  Phi:= subs(Phi,y(x)=y);
  if iszero(Phi) then 
    return(FAIL);
  end_if;  
  tmp2:= ode::normal(Phi*diff(1/Phi,x,x));
  if iszero(tmp2) then 
    return(FAIL);
  end_if;  
  SBP:= ode::separateByProduct(diff(1/tmp2,y),y,x);
  if SBP <> FAIL and not iszero(SBP[1]) then 
    F_xx:= 1/SBP[1]; // those factors only depenending on 'x' 
    F_plus_G:= ode::normal(F_xx/tmp2);
    if not iszero(F_plus_G*Phi) and 
       ode::odeIszero((tmp:= diff(F_plus_G*Phi,x)+Phi^2*diff(F_plus_G,y))) and 
       iszero(ode::normal(tmp)) then 
      mu:= 1/lcf*subs(1/(F_plus_G*Phi),y=y(x));
    end_if;   
  end_if;  

  if mu <> FAIL then 
    if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
      return(mu);
    else 
      eq:= int(simplify(mu*eq),x,intOptions);
      eq:= ode::checkIntegration(eq,y,x,1);
      if eq <> FAIL then 
        return(solve(subs(eq,y(x)=y)=genident("C"),y,op(solveOptions)));
      end_if;
    end_if;  
  end_if;
  
  
  return(FAIL);
end_proc:


/* ---------------------------------------------------------------------------------------
   'ode::integratingFactorsOrder1_6' -- computing integrating factors of particular 
                                        Riccati ODEs 
   ---------------------------------------------------------------------------------------
 
   -----------
   PARAMETERS. 
   -----------

             eq -- expression encoding a 2nd order ODE
              y -- dependent variable
              x -- indepenent variable 
   solveOptions -- options for 'solve'
     odeOptions -- options for 'ode' 

  REFERENCE: Theorem 2.12. in "Symbolic algorithms for the computation of 
             integrating factors", internal document 

  EXAMPLES: (1) >> eq:= diff(y(x), x) - y(x)^2/x - x - x^2 - (y(x)*(3*x - 3^(1/2)*x*(x + 
                        1)^(3/2) + 2))/(2*x^2 + 2*x)
                >> ode::integratingFactorsOrder1_6(eq,y,x,{},{}))
                      -(2*x^2*(1/(x*(x^2 + x)))^(1/2)*(x + 1)^3)/((2*x^3*(x + 1)^2)/(x^2 + 
                        x) - 3*x^2*y(x) - 5*x*y(x) - 2*y(x) + (4*x^4*(x + 1)^2)/(x^2 + x) + 
                       (2*x^5*(x + 1)^2)/(x^2 + x) + (2*x*y(x)*(x + 1)^2)/(x^2 + x) + 
                       (2*x*y(x)^2*(x + 1)^2)/(x^2 + x) + (3*x^2*y(x)*(x + 1)^2)/(x^2 + x) + 
                       (2*x^2*y(x)^2*(x + 1)^2)/(x^2 + x) - (3^(1/2)*x^2*y(x)*(x + 
                        1)^(7/2))/(x^2 + x))

  --------------------------------------------------------------------------------------- */

ode::integratingFactorsOrder1_6:=
proc(eq,y,x,solveOptions,odeOptions)
  local intOptions,optIgnoreAnalyticConstraints,Phi,tmp,f0,f1,f2,z,xi,eta,mu,lcf;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  tmp:= ode::quasiLinearForm(eq,y,x,solveOptions,odeOptions);  
  if tmp = FAIL then 
    return(FAIL);
  end_if; 
  mu:= FAIL;
  // Now tmp = [y'-Phi(x,y),Phi(x,y),lcf,y,x,solveOptions,odeOptions]. 
  Phi:= tmp[2];
  lcf:= tmp[3];
  // Need to check if the 'Phi(x,y)' is of the form 'f2(x)*y(x)^2+f1(x)*y(x)+f0(x)'.
  z:= genident();
  tmp:= poly(ode::normal(subs(Phi,y(x)=z)),[z]);
  if tmp <> FAIL and degree(tmp,z) <=2 then 
    f2:= ode::normal(coeff(tmp,z,2));
    f1:= ode::normal(coeff(tmp,z,1));
    f0:= ode::normal(coeff(tmp,z,0));
    if iszero(f0*f2) then 
      return(FAIL);
    end_if;
    // check condition (2.18)
    tmp:= ode::normal((diff(f0,x)*f2-f0*diff(f2,x)-2*f0*f1*f2)^2/(f0*f2)^3);
    if ode::odeIszero((tmp:= diff(tmp,x))) then 
      tmp:= ode::normal(tmp); 
      if not iszero(tmp) then 
        tmp:= simplify(tmp);
      end_if;
      if iszero(tmp) then 
        xi:= ode::normal(1/f2*sqrt(f2/f0),Expand=FALSE);
        eta:= ode::normal((diff(f0,x)*f2-f0*diff(f2,x))/(2*f0^2*f2*sqrt(f2/f0))*y(x),Expand=FALSE);
        mu:= 1/lcf*ode::normal(1/(eta-xi*Phi),Expand=FALSE);
      end_if;
    end_if;
  end_if;

  if mu <> FAIL then 
    if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
      return(mu);
    // else // will be needed when the methods are integrated into the actual solving process
    //  eq:= int(simplify(mu*eq),x,intOptions);
    //  eq:= ode::checkIntegration(eq,y,x,1);
    //  if eq <> FAIL then 
    //    return(solve(subs(eq,y(x)=y)=genident("C"),y,op(solveOptions)));
    //  end_if;
    end_if;  
  end_if;
  
  
  return(FAIL);
end_proc:



/* ---------------------------------------------------------------------------------------
   'ode::integratingFactorsOrder1_7' -- computing integrating factors of particular 
                                        Riccati ODEs 
   ---------------------------------------------------------------------------------------
 
   -----------
   PARAMETERS. 
   -----------

             eq -- expression encoding a 2nd order ODE
              y -- dependent variable
              x -- indepenent variable 
   solveOptions -- options for 'solve'
     odeOptions -- options for 'ode' 

  REFERENCE: Theorem 2.13. in "Symbolic algorithms for the computation of 
             integrating factors", internal document 

  EXAMPLES: (1) >> eq:= 
                >> ode::integratingFactorsOrder1_6(eq,y,x,{},{}))

  --------------------------------------------------------------------------------------- */

ode::integratingFactorsOrder1_7:=
proc(eq,y,x,solveOptions,odeOptions)
  local intOptions,optIgnoreAnalyticConstraints,Phi,tmp,z,xi,eta,mu,Eta,L,Xi,i,lcf;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  tmp:= ode::quasiLinearForm(eq,y,x,solveOptions,odeOptions);  
  if tmp = FAIL then 
    return(FAIL);
  end_if; 
  // Now tmp = [y'-Phi(x,y),Phi(x,y),lcf,y,x,solveOptions,odeOptions]. 
  Phi:= tmp[2];
  lcf:= tmp[3];
  z:= genident();
  L:= coerce(factor(subs(Phi,y(x)=y)),DOM_LIST);
  Xi:= {};
  Eta:= {};
  mu:= FAIL;
  for i from 1 to nops(L) div 2 do 
    if has(L[2*i],x) and not has(L[2*i],y) then 
      Xi:= Xi union {L[2*i]^(-L[2*i+1])};
    elif has(L[2*i],y) and not has(L[2*i],x) then 
      Eta:= Eta union {L[2*i]^L[2*i+1]};
    end_if;
  end_for;  
  xi:= _mult(1/L[1],op(Xi));
  eta:= -subs(_mult(op(Eta)),y=y(x));
  if not iszero(eta-xi*Phi) then
    mu:= 1/lcf*1/(eta-xi*Phi);   
    if not ode::eulerOperator(mu*eq,y,x,solveOptions,odeOptions union {"ExactnessTest"}) then 
      mu:= FAIL;
    end_if;
  end_if;
  if mu = FAIL and L[1] <> 1 then 
    xi:= _mult(op(Xi));
    eta:= subs(_mult(L[1],op(Eta)),y=y(x));
    if not iszero(eta-xi*Phi) then
      mu:= 1/lcf*1/(eta-xi*Phi);   
      if not ode::eulerOperator(mu*eq,y,x,solveOptions,odeOptions union {"ExactnessTest"}) then 
        mu:= FAIL;
      end_if;
    end_if;
  end_if;  

  if mu <> FAIL then 
    if contains(odeOptions,"ReturnIntegratingFactorOnly") then 
      return(mu);
    else 
      eq:= int(simplify(mu*eq),x,intOptions);
      eq:= ode::checkIntegration(eq,y,x,1);
      if eq <> FAIL then 
        return(solve(subs(eq,y(x)=y)=genident("C"),y,op(solveOptions)));
      end_if;
    end_if;  
  end_if;
  
  
  return(FAIL);
end_proc:

  
  