
/*++ ---------------- ExpSol.mu ---------------------

Description:
This file contains functions for computing exponential solutions
of homogeneous linear ordinary differential equations 
over the field of rational functions. 

Functions: 

 - used Parameter:
    ++ Ly, y(x), v: DOM_EXPR
    ++ y, x       : DOM_IDENT
    ++ n          : DOM_INT
    ++ Ly    = ordinary linear homogeneous differential equation 
    ++         over the rational functions.
    ++ y(x)  = the operator (function) of Ly  
    ++ y     = name of the operator of Ly
    ++ x     = dependent variable of the operator y
    ++ n     = order of Ly

 - ode::exponentialSolutions(Ly, y(x), < Generic >)
    ++ determines all exponential solutions of Ly, i.e. solutions z such that
    ++ z'/z is a rational function.
    ++ When the option Generic is given, a generic form of them is returned.
    ++ NOTE: currently this function works properly only if all singularities
    ++ are in Q and no occuring candidate is involved with algebraic numbers.
    ++ Therefore, actually this function is far away from being a 
    ++ decision procedure. :-( 
 - ode::expsols(Ly, y, x, n)
    ++ determines all exponential solutions of Ly, i.e. solutions z such that
    ++ z'/z is a rational function.
    ++ NOTE: Ly should be normalized i.e. denom(Ly)=1.
    ++ NOTE: currently this function works properly only if all singularities
    ++ are in Q and no occuring candidate is involved with algebraic numbers.
    ++ Therefore, actually this function is far away from being a 
    ++ decision procedure. :-(   
 
REFERENCES:
 - Bronstein, M. (1992). Linear Ordinary Differential Equations: breaking
   through the order 2 barrier. In: Proceedings of ISSAC'92, pp. 42-48.
 - Pflgel, E. (1997). An Algorithm for Computing Exponential Solutions of
   First Order Linear Differential Systems. In: Proceedings of ISSAC'97,
   pp. 164-171.


Examples: 

>> Ly:= diff(y(x),x$4)-2*x*diff(y(x),x$3)+(-x+x^2-5)*diff(y(x),x$2)+
        (4*x+2*x^2)*diff(y(x),x)+(2+x-x^3)*y(x)

>> ode::exponentialSolutions(Ly,y(x)),
   ode::exponentialSolutions(Ly,y(x),Generic)

    {exp(x^2/2), x*exp(x^2/2)}, C2*exp(x^2/2) + C1*x*exp(x^2/2)

++*/

ode::exponentialSolutions:= proc(eq,z,o=" ",solveOptions={},odeOptions={}) //o: Option Generic
  local sol,optIgnoreAnalyticConstraints;
begin 
  if args(0) < 2 or args(0) > 5 then 
    error("expecting two or three arguments")
  elif args(0) = 3 and o <> Generic then 
    error("expecting option 'Generic' as third argument");
  elif args(0) = 4 or args(0) = 5 then
    if map({args(args(0)-1),args(args(0))},type) <> {DOM_SET} then 
      error("invalid arguments")    
    end_if;
  end_if;
  if type(z) <> "function" then
    error("expecting a function as the dependent variable")
  elif nops(z) <> 1 then 
    error("dependent variable must be a univariate function") 
  end_if; 
  if domtype(eq) <> DOM_EXPR then 
    error("expecting an ordinary differential equation as first argument")  
  end_if;
  if not has(eq,z) then 
    error("not an ordinary differential equation in ".expr2text(z))
  end_if;
  eq:= ode::isLODE(rewrite(eq, diff), z, HlodeOverRF, solveOptions, odeOptions);
  if nops(eq)<> 4 then
    return(error("not an ordinary homogeneous linear differential ".
                           "equation over the rational functions"))
  end_if;
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  // eq[1] := numer(normal(expr(eq[1]),Rationalize=None));
  eq[1]:= ode::removeDenominator(eq);
  sol:= ode::expsols(eq,solveOptions,odeOptions);
  // K.G.: I have checked whether it gives a pay-off to replace the following
  //       command by a simple map of 'Simplify' to the operands of the elements 
  //       of the set 'sol'. This is not the case. Even when completely removing
  //       the simplification this does not give a significant speed-up, but 
  //       less nice results. So I leave this command untouched.
  sol:= map(map(map(sol,expand,optIgnoreAnalyticConstraints),
                simplify,optIgnoreAnalyticConstraints),
            combine,exp,optIgnoreAnalyticConstraints);
  if o = Generic then
    return(_plus(op(map(sol, e->_mult(e,genident("C"))))));
  else
    return(sol);
  end_if
end_proc:


/* 
  ode::expsols(eq, y, x, n)
    INPUT:  eq - a linear differential equation with polynomial coefficients
            y  - a name (the unknown)
            x  - a name
            n  - an integer (the order of eq)
    OUTPUT: the set of all solutions y of eq with y'/y \in \bar(Q)(x)
*/

ode::expsols:=
proc(eq, y, x, n,solveOptions,odeOptions)
local lode, l, m, i, j, d, p, parts, parts1, part, places,
      f, P, res, u, intOptions;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  // set a flag for correct warnings
  sysassign(ode::possiblyLostSolutionsFlag, FALSE);
  lode:= ode::vectorize(eq,y,x,n,solveOptions,odeOptions);
  l:= factor(poly(lode[n+1],[x]));
  l:= Factored::convert_to(l,DOM_LIST);
  m:= nops(l) div 2;
  // Compute potential singularities 
  for i from 1 to m do 
    p[i]:= expr(l[2*i]) 
  end_for;
  p[m+1]:= 1/x;
  parts:= []; 
  places:= []; 
  for i from 1 to m+1 do
    userinfo(2, "computing p-adic exponential parts at: ", p[i]);
    // Compute p-adic exponential parts 
    part:= ode::padicNewton(lode,x,p[i],n,solveOptions,odeOptions);
    // Eliminate apparent singularities 
    if part <> [[1, 0, 0]] then
      parts:= append(parts, part);
      places:= append(places, p[i])
    end_if;
    userinfo(3, "found: ", part);
    if part = [] then
      userinfo(1, "proof for no exponential solutions");
      return({})
    end_if;
  end_for;
  // compute possible denominator of rational part 
  userinfo(3, "computing rational part");
  d := 1;
  for i from 1 to nops(parts) do
    for j from 1 to nops(parts[i]) do
      d := lcm(d, parts[i][j][1]);
      parts[i][j] := [parts[i][j][2], parts[i][j][3]]
    end_for
  end_for;
  // and bound it 
  userinfo(3, "bounding denominator:", d);
  assert(not iszero(d));
  lode:= ode::changeRiccati(lode,x,-diff(d, x)/d,solveOptions,odeOptions);
  // eliminate equivalent exponential parts 
  parts1:= [];
  j:= 0;
  f:= proc() begin 
               [args(1)[2], args(1)[3]] 
             end_proc;
  for i from 1 to nops(parts) do
    if parts[i] <> [[0, 0]] then
      parts1:= append(parts1, parts[i])
    end_if
  end_for;
  parts:= parts1; 
  // combine all p-adic parts 
  parts:= ode::possiblePadicParts(parts,solveOptions,odeOptions);
  res:= {};
  // test for polynomial solutions 
  for part in parts do
    u:= diff(part[1], x)+part[2];
    userinfo(3, "checking Riccati solution:", u); 
    /*
      The function 'ode::changeRiccati' returns a list. Hence, 'ode::polysols' uses 
      the first branch, where its arguments are checked for 'eq' being a list. Note 
      that when 'ode::polysols' is called with 'eq' being a list, then 'ode::polysols' 
      itself sets the values for 'x', 'n' and 'b'. Hence, we can here give dummy 
      values to these arguments. This must be done, since the last two arguments must 
      be 'solveOptions' and 'odeOptions'. This is a general principle concerning these 
      options used throughout the whole ODE library. Do not change this!!!
    */
    for P in ode::polysols(ode::changeRiccati(lode, x, u,solveOptions,odeOptions),
                           x, 
                           "dummy for x", "dummy for n", "dummy for b", 
                           solveOptions, odeOptions) do
      userinfo(3, "polynomial solution found:", P);
      res := res union 
             {ode::normal(P/d,Rationalize=None)*exp(part[1]+int(part[2],x,intOptions))};
    end_for;
    if nops(res) = n then 
      break 
    end_if;
  end_for;
  // the new solver returns correct generic solutions; but we do not want them
  // unless the option Generic is given to ode::exponentialSolutions
  // therefore, we replace the free variables by 1 again
  l:= [op(indets(res) minus Type::ConstantIdents minus indets(eq))];
  return(evalAt(res,map(l,_equal,1)));
end_proc:


// The p-adic Newton algorithm 
/* 
  ode::padicNewton(lode, x, p, m)                                             
    INPUT:  lode - list of coefficients of a linear differential operator    
            x    - a name                                                     
            p    - an irreducible polynomial or 1/x (representing infinity)   
            m    - an integer                                                 
    OUTPUT: the list of exponential parts of L at p - at most m parts         
*/                                                                            

ode::padicNewton:= 
proc(lode, x, p, m,solveOptions,odeOptions)
  local m1, res, ntPolygon, c, slope, u, aux;
begin
  res:= [];
  // delete u;
  u:= genident();
  userinfo(2, "computing Newton polygon");
  ntPolygon:= ode::newtonPolygon(lode, x, p, u, m,solveOptions,odeOptions)[3];
  m1:= m;
  while m1 > 0 and nops(ntPolygon) > 0 do
    slope:= ntPolygon[1][1];
    c:= ntPolygon[1][2];
    if iszero(slope) then
      userinfo(5,"slope = ",slope,"indicial equation =",c);
      if p <> 1/x then
        if traperror((aux:=ode::padicSingParts(c,x,u,p,solveOptions,odeOptions)))=0 then
          res:= res.aux;
        end_if;
      else
        res:= [[1,0,0]];
      end_if;
      m1:= m1-degree(c,[u]);
    else
      if type(slope) = DOM_INT then
        userinfo(5,"slope = ",slope,"characteristic equation =",c);
        if traperror((
             aux:=ode::padicExpParts(lode,c,x,u,p,slope,solveOptions,odeOptions)
           )) = 0 then
          res:= res.aux;
        end_if;
      end_if;
      m1:= m1-degree(c,[u])*denom(slope);
    end_if;
    delete ntPolygon[1];
  end_while;
  return(res):
end_proc:


/* ode::padicSingParts(c, x, u, p) 
    INPUT:  c - a bivariate polynomial in x and u    
            x - a name                   
            u - a name                                                        
            p - an irreducible polynomial in x (or 1/x, representing infinity)
    OUTPUT: the list of p-adic singular parts associated to the generalized   
            exponents ui of the indicial equation c(ui,x) = 0 mod p(x)        
            only the representant mod Z are returned                          
*/                                                                             
ode::padicSingParts:=
proc(c, x, u, p,solveOptions,odeOptions)
  local res, numers, num, i, j, QPhi, P, R, num1, rest, tr,h, Phi, indp, Rp, Fp;
begin 
  res:= [];
  // delete Phi;
  Phi:= genident();
  numers:= ode::modSolve(c,x,u,p,solveOptions,odeOptions);
  for i from 1 to nops(numers) div 2 do
    delete numers[i+1];
  end_for;
  numers:= ode::reps(numers,solveOptions,odeOptions);
  for i from 1 to nops(numers) do
    num:= numers[i];
    if (type(num) = DOM_INT) then
      if (num < 0) then
        res:= append(res,[p^(-num),0,0])
      else
        res:= append(res,[1,0,0])
      end_if
    else
      if degree(p,x) > 1 then
        /* 
           We now interprete 'num' as algebraic number: replace in 'num' 
           the variable 'x' by 'Phi' where 'p(Phi) = 0'. We then have to 
           compute the algebraic trace i.e. the sum of all conjugates 
           of 'num/(x-Phi)'. 
        */
        indp:= indets(p) minus {x} minus Type::ConstantIdents;
        Rp:= Dom::DistributedPolynomial([op(indp)],Dom::Rational);
        Fp:= Dom::Fraction(Rp);
        QPhi:= Dom::AlgebraicExtension(Fp,subs(p,x=Phi),Phi);
        P:= Dom::DistributedPolynomial([x], QPhi);
        R:= Dom::Fraction(P);
        // We divide 'p' by 'x-Phi' in 'QPhi'.  
        rest:= R::numer(R::normalize(P(p),P(x-Phi)));
        // We then make the denominator rational by multiplying with rest 
        num1:= P::_mult(P(subs(num, x=Phi)), rest);
        // Now we compute the algebraic trace of all coefficients of 'num1' 
        tr:= 0;
        for j from 0 to degree(p, [x])-1 do
          tr:= tr + x^j*ode::conjTrace(P::coeff(num1, x, j), p, x,solveOptions,odeOptions)
        end_for;
        if iszero(traperror((h:=tr/p))) then
          res:= append(res,[1,0,h]);
        else
          sysassign(ode::possiblyLostSolutionsFlag, TRUE);
          res:=append(res,[1,0,0]);
        end_if;
        // This is the corresponding command 
        // res := res,[1, 0, sum(subs(x = a, num[1])/(x-a), a = RootOf(p))];
      else
        // this if statement stands for res := append(res, [1, 0, num/p]) 
        if iszero(traperror((h:=num/p))) and h <> FAIL then
          //res:= append(res, [1, 0, h])
          res:= append(res,[1,0,h*coeff(p,x,1)]);
        else
          sysassign(ode::possiblyLostSolutionsFlag, TRUE);
          res:=append(res,[1,0,0])
        end_if;
      end_if;
    end_if;
  end_for;
  if has(res,FAIL) then 
    res:= [];
  end_if;
  if res <> [] then
    userinfo(5, "p-adic singular parts found: ", res)
  end_if;
  return(res);
end_proc:


/* ode::reps(lambda)                                                           
    INPUT:  lambda - a list of pairs [u, m] where u is a p-adic exponent     
                     and m its multiplicity                                  
    OUTPUT: a sublist l of lambda such if [u, m] in l, then there is         
            no positive integer i with [u+i, m] in l.                        
*/                                                                            

ode::reps:=
proc(lambda,solveOptions,odeOptions)
  local reps,parts,tmp,p,p1,d;
begin
  reps:= {};
  tmp:= {op(lambda)};
  parts:= tmp;
  while parts <> {} do
    p:= op(parts, 1);
    for p1 in parts do
      d:= expr(p-p1);
      if type(expr(d)) = DOM_INT then
        tmp:= tmp minus {p1};
        if d > 0 then
          p:= p1
        end_if
      end_if;
    end_for;
    reps:= reps union {p};
    parts:= tmp
  end_while;
  return([op(reps)]);
end_proc:


/* This is an implementation of the algebraic trace. However, it would be     
   better to include it into the domain "AlgebraicExtension".             */ 

/*                                                                            
  ode::conjTrace(alpha, P, x)                                                 
    INPUT:  alpha - an element of an algebraic extension over the rationals  
            P     - the minimal polynomial of the extension                  
            x     - the variable of P                                        
    OUTPUT: its algebraic trace, i.e. the sum of the conjugates              
*/                                                                            

ode::conjTrace:=
proc(alpha, P, x,solveOptions,odeOptions)
    local tmp, a, Phi, i, tr; 
begin 
  tmp:= extop(alpha, 1); 
  a:= op(tmp, 1);
  Phi:= op(tmp, 2)[1];
  a:= subs(a, Phi = x);
  tr:= coeff(a, x, 0);
  for i from 1 to degree(P)-1 do
    a:= divide(a*x, P, [x], Rem);
    tr:= tr + coeff(a, x, i)
  end_for;
  return(tr);
end_proc:
    

/* ode::padicExpParts(c, x, u, p, k)                                           
   INPUT:  c - a bivariate polynomial in x and u                             
           x - a name                                                        
           u - a name                                                        
           p - an irreducible polynomial in x or 1/x (representing infinity)
           k - an integer                                                    
   OUTPUT: the list of p-adic exponential parts associated to the slope k and
           the p-adic characteristic equation c(u, x)                        
*/ 

ode::padicExpParts:=
proc(lode, c, x, u, p, k,solveOptions,odeOptions)
  local i, c1, numers, part, rparts, res;
begin
  res:= [];
  c1:= subs(c,u=-u*k);
  numers:= ode::modSolve(c1, x, u, p,solveOptions,odeOptions);
  for i from 1 to nops(numers) div 2 do
    if traperror((part:= (numers[2*i-1])/p^k ) ) = 0 and part <> FAIL then
      rparts:= ode::padicNewton(
                              ode::changeRiccati(lode, x, diff(part, x),
                                                 solveOptions,odeOptions), x,
                              p, numers[2*i], solveOptions,odeOptions);
      res:= res.ode::insert(part, rparts,solveOptions,odeOptions);
    else
      sysassign(ode::possiblyLostSolutionsFlag, TRUE);
      res:= res.[[1,0,0]]
    end_if;
  end_for;
  return(res);
end_proc:


/* ode::insert(part, l)                                                        
   INPUT:  part - an expression                                              
           l    - a list of expressions                                      
   OUTPUT: if l is the empty list then [1, part, 0]                          
           else replace each element e of the form [*, e', *] in l           
           by [*, part+e', *].                                               
*/         

ode::insert:=
proc(part, l,solveOptions,odeOptions)
  local f;
begin
  if l = [] then
    [[1,part,0]]
  else
    f:= proc()
        begin
          [args(1)[1], args(1)[2]+args(2), args(1)[3]]
        end_proc ;
    return(map(l,f,part));
  end_if
end_proc:


/* ode::possiblePadicParts
    INPUT:  parts - a list of p-adic exponential parts
    OUTPUT: all possible sums of elements of the different exponential 
            parts
*/
ode::possiblePadicParts:=
proc(part,solveOptions,odeOptions) 
  local p1, p2, pp, parts, res;
begin
  if part = [] then
    return([[0, 0]])
  end_if;
  pp:= part[1];
  delete part[1];
  parts:= ode::possiblePadicParts(part,solveOptions,odeOptions);
  res:= [];
  for p1 in pp do
    for p2 in parts do
      res:= append(res, [p1[1]+p2[1], p1[2]+p2[2]]);
    end_for
  end_for;
  return(res);
end_proc:


// Tools for computing Newton polygons and valuations 
/*
 ode::newtonPolygon(lode, x, p, u, m)                                      
    INPUT:  lode - list of coefficients of linear differential operator       
            x    - a name                                                     
            p    - a polynomial or 1/x                                        
            u    - a name                                                     
            m    - an integer                                      
    OUTPUT: the Newton polygon and of lode at place p as list                 
            [minimal y-value, u, list of couples [slope, newton polynomial]]  
            the polygon is constructed only for lode[i], i = 0,..,m           
*/                                                                             

ode::newtonPolygon:=
proc(lode, x, p, u, m, solveOptions, odeOptions)
  local lode1, n, s, yp, lcs, drop, npoly, d, i, j, k, slope,
  tmp, res, dp;
begin 
  lode1:= lode;
  if type(m) = DOM_INT then 
    n:= m+1 
  else 
    n:= nops(lode) 
  end_if;
  if p = 1/x then 
    s:= -1; 
    dp:= 1 
  else 
    s:= 1; 
    dp:= diff(p, x) 
  end_if;
  yp:= []; 
  lcs:= [];
  for i from 1 to n do
    tmp:= ode::valuation(lode1[i], x, p,solveOptions,odeOptions);
    yp:= append(yp, tmp[1]-s*(i-1));
    lcs:= append(lcs, tmp[2]);
  end_for;
  drop:= min(op(yp));
  npoly:= 0;
  for j from 1 to n do
    if yp[j] = drop then
      npoly:= npoly + ode::tau(lcs[j],x,p,j-1,solveOptions,odeOptions)*_mult(s*u-k+1$ k=1..j-1);
    end_if
  end_for;
  npoly:= ode::normal(npoly);
  d:= degree(npoly, [u]); 
  if d > 0 then
    res:= [[0,collect(ode::normal(polylib::primpart(npoly,[u]),Rationalize=None), [u])]]
  else
    res:= []
  end_if;
  i:= d+1;
  while i < n do 
    /* slope := min(seq((yp[j]-yp[i])/(j-i), j=[$(i+1..n)])); */
    slope:= min((yp[j]-yp[i])/(j-i) $ j in [$(i+1..n)]);
    npoly:= 0;
    for j from i to n do
      if yp[j] -(j-i)*slope = yp[i] then
        npoly:= npoly + ode::tau(lcs[j], x, p, j-i,solveOptions,odeOptions)*
                        (s*u)^((j-i)/denom(slope))
      end_if
    end_for; 
    res:= append(res, [slope,collect(ode::normal(polylib::primpart(npoly, [u]), Rationalize=None), [u])]);
    i:= i+degree(npoly, [u])*denom(slope);
  end_while;
  return([drop, u, res]);
end_proc:


/* ode::val(f, x, p)                                                           
    INPUT:  f - a rational function
            x -  a name
            p - a polynomial or 1/x         
    OUTPUT: the list [v, lc] where v is the p-valuation of f and lc the       
            leading coefficient of f at p.                                    
*/
ode::val:=
proc(f, x, p,solveOptions,odeOptions)
  local t1, t2, G;
begin
  t1:= ode::valuation(numer(f), x, p,solveOptions,odeOptions);
  t2:= ode::valuation(denom(f), x, p,solveOptions,odeOptions);
  if (p = x) or (p = 1/x) then
    return([t1[1]-t2[1], t1[2]/t2[2]])
  else
    G:= gcdex(t2[2], p, x);
    return(t1[1]-t2[1], divide(t1[2]*G[2], p, [x], Rem))
  end_if;
end_proc:

/* ode::valuation(a, x, p)                                                     
    INPUT : a - a polynomial
            x - a name
            p - a polynomial or 1/x                
    OUTPUT: the list [v, lc] where v is the p-valuation of f and lc the       
            leading coefficient of a at p.                                    
*/
ode::valuation:=
proc(a, x, p,solveOptions,odeOptions)
  local ord, tmp, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
              IgnoreAnalyticConstraints;
            else
              null();
            end_if;
  if a = 0 then
    return([infinity, 0])
  elif p = 1/x then
    return([-degree(a, x), lcoeff(a, [x])])
  elif p = x then
    return([ldegree(a, x), tcoeff(a, [x])])
  end_if;
  ord:= 0; 
  tmp:= a;
  repeat
    tmp:= divide(a,p,[x]);    
    if ode::odeIszero(tmp[2]) and  
       iszero(expand(tmp[2],optIgnoreAnalyticConstraints)) then // added an expand
      ord:= ord+1;
      a:= tmp[1]
    else
      break
    end_if;
  until 
    FALSE 
  end_repeat;
  return([ord,divide(a,p,[x],Rem)]);
end_proc:


/* ode::tau(lc, x, p, i)                                                       
    INPUT:  lc - a univariate polynomial in x of degree < degree(p)           
            x  - a name                                                       
            p  - an irreducible polynomial in x                               
            i  - an integer                                                   
    OUTPUT: the ith remainder of a polynomial whose leading coefficient       
            is lc at p                                                        
*/
ode::tau:= 
proc(lc, x, p, i,solveOptions,odeOptions)
begin
  if (p = x) or (p = 1/x) then
    return(lc)
  else
    return(divide(lc*diff(p, x)^i, p, [x], Rem))
  end_if
end_proc:


// Additive change of riccati solution 

/* ode::changeRiccati(L, x, u)                                                 
   INPUT:  L - list of coefficients of linear differential operator          
               with derivation d/dx                                          
           x - a name                                                        
           u - an expression                                                 
   OUTPUT: list of coefficients of linear differential equation which        
           results from the change of unknown variable y(x) to               
           exp(int(u))*z(x).                                                 
*/

ode::changeRiccati:=
proc(L, x, u,solveOptions,odeOptions)
  local n, i, j, T, cs, d, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
              IgnoreAnalyticConstraints;
            else
              null();
            end_if;
  n:= nops(L);
  T:= array(1..n+1);
  T[1]:= 1;
  for i from 2 to n+1 do
    T[i]:= (ode::normal(diff(T[i-1], x)+u*T[i-1],Rationalize=None));
  end_for;
  cs:= array(1..n);
  d:= 1;
  for j from 1 to n do
    cs[j]:= 0;
    for i from 1 to n-j+1 do
      cs[j]:= cs[j] + (binomial(j+i-2, j-1)*T[i]*L[j+i-1]);
    end_for;
    cs[j]:= ode::normal(cs[j],Rationalize=None);
    // next line inserted by S.W: due to normal form change in simplify
    cs[j]:= combine(cs[j],optIgnoreAnalyticConstraints); 
    d:= lcm(d,denom(cs[j]));
  end_for;
  for j from 1 to n do 
    cs[j]:= ode::normal(cs[j]*d,Rationalize=None); 
  end_for;
  return([op(cs)]);
end_proc:
      

// Solving equations modulo an irreducible polynomial 

/* ode::modSolve(c, x, u, p)                                                   
  INPUT:  c - a multivariate polynomial 
          x - a name                                                         
          u - a name                                                         
          p - an irreducible polynomial (or 1/x, representing infinity)      
  OUTPUT: a list of all solutions u \in C[x], deg(u) < deg(p) of c(x, u) = 0 
          (mod p) and their multiplicities                                   
*/                                                                            
ode::modSolve:=
proc(c, x, u, p,solveOptions,odeOptions)
  local res, d, f, i, s, Qphi, P, C, Phi, indp, R, F, tmp;
begin
  if p = 1/x then
    d:= 1
  else
    d:= degree(p, [x])
  end_if;
  res:= [];
  // delete Phi;
  Phi:= genident();
  if d = 1 then
    f:= factor(poly(c, [u]));
    f:= coerce(f, DOM_LIST):
    for i from 1 to (nops(f)-1) div 2 do
      s:= f[2*i];
      if degree(s, u) > 1 then
        // put a traperror because if 'res' is not empty then it's better to return it
        // instead of an error
        if testtype(((tmp:= subs(s,u=Phi,EvalChanges))), 
                    Type::PolyOf(Type::Rational,1)) and 
           traperror((Qphi:= Dom::AlgebraicExtension(Dom::Rational,tmp)))=0 then
          //P:=Dom::DistributedPolynomial([u],Qphi);
          res:= append(res, Qphi(Phi), f[2*i+1])
        end_if;
      else
        s:= op(solvelib::discreteSolve(s, u, op(solveOptions)));
        res:= append(res, s, f[2*i+1]);
        userinfo(5, "solution", s);
      end_if;
    end_for;
  else
    indp:= indets(p) minus {x} minus Type::ConstantIdents;
    R:= Dom::DistributedPolynomial([op(indp)], Dom::Rational);
    F:= Dom::Fraction(R);
    Qphi:= Dom::AlgebraicExtension(F, subs(p, x=Phi), Phi);
    P:= Dom::DistributedPolynomial([u],Qphi);
    C:= coerce(subs(c, x=Phi, EvalChanges),P);
    if C = FAIL then 
      return([]);
    end_if;  
    f:= factor(C);
    f:= coerce(f, DOM_LIST):
    for i from 1 to (nops(f)-1) div 2 do
      s:= f[2*i];
      if P::degree(s, u) > 1 then
        sysassign(ode::possiblyLostSolutionsFlag, TRUE);
        res:= append(res,0,f[2*i+1])
      else
        s:= subs(expr(s), Phi=x);
        s:= op(solvelib::discreteSolve(s, u, op(solveOptions)));
        res:= append(res, s, f[2*i+1]);
        userinfo(5, "solution", s)
      end_if;
    end_for;
  end_if;
  return(map(res, (e)-> if not has(expr(e),Phi) then expr(e) else e end_if))
end_proc:

