
/*++ ---------------- LiouvSol.mu ---------------------
Description:
This file contains functions for computing liouvillian solutions
for linear ordinary differential equations over the field of 
rational functions.
For third order equations currently only the reducible case is implemented.
For higher order equations it is only possible to treat the reducible case
partially.

Functions: 

 - used Parameter:
    ++ Ly, y(x) :DOM_EXPR
    ++ Ly    = ordinary linear homogeneous differential equation 
    ++         over the rational functions
    ++ y(x)  = the operator (function) of Ly  
    ++ y     = name of the operator of Ly
    ++ x     = dependent variable of the operator y

 - ode::liouvillianSolutions(Ly, y(x), < Transform | Irreducible >)
    ++ returns a fundamental system of liouvillian solutions of an ordinary
    ++ homogeneous linear differential equation Ly over the rational functions.
    ++ Currently this is only possible up to third order equations.
    ++ When option Transform is given, the unimodular transform is
    ++ executed unconditionally.
    ++ When option Irreducible is given, Ly is assumed to be irreducible. 
    ++ NOTE: For third order equations only the reducible case is yet 
    ++       implemented!  
    ++ NOTE: currently this function only works properly for equations
    ++       for which the function ode::expsols works properly.
    ++ Otherwise: no warranty!

See:
 - Fakler, W. (1997). On Second Order Homogenous Linear Differential Equations
   with Liouvillian Solutions. Theoretical Computer Science 187, 27-48.
 - Fakler, W. (1997). Algorithms for Solving Linear Ordinary Differential
   Equations. mathPAD 7 No.1, 50-59.
 - Ulmer, F., Weil, J.A. (1996). Note on Kovacic's Algorithm. 
   J. Symb. Comp. 22, 179-200.

Examples:
>> Wy := diff(y(x), x, x)+ diff(y(x), x)/(x*2-1)*(-2) + y(x)*(x*(-86) + x^2*25 
         + x^3*14 + x^4*263 + x^5*(-324) + x^6*108 + 27)/(x^2*144 + x^4*(-576) 
         + x^5*288 + x^6*576 + x^7*(-576) + x^8*144);
>> Hy := diff(y(x), x,x)+27*x/(8*(x^3-2)^2)*y(x);
>> Ky := diff(y(x), x, x) + y(x)*(x*(-27) + x^2*32 + 27)/(x^2*144 + x^3*(-288) 
         + x^4*144);
>> setuserinfo(ode, 10):
>> ode::liouvillianSolutions(Wy, y(x));

++*/

/* Dependencies:

  "isLODE.mu", "2ndOrder.mu", "3rdOrder.mu", "tools.mu" */

ode::liouvillianSolutions:=
proc(eq, z, solveOptions,odeOptions) 
  //args(3..args(0)): option Transform, Irreducible 
  local intOptions,optIgnoreAnalyticConstraints,tmp,eqPoly;
begin
    optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  eq := rewrite(eq, diff);
  if ode::getOrder(eq,z,solveOptions, odeOptions)>1 then
    eq := ode::isLODE(eq, z, HlodeOverRF,solveOptions,odeOptions)
  else
    eq := ode::isLODE(eq, z, Hlode,solveOptions,odeOptions) end_if;
  if eq=FALSE then
    // not an ordinary homogeneous linear differential 
    // equation over the rational functions
    return({});
  end_if;
  case eq[4]
    of 0 do
      return( {op(linsolve({eq[1]}, {z}),[1,2])} );
    of 1 do //return( ode::firstOrder(eq[1],eq[2],eq[3]) );
      eqPoly:= ode::ode2poly(eq,solveOptions,odeOptions);
      tmp:= expr(coeff(eqPoly,diff(eq[2](eq[3]),eq[3]),1));
      if iszero(tmp) then 
        return({});
      else   
        tmp:= exp(int(expr(-coeff(eqPoly,eq[2](eq[3]),1))/tmp,eq[3],intOptions));
        return({simplify(expand(tmp,optIgnoreAnalyticConstraints),
                                    optIgnoreAnalyticConstraints)});
      end_if;                              
    of 2 do
      return(ode::secondOrder(eq[1],eq[2],eq[3],
                              args(5..args(0)),solveOptions,odeOptions));
    of 3 do
      return(ode::thirdOrder(eq[1],eq[2],eq[3],
                             args(5..args(0)),solveOptions,odeOptions));
    otherwise
//      if ode::printWarningsFlag then
//        warning("Only the reducible case will be treated partially");
//      end_if;
      return(ode::reducibleCase(eq,solveOptions,odeOptions))
  end_case
end_proc:



