/*-------------------------------- PolySol.mu -------------------------------

  Description:
   This file contains functions for computing polynomial solutions of linear 
   ordinary differential equations over the field of rational functions. 
   The algorithm used here computes polynomial solutions monomial by monomial.


  Functions:

   - used Parameter:
    ++ Ly, y(x):DOM_EXPR
    ++ A, b    :DOM_EXPR
    ++ y, x    :DOM_IDENT
    ++ n       :DOM_INT
    ++ Ly   = ordinary linear differential equation
    ++ y(x) = the operator (function) of Ly  
    ++ y    = name of the operator of Ly
    ++ x    = dependant variable of the operator y
    ++ n    = order of Ly
    ++ b    = rational function in x, may contain some parameters
    ++ A    = Ly in vectorized form (list of coefficients)

 - ode::polynomialSolutions(Ly, y(x), < Generic >)
    ++ returns a fundamental system of polynomial solutions of an ordinary
    ++ linear differential equation Ly over the rational functions.
    ++ When the option Generic is given, a generic form of them is returned.
    ++ NOTE: !!!this function is in the interface!!!

 - ode::polysols(Ly, y, x, n,< b >)
   ode::polysols(A, x,< b >) 
    ++ finds the polynomial solutions of the linear differential  equation 
    ++ Ly = b. The output is the general polynomial solution and a set of 
    ++ conditions on the parameters that must hold. eq must be normalized, 
    ++ i.e. denom(eq)=1. 
    ++ NOTE: Ly needs to be homogeneous!
    ++ library call: Ly can also be a list of coeffs (from ode::vectorize)
    ++ Then the call is ode::polysols(A, x).
  
References:
 - Abramov, S.A., Kvansenko, K.Yu. (1991). Fast Algorithms to Search for the
   Rational Solutions of Linear Differential Equations with Polynomial 
   Coefficients. In: Proceedings of ISSAC '91, Bonn, Germany, 267-270.
 - Barkatou, M. (1997). An Efficient Algorithm for Computing Rational 
   Solutions of Systems of Linear Differential Equations. 
   Submitted to ISSAC '97

Examples:

>> ode::polynomialSolutions(-8*y(x)+8*x*diff(y(x),x)+156*x^2*diff(y(x),x$2)
	+258*x^3*diff(y(x),x$3)+122*x^4*diff(y(x),x$4)
	+20*x^5*diff(y(x),x$5)+x^6*diff(y(x),x$6),y(x));

                                    {x}

>> ode::polysols([-8,8*x,156*x^2,258*x^3,122*x^4,20*x^5,x^6],x);

                                    {x}

>> ode::polysols([12*x^3+6*x^2-2*x-3,8*x^4+2*x^3-5*x^2-x,x^5-x^3],x);

                                    {}

>> wy := ode::normalize(ode::wronskian([x^2,x+1,x^3-x+1,y(x)],x),y,x,3):
>> ode::polynomialSolutions(wy,y(x));
 
                              3              2
                            {x  + 2, x + 1, x }
 
>> ode::polysols(wy,y,x,3);
 
                              3              2
                            {x  + 2, x + 1, x }
 
>> Ly := y(x)*4 + x*y(x)*3-x*diff(y(x), x) + x*diff(y(x), x, x)*2:
>> ode::polysols(Ly,y,x,2, a+b*x^10);

{                                                              2
{ {       52911104 b }  18129664 b x   13227776 b   1225472 b x
{ { a = - ---------- }, ------------ - ---------- - ------------ -
{ {          729     }      729           729           243

             3            4            5                    7        8
   435520 b x    69208 b x    10888 b x          6   412 b x    5 b x
   ----------- + ---------- + ---------- - 36 b x  - -------- + ------ +
       81            81           27                    27        9

      9 }
   b x  }
   ---- }
    3   }

>> Ln:=n*x*diff(y(x),x,x)-x*diff(y(x),x)+n^2*y(x):
   time((s:=ode::polynomialSolutions(subs(Ln,n=10),y(x))));
 
                                    910
++*/

ode::polynomialSolutions:=
proc(eq, z, o=" ", solveOptions={}, odeOptions={}) //o: Option Generic
  local sol;
begin
  if args(0) < 2 or args(0) > 5 then 
    error("expecting two or three arguments")
  elif args(0) = 3 and o <> Generic then 
    error("expecting option 'Generic' as third argument");
  elif args(0) = 4 or args(0) = 5 then
    if map({args(args(0)-1),args(args(0))},type) <> {DOM_SET} then 
      error("invalid arguments")    
    end_if;
  end_if;
  if type(z) <> "function" then
    error("expecting a function as the dependent variable")
  elif nops(z) <> 1 then 
    error("dependent variable must be a univariate function") 
  end_if; 
  if domtype(eq) <> DOM_EXPR then 
    error("expecting an ordinary differential equation as first argument")  
  end_if;
  if not has(eq,z) then 
    error("not an ordinary differential equation in ".expr2text(z))
  end_if;
  // Now check for the properties related to linear ODEs. 
  eq:= ode::isLODE(rewrite(eq,diff),z,HlodeOverRF,solveOptions,odeOptions);
  // 'eq' is now a sequence with 4 elements: 
  //      -- the expression defining the ODE 
  //      -- the dependent variable    ('y')
  //      -- the independent variable  ('x') 
  //      -- the order of the ODE      ('n')
  case nops(eq)
  of 1 do 
    return(error("not an ordinary homogeneous linear differential ".
                 "equation over the rational functions"));
    of 4 do 
      if eq[4] < 1 then
        return(error("only defined for positive order equations"))
      end_if;
  end_case;
  // We use the utility method 'ode::removeDenominator' instead of computing 
  // 'numer(normal(expr(eq[1]),Rationalize=None))'. This gives a significant 
  // speed-up. 
  eq[1]:= ode::removeDenominator(eq, solveOptions, odeOptions);
  // The argument '0' in the call of 'ode::polysols' means we solve 
  // a homogeneous linear ODE. 
  sol:= ode::polysols(eq, 0, solveOptions, odeOptions);
  if o=Generic then
    _plus(op(map(sol, e->_mult(e,genident("C")))))
  else
    sol
  end_if
end_proc:
 
/* 
   BACKGROUND: An implementation of the algorithm which computes 
               polynomial solutions of linear differential equations 
               monomial by monomial. 
 
    REFERENCE: M. Barkatou: An Efficient Algorithm for Computing 
                            Rational Solutions of Systems of Linear 
                            Differential Equations, 
                            submitted to ISSAC '97           

        CALLS: -- polysols(eq, y, x, n, b) 
            
                   finds the polynomial solutions of the linear differential  
                   equation 'eq(y(x)) = b', where 'b' is rational function. 
                   'b' may contain parameters. 'n' denotes the order of 'eq'. 
                   'eq' must be normalized, i.e. 'denom(eq) = 1'

               -- polysols([f0,f1,...,fn],x)

                   finds the polynomial solutions of the ODE 
                   f0*y(x) + f1*y'(x) + ... + fn*y^(n)(x)


       OUTPUT: The output is the general polynomial solution and a set of 
               conditions on the parameters that must hold.      
*/

ode::polysols:=
proc(eq, y, x, n, b, solveOptions={}, odeOptions={})
  local a, params, b1, mon, h, axn, i, P, cP, L, ir, sol, tmp, bb,
        s, conds, N, cis, bbhascis, lambda, c;
begin 
  if type(eq) = DOM_LIST then
    a:= eq;
    x:= y;
    n:= nops(a)-1;
    b:= 0
  else
    a:= ode::vectorize(eq,y,x,n,solveOptions,odeOptions);
  end_if;
  // makes the coefficients polynomial in x 
  userinfo(15, "make coefficients polynomial");
  h:= lcm(op(map(map(a,denom,Expand = FALSE),poly,[x])));
  if degree(h) <> 0 then 
    h:= expr(h); 
    a:= map(map(a,_mult,h),ode::normal,Rationalize=None,Expand = FALSE);
    b:= ode::normal(b*h, Rationalize=None,Expand = FALSE)
  end_if;
  params:= indets(b) minus {x};
  // action of the homogeneous equation on the monomial x^lambda 
  userinfo(15, "precompute action L(x^lambda)");
  mon:=1;
  axn:= a[1]*mon*x^n;
  lambda:= genident();
  for i from 2 to n+1 do
    mon:= mon*(lambda+2-i);
    axn:= axn + a[i]*mon*x^(n+1-i)
  end_for;
  userinfo(5, "computing height of Newton polygon at infinity");
  // height of the Newton polygon of 'eq' at infinity 
  axn:= poly(axn,[x]);
  h:= degree(axn);
  axn:= divide(axn,poly(x^ldegree(axn),[x]),Exact);
  // make h = 0 
  axn:=expr(axn)/x^degree(axn);
  // new rhs member 
  b1:= ode::normal(b/x^(h-n),Rationalize=None,Expand = FALSE);
  // indicial equation of the homogeneous equation at infinity 
  userinfo(5, "computing indicial equation at infinity");
  P:= ode::lc_inf(axn, x, solveOptions, odeOptions);
  cP:= polylib::primpart(poly(P,[lambda]));
  // if cP contains a univariate factor then the previous code may fail
  // when cP contains parameters; so we have to factor cP:
  L:= Factored::convert_to(factor(cP),DOM_LIST);
  cP:= 1;
  for i from 1 to nops(L) div 2 do
    //  if L[2*i] contains parameters then solvelib::iroots() returns an error
    if indets(expr(L[2*i])) minus {lambda, PI, EULER, CATALAN} = {} then
      cP:= cP*expr(L[2*i])
    end_if;
  end_for;
  // list of the integer roots of the indicial equation 
  if traperror((ir:= solvelib::iroots(cP))) <> 0 then 
    ir:= [];
  end_if;
  ir:= sort([op(ir)]);
  while (ir <> []) and (ir[1] < 0) do
    N:= nops(ir);
    ir:= [op(ir, [2..N])];
  end_while;
  userinfo(5, "positive integer roots are: ", ir);
  // call the procedure 'nextMon' 
  c:= genident();
  tmp:= ode::nextMon(axn, b1, x, P, lambda, ir, 0, c, {}, solveOptions,odeOptions);
  cis:= c[i] $ i=0..max(0,op(ir));
  bb:= tmp[2];
  conds:= op(tmp[3]);  
  // normalize 'bb' here, because only the numerator is needed later on. stefanw
  bb:= numer(bb,Expand = FALSE);
  if iszero(bb) then 
    if iszero(nops(conds)) then
      return({coeff(poly(tmp[1],[cis]))} minus {0})
    else
      return({coeff(poly(subs(tmp[1],conds),[cis]))} minus {0})
    end_if;
  else 
    if has(bb, [cis]) then
      h:= {coeff(bb,[x]), conds};
      assert({cis} intersect solvelib::indets(h) <> {});
      s:= solvelib::discreteSolve(h, {cis} intersect solvelib::indets(h),
                                  op(solveOptions));
      s:= subs(s, map([op(indets(s) minus indets(h))], _equal, 1));
      bbhascis := TRUE
    else
      s:= solvelib::discreteSolve({coeff(bb,[x]), conds},
                                  op(solveOptions));    
      bbhascis := FALSE
    end_if;
    if s={} then 
      return({}) 
    end_if;
    s:= op(s);
    sol:= {coeff(poly(subs(tmp[1],s),[cis]))} minus {0};
    if sol={} then
      return({})
    elif bbhascis then
      return(sol)
    else
      return(sol union {conds, s})
    end_if
  end_if;
end_proc:

/* this procedure computes the leading monomial of a possible polynomial
   solution and iterates */
ode::nextMon:= proc(axn, b, x, P, lambda, ir, sol, c, conds, solveOptions,odeOptions)
  local lcb, delta, cc, N, r, g, Indets;
begin 
  N:= nops(ir);
  /* use fraction-free representation: b=b[1]/b[2] where b[1] and b[2]
     are both polynomials, same for axn */
  b:= map(ode::normal(b, List, Rationalize=None, Expand = FALSE),poly,[x]);
  axn:= map(ode::normal(axn, List, Rationalize=None, Expand = FALSE),poly,[x]);
  repeat
    // valuation and leading coefficient of rhs member 
    delta:= (if iszero(b[1]) then 
               -infinity 
             else 
               degree(b[1])-degree(b[2]) 
             end_if);
    lcb:= lcoeff(b[1])/lcoeff(b[2]);
    // stop, when all positive roots are taken into account and the valuation
    //  of b is negative 
    if (ir = [] or ir[N] < 0) and (delta < 0) then
      return([sol, expr(b[1])/expr(b[2]), conds])
    end_if;
    // Case 1: there is a (maximal) positive integer root not treated which 
    //         is > delta 
    if (N > 0) and (ir[N] > delta) then
      g:= multcoeffs(axn[1]*b[2]*poly(x^ir[N],[x]), c[ir[N]]);
      b:= subs([b[1]*axn[2]-g,b[2]*axn[2]], lambda=ir[N]);
      userinfo(12, "free monomial found, degree is", ir[N]);
      sol:= sol+c[ir[N]]*x^(ir[N]); 
      delete ir[N]; 
      N:= N-1;
    // Case 2: there is no positive integer root not treated which is > delta
    //         (this includes the case where all integer roots are treated) 
    elif subs(P, lambda = delta) <> 0 then
      // Case 2a: delta is no root of the indicial equation 
      cc:= lcb/subs(P,lambda=delta);
      g:= (if delta=-infinity then 
             0 
           else
             multcoeffs(axn[1]*b[2]*poly(x^delta,[x]),cc) 
           end_if);
      b:= subs([b[1]*axn[2]-g,b[2]*axn[2]],lambda=delta);
      sol:= sol+cc*x^delta;
      userinfo(12, "monomial found, degree is", delta);
    else
      // Case 2b: delta is a root of the indicial equation 
      userinfo(20, "enter solve, system of ".nops(lcb).
                   " equations, of total size ".length(lcb));
      Indets:= map(indets(lcb,RatExpr),
                   elem -> if freeIndets(elem)<>{} then 
                             elem 
                           end_if);             
      if solvelib::solve_islinear({lcb},Indets) = FALSE then 
        r:= FAIL;             
      else         
        r:= linsolve({lcb},Indets); 
      end_if;
      if r=FAIL then 
        return([sol, expr(b[1])/expr(b[2]), conds])
      else
        userinfo(20, "back from solve, ".nops(r).
                     " solution(s) of total size ".length(r));
	      b:= subs(b, r); 
        sol:= subs(sol, r); 
        conds:= conds union {op(r)};
      end_if;
    end_if
  until 
    FALSE 
  end_repeat
end_proc:

// leading coefficient of a rational function f at infinity 
ode::lc_inf:=
proc(f, x, solveOptions,odeOptions)
begin
  if iszero(f) then
    return(0)
  else
    f:= ode::normal(f,List,Expand=FALSE);
    return(lcoeff(f[1], [x])/lcoeff(f[2], [x]))
  end_if;
end_proc:












