/*++ ---------------- mkUnimodular.mu ---------------------
Description:
This file contains a function for transforming an ordinary linear
(homogeneous) differential equation into a unimodular equations, i.e. the
Wronskian lies in the coefficient field of the equation.

Functions: 

 - used Parameter:
    ++ Ly, y(x) :DOM_EXPR
    ++ W, Wn    :DOM_EXPR
    ++ y, x     :DOM_IDENT
    ++ n        :DOM_INT
    ++ a        :DOM_LIST
    ++ Ly    = ordinary linear homogeneous differential equation 
    ++         over the rational functions
    ++ y(x)  = the operator (function) of Ly  
    ++ y     = name of the operator of Ly
    ++ x     = dependent variable of the operator y
    ++ n     = order of Ly
    ++ W     = Wronskian of the original equation Ly
    ++ a     = list of the coefficients of the unique  
    ++         representation of transformed Ly 
    ++         (i.e. leading coefficient of ly is 1)
    ++ Wn    = W^(1/n) or 1, the factor of the transformation

 - ode::unimodular(Ly, y(x), < Transform >)
    ++ tests, if Ly has a unimodular Galois group (i.e. W is a rational 
    ++ function), transforms Ly if not (i.e. the second highest coefficient is
    ++ zero) and returns the 
    ++    table(equation='transformed Ly', factorOfTransformation=Wn)
    ++ i.e. a solution of 'transformed Ly' multiplied with Wn is a solution 
    ++ of Ly.
    ++ When the option Transform is given, Ly is transformed unconditionally
    ++ (even if Ly already has an unimodular Galois group).
 - ode::mkUnimodular(Ly, y, x, n, < Transform >)
    ++ tests, if Ly has a unimodular Galois group (i.e. W is a rational 
    ++ function), transforms Ly if not (i.e. the second highest coefficient is
    ++ zero) and returns the sequence 
    ++ Ly (possibly transformed), W, a (possibly transformed), Wn.
    ++ When the option Transform is given, Ly is transformed unconditionally
    ++ (even if Ly has yet an unimodular Galois group).
 
See:
 - Kaplansky, I. (1957). Introduction to differential algebra. Paris: Hermann.
   p. 41.
 
Examples:

>> Ly := y(x)*6 + x*diff(y(x), x)*(-2) + diff(y(x), x, x)*(-x^2 + 1):
>> ode::unimodular(Ly,y(x));

   table(
     factorOfTransformation = 1,
                                                                      2
     equation = 6 y(x) - 2 x diff(y(x), x) + diff(y(x), x, x) (1 - x )
   )


>> ode::mkUnimodular(Ly,y,x,2,Transform);                          

                              2               --        2              --
                   y(x) (- 6 x  + 7)    1     |    - 6 x  + 7           |
diff(y(x), x, x) + -----------------, ------, |  ---------------, 0, 1  |,
                         2    4        2      |       2    4            |
                    - 2 x  + x  + 1   x  - 1  -- - 2 x  + x  + 1       --

     1
-----------
  2     1/2
(x  - 1)

++*/

/* Dependencies:
  isLODE.mu, vectoriz.mu, wronsk.mu */

ode::unimodular:=
proc(eq,z,o=" ",solveOptions={},odeOptions={}) // option: Transform 
  local l;
begin
  eq:= ode::isLODE(rewrite(eq,diff),z,HlodeOverRF,solveOptions,odeOptions);
  case nops(eq)
    of 1 do return(error("not an ordinary homogeneous linear differential ".
                         "equation over the rational functions"));
    of 4 do if eq[4] < 1 then
              return(error("only defined for positive order equations"))
            end_if;
  end_case;
  l:= ode::mkUnimodular(eq, o, solveOptions, odeOptions);
  table(hold(equation)=l[1], hold(factorOfTransformation)=l[4])
end_proc:


ode::mkUnimodular:=
proc(eq, y, x, n, o=" ", solveOptions={},odeOptions={}) // option: Transform 
  local a, k, W, vars, L, deriv, dy, i, intOptions, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
    intOptions:= null();            
    if has(solveOptions, IgnoreSpecialCases) then 
      intOptions:= intOptions,IgnoreSpecialCases;
    end_if;
    if has(solveOptions, IgnoreAnalyticConstraints) then   
      intOptions:= intOptions,IgnoreAnalyticConstraints;
    end_if;   
    a:= ode::vectorize(eq, y, x, n, solveOptions, odeOptions); 
    assert(not iszero(a[n+1]));
    a:= map(a, _mult, 1/a[n+1]);
    W:= exp(-int(a[n],x,intOptions));
    vars:= [diff(y(x), x$k) $ k=0..n];
    if not(o=Transform) and nops(rationalize(W)[2])=0 then
      return(expr(eq),W,a,1)
    end_if;
    case n 
      of 2 do 
        L:= poly(diff(y(x), x$2)+(a[1]-a[2]^2/4-diff(a[2],x)/2)*y(x), vars); 
        break;
      of 3 do
        L:= poly(diff(y(x), x$3)+(a[2]-a[3]^2/3-diff(a[3],x))*diff(y(x), x)+
                 (a[1]-a[2]*a[3]/3-diff(a[3], x$2)/3+2*a[3]^3/27)*y(x), vars);
        break;        
      otherwise
        deriv:= proc() begin 
                  diff(args(1), x)-a[n]*args(1)/n
                end_proc;
        L:= a[1]*y(x); 
        dy:= y(x);
        for i from 2 to n+1 do
          dy:= deriv(dy);
          L:= L+a[i]*dy
        end_for;
        L:= poly(L, vars);
    end_case;
    L:= expr(mapcoeffs(L, ode::normal));
    return(L,W,ode::vectorize(L,y,x,n,solveOptions,odeOptions),W^(1/n));
end_proc:


