/*++ ---------------- fuchsian.mu ---------------------
Description:
This file contains function for testing and dealing with homogeneous linear 
differential equations of Fuchsian type.
A linear differential equation which has at worst only finitely many regular 
singular points (including infinity) is said to be of Fuchsian type.
 
Functions:

 - used Parameter:
    ++ Ly, y(x):DOM_EXPR
    ++ p       :DOM_EXPR
    ++ y, x, nu:DOM_IDENT
    ++ n       :DOM_INT
    ++ v       :DOM_LIST
    ++ Ly   = ordinary linear differential equation
    ++ y(x) = the operator (function) of Ly  
    ++ y    = name of the operator of Ly
    ++ x    = dependant variable of the operator y
    ++ n    = order of Ly
    ++ p    = place
    ++ nu   = indeterminate of indicial equation
    ++ v    = Ly in vectorized form

 - ode::exponents(Ly, y(x), p)
    ++ returns the exponents at place p of the given equation Ly. Hereby
    ++ p is an irreducible polynomial expression in x or for infinity 1/x.
 - ode::indicialEquation(Ly, y(x), p, nu)
    ++ returns the indicial equation in nu of equation Ly at place p. Hereby
    ++ p is an irreducible polynomial expression in x or for infinity 1/x. 
 - ode::indicialEq(v, x, p, nu)
    ++ see ode::indicialEquation.
 - ode::isFuchsian(Ly, y(x), <AllExponents>)
    ++ tests, if the given equation Ly is fuchsian. When option 
    ++ AllExponents is given, a list of table[place,indicialEq,exponents]
    ++ or FALSE is returned.

Examples:
>> Ly:=x*(1-x)*diff(y(x),x$2)+(1-x)*diff(y(x),x)+10*y(x):
>> ode::isFuchsian(Ly,y(x),AllExponents);              

--
|
|  table(                 table(
|                     2                               2
|    indicial = lambda ,,   indicial = lambda - lambda ,,
|    exponents = {0},       exponents = {0, 1},
|    place = x              place = x - 1
|  )                      )
--

   table(                          --
                           2        |
     indicial = 10 - lambda ,       |
                    1/2      1/2    |
     exponents = {10   , - 10   },  |
             1                      |
     place = -                      |
             x                      |
   )                               --

>> ode::indicialEquation(Ly,y(x),x-1,nu);

                                        2
                                 nu - nu

>> ode::exponents(Ly,y(x),1/x);          

                                1/2      1/2
                             {10   , - 10   }
++*/ 

ode::isFuchsian:=
proc(eq, z, o=" ", solveOptions={}, odeOptions={}) 
  local lode,l,m,i,p,t,indeq,all, lambda;
 // save lambda;
begin
  eq := ode::isLODE(rewrite(eq, diff), z, HlodeOverRF, solveOptions, odeOptions);
  if nops(eq)<> 4 then
    return(FALSE)
  end_if;
  lode := ode::vectorize(numer(normal(expr(eq[1]))),eq[2],eq[3],eq[4],
                         solveOptions, odeOptions);
  t:=eq[3];
  all:=[];
  l:=factor(lode[eq[4]+1]);
  l:=Factored::convert_to(l,DOM_LIST);
  m := nops(l) div 2;
  (p[i]:=l[2*i]) $ i=1..m ;
  p[m+1] := 1/t;
  // delete lambda;
  lambda := genident();
  for i from 1 to m+1 do
    indeq:=ode::indicialEq(lode,t,p[i],lambda, solveOptions, odeOptions);
    if indeq=FAIL then
      return(FALSE)
    end_if;
    if o=AllExponents then
      all := append(all,
                    table(hold(place)=p[i],
                          hold(indicialEq)=indeq,
                          hold(exponents)=solve(indeq,lambda,op(solveOptions))))
    end_if
  end_for;
  if o=AllExponents then
    all
  else
    TRUE
  end_if
end_proc:


ode::indicialEquation:=
proc(eq, z, p, u, solveOptions={}, odeOptions={}) 
begin
  eq := ode::isLODE(rewrite(eq, diff), z, HlodeOverRF, solveOptions, odeOptions);
  if nops(eq)<> 4 then
    return(error("not an ordinary homogeneous linear differential ".
                 "equation over the rational functions"))
  end_if;
  eq[1] := numer(normal(expr(eq[1])));
  ode::indicialEq(ode::vectorize(eq,solveOptions,odeOptions),eq[3],p,u, solveOptions, odeOptions);
end_proc:

ode::indicialEq:=
proc(eq, x, p, u, solveOptions, odeOptions)
  local indeq;
begin
  /* Note by Kai: Here we have no value for the integer parameter 'm'
                  which has to be given to 'ode::newtonPolygon'. Hence, 
                  we use the string "dummy" as a dummy value for 'm'. 
                  The function 'ode::newtonPolygon' sets an appropriate 
                  value for 'm', when the 5th argument passed over to
                  'ode::newtonPolygon' is not an integer.  
  */
  indeq := op(ode::newtonPolygon(eq,x,p,u,"dummy",solveOptions,odeOptions), 3);
  if nops(indeq)=1 and iszero(indeq[1][1])
    then indeq[1][2]
  else
    FAIL
  end_if
end_proc:

ode::exponents:=
proc(eq, z, p, solveOptions={}, odeOptions={})
  local indeq, lambda;
  // save lambda;
begin
    //delete lambda;
  lambda := genident();
  indeq:=ode::indicialEquation(eq,z,p,lambda, solveOptions, odeOptions);
  if indeq=FAIL then
    return({})
  else
    solvelib::discreteSolve(indeq,lambda, op(solveOptions))
  end_if
end_proc:
 
