/*++ ---------------- isLODE.mu ---------------------
Description:
isLODE(Ly, y(x), < Lode|LodeOverRF|Hlode|HlodeOverRF|Homogeneous >) 
tests, if Ly is an ordinary linear differential equation in y(x).

Functions:

 - used Parameter:
    ++ Ly, y(x):DOM_EXPR
    ++ y, x    :DOM_IDENT
    ++ n       :DOM_INT
    ++ Ly   = ordinary linear differential equation
    ++ y(x) = the operator (function) of Ly  
    ++ y    = name of the operator of Ly
    ++ x    = dependant variable of the operator y
    ++ n    = order of Ly
    ++ dy   = set of all differential expressions of y(x) in Ly

 - ode::isLODE(Ly, y(x), 
                <Lode|LodeOverRF|Hlode|HlodeOverRF|Homogeneous>)
    ++ tests, if Ly is an ordinary linear differential equation in y(x).
    ++ When the option 
    ++  o Homogeneous is given, Ly is further tested if it is homogeneous.
    ++  o HlodeOverRF is given, FALSE or the homogeneous Ly, y, x, n 
    ++    is returned, where Ly is an equation over the rational functions. 
    ++  o Hlode is given, FALSE or the homogeneous Ly, y, x, n is returned.
    ++  o LodeOverRF is given, FALSE or Ly, y, x, n is returned, 
    ++    where Ly is an equation over the rational functions. 
    ++  o Lode is given, FALSE or Ly, y, x, n is returned.

 - ode::isHomogeneous(Ly, dy, n)
    ++ tests, if the equation Ly is homogeneous in the set of variables dy.

Examples:
>> eq := diff(y(x),x,x)+1/x*diff(y(x),x)-1/(4*x^2)*y(x):
>> ode::isLODE(eq, y(x));
 
                                   TRUE
 
>> ode::isLODE(eq, y(x), Homogeneous);
 
                                   TRUE
 
>> ode::isLODE(eq-1, y(x), Homogeneous);
 
                                   FALSE

>> ode::isLODE(eq, y(x), Hlode);  
 
           diff(y(x), x)                        y(x)
           ------------- + diff(y(x), x, x) - --------, y, x, 2
                 x                               2
                                              4 x  - 1
++*/ 

ode::isLODE:=
  proc(eq, z, o=" ") 
    //o: Options Homogeneous, Lode, LodeOverRF, Hlode, HlodeOverRF 
    local v0, vars, x, n, k, solveOptions, odeOptions;
  begin
    if args(0) = 2 or args(0) = 3 then 
      solveOptions:= {}; 
      odeOptions:= {};
    else  
      solveOptions:= args(args(0)-1); 
      odeOptions:= args(args(0));
    end_if;   
    x := op(z); 
    vars := split((v0 := indets(eq, RatExpr) minus indets(eq)), has, z)[1];
    n := max(-infinity, op(map(vars, nops)))-1;
    if o=HlodeOverRF then vars := v0; o := Hlode 
    elif o=LodeOverRF then vars := v0; o := Lode end_if;
    case n
    of -infinity do 
      if vars <> {} then 
        return( FALSE ) 
      end_if; 
      break; 
    otherwise
      if (vars minus {diff(z, x$k) $ k=0..n} <> {}) or
         (degree(poly(eq, [op(vars)])) <> 1) then
        return( FALSE ) 
      end_if
    end_case;
    // now eq is a linear ordinary differential equation 
    case o
    of Homogeneous do return( ode::isHomogeneous(eq, vars, n, 
                                                 solveOptions, odeOptions) );
    of Hlode do 
      if not( ode::isHomogeneous(eq, vars, n, solveOptions, odeOptions) ) then 
        return( FALSE ) 
      end_if; 
    of Lode do 
      return(eq, op(z, 0), x, n); 
    otherwise 
      return( TRUE ) 
    end_case
  end_proc:

ode::isHomogeneous:=
  proc(eq, vars, n, solveOptions, odeOptions)
  begin
    if iszero(eq) then return( TRUE ) end_if;
    if n=-infinity or vars={} then return( FALSE ) end_if;
    has(nthmonomial(eq, [op(vars)], nterms(eq,[op(vars)])), vars)
  end_proc:

