/*

  =======================================================
  LOOK-UP METHODS FOR VARIOUS NON-LINEAR FIRST ORDER ODEs
  =======================================================

  REFERENCE. Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------

            eq -- expression encoding a first order ODE
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    odeOptions -- options for 'ode' 

 */
 
ode::lookUp1stOrderNonLinearVariousODEs:=
proc(eq, y, x, solveOptions={}, odeOptions={}) 
  local z, intOptions, optIgnoreAnalyticConstraints, y0, y1, lcoeff, t, ind, k, xpow,
        A, B, a, b, exp_pow, lambda, sin_arg, eqNormalized, C, f, ff, n, y1pow, res,
        Subst;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  z:= solvelib::getIdent(Any, indets([eq,y,x]));
  y0:= solvelib::getIdent(Any, indets([eq,y,x,z]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,z,y0]));
  eq:= evalAt(eq, [diff(y(x),x) = y1, y(x) = y0]);
  eqNormalized:= ode::normal(eq);
  // -------------------------
  // look-up for g459678
  // -------------------------
  a:= diff(eq,y1);
  if not has(a,x) and not has(a,y1) and testtype(eq, Type::PolyExpr(y0)) then 
    b:= -evalAt(diff(eq,y0,y0),y0=0)/2;
    if not has(b,x) and not has(b,y1) and 
       ode::odeIszero(eq-(a*y1-b*y0^2+b^2*y0^3)) and 
       iszero(expand(eq-(a*y1-b*y0^2+b^2*y0^3),optIgnoreAnalyticConstraints)) and 
       traperror((res:= {0,1/b,1/(b*(lambertW(-1/(genident("C")*exp(x/a+1)))+1))})) = 0 then 
      return(res);
    end_if;  
  end_if;   
  // -------------------------
  // look-up for eq. 6, p. 111
  // -------------------------
  if has(eq,y1) and has(eq,y0) then 
    lcoeff:= diff(diff(eq,y1),y0);        
    if not iszero(lcoeff) and not has(lcoeff,[x,y0,y1]) then
      if lcoeff <> 1 then 
        eq:= eq/lcoeff;
      end_if;
      if not has((t:= combine(expand(eq-y0*y1+y0,optIgnoreAnalyticConstraints),
                              optIgnoreAnalyticConstraints)),[y0,y1]) then 
        xpow:= {};
        misc::maprec(t,
                     {"_power"} = proc(elem)
                                    begin
                                      if op(elem,1) = x and not has(op(elem,2),x) then 
                                        xpow:= xpow union {op(elem,2)}
                                      end_if;
                                      elem;
                                      end_proc);  
        xpow:= [op(xpow)];
        k:= FAIL;
        if {op(xpow)} = {-1/2,1/2} then 
          k:= 1/2;  
        end_if;
        if nops(xpow) = 3 then 
          for ind in [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]] do 
            if iszero(expand(xpow[ind[1]]-2*xpow[ind[2]]+1,optIgnoreAnalyticConstraints)) and 
               iszero(expand(xpow[ind[2]]-xpow[ind[3]]-1,optIgnoreAnalyticConstraints)) then 
              k:= xpow[ind[2]];
              break;
            end_if;     
          end_for;
        end_if;
        if k <> FAIL then 
          if traperror((B:= ode::normal(diff(subs(t,x^k=z),z)/k))) = 0 then 
            if not has(B,[x,z]) and not iszero(B) then 
              A:= -diff(subs(t,x^(k-1)=z),z);
              if not has(A,[x,z]) and 
                 ode::odeIszero(t+A*x^(k-1)-k*B*x^k+k*B^2*x^(2*k-1)) and 
                 iszero(expand(t+A*x^(k-1)-k*B*x^k+k*B^2*x^(2*k-1),optIgnoreAnalyticConstraints)) then 
                return({x-B*x^k-A/(k*B)})       
              end_if;
            end_if;
          end_if;
        end_if;
        // -------------------------- 
        // look-up for eq. 73, p. 120
        // --------------------------
        exp_pow:= {}; 
        t:= combine(t,exp);
        misc::maprec(t,
                     {"exp"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                     exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                   end_if;
                                   elem;
                                 end_proc);  
        exp_pow:= [op(exp_pow)];
        lambda:= FAIL;
        if nops(exp_pow) = 2 then 
          if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
            lambda:= exp_pow[2];
          elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
            lambda:= exp_pow[1];
          end_if;
        end_if;
        if lambda <> FAIL then
          if traperror((
             A:= combine(sqrt(ode::normal(
                        diff(-subs(t,[exp(2*lambda*x)=z,
                                     exp(expand(2*lambda*x,optIgnoreAnalyticConstraints))=z]),z)/lambda)),
                        IgnoreAnalyticConstraints))) = 0 then 
            if not has(A,[x,z]) then
              for a in [-A,A] do
                if not iszero(a*lambda) then  
                  if traperror((Subst:= subs(t,[exp(lambda*x)=z,
                                  exp(expand(lambda*x,optIgnoreAnalyticConstraints))=z]))) = 0 then                                  
                    b:= (diff(Subst,z)-a)/(a*lambda);
                    if not has(b,[x,z]) and 
                       ode::odeIszero(t-(-a^2*lambda*exp(2*lambda*x)+a*(b*lambda+1)*exp(lambda*x)-b)) and 
                       iszero(expand(t-(-a^2*lambda*exp(2*lambda*x)+a*(b*lambda+1)*exp(lambda*x)-b),
                                     optIgnoreAnalyticConstraints)) then 
                      return({a*exp(lambda*x)-b})  
                    end_if;
                    // -------------------------- 
                    // look-up for eq. 74, p. 120
                    // --------------------------
                    b:= -(diff(Subst,z)+a*lambda*x);
                    if not has(b,[x,z]) and 
                       ode::odeIszero(t+a^2*lambda*exp(2*lambda*x)+a*lambda*x*exp(lambda*x)+b*exp(lambda*x)) and 
                       iszero(expand(t+a^2*lambda*exp(2*lambda*x)+a*lambda*x*exp(lambda*x)+b*exp(lambda*x),
                                     optIgnoreAnalyticConstraints)) then 
                      return({a*exp(lambda*x)+x+b/(a*lambda)})  
                    end_if;  
                  end_if;
                end_if;
              end_for;
            end_if;
          end_if;  
        end_if;
        // -------------------------- 
        // look-up for eq. 75, p. 120
        // --------------------------
        if has(t,sin) and has(t,cos) then 
          t:= combine(t,sincos);
        end_if;
        sin_arg:= {};
        misc::maprec(t,
                     {"sin"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                     sin_arg:= sin_arg union {diff(op(elem,1),x)}
                                   end_if;
                                   elem;
                                 end_proc);  
        sin_arg:= [op(sin_arg)];
        lambda:= FAIL;
        if nops(sin_arg) = 1 then 
          lambda:= sin_arg[1];
        elif nops(sin_arg) = 2 then 
          if iszero(expand(sin_arg[1]-2*sin_arg[2],optIgnoreAnalyticConstraints)) then 
            lambda:= sin_arg[2];
          elif iszero(expand(sin_arg[2]-2*sin_arg[1],optIgnoreAnalyticConstraints)) then 
            lambda:= sin_arg[1];
          end_if;
        end_if;
        if lambda <> FAIL then
          if traperror((A:= combine(sqrt(ode::normal(
                       diff(-subs(t,[sin(2*lambda*x)=z,
                                     sin(expand(2*lambda*x,optIgnoreAnalyticConstraints))=z]),z)/(2*lambda))),
                       IgnoreAnalyticConstraints))) = 0 then 
            if not has(A,[x,z]) then
              for a in [-A,A] do
                if ode::odeIszero(t+2*a^2*lambda*sin(2*lambda*x)+2*a*sin(lambda*x)) and 
                   iszero(expand(t+2*a^2*lambda*sin(2*lambda*x)+2*a*sin(lambda*x),optIgnoreAnalyticConstraints)) then  
                  return({-2*a*sin(lambda*x)});
                end_if;
              end_for;
            end_if;
          end_if;  
        end_if;         
      end_if;
    end_if;
  end_if;
  // --------------------------
  // look-up for eq. 17, p. 205
  // --------------------------
  a:= -diff(subs(eqNormalized,y1^2=z),z);
  if not iszero(a) and 
     not has(expand(a,optIgnoreAnalyticConstraints),x) and 
     traperror((f:= ode::normal(-evalAt(eqNormalized-y0,y1=0)))) = 0 and
     not has(f,[y0,y1]) and 
     traperror((ff:= evalAt(f,x=x-2*a*y1))) = 0 and 
     ode::odeIszero(eqNormalized-y0+a*y1^2+ff) and 
     iszero(expand(eqNormalized-y0+a*y1^2+ff)) then 
    C:= genident("C"); 
    return({evalAt(f,x=C) + 1/(4*a)*(x-C)^2});   
  end_if;  
  // --------------------------
  // look-up for eq. 51, p. 209
  // --------------------------
  ff:= -eqNormalized+y0-2*x*y1;  
  if traperror((f:= evalAt(ff,y1=1))) = 0 and 
     not has(f,[y0,y1]) and 
     ode::odeIszero(eqNormalized-y0+2*x*y1+evalAt(f,x=x*y1^2)) and 
     iszero(expand(eqNormalized-y0+2*x*y1+evalAt(f,x=x*y1^2),optIgnoreAnalyticConstraints)) then 
    C:= genident("C"); 
    return(solve((y(x)-evalAt(f,x=C))^2 = 4*C*x,y(x)));    
  end_if;  
  // --------------------------
  // look-up for eq. 52, p. 209
  // --------------------------
  a:= -1/2*diff(subs(eqNormalized,y1^3=z),z);
  if not iszero(a) and 
     not has(expand(a,optIgnoreAnalyticConstraints),x) and 
     traperror((f:= ode::normal(-evalAt(eqNormalized-y0,y1=0)))) = 0 and
     not has(f,[y0,y1]) and 
     traperror((ff:= evalAt(f,x=x-3*a*y1^2))) = 0 and 
     ode::odeIszero(eqNormalized-y0+2*a*y1^3+ff) and 
     iszero(expand(eqNormalized-y0+2*a*y1^3+ff)) then 
    C:= genident("C"); 
    return({evalAt(f,x=C) + 2*a*((x-C)/(3*a))^(3/2)});       
  end_if;
  // --------------------------
  // look-up for eq. 71, p. 210
  // --------------------------
  y1pow:= {}; 
  misc::maprec(combine(expand(eqNormalized)),
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = y1 and not has(op(elem,2),x) then 
                                  y1pow:= y1pow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc);  
  for n in y1pow do 
    if traperror((f:= ode::normal(-evalAt(eqNormalized-y0+n/(n-1)*x*y1,y1=1)))) = 0 and 
       not has(f,[y0,y1]) and 
       ode::odeIszero(eqNormalized-y0+evalAt(f,x=x*y1^n)+n/(n-1)*x*y1) and 
       iszero(ode::normal(eqNormalized-y0+evalAt(f,x=x*y1^n)+n/(n-1)*x*y1)) then   
      C:= genident("C"); 
      return({evalAt(f,x=C^n) + n*C/(n-1)*x^((n-1)/n)});       
    end_if;   
  end_for;  
  
  
  return(FAIL);
end_proc:

