/*
  ================================
  LOOK-UP METHODS FOR Riccati ODEs
  ================================

  REFERENCE. Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 

  -----------
  PARAMETERS. 
  -----------

            eq -- expression of the form g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    odeOptions -- options for 'ode' 

 */


ode::lookUp1stOrderRiccati:=
proc(eq, y, x, solveOptions={}, odeOptions={}) 
  local z, g, f0, f1, f2, p, res, intOptions, optIgnoreAnalyticConstraints, inits;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  if has(odeOptions,#inits) then 
    inits:= select(odeOptions,has,#inits)[1];
  else 
    inits:= {};
  end_if;  
  
  z:= solvelib::getIdent(Any, indets([eq,y,x]));

  // compute the coefficient function g(x) of y'(x) 
  g:= diff(evalAt(eq, diff(y(x),x)=z),z);
  // check whether eq is linear in y'(x), i.e. g must depend on z anymore
  if has(g,z) or has(g,y) then  
    return(FAIL);
  end_if;

  // now check whether eq - g*diff(y(x),x) is a quadratic polynomial in y(x)
  p:= -(eq - g*diff(y(x),x));
  if has(p,diff(y(x),x)) then 
    p:= expand(-(eq - g*diff(y(x),x))/*,optIgnoreAnalyticConstraints*/);
  end_if;
  if not iszero(expand(diff(evalAt(p,y(x)=z),z,z,z),optIgnoreAnalyticConstraints)) or 
     has(expand(diff(evalAt(p,y(x)=z),z,z,z),optIgnoreAnalyticConstraints),diff(y(x),x)) then 
    return(FAIL);
  end_if;

  // extract the coefficient functions f0, f1, and f2 from the quadratic 
  // polynomial part p of eq
  if not has(p,y(x)^2) then 
    p:= expand(p/*,optIgnoreAnalyticConstraints*/);
  end_if;
  f2:= diff(evalAt(p,y(x)^2=z),z);  
  p:= p - f2*y(x)^2;
  f1:= diff(evalAt(p,y(x)=z),z);
  if has(f1,y(x)) then 
    f1:= expand(f1/*,optIgnoreAnalyticConstraints*/);
    if has(f1,y(x)) then 
      return(FAIL);
    end_if;
  end_if;
  f0:= p - f1*y(x);
  if has(f0,y(x)) then 
    f0:= expand(f0/*,optIgnoreAnalyticConstraints*/);
    if has(f0,y(x)) then 
      return(FAIL);
    end_if;
  end_if;

  // Now try the look-up methods available for ODEs of Riccati 
  res:= FAIL;
  if res = FAIL then 
    res:= ode::PZ_Sec128_Riccati(g, f0, f1, f2, x, y, solveOptions, odeOptions);
    if res <> FAIL and inits <> {} and 
       ode::insertInits(res,x,inits,solveOptions,odeOptions) = FALSE then 
      res:= FAIL;              
    end_if;
  end_if;
  
  if res = FAIL then 
    res:= ode::PZ_Sec122_Riccati(g, f0, f1, f2, x, y, solveOptions, odeOptions);
    if res <> FAIL and inits <> {} and 
       ode::insertInits(res,x,inits,solveOptions,odeOptions) = FALSE then 
      res:= FAIL;              
    end_if;
  end_if;

  if res = FAIL then 
    res:= ode::PZ_Sec123_Riccati(g, f0, f1, f2, x, y, solveOptions, odeOptions);
    if res <> FAIL and inits <> {} and 
       ode::insertInits(res,x,inits,solveOptions,odeOptions) = FALSE then 
      res:= FAIL;              
    end_if;
  end_if;

  if res = FAIL then 
    res:= ode::PZ_Sec124_Riccati(g, f0, f1, f2, x, y, solveOptions, odeOptions);
    if res <> FAIL and inits <> {} and 
       ode::insertInits(res,x,inits,solveOptions,odeOptions) = FALSE then 
      res:= FAIL;              
    end_if;
  end_if;  

  if res = FAIL then 
    res:= ode::PZ_Sec125_Riccati(g, f0, f1, f2, x, y, solveOptions, odeOptions);
    if res <> FAIL and inits <> {} and 
       ode::insertInits(res,x,inits,solveOptions,odeOptions) = FALSE then 
      res:= FAIL;              
    end_if;
  end_if;  

  if res = FAIL then 
    res:= ode::PZ_Sec126_Riccati(g, f0, f1, f2, x, y, solveOptions, odeOptions);
    if res <> FAIL and inits <> {} and 
       ode::insertInits(res,x,inits,solveOptions,odeOptions) = FALSE then 
      res:= FAIL;              
    end_if;
  end_if;  
  
  return(res);
end_proc:



/*
----------
REFERENCE. Implementation of the method for (Section 1.2.1 p. 82)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 
  
           to generate the general solution of a Riccati ODE from 
           a particular one. 

-----------
PARAMETERS. 
-----------

        y0   -- particular solution of a Riccati ODE given 
                by g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
g,f2,f1,f0   -- coefficients of the Riccati equation, i.e. coefficients 
                of g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 

*/

ode::RiccatiSolutionGenerator:= proc(y0, g, f0, f1, f2, x, solveOptions={}, odeOptions={})
  local Phi, intOptions;
begin
  intOptions:= null();      
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if; 
  Phi:= exp(int((2*f2*y0+f1)/g,x,intOptions)); // print((2*f2*y0+f1)/g); print(Phi*f2/g);
  return({y0,y0 + Phi/(genident("C") - int(Phi*f2/g,x,intOptions))});
end_proc:

/*
----------
REFERENCE. Implementation of (Section 1.2.8-1. pp. 102)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
  
  g,f2,f1,f0 -- coefficients of the Riccati equation 
                g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
           x -- the independent variable 
           y -- the dependent variable
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 
*/

ode::PZ_Sec128_Riccati:= proc(g, f0, f1, f2, x, y, solveOptions={}, odeOptions={})
  local a, f, L, z, y0, h, xpow, n, g_orig, f0_orig, f1_orig, f2_orig, gg,
        ind, exp_pow, tanh_arg, coth_arg, tan_arg, cot_arg, sinh_arg, sin_arg,
        cos_arg, lambda, intOptions, optIgnoreAnalyticConstraints;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;
  z:= solvelib::getIdent(Any, indets([g, f0, f1, f2, x, y])); 
  g_orig := g;
  f0_orig:= f0;
  f1_orig:= f1;
  f2_orig:= f2; 
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> 1 then
    g := 1;
    f0:= f0_orig/g_orig; 
    f1:= f1_orig/g_orig; 
    f2:= f2_orig/g_orig;  
  else 
     g:= g_orig;
    f0:= f0_orig;
    f1:= f1_orig;
    f2:= f2_orig;
  end_if; 
  // -----------------
  // look-up for eq. 1  
  // -----------------
  if ode::normal(f2) = 1 then 
    f:= f1;
    L:= ode::solveWithoutProperties(-z^2-z*f = f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = piecewise then 
      L:= piecewise::disregardPoints(L);
    end_if;
    if type(L) = DOM_SET then
      L:= select(ode::normal(L), _not@has, x);
      if L <> {} then 
        y0:= L[1];
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;
    end_if;
  end_if;  
  // -----------------
  // look-up for eq. 2  
  // -----------------
  f:= f2;
  a:= f1;
  if not has(a,x) then 
    L:= ode::solveWithoutProperties(-a*z - z^2*f = f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = piecewise then 
      L:= piecewise::disregardPoints(L);
    end_if;
    if type(L) = DOM_SET then
      L:= select(ode::normal(L), _not@has, x);
      if L <> {} then 
        y0:= L[1];
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;
    end_if;
  end_if;
  // -----------------
  // look-up for eq. 3  
  // -----------------
  if ode::normal(f2) = 1 then 
    f:= f1/x;
    if ode::odeIszero(f0-f) and 
       iszero(ode::normal(expand(f0-f,optIgnoreAnalyticConstraints))) then 
      y0:= -1/x;
      return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
    end_if;
  end_if;  
  // -----------------
  // look-up for eq. 4  
  // -----------------
  // Note. Pattern has been generalized using generic 'h' instead 'a*x^n'. 
  //        Then the pattern to be matched is 
  //               y'(x) = f(x)*y(x)^2 - h(x)*f(x) + h'(x)
  f:= f2; 
  if not iszero(ode::normal(f)) then 
    h:= -f1/f;
    if ode::odeIszero(f0-diff(h,x)) and 
       iszero(ode::normal(expand(f0-diff(h,x),optIgnoreAnalyticConstraints))) then 
      y0:= h;
      return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
    end_if;  
  end_if;
  // -----------------
  // look-up for eq. 5  
  // -----------------
  f:= f2; 
  if ode::odeIszero(f1) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
    xpow:= {}; 
    misc::maprec(combine(expand(f0/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/),
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                  end_proc);  
    xpow:= [op(xpow)]; 
    if xpow = [-1/2] then 
      xpow:= [-1/2,1];
    elif xpow = [2] then 
      xpow:= [0,2];
    elif xpow = [4] then   
      xpow:= [4,1];
    end_if;      
    if nops(xpow) = 2 then 
      n:= FAIL; 
      if iszero(expand(2*(xpow[1]+1)-xpow[2],optIgnoreAnalyticConstraints)) then 
        n:= xpow[1]+1;
      elif iszero(expand(2*(xpow[2]+1)-xpow[1],optIgnoreAnalyticConstraints)) then   
        n:= xpow[2]+1; 
      end_if; 
      if n <> FAIL then  
        if n = 1 and traperror((evalAt(f0,x^2=0))) = 0 then 
          a:= evalAt(f0,x^2=0);
        elif n = 2 and traperror((evalAt(f0,x^4=0))) = 0 then 
          a:= eval(diff(subs(combine(expand(f0/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints),x^4=0),x)/n);
        else 
          a:= eval(diff(subs(combine(expand(f0/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints),x^(n-1)=z),z)/n);
        end_if;
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(f0-a*n*x^(n-1)+a^2*x^(2*n)*f) and 
           iszero(ode::normal(expand(f0-a*n*x^(n-1)+a^2*x^(2*n)*f,optIgnoreAnalyticConstraints))) then 
          y0:= a*x^n;
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
        end_if;
      end_if;
    end_if;
  end_if; 
  // -----------------
  // look-up for eq. 6  
  // -----------------
  f:= -f0;
  n:= FAIL;
  if ode::odeIszero(f2+2*x) and iszero(expand(f2+2*x,optIgnoreAnalyticConstraints)) then 
    n:= 1;
  else 
    xpow:= {};
    misc::maprec(combine(expand(f2/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/),
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                  end_proc);  
    if nops(xpow) = 1 then 
      n:= xpow[1];
    end_if;    
  end_if;  
  if n <> FAIL and 
     ode::odeIszero(f2+(n+1)*x^n) and 
     ode::odeIszero(f1-x^(n+1)*f) and 
     iszero(expand(f2+(n+1)*x^n,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f1-x^(n+1)*f,optIgnoreAnalyticConstraints)) then 
    y0:= 1/x^(n+1);
    return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
  end_if;
  // -----------------
  // look-up for eq. 9  
  // -----------------
  f:= f2;
  gg:= f1;
  L:= ode::solveWithoutProperties(-z^2*f-z*gg = f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  if type(L) = piecewise then 
    L:= piecewise::disregardPoints(L);
  end_if;
  if type(L) = DOM_SET then
    L:= select(ode::normal(L), _not@has, x);
    if L <> {} then 
      y0:= L[1];
      return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
    end_if;
  end_if;
  // ------------------
  // look-up for eq. 10  
  // ------------------
  f:= f2;
  gg:= f1;
  xpow:= {};
  misc::maprec(combine(expand(f0/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/),
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                                end_proc);  
  n:= FAIL;
  if xpow = {2} then 
    n:= 1;
  elif xpow = {-1/2,1/2} then 
    n:= 1/2;
  elif nops(xpow) = 3 then 
    xpow:= [op(xpow)];
    for ind in [[2,3,1],[3,2,1],[1,3,2],[3,1,2],[1,2,3],[2,1,3]] do
      if iszero(expand(xpow[ind[1]]-(xpow[ind[2]]+1),optIgnoreAnalyticConstraints)) and 
         iszero(expand(2*xpow[ind[1]]-xpow[ind[3]],optIgnoreAnalyticConstraints)) then 
        n:= xpow[ind[1]];
        break;
      end_if;
    end_for;
  end_if;    
  if n <> FAIL then 
    L:= ode::solveWithoutProperties(z*n*x^(n-1)-z*x^n*gg-z^2*f*x^(2*n) = f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = piecewise then 
      L:= piecewise::disregardPoints(L);
    end_if;
    if type(L) = DOM_SET then
      L:= select(ode::normal(L), _not@has, x);
      if L <> {} then 
        y0:= L[1]*x^n;
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;
    end_if;
  end_if;
  // ------------------
  // look-up for eq. 11  
  // ------------------
  f:= f2;
  n:= FAIL;
  xpow:= {};
  misc::maprec(combine(expand(f1/*,optIgnoreAnalyticConstraints*/)/*,optIgnoreAnalyticConstraints*/),
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                                end_proc);  
  if nops(xpow) = 1 then 
    n:= xpow[1];
    y0:= z*x^n;
    L:= ode::solveWithoutProperties(g*diff(y0,x)-f2*y0^2-f1*y0-f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = piecewise then 
      L:= piecewise::disregardPoints(L);
    end_if;
    if type(L) = DOM_SET then
      L:= select(ode::normal(L), _not@has, x);
      if L <> {} then 
        y0:= L[1]*x^n;
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;
    end_if;
  end_if;    
  // ------------------
  // look-up for eq. 12  
  // ------------------
  exp_pow:= {};
  if has(f2,exp) then 
    misc::maprec(f2,
                 {"exp"} = proc(elem)
                             begin
                               if has(elem,x) and 
                                  iszero(expand(diff(op(elem,1),x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      a:= diff(subs(f2,[exp(lambda*x)=z,exp(expand(lambda*x))=z]),z);
      if not has(a,x) and not has(a,z) and not iszero(a) then 
        f:= ode::normal(f1/(a*exp(lambda*x)));
        if ode::odeIszero(f0-lambda*f) and ode::odeIszero(f2-a*exp(lambda*x)) and 
           iszero(expand(f0-lambda*f,optIgnoreAnalyticConstraints)) and 
           iszero(expand(f2-a*exp(lambda*x),optIgnoreAnalyticConstraints)) then 
          y0:= -lambda/a*exp(-lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
        end_if;
      end_if; 
    end_if; 
  end_if;
  // ------------------
  // look-up for eq. 14  
  // ------------------
  f:= f2; 
  if has(f0,exp) then 
    exp_pow:= {};
    misc::maprec(combine(f0,exp),
                 {"exp"} = proc(elem)
                             begin
                               if has(elem,x) and 
                                  iszero(expand(diff(op(elem,1),x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
    if nops(exp_pow) = 2 then 
      lambda:= FAIL;
      if ode::odeIszero(exp_pow[1]-2*exp_pow[2]) and 
         iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[2];
      elif ode::odeIszero(exp_pow[2]-2*exp_pow[1]) and 
           iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[1];
      end_if;
      if lambda <> FAIL then 
        if ode::odeIszero(f1) and 
           iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
          L:= ode::solveWithoutProperties(z*lambda*exp(lambda*x)-z^2*exp(2*lambda*x)*f = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= L[1]*exp(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if;
          end_if;
        end_if;
        // ------------------
        // look-up for eq. 18  
        // ------------------
        gg:= f1; 
        L:= ode::solveWithoutProperties(z*lambda*exp(lambda*x)-z*exp(lambda*x)*gg-z^2*exp(2*lambda*x)*f = combine(f0,exp),z,
                  IgnoreSpecialCases,optIgnoreAnalyticConstraints);
        if type(L) = piecewise then 
          L:= piecewise::disregardPoints(L);
        end_if;
        if type(L) = DOM_SET then
          L:= select(ode::normal(L), _not@has, x);
          if L <> {} then 
            y0:= L[1]*exp(lambda*x);
            return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
          end_if;
        end_if;
      end_if;
    end_if;    
  end_if; 
  // ------------------
  // look-up for eq. 20  
  // ------------------
  f:= f2;
  if has(f0,exp) then 
    exp_pow:= {};
    misc::maprec(combine(f0,exp), // before just 'f0'
                 {"exp"} = proc(elem)
                             begin
                               if has(elem,x^2) and 
                                  iszero(expand(diff(op(elem,1),x,x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x,x)/2}
                                end_if;
                                elem;
                              end_proc);  
    if nops(exp_pow) = 2 then 
      lambda:= FAIL;
      if iszero(expand(exp_pow[1]-2*exp_pow[2],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[2];
      elif iszero(expand(exp_pow[2]-2*exp_pow[1],optIgnoreAnalyticConstraints)) then 
        lambda:= exp_pow[1];
      end_if;
      if lambda <> FAIL then 
        L:= ode::solveWithoutProperties(2*z*lambda*x*exp(lambda*x^2)-z^2*f*exp(2*lambda*x^2) = f0,z,
                  IgnoreSpecialCases,optIgnoreAnalyticConstraints);
        if type(L) = piecewise then 
          L:= piecewise::disregardPoints(L);
        end_if;
        if type(L) = DOM_SET then
          L:= select(ode::normal(L), _not@has, x);
          if L <> {} then 
            y0:= L[1]*exp(lambda*x^2);
            return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
          end_if; 
        end_if;
      end_if;
    end_if;
  end_if;  
  // ------------------
  // look-up for eq. 22  
  // ------------------
  f:= f2;
  if iszero(f1) then
    if has(f0,tanh) then 
      tanh_arg:= {};
      misc::maprec(f0,
                   {"tanh"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      tanh_arg:= tanh_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(tanh_arg) = 1 then 
        lambda:= tanh_arg[1];
        if has(f0,tanh(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z*tanh(lambda*x)^2*(z*f+lambda)+z*lambda = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= L[1]*tanh(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;
    // ------------------
    // look-up for eq. 23    
    // ------------------
    if has(f0,coth) then 
      coth_arg:= {};
      misc::maprec(f0,
                   {"coth"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      coth_arg:= coth_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(coth_arg) = 1 then 
        lambda:= coth_arg[1];
        if has(f0,coth(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z*coth(lambda*x)^2*(z*f+lambda)+z*lambda = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= L[1]*coth(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;
    // ------------------
    // look-up for eq. 24    
    // ------------------
    if has(f0,sinh) then 
      sinh_arg:= {};
      misc::maprec(f0,
                   {"sinh"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      sinh_arg:= sinh_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(sinh_arg) = 1 then 
        lambda:= sinh_arg[1];
        if has(f0,sinh(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z^2*f+z*lambda*sinh(lambda*x)-z^2*f*sinh(lambda*x)^2 = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= L[1]*cosh(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;
    // ------------------
    // look-up for eq. 30    
    // ------------------
    if has(f0,sin) then 
      sin_arg:= {};
      misc::maprec(f0,
                   {"sin"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      sin_arg:= sin_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(sin_arg) = 1 then 
        lambda:= sin_arg[1];
        if has(f0,sin(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z^2*f+z*lambda*sin(lambda*x)+z^2*f*sin(lambda*x)^2 = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= -L[1]*cos(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;
    // ------------------
    // look-up for eq. 31    
    // ------------------
    if has(f0,cos) then 
      cos_arg:= {};
      misc::maprec(f0,
                   {"cos"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      cos_arg:= cos_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(cos_arg) = 1 then 
        lambda:= cos_arg[1];
        if has(f0,cos(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z^2*f+z*lambda*cos(lambda*x)+z^2*f*cos(lambda*x)^2 = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= L[1]*sin(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;
    // ------------------
    // look-up for eq. 32    
    // ------------------
    if has(f0,tan) then 
      tan_arg:= {};
      misc::maprec(f0,
                   {"tan"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      tan_arg:= tan_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(tan_arg) = 1 then 
        lambda:= tan_arg[1];
        if has(f0,tan(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z*tan(lambda*x)^2*(z*f-lambda)+z*lambda = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= L[1]*tan(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;
    // ------------------
    // look-up for eq. 33    
    // ------------------
    if has(f0,cot) then 
      cot_arg:= {};
      misc::maprec(f0,
                   {"cot"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      cot_arg:= cot_arg union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if nops(cot_arg) = 1 then 
        lambda:= cot_arg[1];
        if has(f0,cot(lambda*x)^2) then 
          L:= ode::solveWithoutProperties(-z*cot(lambda*x)^2*(z*f-lambda)+z*lambda = f0,z,
                    IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            if L <> {} then 
              y0:= -L[1]*cot(lambda*x);
              return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
            end_if; 
          end_if;
        end_if;                
      end_if;
    end_if;     
  end_if; 
  // ------------------
  // look-up for eq. 28  
  // ------------------
  if has(f2,ln) and has(f1,ln) then 
    a:= ode::normal(f2/(-ln(x)));
    if not has(a,x) then 
      f:= -f0;
      if ode::odeIszero(f1-a*f*(x*ln(x)-x)) and 
         iszero(expand(f1-a*f*(x*ln(x)-x),optIgnoreAnalyticConstraints)) then
        y0:= 1/(a*(x*ln(x)-x));
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;
    end_if;
  end_if;
  // ------------------
  // look-up for eq. 29  
  // ------------------
  if has(f2,sin) and has(f1,cos) then 
    cos_arg:= {};
    misc::maprec(f1,
                 {"cos"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    cos_arg:= cos_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    if nops(cos_arg) = 1 then 
      lambda:= cos_arg[1];
      f:= -f0;
      if ode::odeIszero(f2-lambda*sin(lambda*x)) and ode::odeIszero(f1-f*cos(lambda*x)) and 
         iszero(expand(f2-lambda*sin(lambda*x),optIgnoreAnalyticConstraints)) and
         iszero(expand(f1-f*cos(lambda*x),optIgnoreAnalyticConstraints)) then 
        y0:= 1/cos(lambda*x);
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if; 
    end_if;
  end_if;
  // ------------------
  // look-up for eq. 37  
  // ------------------
  gg:= -f0;
  f:= ode::normal(f1/gg);
  if not iszero(f) and ode::odeIszero(f2+diff(f,x)) and 
     iszero(expand(f2+diff(f,x),optIgnoreAnalyticConstraints)) then 
    y0:= 1/f;
    return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
  end_if;
  gg:= f2;
  if not iszero(gg) then 
    f:= ode::normal(-f1/(2*gg));
    if ode::odeIszero(f0-gg*f^2-diff(f,x)) and 
       iszero(expand(f0-gg*f^2-diff(f,x),optIgnoreAnalyticConstraints)) then
      y0:= f;
      return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
    end_if;
  end_if;
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> x then 
     g:= x;
    f0:= g*f0_orig/g_orig; 
    f1:= g*f1_orig/g_orig; 
    f2:= g*f2_orig/g_orig;  
  else 
     g:= g_orig;
    f0:= f0_orig;
    f1:= f1_orig;
    f2:= f2_orig;
  end_if; 
  // -----------------
  // look-up for eq. 7  
  // -----------------
  f:= f2;
  n:= f1; 
  if not has(n,x) and not iszero(f) and not iszero(n) then 
    a:= ode::normal(combine(f0/(x^(2*n)*f),IgnoreAnalyticConstraints));
    if not has(a,x) then 
      if is(a<0) = TRUE then 
        return({sqrt(abs(a))*x^n*tanh(-sqrt(abs(a))*int(x^(n-1)*f,x,intOptions) + genident("C"))})
      else 
        if n <> 1 then 
          return({sqrt(a)*x^n*tan(sqrt(a)*int(x^(n-1)*f,x,intOptions) + genident("C"))})
        else // this case is not in the reference book; using the generic method here 
             // helps to avoid loss of "singular" solutions
          L:= ode::riccati(g*diff(z(x),x)-f2*z(x)^2-f1*z(x)-f0,z,x,solveOptions,odeOptions);
          if L <> FAIL and nops(L) > 1 then 
            return(L)
          end_if;
        end_if;
      end_if;
    end_if;    
  end_if;  
  // ------------------
  // look-up for eq. 25    
  // ------------------
  f:= f2;
  if has(f0,ln) then 
    L:= ode::solveWithoutProperties(z-z^2*f*ln(x)^2 = f0,z,
              IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = piecewise then 
      L:= piecewise::disregardPoints(L);
    end_if;
    if type(L) = DOM_SET then
      L:= select(ode::normal(L), _not@has, x);
      if L <> {} then 
        y0:= L[1]*ln(x);
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;        
    end_if;
  end_if;

  //--------------------

  return(FAIL);
end_proc:


/*
----------
REFERENCE. Implementation of (Section 1.2.2. pp. 82)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
  
  g,f2,f1,f0 -- coefficients of the Riccati equation 
                g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
           x -- the independent variable 
           y -- the dependent variable
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 
*/

ode::PZ_Sec122_Riccati:= proc(g, f0, f1, f2, x, y, solveOptions={}, odeOptions={})
  local a, b, z, w, xpow, n, m, k, g_orig, f0_orig, f1_orig, f2_orig, t, L, y0,
        c, lambda, mu, optIgnoreAnalyticConstraints;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  z:= solvelib::getIdent(Any, indets([g, f0, f1, f2, x, y])); 
  g_orig := g;
  f0_orig:= f0;
  f1_orig:= f1;
  f2_orig:= f2; 
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> 1 then
    g := 1;
    f0:= f0_orig/g_orig; 
    f1:= f1_orig/g_orig; 
    f2:= f2_orig/g_orig;  
  else 
     g:= g_orig;
    f0:= f0_orig;
    f1:= f1_orig;
    f2:= f2_orig;
  end_if; 
  // -----------------
  // look-up for eq. 4  
  // -----------------
  a:= f2;
  if not iszero(a) and iszero(f1) and not has(a,x) then 
    xpow:= {}; 
    t:= combine(expand(f0/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    misc::maprec(t,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    n:= FAIL;                            
    if nops(xpow) = 1 then 
      n:= xpow[1];
    elif xpow = {} then 
      n:= 1;
    end_if; 
    if n<>FAIL then 
      b:= diff(subs(t,x^n=z),z);
      if iszero(expand(f0-b*x^n,optIgnoreAnalyticConstraints)) and 
         ode::odeIszero(expand(f0-b*x^n,optIgnoreAnalyticConstraints)) then
        k:= ode::normal(1/2*(n+2));
        if not iszero(k) and not has(b,x) and not has(b,z) then 
          if traperror((
                w:= sqrt(x)*(genident("C")*besselJ(1/(2*k),1/k*sqrt(a*b)*x^k)+
                             genident("C")*besselY(1/(2*k),1/k*sqrt(a*b)*x^k))
                      )) = 0 then 
            return({-1/a*diff(w,x)/w})             
          end_if;              
        end_if;
      end_if;
    end_if;
  end_if;
  // ----------------------
  // look-up for eq. 8 & 10  
  // ----------------------
  if iszero(f1) then 
    xpow:= {}; 
    t:= combine(expand(f2/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
    misc::maprec(t,
                 {"_power"} = proc(elem)
                                begin
                                  if op(elem,1) = x then 
                                    xpow:= xpow union {op(elem,2)}
                                  end_if;
                                  elem;
                                end_proc);  
    n:= FAIL;                            
    if nops(xpow) = 1 then 
      n:= xpow[1];
    elif xpow = {} then 
      n:= 1;
    end_if; 
    a:= diff(subs(t,x^n=z),z);
    if not has(a,x) and not has(a,z) and ode::odeIszero(f2-a*x^n) and 
       iszero(expand(f2-a*x^n,optIgnoreAnalyticConstraints)) then 
      t:= combine(expand(f0/*,optIgnoreAnalyticConstraints*/),optIgnoreAnalyticConstraints);
      xpow:= {};
      misc::maprec(t,
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = x then 
                                      xpow:= xpow union {op(elem,2)}
                                    end_if;
                                    elem;
                                  end_proc);  
      m:= FAIL;                            
      // -----------------
      // look-up for eq. 8  
      // -----------------
      if nops(xpow) = 1 then 
        m:= xpow[1];
      elif xpow = {} then 
        m:= 1;
      end_if; 
      b:= diff(subs(t,x^m=z),z);
      if not has(b,x) and not has(b,z) and ode::odeIszero(f0-b*x^m) and 
         iszero(expand(f0-b*x^m,optIgnoreAnalyticConstraints)) then 
        if n <> -1 and not iszero(a) then
          L:= ode::lookUp1stOrderRiccati(diff(y(z),z) - a/(n+1)*y(z)^2 - b/(n+1)*z^((m-n)/(n+1)), 
                                         y,z,solveOptions,odeOptions);      
          if traperror((L:= map(L, elem -> evalAt(elem,z=x^(n+1))))) = 0 then 
            return(L);
          end_if;    
        else // n = -1
          if m <> -1 then 
            L:= ode::lookUp1stOrderRiccati(diff(y(z),z) - b/(m+1)*y(z)^2 - a/(m+1)/z, 
                                           y,z,solveOptions,odeOptions);      
            if traperror((L:= map(L, elem -> -1/evalAt(elem,z=x^(m+1))))) = 0 then 
              return(L);
            end_if;              
          end_if; // in case 'm = n = -1' equations is separable and should be tackled by 
                  // method for separation of variables 
        end_if;  
      end_if;    
      // ------------------
      // look-up for eq. 10  
      // ------------------
      if nops(xpow) = 2 then 
        m:= FAIL;
        if iszero(expand(xpow[1]+1-(xpow[2]-n)/2,optIgnoreAnalyticConstraints)) then 
          m:= xpow[1]+1;
        elif iszero(expand(xpow[2]+1-(xpow[1]-n)/2,optIgnoreAnalyticConstraints)) then 
          m:= xpow[2]+1;
        end_if;
        if m <> FAIL then 
          b:= diff(subs(t,x^(m-1)=z),z)/m;
          if not has(b,x) and not has(b,z) and ode::odeIszero(t-b*m*x^(m-1)+a*b^2*x^(n+2*m)) and 
             iszero(expand(t-b*m*x^(m-1)+a*b^2*x^(n+2*m),optIgnoreAnalyticConstraints)) then 
            y0:= b*x^m;
            return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
          end_if;
        end_if;
      end_if;  
    end_if;  
  end_if;  
  // ------------------
  // look-up for eq. 23  
  // ------------------
  /*
    NOTE. This is a strong generalization of the above cited pattern. 
  */ 
  if ode::odeIszero(f1_orig) and 
     iszero(expand(f1_orig,optIgnoreAnalyticConstraints)) and
     not iszero(expand(g_orig,optIgnoreAnalyticConstraints)) and 
     ode::odeIszero(g_orig-f2_orig) and ode::odeIszero(f0_orig+diff(g_orig,x,x)) and 
     iszero(expand(g_orig-f2_orig,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f0_orig+diff(g_orig,x,x),optIgnoreAnalyticConstraints)) then 
    y0:= -diff(g_orig,x)/g_orig;
    return(ode::RiccatiSolutionGenerator(y0,g_orig,f0_orig,f1_orig,f2_orig,x,solveOptions,odeOptions));
  end_if; 
  //=======================
  // no ode::normalization here
  //=======================
   g:= g_orig;
  f0:= f0_orig;
  f1:= f1_orig;
  f2:= f2_orig;
  // ------------------
  // look-up for eq. 59  
  // ------------------
  if ode::odeIszero(f2-1) and 
     ode::odeIszero(diff(g,x,x,x)) and 
     ode::odeIszero(diff(f1,x,x)) and 
     ode::odeIszero(diff(f0,x,x,x)) and 
     iszero(expand(f2-1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(diff(g,x,x,x),optIgnoreAnalyticConstraints)) and 
     iszero(expand(diff(f1,x,x),optIgnoreAnalyticConstraints)) and 
     iszero(expand(diff(f0,x,x,x),optIgnoreAnalyticConstraints)) then 
    a:= diff(g,x,x)/2;
    b:= diff(expand(g-a*x^2/*,optIgnoreAnalyticConstraints*/),x);
    c:= expand(g-a*x^2-b*x/*,optIgnoreAnalyticConstraints*/);
    if not has(a,x) and not has(b,x) and not has(c,x) and ode::odeIszero(g-a*x^2-b*x-c) and 
       iszero(expand(g-a*x^2-b*x-c,optIgnoreAnalyticConstraints)) then 
      lambda:= diff(f1,x)/2;
      if traperror((mu:= (f0 | x=0))) = 0 and 
         ode::odeIszero(f1-(2*lambda*x+b)) and 
         ode::odeIszero(f0-(lambda*(lambda-a)*x^2+mu)) and 
         iszero(expand(f1-(2*lambda*x+b),optIgnoreAnalyticConstraints)) and 
         iszero(expand(f0-(lambda*(lambda-a)*x^2+mu),optIgnoreAnalyticConstraints)) then
        y0:= -lambda*x+1/2*(-b+sqrt(b^2-4*mu-4*lambda*c));
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));
      end_if;  
    end_if;   
  end_if;  
  
  
  //--------------------

  return(FAIL);
end_proc:

/*
----------
REFERENCE. Implementation of (Section 1.2.3. pp. 90)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
  
  g,f2,f1,f0 -- coefficients of the Riccati equation 
                g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
           x -- the independent variable 
           y -- the dependent variable
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 
*/

ode::PZ_Sec123_Riccati:= proc(g, f0, f1, f2, x, y, solveOptions={}, odeOptions={})
  local g_orig, f0_orig, f1_orig, f2_orig, optIgnoreAnalyticConstraints, z, 
        a, b, exp_pow, lambda, mu, y0, L, c, d, k, s, t, n;
begin 
  // If the coefficients do not contain exponential functions nothing more needs
  // to be done here. We only consider ODEs with coefficients involving 'exp'. 
  if not has([g,f0,f1,f2],exp) then 
    return(FAIL)
  end_if;
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  z:= solvelib::getIdent(Any, indets([g, f0, f1, f2, x, y])); 
  g_orig := g;
  f0_orig:= f0;
  f1_orig:= f1;
  f2_orig:= f2; 
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> 1 then
    g := 1;
    f0:= combine(f0_orig/g_orig,exp); 
    f1:= combine(f1_orig/g_orig,exp);  
    f2:= combine(f2_orig/g_orig,exp);   
  else 
     g:= combine(g_orig,exp); 
    f0:= combine(f0_orig,exp); 
    f1:= combine(f1_orig,exp); 
    f2:= combine(f2_orig,exp); 
  end_if; 
  // ------------------
  // look-up for eq. 10  
  // ------------------
  if ode::odeIszero(f1) and iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     has(f2,exp) then  
    exp_pow:= {};
    misc::maprec(f2,
                 {"exp"} = proc(elem)
                             begin
                               if has(elem,x) and 
                                  iszero(expand(diff(op(elem,1),x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
    if nops(exp_pow) = 1 then 
      mu:= exp_pow[1];
      b:= diff(subs(f2,[exp(expand(mu*x))=z,exp(mu*x)=z]),z);
      if not has(b,x) and not has(b,z) and ode::odeIszero(f2-b*exp(mu*x)) and 
         iszero(expand(f2-b*exp(mu*x),optIgnoreAnalyticConstraints)) then 
        exp_pow:= {};
        misc::maprec(f0,
                     {"exp"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                      exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
        lambda:= FAIL;
        if nops(exp_pow) = 2 then 
          if iszero(expand(exp_pow[1]-(exp_pow[2]-mu)/2,optIgnoreAnalyticConstraints)) then 
            lambda:= exp_pow[1];
          elif iszero(expand(exp_pow[2]-(exp_pow[1]-mu)/2,optIgnoreAnalyticConstraints)) then 
            lambda:= exp_pow[2];
          end_if;
        end_if;   
        if lambda <> FAIL then 
          a:= ode::normal(diff(subs(f0,[exp(expand(lambda*x))=z,exp(lambda*x)=z]),z)/lambda);
          if not has(a,x) and not has(a,z) and 
             ode::odeIszero(f0-a*lambda*exp(lambda*x)+a^2*b*exp((mu+2*lambda)*x)) and 
             iszero(combine(expand(f0-a*lambda*exp(lambda*x)+a^2*b*exp((mu+2*lambda)*x),
                                   optIgnoreAnalyticConstraints),exp)) then 
            y0:= a*exp(lambda*x);
            return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
          end_if;  
        end_if;
      end_if;
    end_if; 
  end_if;
  // -----------------------
  // look-up for eq. 16 & 18  
  // -----------------------
  if has(f0,exp) and not has(f1,x) and has(f2,exp) then 
    exp_pow:= {};
    misc::maprec(f2,
                 {"exp"} = proc(elem)
                             begin
                               if has(elem,x) and 
                                  iszero(expand(diff(op(elem,1),x,x),
                                                optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
    if nops(exp_pow) = 1 then 
      k:= exp_pow[1];
      a:= diff(subs(f2,[exp(expand(k*x))=z,exp(k*x)=z]),z);
      if not has(a,x) and not has(a,z) and ode::odeIszero(f2-a*exp(k*x)) and 
         iszero(expand(combine(f2-a*exp(k*x),exp),optIgnoreAnalyticConstraints)) then 
        b:= f1;
        // ------------------
        // look-up for eq. 16  
        // ------------------
        d:= diff(subs(f0,[exp(expand(-k*x))=z,exp(-k*x)=z]),z);
        if not has(d,x) and not has(d,z) then 
          t:= combine(expand(combine(f0-d*exp(-k*x),exp)/*,optIgnoreAnalyticConstraints*/),exp);
          exp_pow:= {};
          misc::maprec(combine(t,exp),
                       {"exp"} = proc(elem)
                                   begin
                                     if has(elem,x) and 
                                        iszero(expand(diff(op(elem,1),x,x),
                                                      optIgnoreAnalyticConstraints)) then 
                                        exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                      end_if;
                                      elem;
                                    end_proc);  
          if nops(exp_pow) = 1 then 
            s:= exp_pow[1];
            c:= diff(subs(combine(t,exp),[exp(expand(s*x))=z,exp(s*x)=z]),z);
            if not has(c,x) and not has(c,z) and ode::odeIszero(f0-c*exp(s*x)-d*exp(-k*x)) and 
               iszero(expand(combine(f0-c*exp(s*x)-d*exp(-k*x),exp),optIgnoreAnalyticConstraints)) then 
              L:= solve(ode(k*x^2*diff(y(x),x)-a*x^2*y(x)^2-b*x*y(x)-c*x^((k+s)/k)-d,y(x)));
              if traperror((L:= L | x=exp(k*x))) = 0 then 
                return(L);  
              end_if;
            end_if;
          end_if;          
        end_if;
        // ------------------
        // look-up for eq. 18  
        // ------------------
        exp_pow:= {};
        misc::maprec(f0,
                     {"exp"} = proc(elem)
                                 begin
                                   if has(elem,x) and 
                                      iszero(expand(diff(op(elem,1),x,x),
                                                    optIgnoreAnalyticConstraints)) then 
                                      exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
        if nops(exp_pow) = 2 then 
          n:= FAIL;
          if iszero(expand(2*exp_pow[1]/k+1-exp_pow[2]/k,optIgnoreAnalyticConstraints)) then 
            n:= ode::normal(exp_pow[1]/k);
          elif iszero(expand(2*exp_pow[2]/k+1-exp_pow[1]/k,optIgnoreAnalyticConstraints)) then 
            n:= ode::normal(exp_pow[2]/k);                 
          end_if;                    
          if n <> FAIL then 
            c:= diff(subs(f0,[exp(expand(k*n*x))=z,exp(k*n*x)=z]),z);
            d:= diff(subs(f0,[exp(expand(k*(2*n+1)*x))=z,exp(k*(2*n+1)*x)=z]),z);
            if ode::odeIszero(f0-c*exp(k*n*x)-d*exp(k*(2*n+1)*x)) and  
               iszero(expand(combine(f0-c*exp(k*n*x)-d*exp(k*(2*n+1)*x),exp),
                             optIgnoreAnalyticConstraints)) then 
              L:= solve(ode(k*x^2*diff(y(x),x)-a*x^2*y(x)^2-b*x*y(x)-c*x^(n+1)-d*x^(2*(n+1)),y(x)));
              if traperror((L:= L | x=exp(k*x))) = 0 then 
                return(L);  
              end_if;
            end_if;                   
          end_if;
        end_if;  
      end_if;   
    end_if;  
  end_if;  
  
  //--------------------

  return(FAIL);
end_proc:

/*
----------
REFERENCE. Implementation of (Section 1.2.4. pp. 92)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
  
  g,f2,f1,f0 -- coefficients of the Riccati equation 
                g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
           x -- the independent variable 
           y -- the dependent variable
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 
*/

ode::PZ_Sec124_Riccati:= proc(g, f0, f1, f2, x, y, solveOptions={}, odeOptions={})
  local g_orig, f0_orig, f1_orig, f2_orig, optIgnoreAnalyticConstraints, z, 
        exp_arg, lambda, y0, a, b, t, L, coth_arg, L1, L2, ff0, ff1, ff2;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  z:= solvelib::getIdent(Any, indets([g, f0, f1, f2, x, y])); 
  g_orig := g;
  f0_orig:= f0;
  f1_orig:= f1;
  f2_orig:= f2; 
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> 1 then
    g := 1;
    f0:= f0_orig/g_orig; 
    f1:= f1_orig/g_orig; 
    f2:= f2_orig/g_orig;  
  else 
     g:= g_orig;
    f0:= f0_orig;
    f1:= f1_orig;
    f2:= f2_orig;
  end_if; 
  ff0:= combine(ode::normal(rewrite(f0,exp)),exp);
  ff1:= combine(ode::normal(rewrite(f1,exp)),exp);
  ff2:= combine(ode::normal(rewrite(f2,exp)),exp);
  // -----------------
  // look-up for eq. 4  
  // -----------------
  if ode::odeIszero(f1) and iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
    exp_arg:= {};
    misc::maprec(ff2,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x))}
                                  end_if;
                                  elem;
                                end_proc);                                          
    exp_arg:= [op(exp_arg)];
    for lambda in exp_arg do  
      if ode::odeIszero(ff2-lambda*(exp(lambda*x)/2-1/(2*exp(lambda*x)))) and 
         ode::odeIszero(ff0+lambda*(exp(lambda*x)/2-1/(2*exp(lambda*x)))^3) and 
         iszero(ode::normal(ff2-lambda*(exp(lambda*x)/2-1/(2*exp(lambda*x))))) and 
         iszero(ode::normal(ff0+lambda*(exp(lambda*x)/2-1/(2*exp(lambda*x)))^3)) then 
        y0:= cosh(lambda*x);      
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
      end_if;  
    end_for;
    for lambda in map(exp_arg, elem -> ode::normal(elem/2)) do  
      if not iszero(exp(lambda*x)/2-1/(2*exp(lambda*x))) then 
        // -----------------
        // look-up for eq. 5  
        // -----------------
        a:= ode::normal((ff2+lambda)/(exp(lambda*x)/2-1/(2*exp(lambda*x)))^2);
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(ff2+lambda-a*(exp(lambda*x)/2-1/(2*exp(lambda*x)))^2) and 
           ode::odeIszero(a+ff0-lambda+a*(exp(lambda*x)/2-1/(2*exp(lambda*x)))^2) and 
           iszero(ode::normal(ff2+lambda-a*(exp(lambda*x)/2-1/(2*exp(lambda*x)))^2)) and 
           iszero(ode::normal(a+ff0-lambda+a*(exp(lambda*x)/2-1/(2*exp(lambda*x)))^2)) then 
          y0:= coth(lambda*x);      
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
        // ------------------
        // look-up for eq. 11  
        // ------------------
        a:= ode::normal((ff2+lambda)/(exp(lambda*x)/2+1/(2*exp(lambda*x)))^2);
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(ff2+lambda-a*(exp(lambda*x)/2+1/(2*exp(lambda*x)))^2) and 
           ode::odeIszero(ff0-a-lambda+a*(exp(lambda*x)/2+1/(2*exp(lambda*x)))^2) and 
           iszero(ode::normal(ff2+lambda-a*(exp(lambda*x)/2+1/(2*exp(lambda*x)))^2)) and 
           iszero(ode::normal(ff0-a-lambda+a*(exp(lambda*x)/2+1/(2*exp(lambda*x)))^2)) then 
          y0:= tanh(lambda*x);      
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
      end_if;  
    end_for;
  end_if;   
  // ------------------
  // look-up for eq. 19  
  // ------------------
  if ode::odeIszero(f1) and ode::odeIszero(f2-1) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f2-1,optIgnoreAnalyticConstraints)) then 
    exp_arg:= {};
    misc::maprec(ff0,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x))}
                                  end_if;
                                  elem;
                                end_proc);                                          
    exp_arg:= map([op(exp_arg)], elem -> ode::normal(elem/2));
    for lambda in exp_arg do  
      L:= ode::solveWithoutProperties(3*z*lambda-lambda^2-z*(z+lambda)*
                (exp(2*lambda*x)-1)^2/(exp(2*lambda*x)+1)^2=ff0,z,
                IgnoreSpecialCases,optIgnoreAnalyticConstraints);
      L:= select(ode::normal(L), _not@has, x);
      for a in L do 
        if ode::odeIszero(ff0-3*a*lambda+lambda^2+(a*(exp(2*lambda*x)-1)^2*(a+lambda))/(exp(2*lambda*x)+1)^2) and 
           iszero(ode::normal(ff0-3*a*lambda+lambda^2+(a*(exp(2*lambda*x)-1)^2*
                                      (a+lambda))/(exp(2*lambda*x)+1)^2)) then 
          y0:= a*tanh(lambda*x)-lambda*coth(lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;
      end_for; 
      // ------------------
      // look-up for eq. 23  
      // ------------------
      L:= ode::solveWithoutProperties(3*z*lambda-lambda^2-z*(z+lambda)*
                (exp(2*lambda*x)+1)^2/(exp(2*lambda*x)-1)^2=ff0,z,
                IgnoreSpecialCases,optIgnoreAnalyticConstraints);
      L:= select(ode::normal(L), _not@has, x);
      for a in L do 
        if ode::odeIszero(ff0-3*a*lambda+lambda^2+(a*(exp(2*lambda*x)+1)^2*(a+lambda))/(exp(2*lambda*x)-1)^2) and 
           iszero(ode::normal(ff0-3*a*lambda+lambda^2+(a*(exp(2*lambda*x)+1)^2*
                                      (a+lambda))/(exp(2*lambda*x)-1)^2)) then 
          y0:= a*coth(lambda*x)-lambda*tanh(lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;
      end_for; 
      
    end_for;
  end_if;
  // ---------------------------------------------------
  // look-up for eq. 27 (includes 26 for lambda = a = b)  
  // ---------------------------------------------------
  if ode::odeIszero(f1) and 
     ode::odeIszero(f2-1) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f2-1,optIgnoreAnalyticConstraints)) then 
    coth_arg:= {};
    misc::maprec(f0,
                 {"coth"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    coth_arg:= coth_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    coth_arg:= [op(coth_arg)];
    if nops(coth_arg) = 1 then 
      lambda:= coth_arg[1];
      t:= eval(diff(subs(f0,coth(lambda*x)^2=z),z));
      L1:= ode::solveWithoutProperties(-z*(z+lambda)=t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
      L1:= select(ode::normal(L1), _not@has, x);      
      t:= eval(diff(subs(f0,tanh(lambda*x)^2=z),z));
      L2:= ode::solveWithoutProperties(-z*(z+lambda)=t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
      L2:= select(ode::normal(L2), _not@has, x);      
      for b in L1 do 
        for a in L2 do 
          if ode::odeIszero(f0-a*lambda-b*lambda+2*a*b+a*(a+lambda)*tanh(lambda*x)^2+
                           b*(b+lambda)*coth(lambda*x)^2) and 
             iszero(expand(f0-a*lambda-b*lambda+2*a*b+a*(a+lambda)*tanh(lambda*x)^2+
                           b*(b+lambda)*coth(lambda*x)^2,optIgnoreAnalyticConstraints)) then 
            y0:= a*tanh(lambda*x)+b*coth(lambda*x);
            return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
          end_if;
        end_for;
      end_for;  
    end_if;  
  end_if;
     
  //--------------------

  return(FAIL);
end_proc:

/*
----------
REFERENCE. Implementation of (Section 1.2.5. pp. 94)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
  
  g,f2,f1,f0 -- coefficients of the Riccati equation 
                g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
           x -- the independent variable 
           y -- the dependent variable
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 
*/

ode::PZ_Sec125_Riccati:= proc(g, f0, f1, f2, x, y, solveOptions={}, odeOptions={})
  local g_orig, f0_orig, f1_orig, f2_orig, optIgnoreAnalyticConstraints, z, L, a, 
        b, c, bb, k, ln_arg, ln_pow, t, y0, aa;
  begin 
  // If the coefficients do no contain 'ln' nothing more needs to be done here. We 
  // only consider ODEs with coefficients in these class ogf functions.  
  if not has([g,f0,f1,f2],ln) then  
    return(FAIL)
  end_if;
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  z:= solvelib::getIdent(Any, indets([g, f0, f1, f2, x, y])); 
  g_orig := g;
  f0_orig:= f0;
  f1_orig:= f1;
  f2_orig:= f2; 
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> x then
    g := x;
    f0:= g*f0_orig/g_orig; 
    f1:= g*f1_orig/g_orig; 
    f2:= g*f2_orig/g_orig;  
  else 
     g:= g_orig;
    f0:= f0_orig;
    f1:= f1_orig;
    f2:= f2_orig;
  end_if; 
  // -----------------
  // look-up for eq. 2  
  // -----------------
  if ode::odeIszero(diff(f2,x)) and ode::odeIszero(f1) and 
     iszero(expand(diff(f2,x),optIgnoreAnalyticConstraints)) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     has(f0,ln) then 
    a:= f2;
    b:= diff(evalAt(f0,ln(x)=z),z);
    if not has(b,x) and not has(b,z) then 
      c:= f0-b*ln(x);  
      if not has(c,x) and ode::odeIszero(f0-b*ln(x)-c) and 
         iszero(expand(f0-b*ln(x)-c,optIgnoreAnalyticConstraints)) then 
        L:= solve(ode(diff(y(z),z)=a*y(z)^2+b*z+c,y(z)),op(solveOptions),op(odeOptions)); 
        if has(L,int) then 
          L:= map(L,intlib::changevar,z=ln(x),x);
        end_if;             
        return(subs(L,z=ln(x)));  
      end_if;  
    end_if;
  end_if;   
  // ---------------------
  // look-up for eq. 4 & 5  
  // ---------------------
  if ode::odeIszero(f2-x) and ode::odeIszero(f1) and  
     iszero(expand(f2-x,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
    ln_arg:= {};
    t:= combine(f0,ln);
    misc::maprec(t,
                 {"ln"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    ln_arg:= ln_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    ln_arg:= [op(ln_arg)];
    if nops(ln_arg) = 1 then 
      bb:= ln_arg[1];
      ln_pow:= {};
      misc::maprec(t,
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = ln(bb*x) or 
                                       op(elem,1) = ln(expand(bb*x,optIgnoreAnalyticConstraints)) then 
                                      ln_pow:= ln_pow union {op(elem,2)}
                                    end_if;
                                    elem;
                                    end_proc);  
      ln_pow:= [op(ln_pow)];
      k:= FAIL;
      if ln_pow = [2] then 
        k:= 1;
      elif ln_pow = [4] then 
        k:= 2;
      elif ln_pow = [-2] then 
        k:= -1;
      elif ln_pow = [-1/2] then 
        k:= 1/2;
      elif nops(ln_pow) = 2 then 
        if iszero(expand(2*ln_pow[2]+2-ln_pow[1],optIgnoreAnalyticConstraints)) then 
          k:= ln_pow[2]+1;
        elif iszero(expand(2*ln_pow[1]+2-ln_pow[2],optIgnoreAnalyticConstraints)) then 
          k:= ln_pow[1]+1;
        end_if;
      end_if;
      if k <> FAIL and not has(k,x) then 
        a:= combine(sqrt(-diff(subs(t,[ln(bb*x)^(2*k)=z,
                                       ln(expand(bb*x/*,optIgnoreAnalyticConstraints*/))^
                                       expand(2*k/*,optIgnoreAnalyticConstraints*/)=z]),z)/x),
                    IgnoreAnalyticConstraints);
        if not has(a,x) and not has(a,z) and ode::odeIszero(t+a^2*x*ln(bb*x)^(2*k)-a*k*ln(bb*x)^(k-1)) and 
           iszero(expand(t+a^2*x*ln(bb*x)^(2*k)-a*k*ln(bb*x)^(k-1),optIgnoreAnalyticConstraints)) then 
          y0:= a*ln(bb*x)^k;
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;     
      end_if;  
    end_if;    
  end_if;   
  // ------------------
  // look-up for eq. 17  
  // ------------------
  if not has(f2,x) and has(f1,ln) and has(f0,ln) then 
    aa:= combine(sqrt(f2),IgnoreAnalyticConstraints);
    for a in [-aa,aa] do 
      if traperror(( b:= ode::normal(evalAt(diff(subs(f1,ln(x)=z),z),z=0)/(2*a)) )) = 0 and 
         ode::odeIszero(f1-2*a*b*ln(x)) and ode::odeIszero(f0-b^2*ln(x)^2) and 
         iszero(ode::normal(expand(f1-2*a*b*ln(x),optIgnoreAnalyticConstraints))) and 
         iszero(ode::normal(expand(f0-b^2*ln(x)^2,optIgnoreAnalyticConstraints))) and 
         traperror(( L:= {-(b*ln(x)*a-tan(ln(x)*(b*a)^(1/2)+
                           genident("C")*a*(b*a)^(1/2))*(b*a)^(1/2))/a^2} )) = 0 then 
        return(L);          
      end_if;   
    end_for;  
  end_if;
  
  //--------------------

  return(FAIL);
end_proc:


/*
----------
REFERENCE. Implementation of (Section 1.2.6. pp. 96)
----------                   
             from   
             
           Polyanin & Zaitsev: Handbook of exact solutions for 
                               ordinary differential equations, 
                               Chapman & Hall/CRC, 2nd edition, 
                               2003 

-----------
PARAMETERS. 
-----------
  
  g,f2,f1,f0 -- coefficients of the Riccati equation 
                g(x)*y'(x) = f2(x)*y(x)^2 + f1(x)*y(x) + f0(x)
           x -- the independent variable 
           y -- the dependent variable
solveOptions -- options for 'solve'
  odeOptions -- options for 'ode' 
*/

ode::PZ_Sec126_Riccati:= proc(g, f0, f1, f2, x, y, solveOptions={}, odeOptions={})
  local g_orig, f0_orig, f1_orig, f2_orig, optIgnoreAnalyticConstraints, z, lambda, 
        y0, t, a, tan_arg, L, b, cos_pow, n, sin_pow, La, Lb, exp_arg, ff0, ff1, ff2;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  z:= solvelib::getIdent(Any, indets([g, f0, f1, f2, x, y])); 
  g_orig := g;
  f0_orig:= f0;
  f1_orig:= f1;
  f2_orig:= f2; 
  //=======================
  // Normalization strategy
  //=======================
  if g_orig <> 1 then
    g := 1;
    f0:= f0_orig/g_orig; 
    f1:= f1_orig/g_orig; 
    f2:= f2_orig/g_orig;  
  else 
     g:= g_orig;
    f0:= f0_orig;
    f1:= f1_orig;
    f2:= f2_orig;
  end_if; 
  ff0:= combine(ode::normal(rewrite(f0,exp)),exp);
  ff1:= combine(ode::normal(rewrite(f1,exp)),exp);
  ff2:= combine(ode::normal(rewrite(f2,exp)),exp);
  // ------------------------------
  // look-up for eq. 6,7,8,19,20,21  
  // ------------------------------
  if ode::odeIszero(f1) and iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
    t:= ff2;
    exp_arg:= {};
    misc::maprec(t,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x)/I)}
                                  end_if;
                                  elem;
                                end_proc);                                          
    exp_arg:= [op(exp_arg)];
    for lambda in exp_arg do  
      // -----------------
      // look-up for eq. 6  
      // -----------------
      if ode::odeIszero(ff2-lambda*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)) and 
         ode::odeIszero(ff0-lambda*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)^3) and 
         iszero(ode::normal(ff2-lambda*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))) and 
         iszero(ode::normal(ff0-lambda*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)^3)) then 
        y0:= -cos(lambda*x);
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
      end_if;
      // -----------------
      // look-up for eq. 7  
      // -----------------
      if not iszero((1-(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))) then 
        a:= ode::normal((2*t-lambda)/(1-(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)));
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(ff2-a/2-lambda/2+(a*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))/2) and 
           ode::odeIszero(a/2+ff0-lambda/2+(a*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))/2) and 
           iszero(ode::normal(ff2-a/2-lambda/2+(a*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))/2)
                         ) and 
           iszero(ode::normal(a/2+ff0-lambda/2+(a*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))/2)
                         ) then  
          y0:= tan(1/2*lambda*x+1/4*PI);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
      end_if;  
    end_for;        
    // -----------------
    // look-up for eq. 8  
    // -----------------
    for lambda in map(exp_arg,elem -> ode::normal(elem/2)) do 
      if not iszero((((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)) then 
        a:= ode::normal((ff2-lambda)/(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)^2);
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(t-lambda+a*((1/exp(lambda*x*2*I))/4+exp(lambda*x*2*I)/4-1/2)) and 
           ode::odeIszero(a+ff0-lambda-a*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)^2) and 
           iszero(ode::normal(t-lambda+a*((1/exp(lambda*x*2*I))/4+exp(lambda*x*2*I)/4-1/2))
                         ) and 
           iszero(ode::normal(a+ff0-lambda-a*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2)^2)
                         ) then 
          y0:= -cot(lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
      end_if;  
    end_for;   
    // ------------------
    // look-up for eq. 19 
    // ------------------
    for lambda in exp_arg do 
      if ode::odeIszero(ff2-lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)) and 
         ode::odeIszero(ff0-lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^3) and 
         iszero(ode::normal(ff2-lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2))) and 
         iszero(ode::normal(ff0-lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^3))
       then 
        y0:= sin(lambda*x);
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
      end_if;
      // Try again with negative values for 'lambda' due to the symmetry properties of 
      // cosine. 
      if ode::odeIszero(ff2+lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)) and 
         ode::odeIszero(ff0+lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^3) and 
         iszero(ode::normal(ff2+lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2))) and 
         iszero(ode::normal(ff0+lambda*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^3)) then                       
        y0:= sin(-lambda*x);
        return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
      end_if;
      // ------------------
      // look-up for eq. 20  
      // ------------------
      if not iszero((1+((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2))) then 
        a:= ode::normal((2*t-lambda)/(1+((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)));
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(f2-(lambda+a+a*cos(lambda*x))/2) and 
           ode::odeIszero(f0+(-lambda+a-a*cos(lambda*x))/2) and 
           iszero(expand(f2-(lambda+a+a*cos(lambda*x))/2,optIgnoreAnalyticConstraints)) and 
           iszero(expand(f0+(-lambda+a-a*cos(lambda*x))/2,optIgnoreAnalyticConstraints)) then  
          y0:= tan(1/2*lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
        // Try again with negative values for 'lambda' due to the symmetry properties of 
        // cosine. 
        // if not has(a,x) and not has(a,z) and 
        //    iszero(expand(f2-(-lambda+a+a*cos(lambda*x))/2,optIgnoreAnalyticConstraints)) and 
        //    iszero(expand(f0+(lambda+a-a*cos(lambda*x))/2,optIgnoreAnalyticConstraints)) then  
        //   y0:= tan(-1/2*lambda*x);  
        //   return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        // end_if; 
      end_if;  
    end_for;  
    // ------------------
    // look-up for eq. 21  
    // ------------------
    for lambda in map(exp_arg,elem -> ode::normal(elem/2)) do 
      if not iszero((((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^2)) then
        a:= ode::normal((ff2-lambda)/(((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^2));
        if not has(a,x) and not has(a,z) and 
           ode::odeIszero(ff2-lambda-a*((1/exp(lambda*x*2*I))/4+exp(lambda*x*2*I)/4+1/2)) and 
           ode::odeIszero(a+ff0-lambda-a*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^2) and 
           iszero(ode::normal(ff2-lambda-a*((1/exp(lambda*x*2*I))/4+exp(lambda*x*2*I)/4+1/2))) and 
           iszero(ode::normal(a+ff0-lambda-a*((1/exp(lambda*x*I))/2+exp(lambda*x*I)/2)^2)) then 
          y0:= tan(lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
      end_if;  
    end_for;      
  end_if;    
  // ---------------------
  // look-up for eq. 28,39  
  // ---------------------
  if ode::odeIszero(f1) and ode::odeIszero(f2-1) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f2-1,optIgnoreAnalyticConstraints)) then 
    exp_arg:= {};
    misc::maprec(ff0,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x)/I)}
                                  end_if;
                                  elem;
                                end_proc);                                          
    for lambda in map(exp_arg,elem -> ode::normal(elem/2)) do 
      if not iszero(expand(lambda,optIgnoreAnalyticConstraints)) then 
        L:= ode::solveWithoutProperties(ff0-lambda^2-3*z*lambda-z*(lambda-z)*((exp(lambda*x*2*I)*I-I)^2/
                                                        (exp(lambda*x*2*I)+1)^2),
                  z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
        if type(L) = piecewise then 
          L:= piecewise::disregardPoints(L);
        end_if;
        if type(L) = DOM_SET then
          L:= select(ode::normal(L), _not@has, x);
          for a in L do 
            // ------------------
            // look-up for eq. 28  
            // ------------------
            if not iszero((exp(lambda*x*2*I)+1)) then 
              if ode::odeIszero(ff0-3*a*lambda-lambda^2+(a*(exp(lambda*x*2*I)*I-I)^2*
                                         (a-lambda))/(exp(lambda*x*2*I)+1)^2) and 
                 iszero(ode::normal(ff0-3*a*lambda-lambda^2+(a*(exp(lambda*x*2*I)*I-I)^2*
                                         (a-lambda))/(exp(lambda*x*2*I)+1)^2)) then 
                y0:= a*tan(lambda*x)-lambda*cot(lambda*x);
                return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
              end_if;
            end_if;  
          end_for;
        end_if;
        if not iszero((exp(lambda*x*2*I)-1)) then
          L:= ode::solveWithoutProperties(ff0-lambda^2-3*z*lambda-z*(lambda-z)*((exp(lambda*x*2*I)*I+I)^2/
                                                          (exp(lambda*x*2*I)-1)^2),
                    z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
          if type(L) = piecewise then 
            L:= piecewise::disregardPoints(L);
          end_if;
          if type(L) = DOM_SET then
            L:= select(ode::normal(L), _not@has, x);
            for a in L do 
              // ------------------
              // look-up for eq. 39  
              // ------------------
              if ode::odeIszero(ff0-3*a*lambda-lambda^2+(a*(exp(lambda*x*2*I)*I+I)^2*
                                        (a-lambda))/(exp(lambda*x*2*I)-1)^2) and 
                 iszero(ode::normal(ff0-3*a*lambda-lambda^2+(a*(exp(lambda*x*2*I)*I+I)^2*
                                        (a-lambda))/(exp(lambda*x*2*I)-1)^2)
                               ) then 
                y0:= lambda*tan(lambda*x)-a*cot(lambda*x);
                return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
              end_if;                  
            end_for;  
          end_if;                  
        end_if;  
      end_if;  
    end_for;  
  end_if;
  // ------------------
  // look-up for eq. 48  
  // ------------------
  if ode::odeIszero(f1) and iszero(expand(f1,optIgnoreAnalyticConstraints)) then
    exp_arg:= {};
    misc::maprec(ff2,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x)/I)}
                                  end_if;
                                  elem;
                                end_proc);                                          
    exp_arg:= [op(exp_arg)];
    for lambda in exp_arg do  
      if not iszero(sin(lambda*x)) then 
        a:= ode::normal(f2/sin(lambda*x));
        if not has(a,x) and not has(a,z) then 
          cos_pow:= {};
          misc::maprec(f0,
                       {"_power"} = proc(elem)
                                      begin
                                        if op(elem,1) = cos(lambda*x) or 
                                           op(elem,1) = cos(expand(lambda*x/*,optIgnoreAnalyticConstraints*/)) then 
                                          cos_pow:= cos_pow union {op(elem,2)}
                                        end_if;
                                        elem;
                                        end_proc);  
          cos_pow:= [op(cos_pow)];          
          if cos_pow = [] and has(f0,cos(lambda*x)) then 
            cos_pow:= [1];
          end_if;  
          if nops(cos_pow) = 1 then 
            n:= cos_pow[1];
            if traperror((b:= ode::normal(f0/(cos(lambda*x)^n*sin(lambda*x))))) = 0 then 
              if not has(b,x) and not has(b,z) and 
                 not iszero(expand(a,optIgnoreAnalyticConstraints)) and 
                 not iszero(expand(lambda,optIgnoreAnalyticConstraints)) and 
                 ode::odeIszero(f0-b*sin(lambda*x)*cos(lambda*x)^n) and 
                 iszero(expand(f0-b*sin(lambda*x)*cos(lambda*x)^n,optIgnoreAnalyticConstraints)) then 
                L:= ode::PZ_Sec122_Riccati(1, a*b/lambda^2*x^n, 0, 1, x, y, solveOptions={}, odeOptions={});      
                L:= {-lambda/a*evalAt(L[1],x=cos(lambda*x))};
                return(L);
              end_if;  
            end_if;
          end_if;  
        end_if;  
      end_if;  
    end_for;  
  end_if;
  // ------------------
  // look-up for eq. 50  
  // ------------------
  if ode::odeIszero(f1) and iszero(expand(f1,optIgnoreAnalyticConstraints)) then
    exp_arg:= {};
    misc::maprec(ff2,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x)/I)}
                                  end_if;
                                  elem;
                                end_proc);                                          
    exp_arg:= [op(exp_arg)];
    for lambda in exp_arg do  
      if not iszero(cos(lambda*x)) then 
        a:= ode::normal(f2/cos(lambda*x));
        if not has(a,x) and not has(a,z) and ode::odeIszero(f2-a*cos(lambda*x)) and 
           iszero(expand(f2-a*cos(lambda*x),optIgnoreAnalyticConstraints)) then 
          sin_pow:= {};
          misc::maprec(f0,
                       {"_power"} = proc(elem)
                                      begin
                                        if op(elem,1) = sin(lambda*x) or 
                                           op(elem,1) = sin(expand(lambda*x/*,optIgnoreAnalyticConstraints*/)) then 
                                          sin_pow:= sin_pow union {op(elem,2)}
                                        end_if;
                                        elem;
                                        end_proc);  
          sin_pow:= [op(sin_pow)];          
          if sin_pow = [] and has(f0,sin(lambda*x)) then 
            sin_pow:= [1];
          end_if;  
          if nops(sin_pow) = 1 then 
            n:= sin_pow[1];
            if traperror((b:= ode::normal(f0/(sin(lambda*x)^n*cos(lambda*x))))) = 0 then 
              if not has(b,x) and not has(b,z) and 
                 not iszero(expand(a,optIgnoreAnalyticConstraints)) and 
                 not iszero(expand(lambda,optIgnoreAnalyticConstraints)) and 
                 ode::odeIszero(f0-b*cos(lambda*x)*sin(lambda*x)^n) and 
                 iszero(expand(f0-b*cos(lambda*x)*sin(lambda*x)^n,optIgnoreAnalyticConstraints)) then 
                L:= ode::PZ_Sec122_Riccati(1, a*b/lambda^2*x^n, 0, 1, x, y, solveOptions={}, odeOptions={});      
                if L <> FAIL then 
                  L:= {lambda/a*evalAt(L[1],x=sin(lambda*x))};
                  return(L);
                end_if;  
              end_if;  
            end_if;
          end_if;  
        end_if;  
      end_if;  
    end_for;  
  end_if;
  // ------------------
  // look-up for eq. 56  
  // ------------------
  if ode::odeIszero(f1) and ode::odeIszero(f2-1) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f2-1,optIgnoreAnalyticConstraints)) then 
    exp_arg:= {};
    misc::maprec(ff0,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x)/I)}
                                  end_if;
                                  elem;
                                end_proc);                                          
    exp_arg:= [op(exp_arg)];
    for lambda in map(exp_arg,elem -> ode::normal(elem/4)) do  
      if not iszero((1+exp(lambda*x*8*I)-2*exp(lambda*x*4*I))) then 
        if ode::odeIszero(ff0-(4*lambda^2+24*lambda^2*exp(lambda*x*4*I)+
                                     4*lambda^2*exp(lambda*x*8*I))/(1+exp(lambda*x*8*I)-
                                       2*exp(lambda*x*4*I))) and 
           iszero(ode::normal(ff0-(4*lambda^2+24*lambda^2*exp(lambda*x*4*I)+
                                     4*lambda^2*exp(lambda*x*8*I))/(1+exp(lambda*x*8*I)-
                                       2*exp(lambda*x*4*I)))) then 
          y0:= lambda*cot(lambda*x)-lambda*tan(lambda*x);
          return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
        end_if;  
      end_if;  
    end_for;  
  end_if;
  // ------------------
  // look-up for eq. 57  
  // ------------------
  if ode::odeIszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     has(f0,tan) then  
    tan_arg:= {};
    misc::maprec(f0,
                 {"tan"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    tan_arg:= tan_arg union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    tan_arg:= [op(tan_arg)];
    if nops(tan_arg) = 1 then 
      lambda:= tan_arg[1];  
      t:= eval(diff(subs(f0,[tan(lambda*x)^2=z,tan(expand(lambda*x))^2=z]),z));
      if not has(t,x) and not has(t,z) then 
        La:= ode::solveWithoutProperties(z*(lambda-z) = t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
        La:= select(ode::normal(La), _not@has, x);
        if nops(La) > 0 then 
          t:= eval(diff(subs(f0,[cot(lambda*x)^2=z,cot(expand(lambda*x))^2=z]),z));  
          if not has(t,x) and not has(t,z) then 
            Lb:= ode::solveWithoutProperties(z*(lambda-z) = t,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
            Lb:= select(ode::normal(Lb), _not@has, x);
            if nops(Lb) > 0 then           
              for a in La do 
                for b in Lb do 
                  if ode::odeIszero(f0-lambda*a-lambda*b-2*a*b-a*(lambda-a)*tan(lambda*x)^2
                                     -b*(lambda-b)*cot(lambda*x)^2) and 
                     iszero(expand(f0-lambda*a-lambda*b-2*a*b-a*(lambda-a)*tan(lambda*x)^2
                                     -b*(lambda-b)*cot(lambda*x)^2,
                                    optIgnoreAnalyticConstraints)) then 
                    y0:= a*tan(lambda*x)-b*cot(lambda*x);
                    return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
                  end_if;                  
                end_for;  
              end_for;    
            end_if;  
          end_if;  
        end_if;  
      end_if;
    end_if;  
  end_if;   
  // ------------------
  // look-up for eq. 59  
  // ------------------  
  exp_arg:= {};
  misc::maprec(ff2,
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_arg:= exp_arg union {ode::normal(diff(op(elem,1),x)/I)}
                                end_if;
                                elem;
                              end_proc);                                          
  exp_arg:= [op(exp_arg)];
  for lambda in exp_arg do  
    a:= ode::normal(ff1/(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2));
    if not has(a,x) and not iszero((exp(lambda*x*2*I)+1)) and not iszero(cos(lambda*x)) and 
       ode::odeIszero((a*I+ff0-a*exp(lambda*x*2*I)*I+
                             ff0*exp(lambda*x*2*I))/(exp(lambda*x*2*I)+1)) and 
       iszero(ode::normal((a*I+ff0-a*exp(lambda*x*2*I)*I+
                             ff0*exp(lambda*x*2*I))/(exp(lambda*x*2*I)+1))) then 
      y0:= 1/cos(lambda*x);
      return(ode::RiccatiSolutionGenerator(y0,g,f0,f1,f2,x,solveOptions,odeOptions));              
    end_if;
  end_for; 
  
  
  //--------------------

  return(FAIL);
end_proc:





