/*
  =========================================
  LOOK-UP METHODS FOR 3rd ORDER LINEAR ODEs
  =========================================
 
  REFERENCE. Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 
  -----------
  PARAMETERS. 
  -----------

            eq -- expression of the form f3(x)*y'''(x)+f2(x)*y''(x)+
                                         f1(x)*y'(x)+f0(x)*y(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    odeOptions -- options for 'ode' may contain 'hypergeom' for 
                  returning the solution in terms of 'hypergeom'
                  in the most general case

 */
   
ode::lookUp3rdOrderLinear:=
proc(eq, y, x, solveOptions={}, odeOptions={}) 
  local sol, y0, y1, y2, intOptions, optIgnoreAnalyticConstraints, 
        inits, coeffList, y3, eqOrig;
  save MAXEFFORT; 
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(odeOptions,#inits) then 
    inits:= select(odeOptions,has,#inits)[1];
  else 
    inits:= {};
  end_if;  
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  y3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
   
  eqOrig:= eq; // save the original input 
  eq:= subs(eq, [diff(y(x),x,x,x) = y3, diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]); 
  if not has(eq,y3) then 
    return(FAIL)
  end_if;

  sol:= FAIL; 
  coeffList:= [diff(eq,y0),diff(eq,y1),diff(eq,y2),diff(eq,y3)];

  if sol = FAIL then 
    sol:= ode::PZ_Sec_319_1_p_550(coeffList, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL then 
    sol:= ode::lookUpHypergeomOrder3(eqOrig, y, x, solveOptions, odeOptions);
    if sol <> FAIL and inits <> {} and 
       ode::insertInits(sol,x,inits,solveOptions,odeOptions) = FALSE then 
      sol:= FAIL;              
    end_if;
  end_if;
  if sol = FAIL or sol = {} then 
    return({});
  else 
    return(sol);
  end_if;
end_proc:


/*
  Generate a 3rd independent solution from two given indepenent solutions of 
  a 3rd order linear ODE. 

  REFERENCE. Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 
  -----------
  PARAMETERS. 
  -----------

     coeffList -- a list [f0,f1,f2,f3] corresponding to the coefficients of the ODE 
                  f3(x)*y'''(x) + f2(x)*y''(x) + f1(x)y'(x) + f0(x)
       solList -- two independent solutions
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    odeOptions -- options for 'ode' may contain 'hypergeom' for 
                  returning the solution in terms of 'hypergeom'
                  in the most general case

  -------------
  RETURN VALUE.
  -------------

  A third independent solution (as an expression).  

*/

ode::solutionGenerator3rdOrderLinear:= 
proc(coeffList, solList, x, solveOptions={}, odeOptions={}) 
  local res, y1, y2, psi, intOptions;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;
  y1:= solList[1];
  y2:= solList[2];
  psi:= exp(-int(coeffList[3]/coeffList[4],x,intOptions))/(y1*diff(y2,x)-diff(y1,x)*y2)^2;
  res:= y2*int(y1*psi,x,intOptions)-y1*int(y2*psi,x,intOptions);
  return(res);       
end_proc:  
  
/*
  Reduce a 3rd order linear ODE to a 2nd order linear ODE using one solution.   

  REFERENCE. Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 
  -----------
  PARAMETERS. 
  -----------

     coeffList -- a list [f0,f1,f2,f3] corresponding to the coefficients of the ODE 
                  f3(x)*y'''(x) + f2(x)*y''(x) + f1(x)y'(x) + f0(x)
       solList -- two independent solutions
             x -- indepenent variable 
             y -- depenent variable 
  solveOptions -- options for 'solve'
    odeOptions -- options for 'ode' may contain 'hypergeom' for 
                  returning the solution in terms of 'hypergeom'
                  in the most general case

  -------------
  RETURN VALUE.
  -------------

  A (possibly incomplete) set of solutions or 'FAIL'. 

*/

ode::reduce3rdOrderLinear:= 
proc(coeffList, y0, y, x, solveOptions={}, odeOptions={}) 
  local res, f1, f2, f3, csts, redSol, intOptions;
begin
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;
  res:= FAIL;
  f3:= coeffList[4];
  f2:= coeffList[3];
  f1:= coeffList[2]; 
  // We need to compute  
  // 
  //    redODE:= f3*y0*diff(y(x),x,x) + (3*f3*diff(y0,x) + f2*y0)*diff(y(x),x) + 
  //             (3*f3*diff(y0,x,x) + 2*f2*diff(y0,x) + f1*y0)*y(x);
  //    redSol:= solve(ode(redODE,y(x)),op(solveOptions));   
  //
  // In order to save some time we call the linear solver directly such
  // that the non-linear methods which may also be applicable to linear
  // ODEs are not tried out. 
  redSol:= ode::solve_linear([(3*f3*diff(y0,x,x)+2*f2*diff(y0,x)+f1*y0),
                              (3*f3*diff(y0,x)+f2*y0),
                              (f3*y0),
                              0],x,2,solveOptions,odeOptions);
  if type(redSol) = DOM_SET then 
    csts:= [op(freeIndets(redSol) minus (indets([coeffList, y0, y, x])))];
    if nops(csts) = 2 then 
      res:= evalAt(redSol,csts[1]=1,csts[2]=0) union evalAt(redSol,csts[1]=0,csts[2]=1);
      res:= {y0,op(map(res, f -> y0*int(f,x,intOptions)))};
    else 
      res:= FAIL;
    end_if;    
  else    
    return({y0})
  end_if;  
  
  return(res);       
end_proc:  
  

/*
  ----------
  REFERENCE. Implementation of (Section 3.1.9 pp. 550) 
               from   
               
             Polyanin & Zaitsev: Handbook of exact solutions for 
                                 ordinary differential equations, 
                                 Chapman & Hall/CRC, 2nd edition, 
                                 2003 
  
  -----------
  PARAMETERS. 
  -----------
     coeffList -- a list [f0,f1,f2,f3] corresponding to the coefficients of the ODE 
                  f3(x)*y'''(x) + f2(x)*y''(x) + f1(x)y'(x) + f0(x)
             y -- dependent variable
             x -- indepenent variable 
  solveOptions -- options for 'solve'
    pdeOptions -- options for 'ode' may contain 'hypergeom' for 
                  returning the solution in terms of 'hypergeom'
                  in the most general case

  -------------
  RETURN VALUE.
  -------------

  A (possibly incomplete) set of solutions or 'FAIL'. 

*/

ode::PZ_Sec_319_1_p_550:= proc(coeffList,y,x,solveOptions={},odeOptions={})
  local optIgnoreAnalyticConstraints, intOptions, f, y0, f0, f1, f2, f3, z, 
        L, aa, a, xpow, n, tmp, exp_pow, lambda, b, csts, y1, y2, eq, c, g,
        n1, n2, bb, cc, mu, i;
  begin   
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;
  z:= solvelib::getIdent(Any, indets([coeffList,y,x]));
  f0:= coeffList[1];
  f1:= coeffList[2];
  f2:= coeffList[3];
  f3:= coeffList[4];
  // ---------------------------
  // look-up for eq. 3
  // ---------------------------
  if ode::odeIszero(f1) and ode::odeIszero(f2) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f2,optIgnoreAnalyticConstraints)) then  
    f:= f3;
    if ode::odeIszero(diff(f,x,x,x)+f0) and 
       iszero(expand(diff(f,x,x,x)+f0,optIgnoreAnalyticConstraints)) then 
      y0:= f;
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_if;  
    // ---------------------------
    // look-up for eq. 4
    // ---------------------------
    if ode::odeIszero(diff(f,x,x,x)-f0) and 
       iszero(expand(diff(f,x,x,x)-f0,optIgnoreAnalyticConstraints)) then 
      return(solve(ode(f*diff(y(x),x,x)-diff(f,x)*diff(y(x),x)+diff(f,x,x)*y(x)+genident("C"),
                       y(x)),op(solveOptions)));
    end_if;  
  end_if;
  // ---------------------------
  // look-up for eq. 67
  // --------------------------- 
  a:= diff(f3,x); 
  if not has(a,x) and not iszero(a) then 
    f:= ode::normal(f0/(2*a*(a*x-1)));
    if ode::odeIszero(f2-x*((a*x-2)*f-a^2)) and ode::odeIszero(f1-((2-a^2*x^2)*f+a^2)) and 
       iszero(expand(f2-x*((a*x-2)*f-a^2),optIgnoreAnalyticConstraints)) and 
       iszero(expand(f1-((2-a^2*x^2)*f+a^2),optIgnoreAnalyticConstraints)) then
      y0:= x^2;
      y1:= exp(a*x);
      y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
      return({y0,y1,y2})
    end_if;  
  end_if;  
  // ode::normalization 
  if not iszero(expand(coeffList[4]-1,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= 1;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;  
  if ode::odeIszero(f2) and iszero(expand(f2,optIgnoreAnalyticConstraints)) then 
    f:= f1;
    // ---------------------------
    // look-up for eq. 5
    // ---------------------------
    L:= ode::solveWithoutProperties(-(z*f+z^3)=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do 
      y0:= exp(a*x);
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_for;
    // ---------------------------
    // look-up for eq. 6
    // ---------------------------
    L:= ode::solveWithoutProperties(z*x*(f+z^2*x^2-3*z)=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do 
      y0:= exp(-1/2*a*x^2);
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_for;
    // ---------------------------
    // look-up for eq. 8
    // ---------------------------
    f:= ode::normal(-1/2*f0);
    if ode::odeIszero(f1-x*f) and iszero(expand(f1-x*f,optIgnoreAnalyticConstraints)) then 
      y0:= x^2;
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_if;  
    // ---------------------------
    // look-up for eq. 9
    // ---------------------------
    if not iszero(expand(f0,optIgnoreAnalyticConstraints)) then 
      y0:= ode::normal(-f1/f0);
      if ode::odeIszero(diff(y0-x,x)) and 
         iszero(expand(diff(y0-x,x),optIgnoreAnalyticConstraints)) then 
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      end_if;  
      // ---------------------------
      // look-up for eq. 10
      // ---------------------------
      y0:= ode::normal(-2*f1/f0);
      if ode::odeIszero(diff(y0-x,x)) and 
         iszero(expand(diff(y0-x,x),optIgnoreAnalyticConstraints)) then 
        return(ode::reduce3rdOrderLinear(coeffList, y0^2, y, x, solveOptions, odeOptions));
      end_if;  
      // ---------------------------
      // look-up for eq. 12
      // ---------------------------
      xpow:= {};
      misc::maprec(combine(f1,optIgnoreAnalyticConstraints),
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = x then 
                                      xpow:= xpow union {op(elem,2)}
                                    end_if;
                                    elem;
                                  end_proc);  
      xpow:= [op(xpow)];
      if nops(xpow) = 1 and xpow[1] <> -2 then 
        n:= ode::normal(xpow[1]/2);
        tmp:= diff(subs(f1,[x^(2*n)=z,x^(expand(2*n))=z]),z);
        if ode::odeIszero(diff(tmp,x)) and ode::odeIszero(diff(tmp,z)) and 
           iszero(expand(diff(tmp,x),optIgnoreAnalyticConstraints)) and 
           iszero(expand(diff(tmp,z),optIgnoreAnalyticConstraints)) then 
          tmp:= combine(sqrt(-tmp),IgnoreAnalyticConstraints);
          for a in [tmp,-tmp] do 
            f:= f1+a^2*x^(2*n);
            if ode::odeIszero(f0+a*(x^n*f+3*a*n*x^(2*n-1)+n*(n-1)*x^(n-2))) and 
               iszero(expand(f0+a*(x^n*f+3*a*n*x^(2*n-1)+n*(n-1)*x^(n-2)),optIgnoreAnalyticConstraints)) then
              y0:= exp(a/(n+1)*x^(n+1));
              return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
            end_if;  
          end_for;  
        end_if;   
      end_if;  
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 83
  // ---------------------------  
  if not has(f1,x) and not has(f0,x) then 
    lambda:= combine((f0/2)^(1/3),IgnoreAnalyticConstraints);  
    if not has(lambda,x) and ode::odeIszero(f1+3*lambda^2) and 
       iszero(expand(f1+3*lambda^2,optIgnoreAnalyticConstraints)) then 
      a:= ode::normal(f2/exp(lambda*x));
      if not has(a,x) then 
        return(solve(ode(exp(-lambda*x)*diff(y(x),x)+(a+2*lambda*exp(-lambda*x))*y(x)+
                           genident("C")*x+genident("C"),
                       y(x)),op(solveOptions)))  
      end_if;  
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 84,85
  // ---------------------------  
  exp_pow:= {};
  misc::maprec(combine(f1,exp,optIgnoreAnalyticConstraints),
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  if nops(exp_pow) = 1 then 
    a:= exp_pow[1];
    // ---------------------------
    // look-up for eq. 85
    // ---------------------------  
    f:= ode::normal(-f0/(2*a^3));
    if not iszero(f) then 
      b:= combine(ode::normal((f2/f-2*a)/(exp(a*x))),exp,optIgnoreAnalyticConstraints);
      if not has(b,x) and ode::odeIszero(f1+a*(b*exp(a*x)*f+a)) and 
         iszero(expand(f1+a*(b*exp(a*x)*f+a),optIgnoreAnalyticConstraints)) then 
        y0:= exp(a*x);
        y1:= exp(-a*x)+b/a;
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;  
    end_if;  
    // ---------------------------
    // look-up for eq. 84
    // ---------------------------  
    g:= ode::normal(f0/a);
    f:= f2-a;
    b:= combine(ode::normal((ode::normal((f1-a*f)/g)-1)/exp(a*x)),exp,optIgnoreAnalyticConstraints);
    if not has(b,x) then 
      y0:= exp(-a*x)+b;
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_if;
  end_if; 
  exp_pow:= {}; 
  tmp:= combine(f2,exp,optIgnoreAnalyticConstraints);
  misc::maprec(tmp,
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  if nops(exp_pow) = 1 then 
    // ---------------------------
    // look-up for eq. 86
    // ---------------------------  
    lambda:= exp_pow[1];
    a:= ode::normal(-diff(subs(tmp,[exp(lambda*x)=z,exp(expand(lambda*x))=z]),z)/2);
    if not has(a,x) and not has(a,z) then 
      f:= f2+2*a*exp(lambda*x);
      if ode::odeIszero(f1+a*exp(lambda*x)*(2*f-a*exp(lambda*x)+3*lambda)) and 
         ode::odeIszero(f0-a*exp(lambda*x)*((a*exp(lambda*x)-lambda)*f+
                       2*a*lambda*exp(lambda*x)-lambda^2)) and 
         iszero(expand(f1+a*exp(lambda*x)*(2*f-a*exp(lambda*x)+3*lambda),
                       optIgnoreAnalyticConstraints)) and 
         iszero(expand(f0-a*exp(lambda*x)*((a*exp(lambda*x)-lambda)*f+
                       2*a*lambda*exp(lambda*x)-lambda^2),
                       optIgnoreAnalyticConstraints)) then
        y0:= exp(a/lambda*exp(lambda*x));
        y1:= x*exp(a/lambda*exp(lambda*x));
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;
    end_if;
    // ---------------------------
    // look-up for eq. 87
    // ---------------------------  
    a:= -diff(subs(tmp,[exp(lambda*x)=z,exp(expand(lambda*x))=z]),z);
    if not has(a,x) and not has(a,z) then 
      f:= combine(f2+a*exp(lambda*x),exp);
      g:= ode::normal(combine(f1+2*a*lambda*exp(lambda*x),exp));
      if ode::odeIszero(f0+a*exp(lambda*x)*((a*exp(lambda*x)+lambda)*f+g+lambda^2)) and 
         iszero(ode::normal(combine(expand(f0+a*exp(lambda*x)*((a*exp(lambda*x)+lambda)*f+g+lambda^2),
                                      optIgnoreAnalyticConstraints),exp))) then 
        y0:= exp(a/lambda*exp(lambda*x));
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      end_if;  
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 89
  // ---------------------------  
  if has(coeffList[4],exp) and has(coeffList[1],exp) then 
    exp_pow:= {};
    misc::maprec(combine(coeffList[4],exp,optIgnoreAnalyticConstraints),
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    if nops(exp_pow) = 1 then 
      lambda:= exp_pow[1];
      exp_pow:= {};
      misc::maprec(combine(coeffList[1],exp,optIgnoreAnalyticConstraints),
                   {"exp"} = proc(elem)
                                  begin
                                    if has(elem,x) and 
                                       iszero(expand(diff(op(elem,1),x,x),
                                                     optIgnoreAnalyticConstraints)) then 
                                      exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                    end_if;
                                    elem;
                                  end_proc);  
      if not iszero(ode::normal(coeffList[4]-exp(lambda*x))) then 
        tmp:= ode::normal(coeffList[4]/exp(lambda*x));
        f0:= ode::normal(coeffList[1]/tmp);  
        f1:= ode::normal(coeffList[2]/tmp);
        f2:= ode::normal(coeffList[3]/tmp);
      else 
        f0:= coeffList[1];  
        f1:= coeffList[2];
        f2:= coeffList[3];
      end_if;  
      if nops(exp_pow) = 1 then 
        mu:= exp_pow[1];
        bb:= ode::normal(f0/(mu^2*exp(mu*x)));
        if not has(bb,x) then 
          if ode::odeIszero(f1-lambda^2*exp(lambda*x)-2*bb*mu*exp(mu*x)) and 
             iszero(ode::normal(f1-lambda^2*exp(lambda*x)-2*bb*mu*exp(mu*x))) then 
            cc:= combine(ode::normal(f2-2*lambda*exp(lambda*x)-bb*exp(mu*x)),
                         exp,optIgnoreAnalyticConstraints);
            if not has(cc,x) then 
              return(solve(ode(exp(lambda*x)*diff(y(x),x)+(bb*exp(mu*x)+cc)*y(x)+
                               genident("C")*x+genident("C"),y(x)),op(solveOptions)));  
            end_if;  
          end_if;
        end_if;  
      end_if;   
    end_if;    
  end_if;
  // ---------------------------
  // look-up for eq. 90
  // ---------------------------    
  // We have 'tanh(lambda*x) = (exp(2*lambda*x) - 1)/(exp(2*lambda*x) + 1)' and 
  // 'coth(lambda*x) = (exp(2*lambda*x) + 1)/(exp(2*lambda*x) - 1)'. 
  // Create ode::normal form rewriting to 'exp'. 
  //
  // NOTE. Do not change the order of the methods enclosed by the next 'if'-statement.
  //       The methods have been well-arranged.  
  tmp:= combine(rewrite(f0,exp),exp,optIgnoreAnalyticConstraints); 
  if has(tmp,exp) then 
    exp_pow:= {};
    misc::maprec(tmp,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    for i from 1 to nops(exp_pow) do  
      lambda:= exp_pow[i]/2;
      // ---------------------------
      // look-up for eq. 91
      // ---------------------------    
      if ode::odeIszero(f1+lambda*(2*f2*tanh(lambda*x)+3*lambda)) and 
         ode::odeIszero(f0-lambda^2*((2*tanh(lambda*x)^2-1)*f2+2*lambda*tanh(lambda*x))) and 
         iszero(expand(ode::normal(rewrite(f1+lambda*(2*f2*tanh(lambda*x)+3*lambda),exp)))) and 
         iszero(expand(ode::normal(rewrite(f0-lambda^2*((2*tanh(lambda*x)^2-1)*f2+
                                      2*lambda*tanh(lambda*x)),exp)))) then  
        if has(lambda,I) and not has(ode::normal(lambda/I),I) then 
          y0:= rewrite(cosh(lambda*x),cos);
          y1:= rewrite(x*cosh(lambda*x),cos);
        else 
          y0:= cosh(lambda*x);
          y1:= x*cosh(lambda*x);
        end_if;  
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;
      // ---------------------------
      // look-up for eq. 92
      // ---------------------------    
      if ode::odeIszero(f1+lambda*(2*f2*coth(lambda*x)+3*lambda)) and 
         ode::odeIszero(f0-lambda^2*((2*coth(lambda*x)^2-1)*f2+2*lambda*coth(lambda*x))) and 
         iszero(expand(ode::normal(rewrite(f1+lambda*(2*f2*coth(lambda*x)+3*lambda),exp)))) and 
         iszero(expand(ode::normal(rewrite(f0-lambda^2*((2*coth(lambda*x)^2-1)*f2+
                                      2*lambda*coth(lambda*x)),exp)))) then  
        if has(lambda,I) and not has(ode::normal(lambda/I),I) then 
          y0:= ode::normal(rewrite(sinh(lambda*x),sin)/I);
          y1:= ode::normal(rewrite(x*sinh(lambda*x),sin)/I);
        else 
        y0:= sinh(lambda*x);
        y1:= x*sinh(lambda*x);
        end_if;  
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;
      // ---------------------------
      // look-up for eq. 94
      // ---------------------------    
      if ode::odeIszero(f0+lambda*(lambda*f2+coth(lambda*x)*(f1+lambda^2))) and 
         iszero(expand(ode::normal(rewrite(f0+lambda*(lambda*f2+coth(lambda*x)*(f1+lambda^2)),exp)))) then 
        if has(f2,coth) then 
          f:= diff(subs(f2,coth(x)=z),z);
          if not has(f,z) then 
            L:= ode::solveWithoutProperties((coth(x)-z)*f-z = f2,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
            L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
            for a in L do 
              if ode::odeIszero(f1-((a^2-1)*f-1)) and ode::odeIszero(f0-a*((1-a*coth(x))*f+1)) and 
                 iszero(expand(ode::normal(f1-((a^2-1)*f-1)),optIgnoreAnalyticConstraints)) and 
                 iszero(expand(ode::normal(f0-a*((1-a*coth(x))*f+1)),optIgnoreAnalyticConstraints)) then 
                y0:= exp(a*x);
                y1:= sinh(x);
                y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
                return({y0,y1,y2})
              end_if;  
            end_for;
          end_if;  
        end_if;  
      end_if;  
      // ---------------------------
      // look-up for eq. 90
      // ---------------------------   
      if ode::odeIszero(f0+lambda*(lambda*f2+tanh(lambda*x)*(f1+lambda^2))) and 
         iszero(expand(ode::normal(rewrite(f0+lambda*(lambda*f2+tanh(lambda*x)*(f1+lambda^2)),exp)))) then 
        // --------------------------
        // look-up for eq. 93
        // --------------------------
        if has(f2,tanh) then 
          f:= diff(subs(f2,tanh(x)=z),z);
          if not has(f,z) and not iszero(f+1) then 
            L:= {-(f2-f*tanh(x))/(f+1)}; //solve((tanh(x)-z)*f-z = f2,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
            L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
            for a in L do 
              if ode::odeIszero(f1-((a^2-1)*f-1)) and ode::odeIszero(f0-a*((1-a*tanh(x))*f+1)) and 
                 iszero(expand(ode::normal(f1-((a^2-1)*f-1)),optIgnoreAnalyticConstraints)) and 
                 iszero(expand(ode::normal(f0-a*((1-a*tanh(x))*f+1)),optIgnoreAnalyticConstraints)) then 
                y0:= exp(a*x);
                y1:= cosh(x);
                y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
                return({y0,y1,y2})
              end_if;  
            end_for;
          end_if;  
        end_if; 
        // --------------------------
        // look-up for eq. 101
        // --------------------------
        if has(f2,tan) then 
          f:= -diff(subs(f2,tan(x)=z),z);
          if not has(f,z) and not iszero(f+1) then 
            L:= {-(f2+f*tan(x))/(f+1)};
            L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
            for a in L do
              if ode::odeIszero(f1-((a^2+1)*f+1)) and ode::odeIszero(f0-a*((a*tan(x)-1)*f-1)) and 
                 iszero(expand(ode::normal(f1-((a^2+1)*f+1)),optIgnoreAnalyticConstraints)) and 
                 iszero(expand(ode::normal(f0-a*((a*tan(x)-1)*f-1)),optIgnoreAnalyticConstraints)) then 
                y0:= exp(a*x);
                y1:= cos(x);
                y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
                return({y0,y1,y2})
              end_if;  
            end_for;
          end_if;  
        end_if; 
        // --------------------------
        // look-up for eq. 107
        // --------------------------
        y1:= FAIL;
        L:= ode::solveWithoutProperties(z^3+f2*z^2+f1*z+f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
        if type(L) = DOM_SET then 
          L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
          if nops(L) > 0 then 
            y1:= exp(L[1]*x);
          end_if;  
        end_if; 
        // ---------------------------
        if has(lambda,I) and not has(ode::normal(lambda/I),I) then 
          y0:= rewrite(cosh(lambda*x),cos);
        else 
          y0:= cosh(lambda*x);
        end_if;  
        if y1 = FAIL then 
          return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, 
                                           solveOptions, odeOptions));                                       
        else                                    
          y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
          return({y0,y1,y2})
        end_if;  
      end_if;  
      // --------------------------
      // look-up for eq. 102
      // --------------------------
      if has(f2,cot) then 
        f:= diff(subs(f2,cot(x)=z),z);
        if not has(f,z) and not iszero(f+1) then 
          L:= {(f2-f*cot(x))/(f+1)};
          L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
          for a in L do 
            if ode::odeIszero(f1-((a^2+1)*f+1)) and ode::odeIszero(f0-a*((1-a*cot(x))*f+1)) and 
               iszero(expand(ode::normal(f1-((a^2+1)*f+1)),optIgnoreAnalyticConstraints)) and 
               iszero(expand(ode::normal(f0-a*((1-a*cot(x))*f+1)),optIgnoreAnalyticConstraints)) then 
              y0:= exp(-a*x);
              y1:= sin(x);
              y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
              return({y0,y1,y2})
            end_if;  
          end_for;
        end_if;  
      end_if; 
    end_for;  
    // ---------------------------
    // look-up for eq. 98
    // ---------------------------
    /* Matched by other patterns. 
    f:= f2;
    g:= f1;
    for lambda in exp_pow do 
      l:= ode::normal(lambda/(2*I));
      if iszero(expand(f0-l*(l*f+tan(l*x)*(g-l^2)),optIgnoreAnalyticConstraints)) then 
        y0:= cos(l*x);
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      end_if;  
    end_for;  
    */
  end_if;  
  // ---------------------------
  // look-up for eq. 95,96
  // ---------------------------
  tmp:= combine(rewrite(f2,exp),exp,optIgnoreAnalyticConstraints); 
  if has(tmp,exp) then 
    exp_pow:= {};
    misc::maprec(tmp,
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    for i from 1 to nops(exp_pow) do  
      lambda:= exp_pow[i]/2;
      f:= ode::normal(f0/lambda^2);
      if ode::odeIszero(f2-(lambda*tanh(lambda*x)*(x*f-1)-f)) and 
         ode::odeIszero(f1+(lambda^2*x*f)) and 
         iszero(expand(f2-(lambda*tanh(lambda*x)*(x*f-1)-f),optIgnoreAnalyticConstraints)) and 
         iszero(expand(f1+(lambda^2*x*f),optIgnoreAnalyticConstraints)) then 
        // ---------------------------
        // look-up for eq. 95
        // ---------------------------         
        if has(lambda,I) and not has(ode::normal(lambda/I),I) then  
          y0:= x;
          y1:= rewrite(cosh(lambda*x),cos);
          y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        else   
          y0:= x;
          y1:= cosh(lambda*x);
          y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        end_if;  
        return({y0,y1,y2})
      end_if;   
      if ode::odeIszero(f2-(lambda*coth(lambda*x)*(x*f-1)-f)) and ode::odeIszero(f1+(lambda^2*x*f)) and 
         iszero(expand(f2-(lambda*coth(lambda*x)*(x*f-1)-f),optIgnoreAnalyticConstraints)) and 
         iszero(expand(f1+(lambda^2*x*f),optIgnoreAnalyticConstraints)) then 
        // ---------------------------
        // look-up for eq. 96
        // ---------------------------         
        if has(lambda,I) and not has(ode::normal(lambda/I),I) then 
          y0:= x; 
          y1:= ode::normal(rewrite(sinh(lambda*x),sin)/I);
          y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        else 
          y0:= x; 
          y1:= sinh(lambda*x);
          y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        end_if;  
        return({y0,y1,y2})
      end_if; 
      /* Matched by other patterns. 
      f:= -f;
      lambda:= ode::normal(lambda/I);
      if iszero(expand(f2-(f-lambda*cot(lambda*x)*(x*f+1)),optIgnoreAnalyticConstraints)) and 
         iszero(expand(f1+(lambda^2*x*f),optIgnoreAnalyticConstraints)) and 
         iszero(expand(f0-(lambda^2*f),optIgnoreAnalyticConstraints)) then 
        // ---------------------------
        // look-up for eq. 104
        // ---------------------------         
        y0:= x; 
        y1:= sin(lambda*x);
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;   
      */
    end_for;
  end_if;  
  // ---------------------------
  // look-up for eq. 108
  // ---------------------------
  f:= f2; 
  g:= f1;
  if ode::odeIszero(f0-f*g-diff(g,x)) and iszero(expand(f0-f*g-diff(g,x),optIgnoreAnalyticConstraints)) then 
    tmp:= diff(y(x),x,x)+g*y(x)+genident("C")*exp(-int(f,x,intOptions));
    if not hastype(tmp,piecewise) then 
      L:= solve(ode(tmp,y(x)),op(solveOptions));
      if type(L) = DOM_SET then 
        csts:= [op(freeIndets(L) minus (freeIndets(f1)))];
        if nops(csts) = 3  then 
          L:= evalAt(L,csts[1]=1,csts[2]=0,csts[3]=0) union evalAt(L,csts[1]=0,csts[2]=1,csts[3]=0) union 
              evalAt(L,csts[1]=0,csts[2]=0,csts[3]=1);
          return(L);  
        end_if;  
      end_if;          
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 109
  // ---------------------------
  f:= ode::normal(f2/3); 
  g:= ode::normal((f1-diff(f,x)-2*f^2)/2);  
  if ode::odeIszero(f0-2*f*g-diff(g,x)) and 
     iszero(expand(f0-2*f*g-diff(g,x),optIgnoreAnalyticConstraints)) then  
    tmp:= diff(y(x),x,x)+f*diff(y(x),x)+1/2*g*y(x); 
    if not hastype(tmp,piecewise) then 
      L:= solve(ode(tmp,y(x)),op(solveOptions));
      if type(L) = DOM_SET then 
        csts:= [op(freeIndets(L) minus (freeIndets(f1)))];
        if nops(csts) = 2  then 
          L:= evalAt(L,csts[1]=1,csts[2]=0) union evalAt(L,csts[1]=0,csts[2]=1);
          return({L[1]^2, L[1]*L[2], L[2]^2});  
        end_if;  
      end_if;  
    end_if;  
  end_if;  
  // ode::normalization 
  if not iszero(expand(ode::normal(coeffList[4]/x)-1,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(x*coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(x*coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(x*coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= x;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;  
  // ---------------------------
  // look-up for eq. 13
  // ---------------------------
  if ode::odeIszero(f2) and 
     iszero(expand(f2,optIgnoreAnalyticConstraints)) then 
    f:= ode::normal(f1/x);
    L:= ode::solveWithoutProperties(-((z*x+1)*f+z^3*x+3*z^2)=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do 
      y0:= x*exp(a*x);
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions
      ));
    end_for;
  end_if;
  // ode::normalization 
  if not iszero(expand(ode::normal(coeffList[4]/x^2)-1,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(x^2*coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(x^2*coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(x^2*coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= x^2;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;  
  // ---------------------------
  // look-up for eq. 14
  // ---------------------------
  if ode::odeIszero(f2) and iszero(expand(f2,optIgnoreAnalyticConstraints)) then 
    L:= ode::solveWithoutProperties(f1-(z*f1-x*f0)=z^3-z,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = DOM_SET then 
      L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
      L:= L minus {1};
      for a in L do 
        f:= ode::normal(f0/(a-1)); 
        if ode::odeIszero(f1-(x*f-a^2-a)) and ode::odeIszero(f0-(a-1)*f) and 
           iszero(expand(f1-(x*f-a^2-a),optIgnoreAnalyticConstraints)) and 
           iszero(expand(f0-(a-1)*f,optIgnoreAnalyticConstraints)) then 
          y0:= x^(1-a);
          return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
        end_if;  
      end_for;
    end_if;
    // ---------------------------
    // look-up for eq. 15
    // ---------------------------
    f:= f0;
    if not iszero(expand(f,optIgnoreAnalyticConstraints)) then 
      a:= ode::normal(((f1+6)/(x*f)-1)/x);
      if not has(a,x) then 
        y0:= a+1/x;
        return({y0});
      end_if;  
    end_if;
  end_if;  
  // ode::normalization 
  if not iszero(expand(ode::normal(coeffList[4]/x^2)-1,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(x^6*coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(x^6*coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(x^6*coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= x^6;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;  
  // ---------------------------
  // look-up for eq. 18
  // ---------------------------
  if ode::odeIszero(f2) and iszero(expand(f2,optIgnoreAnalyticConstraints)) then 
    f:= ode::normal(f1/x^2);
    L:= ode::solveWithoutProperties(z^3+z*f-2*x*f=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = DOM_SET then 
      L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
      for a in L do 
        y0:= x^2*exp(a/x);
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      end_for;  
    end_if;  
  end_if;
  // ode::normalization 
  if not iszero(expand(coeffList[4]-1,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= 1;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;    
  // ---------------------------
  // look-up for eq. 19
  // --------------------------- 
  if ode::odeIszero(f2) and iszero(expand(f2,optIgnoreAnalyticConstraints)) then 
    exp_pow:= {};
    misc::maprec(combine(f1,exp,optIgnoreAnalyticConstraints),
                 {"exp"} = proc(elem)
                                begin
                                  if has(elem,x) and 
                                     iszero(expand(diff(op(elem,1),x,x),
                                                   optIgnoreAnalyticConstraints)) then 
                                    exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                  end_if;
                                  elem;
                                end_proc);  
    exp_pow:= [op(exp_pow)];
    if nops(exp_pow) = 1 then 
      lambda:= ode::normal(exp_pow[1]/2);
      aa:= combine(sqrt(-diff(subs(f1,[exp(2*lambda*x)=z,exp(expand(2*lambda*x))=z]),z)),
                   IgnoreAnalyticConstraints);
      if not has(aa,x) then 
        for a in [-aa,aa] do 
          f:= f1+a^2*exp(2*lambda*x);
          if ode::odeIszero(f0+a*exp(lambda*x)*(f+3*a*lambda*exp(lambda*x)+lambda^2)) and 
             iszero(expand(f0+a*exp(lambda*x)*(f+3*a*lambda*exp(lambda*x)+lambda^2), 
                           optIgnoreAnalyticConstraints)) then 
            y0:= exp(a/lambda*exp(lambda*x));               
            return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
          end_if;                 
        end_for;  
      end_if;  
      /* Matched by other patterns. 
      // ---------------------------
      // look-up for eq. 20
      // --------------------------- 
      a:= exp_pow[1];
      f:= f0/a;
      b:= ode::normal(combine(((f1+a^2)/f-1)/exp(a*x),optIgnoreAnalyticConstraints));
      if not has(b,x) then 
        y0:= exp(-a*x)+b;
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      end_if;  
      */
    end_if;  
    // ---------------------------
    // look-up for eq. 23
    // --------------------------- 
    f:= f1; 
    if iszero(expand(f0-tan(x)*(f-1),optIgnoreAnalyticConstraints)) or 
       iszero(ode::normal(rewrite(f0-tan(x)*(f-1),exp))) then 
      y0:= cos(x);
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_if;  
    // ---------------------------
    // look-up for eq. 24
    // --------------------------- 
    f:= f1;
    if iszero(expand(f0-cot(x)*(1-f),optIgnoreAnalyticConstraints)) or 
       iszero(ode::normal(rewrite(f0-cot(x)*(1-f),exp),optIgnoreAnalyticConstraints)) then 
      y0:= sin(x);
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
    end_if;  
    /* Matched by other patterns. 
    // ---------------------------
    // look-up for eq. 25
    // --------------------------- 
    if iszero(expand(diff(f1,x)-f0,optIgnoreAnalyticConstraints)) then 
      return(solve(ode(diff(y(x),x,x)+f1*y(x)+genident("C"),
                       y(x)),op(solveOptions)))  
    end_if;  
    */
    /* Matched by other patterns. 
    // ---------------------------
    // look-up for eq. 26
    // --------------------------- 
    if iszero(expand(diff(f1/2,x)-f0,optIgnoreAnalyticConstraints)) then 
      L:= solve(ode(2*diff(y(x),x,x)+f1/2*y(x),y(x)),op(solveOptions));
      if type(L) = DOM_SET then 
        csts:= [op(freeIndets(L) minus (freeIndets(f1)))];
        if nops(csts) = 2  then 
          L:= evalAt(L,csts[1]=1,csts[2]=0) union evalAt(L,csts[1]=0,csts[2]=1);
          return({L[1]^2, L[1]*L[2], L[2]^2});  
        end_if;  
      end_if;  
    end_if;  
    */
  end_if;
  // ---------------------------
  // look-up for eq. 33
  // --------------------------- 
  if ode::odeIszero(f1) and 
     iszero(expand(f1,optIgnoreAnalyticConstraints)) then 
    f:= f2;
    L:= ode::solveWithoutProperties(-z^2*(f+z)=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = DOM_SET then 
      L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
      for a in L do 
        y0:= exp(a*x);
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      end_for;  
    end_if;  
  end_if;
  // ---------------------------
  // look-up for eq. 34
  // --------------------------- 
  f:= f2;
  a:= f1;
  if not iszero(a) and not has(a,x) and ode::odeIszero(f0-a*f) and 
     iszero(expand(f0-a*f,optIgnoreAnalyticConstraints)) then 
    if is(a<0) = TRUE then 
      y0:= exp(-x*sqrt(-a));
      y1:= exp(x*sqrt(-a));
    else 
      y0:= cos(x*sqrt(a));
      y1:= sin(x*sqrt(a));
    end_if;  
    y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
    return({y0,y1,y2})
  end_if;  
  // ---------------------------
  // look-up for eq. 37
  // --------------------------- 
  f:= f2;
  if not iszero(f) then 
    a:= ode::normal(f1/f);
    if not iszero(a) and not has(a,x) and ode::odeIszero(f0-a^2*(f-a)) and 
       iszero(expand(f0-a^2*(f-a),optIgnoreAnalyticConstraints)) then 
      y0:= exp(-1/2*a*x)*cos(sqrt(3)/2*a*x);
      y1:= exp(-1/2*a*x)*sin(sqrt(3)/2*a*x);
      y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
      return({y0,y1,y2})
    end_if;  
  end_if;
  // ---------------------------
  // look-up for eq. 39
  // --------------------------- 
  L:= ode::solveWithoutProperties(-z*(2*f+3*z)=f1,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  if type(L) = DOM_SET then 
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    if type(L) = DOM_SET then 
      for a in L do 
        if ode::odeIszero(f0-a^2*(f+2*a)) and 
           iszero(expand(f0-a^2*(f+2*a),optIgnoreAnalyticConstraints)) then 
          y0:= exp(a*x);
          y1:= x*exp(a*x);
          y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
          return({y0,y1,y2})
        end_if;
      end_for;  
    end_if;  
  end_if;   
  // ---------------------------------
  // look-up for eq. 41,42,43,44,45,46
  // ---------------------------------
  eq:= diff(y(x),x,x,x)+f2*diff(y(x),x,x)+f1*diff(y(x),x)+f0*y(x);
  L:= ode::solveWithoutProperties(eq | y(x)=exp(z*x),z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  if type(L) = DOM_SET then 
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    if nops(L) > 0 then  
      if nops(L) = 1 then 
        a:= L[1];
        y0:= exp(a*x);
        return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));
      elif nops(L) = 2 then 
        y0:= exp(L[1]*x);
        y1:= exp(L[2]*x);
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;  
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 47
  // --------------------------- 
  f:= ode::normal(f2/x);
  a:= ode::normal((f1+f)/x^2);
  if not has(a,x) and ode::odeIszero(f0-a*x*(x^2*f+3)) and 
     iszero(expand(f0-a*x*(x^2*f+3),optIgnoreAnalyticConstraints)) then 
    y0:= cos(1/2*x^2*sqrt(a));
    y1:= sin(1/2*x^2*sqrt(a));
    y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
    return({y0,y1,y2})
  end_if;
  // ---------------------------
  // look-up for eq. 48
  // --------------------------- 
  f:= ode::normal(-f0/2);
  if not iszero(f) and ode::odeIszero(f1-x*f) and ode::odeIszero(diff(((tmp:= f2/f)),x,x)) and 
     iszero(expand(f1-x*f,optIgnoreAnalyticConstraints)) and 
     iszero(diff(((tmp:= ode::normal(f2/f))),x,x)) then 
    a:= diff(tmp,x);
    b:= tmp-a*x;
    y0:= x^2+2*a*x+b;
    return({y0});
  end_if;   
  // ---------------------------
  // look-up for eq. 49
  // --------------------------- 
  if not has((a:= ode::normal((f1-x*f0)/2)),x) and not iszero(a) then 
    f:= f0/a;
    if ode::odeIszero(f2-f-a*x) and iszero(expand(f2-f-a*x,optIgnoreAnalyticConstraints)) then 
      y0:= exp(-1/2*a*x^2);
      y1:= exp(-1/2*a*x^2)*(2^(1/2)*PI^(1/2)*erf(x*(-a/2)^(1/2)))/(2*(-a)^(1/2));
      y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
      return({y0,y1,y2})
    end_if;              
  end_if;
  // ---------------------------
  // look-up for eq. 50
  // --------------------------- 
  f:= ode::normal(f0/2);
  if ode::odeIszero(f2-x^2*f) and ode::odeIszero(f1+2*x*f) and 
     iszero(expand(f2-x^2*f,optIgnoreAnalyticConstraints)) and 
     iszero(expand(f1+2*x*f,optIgnoreAnalyticConstraints)) then 
    y0:= x;
    y1:= x^2;
    y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
    return({y0,y1,y2})
  end_if;
  // --------------------------------------
  // look-up for eq. 53 (generalized)
  // --------------------------------------
  if iszero(f1) and not iszero(f0) then 
    b:= ode::normal(-(2*f0*diff(f2,x)-2*f2*diff(f0,x)+2*x*f0^2)/f0^2);
    if not has(b,x) then 
      c:= ode::normal(-(b*f0+2*x*f0+2*diff(f2,x)+x^2*diff(f0,x)+b*x*diff(f0,x))/diff(f0,x));
      if not has(c,x) then 
        y0:= x^2+b*x+c;    
        return({y0});
      end_if;  
   end_if;
  end_if;  
  // ---------------------------
  // look-up for eq. 54
  // --------------------------- 
  f:= -f0/2;
  g:= -f1; 
  if ode::odeIszero(f2-x*(x*f+g)) and iszero(expand(f2-x*(x*f+g),optIgnoreAnalyticConstraints)) then 
    y0:= x^2;
    if not iszero(expand(g-2/x^2,optIgnoreAnalyticConstraints)) then  
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));     
    end_if;
    /* Matched by other patterns. 
    if iszero(expand(g-2/x^2,optIgnoreAnalyticConstraints)) then  
      f:= f2*x;
      if iszero(expand(f0-g/x*(2-f),optIgnoreAnalyticConstraints)) then  
        // ---------------------------
        // look-up for eq. 76
        // --------------------------- 
        y1:= 1/x;
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2});       
      end_if;  
    end_if;  
    */
  end_if;  
  // ---------------------------
  // look-up for eq. 55
  // --------------------------- 
  if not iszero(f0) and ode::odeIszero(diff(ode::normal(f2/f0),x,x,x)) and 
     iszero(expand(diff(ode::normal(f2/f0),x,x,x),optIgnoreAnalyticConstraints)) then 
    a:= ode::normal(2*(-evalAt(ode::normal(diff(f2/f0,x)),x=0)-f1/f0));
    if not has(a,x) and not iszero(a) then 
      f:= f0/(2*a);
      if not iszero(f) then 
        b:= ode::normal(f2/(-x*f))-a*x; 
        if not has(b,x) and ode::odeIszero(f1-(b-a^2)*f) and 
           iszero(expand(f1-(b-a^2)*f,optIgnoreAnalyticConstraints)) then 
          y0:= x^2+a*x+1/2*(a^2-b);
          return({y0});
        end_if;  
      end_if;  
    end_if;  
  end_if;
  // ode::normalization 
  if not iszero(expand(coeffList[4]-x,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(x*coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(x*coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(x*coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= x;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;    
  // ---------------------------
  // look-up for eq. 57
  // --------------------------- 
  if f2 = 3 and traperror((a:= ode::normal((f1-f0*x)/(f0*x^3+f1*x^2)))) = 0 
     and not has(a,x) then 
    y0:= a*x+1/x;
    return({y0});
  end_if;  
  // ---------------------------
  // look-up for eq. 58
  // --------------------------- 
  if not has(f0,x) then 
    a:= ode::normal(f0/2);
    b:= expand(f2-a*x^2/*,optIgnoreAnalyticConstraints*/);
    if not has(b,x) and ode::odeIszero(f1-4*a*x) and 
       iszero(expand(f1-4*a*x,optIgnoreAnalyticConstraints)) then 
      return(solve(ode(x*diff(y(x),x)+(a*x^2+b-2)*y(x)+genident("C")*x+genident("C"),y(x)),
                   op(solveOptions)));    
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 60
  // --------------------------- 
  f:= ode::normal((f2-3)/x);
  a:= ode::normal((f1-2*f)/x);
  if ode::odeIszero(f0-a*(x*f+1)) and iszero(expand(f0-a*(x*f+1),optIgnoreAnalyticConstraints)) then 
    y0:= cos(x*sqrt(a))/x;
    y1:= sin(x*sqrt(a))/x;
    y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
    return({y0,y1,y2})
  end_if;
  // ---------------------------
  // look-up for eq. 61
  // --------------------------- 
  if not iszero(f) then 
    a:= ode::normal((f1/f-2)/x);
    if not has(a,x) and ode::odeIszero(f0-a*(a*x*f+f-a^2*x)) and 
       iszero(expand(f0-a*(a*x*f+f-a^2*x),optIgnoreAnalyticConstraints)) then 
      y0:= 1/x*exp(-1/2*a*x)*cos(sqrt(3)/2*a*x);
      y1:= 1/x*exp(-1/2*a*x)*sin(sqrt(3)/2*a*x);
      y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
      return({y0,y1,y2})
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 62
  // --------------------------- 
  L:= ode::solveWithoutProperties(z*(f-z)=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  if type(L) = DOM_SET then 
    L:= select(map(L, elem -> ode::normal(elem)),_not@has, x);
    for a in L do 
      if ode::odeIszero(f1-a*x*f-2*f+a^2*x) and 
         iszero(expand(f1-a*x*f-2*f+a^2*x,optIgnoreAnalyticConstraints)) then 
        y0:= 1/x;
        y1:= 1/x*exp(-a*x);
        y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
        return({y0,y1,y2})
      end_if;  
    end_for;  
  end_if;    
  // ---------------------------
  // look-up for eq. 65
  // --------------------------- 
  if iszero(f1) and not iszero(f0) then 
    f:= ode::normal(-f0/2);
    a:= ode::normal((ode::normal((f2-3)/(x^2*f))-1)/x^2);
    if not has(a,x) then 
      y0:= a*x+1/x;
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));     
    end_if;  
  end_if; 
  // ---------------------------
  // look-up for eq. 97
  // --------------------------- 
  f:= -f0:
  if ode::odeIszero(f1-x*f) and 
     iszero(expand(f1-x*f,optIgnoreAnalyticConstraints)) and
     not has((a:= expand(ode::normal((f2-2)/(x^2*f))+ln(x),optIgnoreAnalyticConstraints)),x) then 
    y0:= x;
    y1:= ln(x)-a+1;
    y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
    return({y0,y1,y2})
  end_if;
  
  // ode::normalization 
  if not iszero(expand(coeffList[4]-x^3,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= 1;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;    
  // ---------------------------
  // look-up for eq. 68,69
  // --------------------------- 
  if not iszero(f0) then 
    a:= ode::normal(-(f0*x^3-f1*x^2+2*f2*x-6)/(f0*x^4));
    if not has(a,x) then
      y0:= a+1/x;
      return({y0})
    end_if;
  end_if;    
  // ode::normalization 
  if not iszero(expand(coeffList[4]-x^3,optIgnoreAnalyticConstraints)) then 
    f0:= combine(ode::normal(x^3*coeffList[1]/coeffList[4]),optIgnoreAnalyticConstraints);
    f1:= combine(ode::normal(x^3*coeffList[2]/coeffList[4]),optIgnoreAnalyticConstraints);
    f2:= combine(ode::normal(x^3*coeffList[3]/coeffList[4]),optIgnoreAnalyticConstraints);
    f3:= x^3;
  else
    f0:= coeffList[1];
    f1:= coeffList[2];
    f2:= coeffList[3];
    f3:= coeffList[4];
  end_if;    
  // ---------------------------
  // look-up for eq. 72
  // --------------------------- 
  c:= f0;
  if not has(c,x) then 
    if ode::odeIszero(diff(f2,x,x,x)) and 
       iszero(expand(diff(f2,x,x,x),optIgnoreAnalyticConstraints)) then 
      a:= ode::normal(diff(f2,x,x)/2);
      if ode::odeIszero(f2-a*x^2) and ode::odeIszero(diff(f1,x,x)) and 
         iszero(expand(f2-a*x^2,optIgnoreAnalyticConstraints)) and 
         iszero(expand(diff(f1,x,x),optIgnoreAnalyticConstraints)) then 
        b:= diff(f1,x);
        if ode::odeIszero(f1-b*x) and iszero(expand(f1-b*x,optIgnoreAnalyticConstraints)) then 
          L:= solve(ode(diff(y(z),z,z,z)+(a-3)*diff(y(z),z,z)+(b-a+2)*diff(y(z),z)+c*y(z),y(z)),
                    op(solveOptions));    
          if traperror((L:= evalAt(L,z=ln(x)))) = 0 then 
            return(L);  
          end_if;  
        end_if;         
      end_if;  
    end_if;
  end_if;  
  // ---------------------------
  // look-up for eq. 73
  // --------------------------- 
  f:= ode::normal(f1/x);
  if not iszero(f) then 
    a:= ode::normal(f0/f);
    if not has(a,x) and ode::odeIszero(f2-(a+2)*x^2) and  
       iszero(expand(f2-(a+2)*x^2,optIgnoreAnalyticConstraints)) then 
      y0:= 1/x^a;
      return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));     
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 74
  // --------------------------- 
  a:= ode::normal(f0/2);
  if not has(a,x) then 
    b:= expand(f2-(a+6)*x^2/*,optIgnoreAnalyticConstraints*/);
    if not has(b,x) and ode::odeIszero(f1-2*(2*a+3)*x) and 
       iszero(expand(f1-2*(2*a+3)*x,optIgnoreAnalyticConstraints)) then 
      return(solve(ode(x^3*diff(y(x),x)+(a*x^2+b)*y(x)+genident("C")+genident("C")*x,y(x)),
                   op(solveOptions)));        
    end_if;  
  end_if;  
  // ---------------------------
  // look-up for eq. 75
  // --------------------------- 
  if not has(f0,x) and iszero(f1) then 
    L:= ode::solveWithoutProperties(2*z*(z+1)*(2*z+1)=f0,z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = DOM_SET then 
      for a in L do 
        tmp:= f2+3*a*x^2;
        b:= ode::normal((ode::normal(f2/x^2)+3*a)/x^(2*a+1));
        if not has(b,x) then 
          return(solve(ode(x^(-2*a)*diff(y(x),x)+(a*x^(-2*a-1)+b)*y(x)+genident("C")+genident("C")*x,y(x)),
                       op(solveOptions)));
        end_if;  
      end_for;  
    end_if;  
  end_if;
  /* Matched by other patterns. 
  // ---------------------------
  // look-up for eq. 77
  // --------------------------- 
  if iszero(expand(f1+6*x,optIgnoreAnalyticConstraints)) then 
    f:= ode::normal(f2/x^2);
    if iszero(expand(f0-6*(2-f),optIgnoreAnalyticConstraints)) then 
      y0:= 1/x^2;
      y1:= x^3;
      y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
      return({y0,y1,y2})
    end_if;  
  end_if;
  */
  // ---------------------------
  // look-up for eq. 78
  // --------------------------- 
  f:= ode::normal(f2/x^2);
  if ode::odeIszero(f1-x*(f-1)) and ode::odeIszero(f0-(f-3)) and 
     iszero(expand(f1-x*(f-1),optIgnoreAnalyticConstraints)) and 
     iszero(expand(f0-(f-3),optIgnoreAnalyticConstraints)) then 
    y0:= cos(ln(x));
    y1:= sin(ln(x));
    y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
    return({y0,y1,y2})
  end_if;   
  // ---------------------------
  // look-up for eq. 79
  // --------------------------- 
  f:= ode::normal(f2/x^2)-1;
  if not iszero(f-2) then 
    a:= ode::normal(-f0/(f-2));
    if not has(a,x) and ode::odeIszero(f1-x*(f-a-1)) and 
       iszero(expand(f1-x*(f-a-1),optIgnoreAnalyticConstraints)) then 
      y0:= x^(-sqrt(a));
      y1:= x^(sqrt(a));
      y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
      return({y0,y1,y2})
    end_if;  
  end_if;
  // ---------------------------
  // look-up for eq. 80
  // --------------------------- 
  L:= ode::solveWithoutProperties(z^3-(f0*x^2+2*f1*x+2*f2)/x^2-(z^2*(3*x^2+f2))/x^2+(z*(2*x^2+f1*x+3*f2))/x^2,
            z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  if type(L) = DOM_SET then 
    for f in L do 
      a:= ode::normal(f2/x^2-f);
      if not has(a,x) then 
        b:= ode::normal((f2+f1*x)/x^2+f^2-(f*(x^2+f2))/x^2);
        if not has(b,x) then 
          n1:= (a^2-2*a-4*b + 1)^(1/2)/2-a/2+1/2;
          n2:= 1/2-(a^2-2*a-4*b+1)^(1/2)/2-a/2;
          if not iszero(n1-n2) then 
            y0:= x^n1;
            y1:= x^n2;
            y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
            return({y0,y1,y2})
          else 
            y0:= x^n1;
            return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));     
          end_if;  
        end_if;  
      end_if;  
    end_for;  
  end_if;  
  // ---------------------------
  // look-up for eq. 81
  // ---------------------------   
  if not iszero(f0) then
    L:= ode::solveWithoutProperties(z^3-f0^2-(z^2*(2*x^2+f1*x-f2))/x^2+(z*(f0*f2-3*f0*x^2))/x^2,
              z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
    if type(L) = DOM_SET then 
      for g in L do 
        a:= ode::normal((x^2*g^2-2*x^2*g-f0*x^2-f1*x*g+f2*g+f0*f2)/(f0*x^2));
        if not has(a,x) then
          y0:= x^(2-a);
          return({y0});
        end_if;  
      end_for;
    end_if;
  end_if;
  // ---------------------------
  // look-up for eq. 82
  // ---------------------------   
  L:= ode::solveWithoutProperties(z^3-(-f2^2+2*f0*x^4+4*f1*x^3)/(2*x^4)-(z^2*(3*x^4+2*f2*x^2))/(2*x^4)+
                (z*(2*f1*x^3+2*f2*x^2))/(2*x^4),
            z,IgnoreSpecialCases,optIgnoreAnalyticConstraints);
  if type(L) = DOM_SET then 
    for f in L do 
      a:= ode::normal((f2-x^2*f)/(2*x^3));
      if not has(a,x) then 
        b:= ode::normal((-f2^2-2*f2*x^2*f+3*x^4*f^2+4*f1*x^3)/(4*x^4));
        if not has(b,x) then 
          n1:= (1-4*b)^(1/2)/2+1/2;
          n2:= 1/2-(1-4*b)^(1/2)/2;
          if not iszero(n1-n2) then 
            y0:= exp(-a*x)*x^n1;
            y1:= exp(-a*x)*x^n2;
            y2:= ode::solutionGenerator3rdOrderLinear(coeffList,[y0,y1],x,solveOptions,odeOptions);
            return({y0,y1,y2})
          /* Matched by other patterns.   
          else 
            y0:= exp(-a*x)*x^n1;
            return(ode::reduce3rdOrderLinear(coeffList, y0, y, x, solveOptions, odeOptions));     
          */
          end_if;  
        end_if;  
      end_if;  
    end_for;
  end_if;
  // ode::normalization 
  f0:= combine(rewrite(coeffList[1],exp),exp,optIgnoreAnalyticConstraints);
  f1:= combine(rewrite(coeffList[2],exp),exp,optIgnoreAnalyticConstraints);
  f2:= combine(rewrite(coeffList[3],exp),exp,optIgnoreAnalyticConstraints);
  f3:= combine(rewrite(coeffList[4],exp),exp,optIgnoreAnalyticConstraints);
  exp_pow:= {};
  misc::maprec(combine(expand(f3),exp),
               {"exp"} = proc(elem)
                              begin
                                if has(elem,x) and 
                                   iszero(expand(diff(op(elem,1),x,x),
                                                 optIgnoreAnalyticConstraints)) then 
                                  exp_pow:= exp_pow union {diff(op(elem,1),x)}
                                end_if;
                                elem;
                              end_proc);  
  exp_pow:= [op(exp_pow)];
  // ---------------------------
  // look-up for eq. 105
  // ---------------------------   
  b:= ode::normal(f2);
  if not has(b,x) then 
    for i from 1 to nops(exp_pow) do 
      lambda:= ode::normal(ode::normal(exp_pow[i]/I));
      a:= ode::normal(f3/(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2));
      if not has(a,x) and 
         ode::odeIszero(f1-(3*a*lambda^2*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))) and 
         ode::odeIszero(f0-(2*a*lambda^3*((1/exp(lambda*x*I))/2 + exp(lambda*x*I)/2))) and 
         iszero(expand(ode::normal(f1-(3*a*lambda^2*(((1/exp(lambda*x*I))*I)/2-(exp(lambda*x*I)*I)/2))),
                       optIgnoreAnalyticConstraints)) and 
         iszero(expand(ode::normal(f0-(2*a*lambda^3*((1/exp(lambda*x*I))/2 + exp(lambda*x*I)/2))),
                       optIgnoreAnalyticConstraints)) then               
        return(solve(ode(a*sin(lambda*x)*diff(y(x),x)+(b-2*a*lambda*cos(lambda*x))*y(x)+
                           genident("C")*x+genident("C"),
                         y(x)),op(solveOptions)))  
      end_if;  
    end_for;
  end_if;    
       
  
  return(FAIL);
end_proc:


