/*
  REFERENCE. 
   [1] Kai Gehrs: Symbolic algorithms for the computation of    
       hypergeometric solutions of linear ODEs, internal paper, 
       2009 
  
  NOTE that especially all references within the code below refer 
  to this paper. The names for parameters, variables and the 
  numbering of equations, identities and facts are chosen due to 
  this work. The comments in this code are possibly not detailed 
  enough to understand the algorithmic ideas. 

   
  PARAMETERS.
   eq -- an expression encoding a 2nd order homogeneous linear ODE
    x -- the independent ODE variable 
    y -- the dependent ODE variable 

   The (standard) options are as usual. 


  OUTPUT. 
   The methods returns solutions of the 3rd order homogeneous 
   linear ODE in terms of hypergeometric functions or 'FAIL' 
   if no such solution could be found.       
*/

ode::lookUpHypergeomOrder3:= proc(eq,y,x,solveOptions={},odeOptions={})
  local intOptions, optIgnoreAnalyticConstraints, y0, y1, y2, y3, lcoeff, 
        I1, J1, xpow, d, n, kTilde, k, I0, numerI0, denomI0, L1, L2, a, 
        aa, b, bb, c, cc, dd, ee, ff, sol1, sol2, sys, fundSys, Sol_eq0F2, 
        i, j, t, A_bar, A_tilde, MoebiusTrans, eqI0, tmp, ind, m, unknownFunctions,
        Subs, J2, eqnF2, eq0F2, specVal, DenomI0Unknowns, DenomI1Unknowns, 
        NumerI0Unknowns, NumerI1Unknowns, Moebius, c0, c1, c2, denomI0Unknowns, 
        denomI1, denomI1Unknowns, numerI0Unknowns, numerI1, numerI1Unknowns, 
        II0, II1, c0_eqHypergeom, c1_eqHypergeom, c2_eqHypergeom, cnt, 
        sysSpecial, fundSys_complete, Sol_eq0F2_replacement, z, Cand;
  save MAXEFFORT;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  y0:= solvelib::getIdent(Any, indets([eq,y,x]));
  y1:= solvelib::getIdent(Any, indets([eq,y,x,y0]));
  y2:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1]));
  y3:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  
  aa:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2]));
  bb:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa]));
  cc:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb]));
  dd:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc]));
  ee:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd]));
  ff:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee]));
   a:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff]));
   b:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a]));
   c:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a,b]));
   t:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a,b,c]));
   eq:= subs(eq, [diff(y(x),x,x,x) = y3, diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  if not has(eq,y3) or (not has(eq,y0) and not has(eq,y1) and not has(eq,y2)) then 
    return(FAIL)
  end_if;
  unassume(x);
  lcoeff:= diff(eq,y3);
  c2:= combine(ode::normal(diff(eq,y2)),optIgnoreAnalyticConstraints);
  c1:= combine(ode::normal(diff(eq,y1)),optIgnoreAnalyticConstraints);
  c0:= combine(ode::normal(diff(eq,y0)),optIgnoreAnalyticConstraints);
  // The algorithm cannot treat ODEs where 'c0, 'c1', 'c2' involve arbitrary 
  // functions like 'f(x)' for a non-specified 'f'. We need to check this first: 
  unknownFunctions:= indets([c0,c1,c2,lcoeff], All) minus indets([c0,c1,c2,lcoeff]);
  unknownFunctions:= select(unknownFunctions, v -> domtype(eval(v)) = DOM_IDENT);
  if unknownFunctions <> {} then 
    return(FAIL);
  end_if;  
  // We consider ODEs of the from (3.1), i.e. if 'lcoeff <> 1' we need to 
  // divide by it.  
  if lcoeff <> 1 then 
    c0:= ode::normal(c0/lcoeff);
    c1:= ode::normal(c1/lcoeff);
    c2:= ode::normal(c2/lcoeff);
    lcoeff:= 1;
  end_if;    
  //------------------------------------------------------------------------------  
  // [1], Step (1)
  //------------------------------------------------------------------------------  
  I1:= ode::normal(diff(c2,x)+c2^2/3-c1); // compute invariant [1], (3.5)
  I0:= ode::normal(diff(c2,x,x)/3-2*c2^3/27+c1*c2/3-c0); // compute invariant [1], (3.4)
  //------------------------------------------------------------------------------  
  // [1], Step (2)
  //------------------------------------------------------------------------------  
  J1:= ode::normal(x^2*I1+1); // shifted invariant due to [1], (3.6)
  J2:= ode::normal(x^3*I0+x^2*I1); // shifted invariant due to [1], (3.7)
  //------------------------------------------------------------------------------  
  // [1], Step (3)
  //------------------------------------------------------------------------------  
  xpow:= {}:
  misc::maprec(combine(J1,IgnoreAnalyticConstraints),
               {"_power"} = proc(elem)
                              begin
                                if op(elem,1) = x then 
                                  xpow:= xpow union {op(elem,2)}
                                end_if;
                                elem;
                              end_proc):  
  // If 'xpow = {}' then we try to use 'J1' to deduce a suitable candidate.  
  if nops(xpow) = 0 then 
    if has(J1,x) then 
      xpow:= {1};
    else
      misc::maprec(combine(J2,IgnoreAnalyticConstraints),
                   {"_power"} = proc(elem)
                                  begin
                                    if op(elem,1) = x then 
                                      xpow:= xpow union {op(elem,2)}
                                    end_if;
                                    elem;
                                  end_proc):  
      if nops(xpow) = 0 then 
        if has(J2,x) then 
          xpow:= {1};
        else  
          return(FAIL);
        end_if;  
      end_if;  
    end_if;  
  end_if;
  d:= lcm(op(map(xpow,denom)));
  xpow:= map(xpow,_mult,d);
  n:= gcd(op(xpow));
  kTilde:= d/n;
  if kTilde <> 1 then 
    Cand:= [1/kTilde,1];
  else 
    Cand:= [1];
  end_if;  
  for k in Cand do  
    //------------------------------------------------------------------------------  
    // [1], Step (4)
    //------------------------------------------------------------------------------  
    // Compute the transformed invariant [1], (3.10)
    II1:= diff(x^(1/k),x)^2*evalAt(I1,x=x^(1/k))-2*(1-(1/k)^2)/(2*x^2); 
    // Compute the transformed invariant [1], (3.11)
    II0:= diff(x^(1/k),x)*diff(x^(1/k),x,x)*evalAt(I1,x=x^(1/k))+
          diff(x^(1/k),x)^3*evalAt(I0,x=x^(1/k))-diff((1-(1/k)^2)/(2*x^2),x);
    numerI0:= numer(II0,Expand=TRUE);
    denomI0:= denom(II0,Expand=TRUE);
    numerI1:= numer(II1,Expand=TRUE);
    denomI1:= denom(II1,Expand=TRUE);
    //------------------------------------------------------------------------------  
    // [1], Step (5)
    //------------------------------------------------------------------------------  
    numerI0:= poly(numerI0,[x]); 
    denomI0:= poly(denomI0,[x]);
    numerI1:= poly(numerI1,[x]);
    denomI1:= poly(denomI1,[x]);
    /*
       Now compute the general form of the invariant of the hypergeometric equation 
       after application of the Moebius transform with general parameters. 
    */
    d:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a,b,c,t]));  
    Moebius:= (a*x+b)/(c*x+d);
    fundSys:= {};
    cnt:= 0;
     
    eqnF2:= [[(aa+bb+1)/x, aa*bb/x^2, -1/x^2], // coefficients from 0F2-equation 
             [(bb+cc+1)/x, -(x-bb*cc)/x^2, -aa/x^2], // coefficients from 1F2-equation 
             [-(x-cc-dd-1)/x, -((aa+bb+1)*x-cc*dd)/x^2, -aa*bb/x^2], // coefficients from 2F2-equation 
             [-(dd+ee+1-(aa+bb+cc+3)*x)/(x*(x-1)), -(ee*dd-((bb+cc+1)*aa+(bb+1)*(cc+1))*x)/(x^2*(x-1)), 
              aa*bb*cc/(x^2*(x-1))] // coefficients from 3F2-equation 
            ];
            
    Sol_eq0F2:= [
           // solutions of 0F2-equation
           {hypergeom([],[aa, bb],x),
            x^(-aa+1)*hypergeom([],[2-aa, 1-aa+bb],x),
            x^(-bb+1)*hypergeom([],[2-bb, 1-bb+aa],x)},
           // solutions of 1F2-equation  
           {hypergeom([aa],[bb, cc],x),
            x^(1-cc)*hypergeom([aa+1-cc],[2-cc, 1-cc+bb],x),
            x^(1-bb)*hypergeom([aa+1-bb],[2-bb, 1-bb+cc],x)}, 
           // solutions of 2F2-equation
           {hypergeom([aa, bb],[cc, dd],x),
            x^(1-dd)*hypergeom([bb+1-dd, aa+1-dd],[2-dd, 1-dd+cc],x),
            x^(1-cc)*hypergeom([aa+1-cc, bb+1-cc],[2-cc, 1-cc+dd],x)}, 
           // solutions of 3F2-equation 
           {x^(-bb)*hypergeom([bb, 1-dd+bb, 1-ee+bb],[1-aa+bb, 1-cc+bb],1/x),
            x^(-aa)*hypergeom([aa, 1-dd+aa, 1-ee+aa],[1-bb+aa, 1-cc+aa],1/x),
            x^(-cc)*hypergeom([cc, 1-ee+cc, 1-dd+cc],[1-bb+cc, 1-aa+cc],1/x)}    ];
  
    // replacements for degenerate cases to fill up the set of solutions     
    Sol_eq0F2_replacement:= [
           // solutions of 0F2-equation
           {meijerG(2,0,[],[0,1-aa,1-bb],x),
            meijerG(3,0,[],[0,1-aa,1-bb],-x),
            meijerG(2,0,[],[1-aa,1-bb,0],x)},
           // solutions of 1F2-equation  
           {meijerG(2,1,[1-aa],[0,1-bb,1-cc],x),
            meijerG(3,1,[1-aa],[0,1-cc,1-bb],-x),
            meijerG(2,1,[1-aa],[1-cc,1-bb,0],x)}, 
           // solutions of 2F2-equation
           {meijerG(2,2,[1-bb,1-aa],[0,1-cc,1-dd],x),
            meijerG(3,2,[1-aa,1-bb],[0,1-cc,1-dd],-x),
            meijerG(2,2,[1-aa,1-bb],[1-cc,1-dd,0],x)}, 
           // solutions of 3F2-equation 
           {meijerG(2,3,[1-bb,1-aa,1-cc],[0,1-dd,1-ee],x),
            meijerG(3,3,[1-bb,1-aa,1-cc],[0,1-dd,1-ee],-x),
            meijerG(2,3,[1-bb,1-aa,1-cc],[1-dd,1-ee,0],x)}    ];        
  
    // explicitly defining the following expressions and not generating them automatically 
    // gives a significant speed-up (12.5% faster for a particular set of tests)
            
    DenomI1Unknowns:= table(4 = poly((3*a^2*c^4 - 6*a^3*c^3 + 3*a^4*c^2)*x^6 + (- 18*a^2*b*c^3 + 
          12*a^3*b*c^2 + 12*a^2*c^3*d - 18*a^3*c^2*d + 6*a*b*c^4 + 6*a^4*c*d)*x^5 + (3*a^4*d^2 + 
          3*b^2*c^4 - 18*a*b^2*c^3 - 18*a^3*c*d^2 + 18*a^2*b^2*c^2 + 18*a^2*c^2*d^2 + 24*a*b*c^3*d + 
          24*a^3*b*c*d - 54*a^2*b*c^2*d)*x^4 + (- 6*a^3*d^3 - 6*b^3*c^3 + 12*a*b^3*c^2 + 12*a^3*b*d^2 + 
          12*a^2*c*d^3 + 12*b^2*c^3*d + 36*a*b*c^2*d^2 - 54*a*b^2*c^2*d - 54*a^2*b*c*d^2 + 
          36*a^2*b^2*c*d)*x^3 + (3*a^2*d^4 + 3*b^4*c^2 - 18*a^2*b*d^3 - 18*b^3*c^2*d + 18*a^2*b^2*d^2 + 
          18*b^2*c^2*d^2 + 24*a*b*c*d^3 + 24*a*b^3*c*d - 54*a*b^2*c*d^2)*x^2 + (- 18*a*b^2*d^3 + 
          12*a*b^3*d^2 + 12*b^2*c*d^3 - 18*b^3*c*d^2 + 6*a*b*d^4 + 6*b^4*c*d)*x + (3*b^2*d^4 - 
          6*b^3*d^3 + 3*b^4*d^2), [x]), 3 = poly((3*a^2*c^4)*x^6 + (12*a^2*c^3*d + 6*a*b*c^4)*x^5 + 
          (3*b^2*c^4 + 18*a^2*c^2*d^2 + 24*a*b*c^3*d)*x^4 + (12*a^2*c*d^3 + 12*b^2*c^3*d + 
          36*a*b*c^2*d^2)*x^3 + (3*a^2*d^4 + 18*b^2*c^2*d^2 + 24*a*b*c*d^3)*x^2 + (12*b^2*c*d^3 + 
          6*a*b*d^4)*x + 3*b^2*d^4, [x]), 2 = poly((3*a^2*c^3)*x^5 + (9*a^2*c^2*d + 6*a*b*c^3)*x^4 + 
          (3*b^2*c^3 + 9*a^2*c*d^2 + 18*a*b*c^2*d)*x^3 + (3*a^2*d^3 + 9*b^2*c^2*d + 18*a*b*c*d^2)*x^2 + 
          (9*b^2*c*d^2 + 6*a*b*d^3)*x + 3*b^2*d^3, [x]), 1 = poly((3*a^2*c^2)*x^4 + (6*a*b*c^2 + 
          6*a^2*c*d)*x^3 + (3*a^2*d^2 + 3*b^2*c^2 + 12*a*b*c*d)*x^2 + (6*a*b*d^2 + 6*b^2*c*d)*x + 
          3*b^2*d^2, [x]));
    
    NumerI1Unknowns:= table(4 = poly((- 3*a^4*d^2 - 2*b^2*c^4 + 3*a*b^2*c^3 + 3*a^3*c*d^2 - 
          b^2*c^4*dd - b^2*c^4*ee - 3*a^2*b^2*c^2 + a^4*aa^2*d^2 + a^4*bb^2*d^2 - 2*a^2*c^2*d^2 + 
          a^4*cc^2*d^2 + b^2*c^4*dd^2 + b^2*c^4*ee^2 - a^2*c^2*d^2*dd - a^2*c^2*d^2*ee + 4*a*b*c^3*d + 
          6*a^3*b*c*d + a^2*aa^2*b^2*c^2 + a^2*b^2*bb^2*c^2 + a^2*b^2*c^2*cc^2 + a^2*c^2*d^2*dd^2 + 
          a^2*c^2*d^2*ee^2 + a*aa*b^2*c^3 + a*b^2*bb*c^3 - a^4*aa*bb*d^2 + a^3*aa*c*d^2 + 
          a*b^2*c^3*cc - a^4*aa*cc*d^2 - 6*a^2*b*c^2*d + a^3*bb*c*d^2 - a^4*bb*cc*d^2 + 
          a^3*c*cc*d^2 - b^2*c^4*dd*ee + 3*a*aa*b^2*bb*c^3 + 3*a*aa*b^2*c^3*cc - 2*a^2*aa*b*c^2*d - 
          2*a^3*aa^2*b*c*d + 3*a^3*aa*bb*c*d^2 - 2*a*aa*b^2*c^3*dd + 3*a*b^2*bb*c^3*cc - 
          2*a^2*b*bb*c^2*d - 2*a^3*b*bb^2*c*d + 3*a^3*aa*c*cc*d^2 - 2*a*aa*b^2*c^3*ee - 
          2*a*b^2*bb*c^3*dd - 2*a^2*b*c^2*cc*d - 2*a^3*b*c*cc^2*d - 2*a^3*aa*c*d^2*dd + 
          3*a^3*bb*c*cc*d^2 - 2*a*b^2*bb*c^3*ee - 2*a*b^2*c^3*cc*dd - 2*a*b*c^3*d*dd^2 - 
          2*a^3*aa*c*d^2*ee - 2*a^3*bb*c*d^2*dd - 2*a*b^2*c^3*cc*ee - 2*a*b*c^3*d*ee^2 - 
          2*a^3*bb*c*d^2*ee - 2*a^3*c*cc*d^2*dd + 3*a*b^2*c^3*dd*ee - 2*a^3*c*cc*d^2*ee + 
          3*a^3*c*d^2*dd*ee - a^2*aa*b^2*bb*c^2 - a^2*aa*b^2*c^2*cc - a^2*b^2*bb*c^2*cc - 
          a^2*c^2*d^2*dd*ee + 2*a*b*c^3*d*dd + 2*a*b*c^3*d*ee + 2*a^3*aa*b*bb*c*d + 
          2*a^3*aa*b*c*cc*d + 2*a^3*b*bb*c*cc*d + 2*a*b*c^3*d*dd*ee - 6*a^2*aa*b*bb*c^2*d - 
          6*a^2*aa*b*c^2*cc*d + 4*a^2*aa*b*c^2*d*dd - 6*a^2*b*bb*c^2*cc*d + 4*a^2*aa*b*c^2*d*ee + 
          4*a^2*b*bb*c^2*d*dd + 4*a^2*b*bb*c^2*d*ee + 4*a^2*b*c^2*cc*d*dd + 4*a^2*b*c^2*cc*d*ee - 
          6*a^2*b*c^2*d*dd*ee)*x^2 + (3*a^3*d^3 + 3*b^3*c^3 - 6*a*b^3*c^2 + aa*b^3*c^3 + 
          a^3*aa*d^3 - 6*a^3*b*d^2 + a^3*bb*d^3 + b^3*bb*c^3 - 4*a^2*c*d^3 + a^3*cc*d^3 + 
          b^3*c^3*cc - 4*b^2*c^3*d + 2*a*aa^2*b^3*c^2 + 2*a^3*aa^2*b*d^2 + 2*a*b^3*bb^2*c^2 + 
          2*a^3*b*bb^2*d^2 + 2*a*b^3*c^2*cc^2 + 2*a^3*b*cc^2*d^2 + 2*a^2*c*d^3*dd^2 + 
          2*b^2*c^3*d*dd^2 + 2*a^2*c*d^3*ee^2 + 2*b^2*c^3*d*ee^2 + 3*aa*b^3*bb*c^3 + 
          3*a^3*aa*bb*d^3 + 3*aa*b^3*c^3*cc + 3*a^3*aa*cc*d^3 + 8*a*b*c^2*d^2 - 3*a*b^2*c^2*d - 
          3*a^2*b*c*d^2 + 12*a^2*b^2*c*d - 2*aa*b^3*c^3*dd - 2*a^3*aa*d^3*dd + 3*a^3*bb*cc*d^3 + 
          3*b^3*bb*c^3*cc - 2*aa*b^3*c^3*ee - 2*a^3*aa*d^3*ee - 2*a^3*bb*d^3*dd - 2*b^3*bb*c^3*dd - 
          2*a^2*c*d^3*dd - 2*a^3*bb*d^3*ee - 2*a^3*cc*d^3*dd - 2*b^3*bb*c^3*ee - 2*b^3*c^3*cc*dd - 
          2*b^2*c^3*d*dd - 2*a^2*c*d^3*ee - 2*a^3*cc*d^3*ee - 2*b^3*c^3*cc*ee - 2*b^2*c^3*d*ee + 
          3*a^3*d^3*dd*ee + 3*b^3*c^3*dd*ee - 2*a*aa*b^3*bb*c^2 - 2*a^3*aa*b*bb*d^2 - 
          2*a*aa*b^3*c^2*cc - a*aa*b^2*c^2*d - a^2*aa*b*c*d^2 - 2*a^3*aa*b*cc*d^2 - 
          2*a*b^3*bb*c^2*cc - a*b^2*bb*c^2*d - a^2*b*bb*c*d^2 - 2*a^3*b*bb*cc*d^2 - 
          a*b^2*c^2*cc*d - a^2*b*c*cc*d^2 + 4*a*b*c^2*d^2*dd + 4*a*b*c^2*d^2*ee - 
          2*a^2*c*d^3*dd*ee - 2*b^2*c^3*d*dd*ee - 4*a^2*aa^2*b^2*c*d - 4*a^2*b^2*bb^2*c*d - 
          4*a^2*b^2*c*cc^2*d - 4*a*b*c^2*d^2*dd^2 - 4*a*b*c^2*d^2*ee^2 - 3*a*aa*b^2*bb*c^2*d - 
          3*a^2*aa*b*bb*c*d^2 + 4*a^2*aa*b^2*bb*c*d - 3*a*aa*b^2*c^2*cc*d - 3*a^2*aa*b*c*cc*d^2 + 
          4*a^2*aa*b^2*c*cc*d + 2*a*aa*b^2*c^2*d*dd - 3*a*b^2*bb*c^2*cc*d + 2*a^2*aa*b*c*d^2*dd - 
          3*a^2*b*bb*c*cc*d^2 + 4*a^2*b^2*bb*c*cc*d + 2*a*aa*b^2*c^2*d*ee + 2*a*b^2*bb*c^2*d*dd + 
          2*a^2*aa*b*c*d^2*ee + 2*a^2*b*bb*c*d^2*dd + 2*a*b^2*bb*c^2*d*ee + 2*a*b^2*c^2*cc*d*dd + 
          2*a^2*b*bb*c*d^2*ee + 2*a^2*b*c*cc*d^2*dd + 2*a*b^2*c^2*cc*d*ee + 2*a^2*b*c*cc*d^2*ee + 
          4*a*b*c^2*d^2*dd*ee - 3*a*b^2*c^2*d*dd*ee - 3*a^2*b*c*d^2*dd*ee)*x + (- 2*a^2*d^4 - 
          3*b^4*c^2 + 3*a^2*b*d^3 + 3*b^3*c^2*d - a^2*d^4*dd - a^2*d^4*ee + aa^2*b^4*c^2 - 
          3*a^2*b^2*d^2 + b^4*bb^2*c^2 + b^4*c^2*cc^2 - 2*b^2*c^2*d^2 + a^2*d^4*dd^2 + 
          a^2*d^4*ee^2 - b^2*c^2*d^2*dd - b^2*c^2*d^2*ee + 4*a*b*c*d^3 + 6*a*b^3*c*d + 
          a^2*aa^2*b^2*d^2 + a^2*b^2*bb^2*d^2 + a^2*b^2*cc^2*d^2 + b^2*c^2*d^2*dd^2 + 
          b^2*c^2*d^2*ee^2 + a^2*aa*b*d^3 - aa*b^4*bb*c^2 + a^2*b*bb*d^3 - aa*b^4*c^2*cc - 
          6*a*b^2*c*d^2 + aa*b^3*c^2*d + a^2*b*cc*d^3 - b^4*bb*c^2*cc + b^3*bb*c^2*d + 
          b^3*c^2*cc*d - a^2*d^4*dd*ee + 3*a^2*aa*b*bb*d^3 - 2*a*aa*b^2*c*d^2 - 2*a*aa^2*b^3*c*d + 
          3*a^2*aa*b*cc*d^3 - 2*a*b^2*bb*c*d^2 - 2*a*b^3*bb^2*c*d + 3*aa*b^3*bb*c^2*d - 
          2*a^2*aa*b*d^3*dd + 3*a^2*b*bb*cc*d^3 - 2*a*b^2*c*cc*d^2 - 2*a*b^3*c*cc^2*d + 
          3*aa*b^3*c^2*cc*d - 2*a^2*aa*b*d^3*ee - 2*a^2*b*bb*d^3*dd - 2*a*b*c*d^3*dd^2 - 
          2*aa*b^3*c^2*d*dd + 3*b^3*bb*c^2*cc*d - 2*a^2*b*bb*d^3*ee - 2*a^2*b*cc*d^3*dd - 
          2*a*b*c*d^3*ee^2 - 2*aa*b^3*c^2*d*ee - 2*b^3*bb*c^2*d*dd - 2*a^2*b*cc*d^3*ee - 
          2*b^3*bb*c^2*d*ee - 2*b^3*c^2*cc*d*dd + 3*a^2*b*d^3*dd*ee - 2*b^3*c^2*cc*d*ee + 
          3*b^3*c^2*d*dd*ee - a^2*aa*b^2*bb*d^2 - a^2*aa*b^2*cc*d^2 - a^2*b^2*bb*cc*d^2 - 
          b^2*c^2*d^2*dd*ee + 2*a*b*c*d^3*dd + 2*a*b*c*d^3*ee + 2*a*aa*b^3*bb*c*d + 
          2*a*aa*b^3*c*cc*d + 2*a*b^3*bb*c*cc*d + 2*a*b*c*d^3*dd*ee - 6*a*aa*b^2*bb*c*d^2 - 
          6*a*aa*b^2*c*cc*d^2 + 4*a*aa*b^2*c*d^2*dd - 6*a*b^2*bb*c*cc*d^2 + 4*a*aa*b^2*c*d^2*ee + 
          4*a*b^2*bb*c*d^2*dd + 4*a*b^2*bb*c*d^2*ee + 4*a*b^2*c*cc*d^2*dd + 4*a*b^2*c*cc*d^2*ee - 
          6*a*b^2*c*d^2*dd*ee), [x]), 3 = poly((a^4*d^2 - 2*b^2*c^4 + a*b^2*c^3 + a^3*c*d^2 - 
          b^2*c^4*cc - b^2*c^4*dd + a^2*b^2*c^2 - 2*a^2*c^2*d^2 + b^2*c^4*cc^2 + b^2*c^4*dd^2 - 
          a^2*c^2*cc*d^2 - a^2*c^2*d^2*dd + 4*a*b*c^3*d - 2*a^3*b*c*d + a^2*c^2*cc^2*d^2 + 
          a^2*c^2*d^2*dd^2 + 3*a*aa*b^2*c^3 + 3*a*b^2*bb*c^3 + 3*a^3*aa*c*d^2 - 2*a*b^2*c^3*cc - 
          2*a^2*b*c^2*d + 3*a^3*bb*c*d^2 - 2*a*b^2*c^3*dd - 2*a^3*c*cc*d^2 - 2*a^3*c*d^2*dd - 
          b^2*c^4*cc*dd - 6*a^2*aa*b*c^2*d - 6*a^2*b*bb*c^2*d + 4*a^2*b*c^2*cc*d - 2*a*b*c^3*cc^2*d + 
          4*a^2*b*c^2*d*dd - 2*a*b*c^3*d*dd^2 - a^2*c^2*cc*d^2*dd + 2*a*b*c^3*cc*d + 2*a*b*c^3*d*dd + 
          2*a*b*c^3*cc*d*dd)*x^2 + (a^3*d^3 + b^3*c^3 + 2*a*b^3*c^2 + 3*aa*b^3*c^3 + 3*a^3*aa*d^3 + 
          2*a^3*b*d^2 + 3*a^3*bb*d^3 + 3*b^3*bb*c^3 - 4*a^2*c*d^3 - 2*a^3*cc*d^3 - 2*b^3*c^3*cc - 
          4*b^2*c^3*d - 2*a^3*d^3*dd - 2*b^3*c^3*dd + 2*a^2*c*cc^2*d^3 + 2*b^2*c^3*cc^2*d + 
          2*a^2*c*d^3*dd^2 + 2*b^2*c^3*d*dd^2 + 8*a*b*c^2*d^2 - a*b^2*c^2*d - a^2*b*c*d^2 - 
          4*a^2*b^2*c*d - 2*a^2*c*cc*d^3 - 2*b^2*c^3*cc*d - 2*a^2*c*d^3*dd - 2*b^2*c^3*d*dd - 
          3*a*aa*b^2*c^2*d - 3*a^2*aa*b*c*d^2 - 3*a*b^2*bb*c^2*d - 3*a^2*b*bb*c*d^2 + 
          4*a*b*c^2*cc*d^2 + 2*a*b^2*c^2*cc*d + 2*a^2*b*c*cc*d^2 + 4*a*b*c^2*d^2*dd + 
          2*a*b^2*c^2*d*dd + 2*a^2*b*c*d^2*dd - 2*a^2*c*cc*d^3*dd - 2*b^2*c^3*cc*d*dd - 
          4*a*b*c^2*cc^2*d^2 - 4*a*b*c^2*d^2*dd^2 + 4*a*b*c^2*cc*d^2*dd)*x + (- 2*a^2*d^4 + 
          b^4*c^2 + a^2*b*d^3 - a^2*cc*d^4 + b^3*c^2*d - a^2*d^4*dd + a^2*b^2*d^2 + a^2*cc^2*d^4 - 
          2*b^2*c^2*d^2 + a^2*d^4*dd^2 - b^2*c^2*cc*d^2 - b^2*c^2*d^2*dd + 4*a*b*c*d^3 - 
          2*a*b^3*c*d + b^2*c^2*cc^2*d^2 + b^2*c^2*d^2*dd^2 + 3*a^2*aa*b*d^3 + 3*a^2*b*bb*d^3 - 
          2*a*b^2*c*d^2 + 3*aa*b^3*c^2*d - 2*a^2*b*cc*d^3 + 3*b^3*bb*c^2*d - 2*a^2*b*d^3*dd - 
          2*b^3*c^2*cc*d - a^2*cc*d^4*dd - 2*b^3*c^2*d*dd - 6*a*aa*b^2*c*d^2 - 6*a*b^2*bb*c*d^2 + 
          4*a*b^2*c*cc*d^2 - 2*a*b*c*cc^2*d^3 + 4*a*b^2*c*d^2*dd - 2*a*b*c*d^3*dd^2 - 
          b^2*c^2*cc*d^2*dd + 2*a*b*c*cc*d^3 + 2*a*b*c*d^3*dd + 2*a*b*c*cc*d^3*dd), [x]), 
          2 = poly((3*a^3*d^2 - 2*b^2*c^3 + 3*a*b^2*c^2 - b^2*bb*c^3 - 2*a^2*c*d^2 - b^2*c^3*cc + 
          b^2*bb^2*c^3 + b^2*c^3*cc^2 + a^2*bb^2*c*d^2 + a^2*c*cc^2*d^2 + 4*a*b*c^2*d - 
          6*a^2*b*c*d - a^2*bb*c*d^2 - b^2*bb*c^3*cc - a^2*c*cc*d^2 - 2*a*b*bb^2*c^2*d - 
          2*a*b*c^2*cc^2*d - a^2*bb*c*cc*d^2 + 2*a*b*bb*c^2*d + 2*a*b*c^2*cc*d + 
          2*a*b*bb*c^2*cc*d)*x + (- 2*a^2*d^3 + 3*b^3*c^2 + 3*a^2*b*d^2 - a^2*bb*d^3 - 
          a^2*cc*d^3 - 2*b^2*c^2*d + a^2*bb^2*d^3 + a^2*cc^2*d^3 + b^2*bb^2*c^2*d + 
          b^2*c^2*cc^2*d + 4*a*b*c*d^2 - 6*a*b^2*c*d - a^2*bb*cc*d^3 - b^2*bb*c^2*d - 
          b^2*c^2*cc*d - 2*a*b*bb^2*c*d^2 - 2*a*b*c*cc^2*d^2 - b^2*bb*c^2*cc*d + 2*a*b*bb*c*d^2 + 
          2*a*b*c*cc*d^2 + 2*a*b*bb*c*cc*d^2), [x]), 1 = poly(- 2*a^2*d^2 - 2*b^2*c^2 - aa*b^2*c^2 - 
          a^2*aa*d^2 - a^2*bb*d^2 - b^2*bb*c^2 + a^2*aa^2*d^2 + aa^2*b^2*c^2 + a^2*bb^2*d^2 + 
          b^2*bb^2*c^2 - aa*b^2*bb*c^2 - a^2*aa*bb*d^2 + 4*a*b*c*d + 2*a*aa*b*c*d + 2*a*b*bb*c*d - 
          2*a*aa^2*b*c*d - 2*a*b*bb^2*c*d + 2*a*aa*b*bb*c*d, [x])); 
          
    DenomI0Unknowns:= table(4 = poly((27*a^3*c^6 - 81*a^4*c^5 + 81*a^5*c^4 - 27*a^6*c^3)*x^9 + 
          (81*a^2*b*c^6 - 324*a^3*b*c^5 + 405*a^4*b*c^4 - 162*a^5*b*c^3 + 162*a^3*c^5*d - 
          405*a^4*c^4*d + 324*a^5*c^3*d - 81*a^6*c^2*d)*x^8 + (81*a*b^2*c^6 - 81*a^6*c*d^2 - 
          486*a^2*b^2*c^5 + 810*a^3*b^2*c^4 - 405*a^4*b^2*c^3 + 405*a^3*c^4*d^2 - 810*a^4*c^3*d^2 + 
          486*a^5*c^2*d^2 + 486*a^2*b*c^5*d - 1620*a^3*b*c^4*d + 1620*a^4*b*c^3*d - 
          486*a^5*b*c^2*d)*x^7 + (- 27*a^6*d^3 + 27*b^3*c^6 - 324*a*b^3*c^5 + 324*a^5*c*d^3 + 
          810*a^2*b^3*c^4 - 540*a^3*b^3*c^3 + 540*a^3*c^3*d^3 - 810*a^4*c^2*d^3 + 1215*a^2*b*c^4*d^2 - 
          2430*a^2*b^2*c^4*d - 3240*a^3*b*c^3*d^2 + 3240*a^3*b^2*c^3*d + 2430*a^4*b*c^2*d^2 - 
          1215*a^4*b^2*c^2*d + 486*a*b^2*c^5*d - 486*a^5*b*c*d^2)*x^6 + (81*a^5*d^4 - 81*b^4*c^5 + 
          405*a*b^4*c^4 - 162*a^5*b*d^3 - 405*a^4*c*d^4 + 162*b^3*c^5*d - 405*a^2*b^4*c^3 + 
          405*a^3*c^2*d^4 + 1215*a*b^2*c^4*d^2 + 1620*a^2*b*c^3*d^3 + 3240*a^2*b^3*c^3*d - 
          3240*a^3*b*c^2*d^3 - 1620*a^3*b^3*c^2*d - 1215*a^4*b^2*c*d^2 - 4860*a^2*b^2*c^3*d^2 + 
          4860*a^3*b^2*c^2*d^2 - 1620*a*b^3*c^4*d + 1620*a^4*b*c*d^3)*x^5 + (- 81*a^4*d^5 + 
          81*b^5*c^4 - 162*a*b^5*c^3 + 405*a^4*b*d^4 + 162*a^3*c*d^5 - 405*b^4*c^4*d - 
          405*a^4*b^2*d^3 + 405*b^3*c^4*d^2 + 1620*a*b^2*c^3*d^3 - 3240*a*b^3*c^3*d^2 + 
          1215*a^2*b*c^2*d^4 - 1215*a^2*b^4*c^2*d + 3240*a^3*b^2*c*d^3 - 1620*a^3*b^3*c*d^2 - 
          4860*a^2*b^2*c^2*d^3 + 4860*a^2*b^3*c^2*d^2 + 1620*a*b^4*c^3*d - 1620*a^3*b*c*d^4)*x^4 + 
          (27*a^3*d^6 - 27*b^6*c^3 - 324*a^3*b*d^5 + 324*b^5*c^3*d + 810*a^3*b^2*d^4 - 
          540*a^3*b^3*d^3 + 540*b^3*c^3*d^3 - 810*b^4*c^3*d^2 + 1215*a*b^2*c^2*d^4 - 
          3240*a*b^3*c^2*d^3 + 2430*a*b^4*c^2*d^2 - 2430*a^2*b^2*c*d^4 + 3240*a^2*b^3*c*d^3 - 
          1215*a^2*b^4*c*d^2 - 486*a*b^5*c^2*d + 486*a^2*b*c*d^5)*x^3 + (81*a^2*b*d^6 - 
          81*b^6*c^2*d - 486*a^2*b^2*d^5 + 810*a^2*b^3*d^4 - 405*a^2*b^4*d^3 + 405*b^3*c^2*d^4 - 
          810*b^4*c^2*d^3 + 486*b^5*c^2*d^2 + 486*a*b^2*c*d^5 - 1620*a*b^3*c*d^4 + 1620*a*b^4*c*d^3 - 
          486*a*b^5*c*d^2)*x^2 + (81*a*b^2*d^6 - 324*a*b^3*d^5 + 405*a*b^4*d^4 - 162*a*b^5*d^3 + 
          162*b^3*c*d^5 - 405*b^4*c*d^4 + 324*b^5*c*d^3 - 81*b^6*c*d^2)*x - (- 27*b^3*d^6 + 
          81*b^4*d^5 - 81*b^5*d^4 + 27*b^6*d^3), [x]), 3 = poly((27*a^3*c^6)*x^9 + (81*a^2*b*c^6 + 
          162*a^3*c^5*d)*x^8 + (81*a*b^2*c^6 + 405*a^3*c^4*d^2 + 486*a^2*b*c^5*d)*x^7 + (27*b^3*c^6 + 
          540*a^3*c^3*d^3 + 1215*a^2*b*c^4*d^2 + 486*a*b^2*c^5*d)*x^6 + (162*b^3*c^5*d + 
          405*a^3*c^2*d^4 + 1215*a*b^2*c^4*d^2 + 1620*a^2*b*c^3*d^3)*x^5 + (162*a^3*c*d^5 + 
          405*b^3*c^4*d^2 + 1620*a*b^2*c^3*d^3 + 1215*a^2*b*c^2*d^4)*x^4 + (27*a^3*d^6 + 
          540*b^3*c^3*d^3 + 1215*a*b^2*c^2*d^4 + 486*a^2*b*c*d^5)*x^3 + (81*a^2*b*d^6 + 
          405*b^3*c^2*d^4 + 486*a*b^2*c*d^5)*x^2 + (81*a*b^2*d^6 + 162*b^3*c*d^5)*x + 
          27*b^3*d^6, [x]), 2 = poly((27*a^3*c^4)*x^7 + (81*a^2*b*c^4 + 108*a^3*c^3*d)*x^6 + 
          (81*a*b^2*c^4 + 162*a^3*c^2*d^2 + 324*a^2*b*c^3*d)*x^5 + (27*b^3*c^4 + 108*a^3*c*d^3 + 
          486*a^2*b*c^2*d^2 + 324*a*b^2*c^3*d)*x^4 + (27*a^3*d^4 + 108*b^3*c^3*d + 486*a*b^2*c^2*d^2 + 
          324*a^2*b*c*d^3)*x^3 + (81*a^2*b*d^4 + 162*b^3*c^2*d^2 + 324*a*b^2*c*d^3)*x^2 + 
          (81*a*b^2*d^4 + 108*b^3*c*d^3)*x + 27*b^3*d^4, [x]), 1 = poly((27*a^3*c^4)*x^7 + 
          (81*a^2*b*c^4 + 108*a^3*c^3*d)*x^6 + (81*a*b^2*c^4 + 162*a^3*c^2*d^2 + 
          324*a^2*b*c^3*d)*x^5 + (27*b^3*c^4 + 108*a^3*c*d^3 + 486*a^2*b*c^2*d^2 + 
          324*a*b^2*c^3*d)*x^4 + (27*a^3*d^4 + 108*b^3*c^3*d + 486*a*b^2*c^2*d^2 + 
          324*a^2*b*c*d^3)*x^3 + (81*a^2*b*d^4 + 162*b^3*c^2*d^2 + 324*a*b^2*c*d^3)*x^2 + 
          (81*a*b^2*d^4 + 108*b^3*c*d^3)*x + 27*b^3*d^4, [x]));    
          
    NumerI0Unknowns:= table(4 = poly((36*a*b^2*c^6 - 54*a^6*c*d^2 - 90*a^2*b^2*c^5 + 108*a^3*b^2*c^4 - 
          54*a^4*b^2*c^3 + 36*a^3*c^4*d^2 - 90*a^4*c^3*d^2 + 108*a^5*c^2*d^2 - 18*a^2*aa*b^2*c^5 + 
          18*a^3*aa*b^2*c^4 - 18*a^2*b^2*bb*c^5 + 18*a^3*b^2*bb*c^4 - 18*a^4*aa*c^3*d^2 + 
          18*a^5*aa*c^2*d^2 + 18*a^6*aa^2*c*d^2 - 18*a^2*b^2*c^5*cc + 18*a^3*b^2*c^4*cc - 
          18*a^4*bb*c^3*d^2 + 18*a^5*bb*c^2*d^2 + 18*a^6*bb^2*c*d^2 - 18*a^2*b^2*c^5*dd - 
          18*a*b^2*c^6*dd^2 - 18*a^4*c^3*cc*d^2 + 18*a^5*c^2*cc*d^2 + 18*a^6*c*cc^2*d^2 - 
          18*a^2*b^2*c^5*ee - 18*a*b^2*c^6*ee^2 + 18*a^3*c^4*d^2*dd - 18*a^4*c^3*d^2*dd + 
          18*a^3*c^4*d^2*ee - 18*a^4*c^3*d^2*ee - 18*a^3*aa^2*b^2*c^4 + 18*a^4*aa^2*b^2*c^3 - 
          18*a^3*b^2*bb^2*c^4 + 18*a^4*b^2*bb^2*c^3 - 18*a^5*aa^2*c^2*d^2 - 18*a^3*b^2*c^4*cc^2 + 
          18*a^4*b^2*c^3*cc^2 - 18*a^5*bb^2*c^2*d^2 + 18*a^2*b^2*c^5*dd^2 - 18*a^5*c^2*cc^2*d^2 + 
          18*a^2*b^2*c^5*ee^2 - 18*a^3*c^4*d^2*dd^2 + 18*a^4*c^3*d^2*dd^2 - 18*a^3*c^4*d^2*ee^2 + 
          18*a^4*c^3*d^2*ee^2 - 72*a^2*b*c^5*d + 180*a^3*b*c^4*d - 216*a^4*b*c^3*d + 108*a^5*b*c^2*d + 
          18*a*b^2*c^6*dd + 18*a*b^2*c^6*ee + 36*a^3*aa*b*c^4*d - 36*a^4*aa*b*c^3*d - 
          18*a^6*aa*bb*c*d^2 + 36*a^3*b*bb*c^4*d - 36*a^4*b*bb*c^3*d - 18*a^6*aa*c*cc*d^2 + 
          36*a^3*b*c^4*cc*d - 36*a^4*b*c^3*cc*d - 18*a^6*bb*c*cc*d^2 - 36*a^2*b*c^5*d*dd + 
          36*a^3*b*c^4*d*dd - 36*a^2*b*c^5*d*ee + 36*a^3*b*c^4*d*ee + 18*a*b^2*c^6*dd*ee - 
          54*a^2*aa*b^2*bb*c^5 + 72*a^3*aa*b^2*bb*c^4 - 18*a^4*aa*b^2*bb*c^3 - 54*a^2*aa*b^2*c^5*cc + 
          72*a^3*aa*b^2*c^4*cc - 18*a^4*aa*b^2*c^3*cc + 36*a^4*aa^2*b*c^3*d - 36*a^5*aa^2*b*c^2*d - 
          54*a^4*aa*bb*c^3*d^2 + 72*a^5*aa*bb*c^2*d^2 + 36*a^2*aa*b^2*c^5*dd - 54*a^2*b^2*bb*c^5*cc - 
          36*a^3*aa*b^2*c^4*dd + 72*a^3*b^2*bb*c^4*cc - 18*a^4*b^2*bb*c^3*cc + 36*a^4*b*bb^2*c^3*d - 
          36*a^5*b*bb^2*c^2*d - 54*a^4*aa*c^3*cc*d^2 + 72*a^5*aa*c^2*cc*d^2 + 36*a^2*aa*b^2*c^5*ee + 
          36*a^2*b^2*bb*c^5*dd - 36*a^3*aa*b^2*c^4*ee - 36*a^3*b^2*bb*c^4*dd + 36*a^4*b*c^3*cc^2*d - 
          36*a^5*b*c^2*cc^2*d + 36*a^4*aa*c^3*d^2*dd - 54*a^4*bb*c^3*cc*d^2 - 36*a^5*aa*c^2*d^2*dd + 
          72*a^5*bb*c^2*cc*d^2 + 36*a^2*b^2*bb*c^5*ee + 36*a^2*b^2*c^5*cc*dd - 36*a^3*b^2*bb*c^4*ee - 
          36*a^3*b^2*c^4*cc*dd + 36*a^2*b*c^5*d*dd^2 - 36*a^3*b*c^4*d*dd^2 + 36*a^4*aa*c^3*d^2*ee + 
          36*a^4*bb*c^3*d^2*dd - 36*a^5*aa*c^2*d^2*ee - 36*a^5*bb*c^2*d^2*dd + 36*a^2*b^2*c^5*cc*ee - 
          36*a^3*b^2*c^4*cc*ee + 36*a^2*b*c^5*d*ee^2 - 36*a^3*b*c^4*d*ee^2 + 36*a^4*bb*c^3*d^2*ee + 
          36*a^4*c^3*cc*d^2*dd - 36*a^5*bb*c^2*d^2*ee - 36*a^5*c^2*cc*d^2*dd - 72*a^2*b^2*c^5*dd*ee + 
          54*a^3*b^2*c^4*dd*ee + 36*a^4*c^3*cc*d^2*ee - 36*a^5*c^2*cc*d^2*ee + 18*a^3*c^4*d^2*dd*ee - 
          72*a^4*c^3*d^2*dd*ee + 54*a^5*c^2*d^2*dd*ee + 108*a^3*aa*b*bb*c^4*d - 144*a^4*aa*b*bb*c^3*d +
          36*a^5*aa*b*bb*c^2*d + 108*a^3*aa*b*c^4*cc*d - 144*a^4*aa*b*c^3*cc*d + 36*a^5*aa*b*c^2*cc*d -
          72*a^3*aa*b*c^4*d*dd + 108*a^3*b*bb*c^4*cc*d + 72*a^4*aa*b*c^3*d*dd - 144*a^4*b*bb*c^3*cc*d +
          36*a^5*b*bb*c^2*cc*d - 72*a^3*aa*b*c^4*d*ee - 72*a^3*b*bb*c^4*d*dd + 72*a^4*aa*b*c^3*d*ee +
          72*a^4*b*bb*c^3*d*dd - 72*a^3*b*bb*c^4*d*ee - 72*a^3*b*c^4*cc*d*dd + 72*a^4*b*bb*c^3*d*ee +
          72*a^4*b*c^3*cc*d*dd - 72*a^3*b*c^4*cc*d*ee + 72*a^4*b*c^3*cc*d*ee - 36*a^2*b*c^5*d*dd*ee +
          144*a^3*b*c^4*d*dd*ee - 108*a^4*b*c^3*d*dd*ee)*x^4 + (- 27*a^6*d^3 + 20*b^3*c^6 - 
          135*a*b^3*c^5 + 144*a^5*c*d^3 + 6*b^3*c^6*dd + 6*b^3*c^6*ee + 288*a^2*b^3*c^4 - 
          189*a^3*b^3*c^3 + 9*a^6*aa^2*d^3 + 2*a^6*aa^3*d^3 + 9*a^6*bb^2*d^3 + 2*a^6*bb^3*d^3 + 
          124*a^3*c^3*d^3 - 225*a^4*c^2*d^3 + 9*a^6*cc^2*d^3 + 2*a^6*cc^3*d^3 - 12*b^3*c^6*dd^2 + 
          2*b^3*c^6*dd^3 - 12*b^3*c^6*ee^2 + 2*b^3*c^6*ee^3 + 45*a^2*aa*b^3*c^4 + 45*a^2*b^3*bb*c^4 - 
          3*a^6*aa*bb^2*d^3 - 3*a^6*aa^2*bb*d^3 - 39*a^4*aa*c^2*d^3 - 15*a^5*aa^2*c*d^3 + 
          45*a^2*b^3*c^4*cc - 3*a^6*aa*cc^2*d^3 - 3*a^6*aa^2*cc*d^3 - 228*a^2*b*c^4*d^2 + 
          45*a^2*b^2*c^4*d + 315*a^3*b*c^3*d^2 - 432*a^3*b^2*c^3*d + 351*a^4*b^2*c^2*d - 
          39*a^4*bb*c^2*d^3 - 15*a^5*bb^2*c*d^3 - 9*a^2*b^3*c^4*dd + 18*a*b^3*c^5*dd^2 - 
          3*a^6*bb*cc^2*d^3 - 3*a^6*bb^2*cc*d^3 - 39*a^4*c^2*cc*d^3 - 15*a^5*c*cc^2*d^3 - 
          9*a^2*b^3*c^4*ee + 18*a*b^3*c^5*ee^2 + 66*a^3*c^3*d^3*dd - 63*a^4*c^2*d^3*dd + 
          66*a^3*c^3*d^3*ee - 63*a^4*c^2*d^3*ee - 3*b^3*c^6*dd*ee^2 - 3*b^3*c^6*dd^2*ee - 
          57*a^2*aa^2*b^3*c^4 + 63*a^3*aa^2*b^3*c^3 - 2*a^3*aa^3*b^3*c^3 - 57*a^2*b^3*bb^2*c^4 + 
          63*a^3*b^3*bb^2*c^3 - 2*a^3*b^3*bb^3*c^3 - 57*a^2*b^3*c^4*cc^2 + 63*a^3*b^3*c^3*cc^2 - 
          2*a^3*b^3*c^3*cc^3 - 60*a^3*c^3*d^3*dd^2 + 54*a^4*c^2*d^3*dd^2 - 2*a^3*c^3*d^3*dd^3 - 
          60*a^3*c^3*d^3*ee^2 + 54*a^4*c^2*d^3*ee^2 - 2*a^3*c^3*d^3*ee^3 - 33*a*aa*b^3*c^5 - 
          33*a*b^3*bb*c^5 - 9*a^6*aa*bb*d^3 + 27*a^5*aa*c*d^3 - 33*a*b^3*c^5*cc - 9*a^6*aa*cc*d^3 + 
          84*a*b^2*c^5*d - 135*a^5*b*c*d^2 + 27*a^5*bb*c*d^3 - 9*a*b^3*c^5*dd - 9*a^6*bb*cc*d^3 + 
          27*a^5*c*cc*d^3 - 9*a*b^3*c^5*ee + 9*a^5*c*d^3*dd + 9*a^5*c*d^3*ee + 21*b^3*c^6*dd*ee - 
          99*a*aa*b^3*bb*c^5 - 99*a*aa*b^3*c^5*cc + 114*a^5*aa*bb*c*d^3 + 69*a*aa*b^3*c^5*dd - 
          99*a*b^3*bb*c^5*cc + 12*a^6*aa*bb*cc*d^3 + 114*a^5*aa*c*cc*d^3 + 69*a*aa*b^3*c^5*ee + 
          69*a*b^3*bb*c^5*dd - 63*a^5*aa*c*d^3*dd + 114*a^5*bb*c*cc*d^3 + 69*a*b^3*bb*c^5*ee + 
          69*a*b^3*c^5*cc*dd + 54*a*b^2*c^5*d*dd - 63*a^5*aa*c*d^3*ee - 63*a^5*bb*c*d^3*dd + 
          69*a*b^3*c^5*cc*ee + 54*a*b^2*c^5*d*ee - 63*a^5*bb*c*d^3*ee - 63*a^5*c*cc*d^3*dd - 
          144*a*b^3*c^5*dd*ee - 63*a^5*c*cc*d^3*ee + 81*a^5*c*d^3*dd*ee + 174*a^2*aa*b^3*bb*c^4 - 
          63*a^3*aa*b^3*bb*c^3 + 174*a^2*aa*b^3*c^4*cc - 63*a^3*aa*b^3*c^3*cc + 27*a^2*aa*b^2*c^4*d + 
          45*a^3*aa*b*c^3*d^2 - 63*a^3*aa*b^2*c^3*d - 9*a^4*aa*b*c^2*d^2 + 45*a^5*aa^2*b*c*d^2 - 
          6*a^5*aa^3*b*c*d^2 - 117*a^4*aa*bb*c^2*d^3 + 9*a^5*aa*bb^2*c*d^3 + 9*a^5*aa^2*bb*c*d^3 - 
          81*a^2*aa*b^3*c^4*dd + 174*a^2*b^3*bb*c^4*cc - 63*a^3*b^3*bb*c^3*cc - 6*a*aa*b^3*c^5*dd^2 + 
          27*a^2*b^2*bb*c^4*d + 45*a^3*b*bb*c^3*d^2 - 63*a^3*b^2*bb*c^3*d - 9*a^4*b*bb*c^2*d^2 + 
          45*a^5*b*bb^2*c*d^2 - 6*a^5*b*bb^3*c*d^2 - 117*a^4*aa*c^2*cc*d^3 + 9*a^5*aa*c*cc^2*d^3 + 
          9*a^5*aa^2*c*cc*d^3 - 81*a^2*aa*b^3*c^4*ee - 81*a^2*b^3*bb*c^4*dd - 6*a*aa*b^3*c^5*ee^2 - 
          6*a*b^3*bb*c^5*dd^2 + 27*a^2*b^2*c^4*cc*d + 45*a^3*b*c^3*cc*d^2 - 63*a^3*b^2*c^3*cc*d - 
          9*a^4*b*c^2*cc*d^2 + 45*a^5*b*c*cc^2*d^2 - 6*a^5*b*c*cc^3*d^2 + 75*a^4*aa*c^2*d^3*dd - 
          117*a^4*bb*c^2*cc*d^3 + 9*a^5*bb*c*cc^2*d^3 - 6*a^5*aa^2*c*d^3*dd + 9*a^5*bb^2*c*cc*d^3 - 
          81*a^2*b^3*bb*c^4*ee - 81*a^2*b^3*c^4*cc*dd - 6*a*b^3*bb*c^5*ee^2 - 6*a*b^3*c^5*cc*dd^2 - 
          126*a^2*b*c^4*d^2*dd - 45*a^2*b^2*c^4*d*dd + 117*a^3*b*c^3*d^2*dd + 27*a^3*b^2*c^3*d*dd - 
          27*a^4*b*c^2*d^2*dd - 36*a*b^2*c^5*d*dd^2 - 6*a*b^2*c^5*d*dd^3 + 75*a^4*aa*c^2*d^3*ee + 
          75*a^4*bb*c^2*d^3*dd - 6*a^5*aa^2*c*d^3*ee - 6*a^5*bb^2*c*d^3*dd - 81*a^2*b^3*c^4*cc*ee - 
          6*a*b^3*c^5*cc*ee^2 - 126*a^2*b*c^4*d^2*ee - 45*a^2*b^2*c^4*d*ee + 117*a^3*b*c^3*d^2*ee + 
          27*a^3*b^2*c^3*d*ee - 27*a^4*b*c^2*d^2*ee - 36*a*b^2*c^5*d*ee^2 - 6*a*b^2*c^5*d*ee^3 + 
          75*a^4*bb*c^2*d^3*ee + 75*a^4*c^2*cc*d^3*dd - 6*a^5*c*cc^2*d^3*dd - 6*a^5*bb^2*c*d^3*ee + 
          135*a^2*b^3*c^4*dd*ee + 9*a*b^3*c^5*dd*ee^2 + 9*a*b^3*c^5*dd^2*ee + 75*a^4*c^2*cc*d^3*ee - 
          6*a^5*c*cc^2*d^3*ee + 51*a^3*c^3*d^3*dd*ee - 144*a^4*c^2*d^3*dd*ee - 9*a^2*aa*b^3*bb^2*c^4 - 
          9*a^2*aa^2*b^3*bb*c^4 + 3*a^3*aa*b^3*bb^2*c^3 + 3*a^3*aa^2*b^3*bb*c^3 - 9*a^2*aa*b^3*c^4*cc^2 - 
          9*a^2*aa^2*b^3*c^4*cc + 3*a^3*aa*b^3*c^3*cc^2 + 3*a^3*aa^2*b^3*c^3*cc + 
          99*a^3*aa^2*b^2*c^3*d - 27*a^4*aa^2*b*c^2*d^2 - 117*a^4*aa^2*b^2*c^2*d + 
          6*a^4*aa^3*b^2*c^2*d + 6*a^2*aa^2*b^3*c^4*dd - 9*a^2*b^3*bb*c^4*cc^2 - 
          9*a^2*b^3*bb^2*c^4*cc + 3*a^3*b^3*bb*c^3*cc^2 + 3*a^3*b^3*bb^2*c^3*cc + 
          99*a^3*b^2*bb^2*c^3*d - 27*a^4*b*bb^2*c^2*d^2 - 117*a^4*b^2*bb^2*c^2*d + 
          6*a^4*b^2*bb^3*c^2*d + 6*a^2*aa^2*b^3*c^4*ee + 6*a^2*b^3*bb^2*c^4*dd + 
          99*a^3*b^2*c^3*cc^2*d - 27*a^4*b*c^2*cc^2*d^2 - 117*a^4*b^2*c^2*cc^2*d + 
          6*a^4*b^2*c^2*cc^3*d + 6*a^4*aa*c^2*d^3*dd^2 + 6*a^2*b^3*bb^2*c^4*ee + 
          6*a^2*b^3*c^4*cc^2*dd + 108*a^2*b*c^4*d^2*dd^2 + 18*a^2*b^2*c^4*d*dd^2 - 
          90*a^3*b*c^3*d^2*dd^2 + 6*a^2*b*c^4*d^2*dd^3 + 6*a^4*aa*c^2*d^3*ee^2 + 
          6*a^4*bb*c^2*d^3*dd^2 + 6*a^2*b^3*c^4*cc^2*ee + 108*a^2*b*c^4*d^2*ee^2 + 
          18*a^2*b^2*c^4*d*ee^2 - 90*a^3*b*c^3*d^2*ee^2 + 6*a^2*b*c^4*d^2*ee^3 + 
          6*a^4*bb*c^2*d^3*ee^2 + 6*a^4*c^2*cc*d^3*dd^2 + 6*a^4*c^2*cc*d^3*ee^2 + 
          3*a^3*c^3*d^3*dd*ee^2 + 3*a^3*c^3*d^3*dd^2*ee - 9*a^4*c^2*d^3*dd*ee^2 - 
          9*a^4*c^2*d^3*dd^2*ee + 27*a^3*aa*b^2*bb^2*c^3*d + 27*a^3*aa^2*b^2*bb*c^3*d - 
          27*a^4*aa*b*bb^2*c^2*d^2 - 9*a^4*aa*b^2*bb^2*c^2*d - 27*a^4*aa^2*b*bb*c^2*d^2 - 
          9*a^4*aa^2*b^2*bb*c^2*d + 27*a^3*aa*b^2*c^3*cc^2*d + 27*a^3*aa^2*b^2*c^3*cc*d - 
          27*a^4*aa*b*c^2*cc^2*d^2 - 9*a^4*aa*b^2*c^2*cc^2*d - 27*a^4*aa^2*b*c^2*cc*d^2 - 
          9*a^4*aa^2*b^2*c^2*cc*d + 18*a^2*aa*b^2*c^4*d*dd^2 - 18*a^3*aa*b*c^3*d^2*dd^2 - 
          18*a^3*aa^2*b^2*c^3*d*dd + 27*a^3*b^2*bb*c^3*cc^2*d + 27*a^3*b^2*bb^2*c^3*cc*d - 
          27*a^4*b*bb*c^2*cc^2*d^2 - 27*a^4*b*bb^2*c^2*cc*d^2 + 18*a^4*aa^2*b*c^2*d^2*dd - 
          9*a^4*b^2*bb*c^2*cc^2*d - 9*a^4*b^2*bb^2*c^2*cc*d + 18*a^2*aa*b^2*c^4*d*ee^2 + 
          18*a^2*b^2*bb*c^4*d*dd^2 - 18*a^3*aa*b*c^3*d^2*ee^2 - 18*a^3*b*bb*c^3*d^2*dd^2 - 
          18*a^3*aa^2*b^2*c^3*d*ee - 18*a^3*b^2*bb^2*c^3*d*dd + 18*a^4*b*bb^2*c^2*d^2*dd + 
          18*a^4*aa^2*b*c^2*d^2*ee + 18*a^2*b^2*bb*c^4*d*ee^2 + 18*a^2*b^2*c^4*cc*d*dd^2 - 
          18*a^3*b*bb*c^3*d^2*ee^2 - 18*a^3*b*c^3*cc*d^2*dd^2 - 18*a^3*b^2*bb^2*c^3*d*ee - 
          18*a^3*b^2*c^3*cc^2*d*dd + 18*a^4*b*bb^2*c^2*d^2*ee + 18*a^4*b*c^2*cc^2*d^2*dd + 
          18*a^2*b^2*c^4*cc*d*ee^2 - 18*a^3*b*c^3*cc*d^2*ee^2 - 18*a^3*b^2*c^3*cc^2*d*ee + 
          18*a^4*b*c^2*cc^2*d^2*ee - 9*a^2*b*c^4*d^2*dd*ee^2 - 9*a^2*b*c^4*d^2*dd^2*ee - 
          27*a^2*b^2*c^4*d*dd*ee^2 - 27*a^2*b^2*c^4*d*dd^2*ee + 27*a^3*b*c^3*d^2*dd*ee^2 + 
          27*a^3*b*c^3*d^2*dd^2*ee - 27*a*aa*b^3*bb*c^5*cc - 45*a^5*aa*b*bb*c*d^2 + 
          9*a*aa*b^3*bb*c^5*dd - 45*a^5*aa*b*c*cc*d^2 - 27*a^5*aa*bb*c*cc*d^3 + 9*a*aa*b^3*bb*c^5*ee + 
          9*a*aa*b^3*c^5*cc*dd - 45*a^5*b*bb*c*cc*d^2 - 3*a^5*aa*bb*c*d^3*dd + 9*a*aa*b^3*c^5*cc*ee + 
          9*a*b^3*bb*c^5*cc*dd - 3*a^5*aa*bb*c*d^3*ee - 3*a^5*aa*c*cc*d^3*dd - 3*a*aa*b^3*c^5*dd*ee + 
          9*a*b^3*bb*c^5*cc*ee - 3*a^5*aa*c*cc*d^3*ee - 3*a^5*bb*c*cc*d^3*dd - 3*a*b^3*bb*c^5*dd*ee + 
          9*a^5*aa*c*d^3*dd*ee - 3*a^5*bb*c*cc*d^3*ee - 3*a*b^3*c^5*cc*dd*ee + 9*a*b^2*c^5*d*dd*ee + 
          9*a^5*bb*c*d^3*dd*ee + 9*a^5*c*cc*d^3*dd*ee + 27*a^2*aa*b^3*bb*c^4*cc - 
          12*a^3*aa*b^3*bb*c^3*cc + 81*a^2*aa*b^2*bb*c^4*d + 135*a^3*aa*b*bb*c^3*d^2 - 
          234*a^3*aa*b^2*bb*c^3*d - 54*a^4*aa*b*bb*c^2*d^2 + 117*a^4*aa*b^2*bb*c^2*d + 
          9*a^5*aa*b*bb^2*c*d^2 + 9*a^5*aa^2*b*bb*c*d^2 + 3*a^2*aa*b^3*bb*c^4*dd + 
          81*a^2*aa*b^2*c^4*cc*d + 135*a^3*aa*b*c^3*cc*d^2 - 234*a^3*aa*b^2*c^3*cc*d - 
          54*a^4*aa*b*c^2*cc*d^2 + 117*a^4*aa*b^2*c^2*cc*d + 9*a^5*aa*b*c*cc^2*d^2 + 
          9*a^5*aa^2*b*c*cc*d^2 + 27*a^4*aa*bb*c^2*cc*d^3 + 3*a^2*aa*b^3*bb*c^4*ee + 
          3*a^2*aa*b^3*c^4*cc*dd - 63*a^2*aa*b^2*c^4*d*dd + 81*a^2*b^2*bb*c^4*cc*d - 
          81*a^3*aa*b*c^3*d^2*dd + 99*a^3*aa*b^2*c^3*d*dd + 135*a^3*b*bb*c^3*cc*d^2 - 
          234*a^3*b^2*bb*c^3*cc*d + 45*a^4*aa*b*c^2*d^2*dd - 54*a^4*b*bb*c^2*cc*d^2 + 
          117*a^4*b^2*bb*c^2*cc*d + 9*a^5*b*bb*c*cc^2*d^2 + 9*a^5*b*bb^2*c*cc*d^2 - 
          9*a^4*aa*bb*c^2*d^3*dd + 3*a^2*aa*b^3*c^4*cc*ee + 3*a^2*b^3*bb*c^4*cc*dd - 
          63*a^2*aa*b^2*c^4*d*ee - 63*a^2*b^2*bb*c^4*d*dd - 81*a^3*aa*b*c^3*d^2*ee + 
          99*a^3*aa*b^2*c^3*d*ee - 81*a^3*b*bb*c^3*d^2*dd + 99*a^3*b^2*bb*c^3*d*dd + 
          45*a^4*aa*b*c^2*d^2*ee + 45*a^4*b*bb*c^2*d^2*dd - 9*a^4*aa*bb*c^2*d^3*ee - 
          9*a^4*aa*c^2*cc*d^3*dd - 9*a^2*aa*b^3*c^4*dd*ee + 3*a^2*b^3*bb*c^4*cc*ee - 
          63*a^2*b^2*bb*c^4*d*ee - 63*a^2*b^2*c^4*cc*d*dd - 81*a^3*b*bb*c^3*d^2*ee - 
          81*a^3*b*c^3*cc*d^2*dd + 99*a^3*b^2*bb*c^3*d*ee + 99*a^3*b^2*c^3*cc*d*dd + 
          45*a^4*b*bb*c^2*d^2*ee + 45*a^4*b*c^2*cc*d^2*dd - 9*a^4*aa*c^2*cc*d^3*ee - 
          9*a^4*bb*c^2*cc*d^3*dd - 9*a^2*b^3*bb*c^4*dd*ee - 63*a^2*b^2*c^4*cc*d*ee - 
          81*a^3*b*c^3*cc*d^2*ee + 99*a^3*b^2*c^3*cc*d*ee + 45*a^4*b*c^2*cc*d^2*ee + 
          3*a^4*aa*c^2*d^3*dd*ee - 9*a^4*bb*c^2*cc*d^3*ee - 9*a^2*b^3*c^4*cc*dd*ee - 
          81*a^2*b*c^4*d^2*dd*ee + 144*a^2*b^2*c^4*d*dd*ee + 144*a^3*b*c^3*d^2*dd*ee - 
          189*a^3*b^2*c^3*d*dd*ee - 27*a^4*b*c^2*d^2*dd*ee + 9*a*b^2*c^5*d*dd*ee^2 + 
          9*a*b^2*c^5*d*dd^2*ee + 3*a^4*bb*c^2*d^3*dd*ee + 3*a^4*c^2*cc*d^3*dd*ee + 
          81*a^2*aa*b^2*bb*c^4*cc*d - 81*a^3*aa*b*bb*c^3*cc*d^2 - 81*a^3*aa*b^2*bb*c^3*cc*d + 
          81*a^4*aa*b*bb*c^2*cc*d^2 + 36*a^4*aa*b^2*bb*c^2*cc*d - 27*a^2*aa*b^2*bb*c^4*d*dd + 
          27*a^3*aa*b*bb*c^3*d^2*dd - 9*a^3*aa*b^2*bb*c^3*d*dd + 9*a^4*aa*b*bb*c^2*d^2*dd - 
          27*a^2*aa*b^2*bb*c^4*d*ee - 27*a^2*aa*b^2*c^4*cc*d*dd + 27*a^3*aa*b*bb*c^3*d^2*ee + 
          27*a^3*aa*b*c^3*cc*d^2*dd - 9*a^3*aa*b^2*bb*c^3*d*ee - 9*a^3*aa*b^2*c^3*cc*d*dd + 
          9*a^4*aa*b*bb*c^2*d^2*ee + 9*a^4*aa*b*c^2*cc*d^2*dd - 27*a^2*aa*b^2*c^4*cc*d*ee - 
          27*a^2*b^2*bb*c^4*cc*d*dd + 27*a^3*aa*b*c^3*cc*d^2*ee - 9*a^3*aa*b^2*c^3*cc*d*ee + 
          27*a^3*b*bb*c^3*cc*d^2*dd - 9*a^3*b^2*bb*c^3*cc*d*dd + 9*a^4*aa*b*c^2*cc*d^2*ee + 
          9*a^4*b*bb*c^2*cc*d^2*dd + 9*a^2*aa*b^2*c^4*d*dd*ee - 27*a^2*b^2*bb*c^4*cc*d*ee - 
          9*a^3*aa*b*c^3*d^2*dd*ee + 27*a^3*aa*b^2*c^3*d*dd*ee + 27*a^3*b*bb*c^3*cc*d^2*ee - 
          9*a^3*b^2*bb*c^3*cc*d*ee - 27*a^4*aa*b*c^2*d^2*dd*ee + 9*a^4*b*bb*c^2*cc*d^2*ee + 
          9*a^2*b^2*bb*c^4*d*dd*ee - 9*a^3*b*bb*c^3*d^2*dd*ee + 27*a^3*b^2*bb*c^3*d*dd*ee - 
          27*a^4*b*bb*c^2*d^2*dd*ee + 9*a^2*b^2*c^4*cc*d*dd*ee - 9*a^3*b*c^3*cc*d^2*dd*ee + 
          27*a^3*b^2*c^3*cc*d*dd*ee - 27*a^4*b*c^2*cc*d^2*dd*ee - 36*a^5*aa*b*bb*c*cc*d^2)*x^3 + 
          (36*a^5*d^4 - 45*b^4*c^5 + 252*a*b^4*c^4 - 15*aa*b^4*c^5 + 9*a^5*aa*d^4 - 81*a^5*b*d^3 + 
          9*a^5*bb*d^4 - 15*b^4*bb*c^5 - 180*a^4*c*d^4 + 9*a^5*cc*d^4 - 15*b^4*c^5*cc + 60*b^3*c^5*d + 
          9*a^5*d^4*dd + 9*b^4*c^5*dd + 9*a^5*d^4*ee + 9*b^4*c^5*ee - 243*a^2*b^4*c^3 + 
          3*a^5*aa^2*d^4 + 3*a^5*bb^2*d^4 + 156*a^3*c^2*d^4 + 3*a^5*cc^2*d^4 - 60*a*aa^2*b^4*c^4 + 
          27*a^5*aa^2*b*d^3 + 6*a^5*aa^3*b*d^3 - 60*a*b^4*bb^2*c^4 + 9*a^5*aa*bb^2*d^4 + 
          9*a^5*aa^2*bb*d^4 + 27*a^5*b*bb^2*d^3 + 6*a^5*b*bb^3*d^3 - 60*a*b^4*c^4*cc^2 + 
          9*a^5*aa*cc^2*d^4 + 9*a^5*aa^2*cc*d^4 + 36*a*b^2*c^4*d^2 - 252*a^2*b*c^3*d^3 - 
          144*a^2*b^3*c^3*d + 45*a^3*b*c^2*d^3 + 405*a^3*b^3*c^2*d - 81*a^4*b^2*c*d^2 + 
          27*a^5*b*cc^2*d^3 + 6*a^5*b*cc^3*d^3 - 6*aa*b^4*c^5*dd^2 + 9*a^5*bb*cc^2*d^4 - 
          6*a^5*aa^2*d^4*dd + 9*a^5*bb^2*cc*d^4 - 6*aa*b^4*c^5*ee^2 - 6*a^5*aa^2*d^4*ee - 
          6*a^5*bb^2*d^4*dd - 6*b^4*bb*c^5*dd^2 + 90*a^3*c^2*d^4*dd + 54*a^4*c*d^4*dd^2 - 
          6*a^5*bb^2*d^4*ee - 6*a^5*cc^2*d^4*dd - 6*b^4*bb*c^5*ee^2 - 6*b^4*c^5*cc*dd^2 - 
          36*b^3*c^5*d*dd^2 + 6*b^3*c^5*d*dd^3 + 90*a^3*c^2*d^4*ee + 54*a^4*c*d^4*ee^2 - 
          6*a^5*cc^2*d^4*ee - 6*b^4*c^5*cc*ee^2 - 36*b^3*c^5*d*ee^2 + 6*b^3*c^5*d*ee^3 + 
          9*b^4*c^5*dd*ee^2 + 9*b^4*c^5*dd^2*ee + 81*a^2*aa^2*b^4*c^3 - 6*a^2*aa^3*b^4*c^3 + 
          81*a^2*b^4*bb^2*c^3 - 6*a^2*b^4*bb^3*c^3 + 81*a^2*b^4*c^3*cc^2 - 6*a^2*b^4*c^3*cc^3 + 
          405*a^2*b^2*c^3*d^2 - 432*a^3*b^2*c^2*d^2 - 72*a^3*c^2*d^4*dd^2 - 6*a^3*c^2*d^4*dd^3 - 
          72*a^3*c^2*d^4*ee^2 - 6*a^3*c^2*d^4*ee^3 + 36*a*aa*b^4*c^4 + 36*a*b^4*bb*c^4 - 
          45*aa*b^4*bb*c^5 + 42*a^5*aa*bb*d^4 - 24*a^4*aa*c*d^4 + 36*a*b^4*c^4*cc - 45*aa*b^4*c^5*cc + 
          42*a^5*aa*cc*d^4 - 225*a*b^3*c^4*d + 288*a^4*b*c*d^3 - 24*a^4*bb*c*d^4 - 18*a*b^4*c^4*dd + 
          33*aa*b^4*c^5*dd - 27*a^5*aa*d^4*dd + 42*a^5*bb*cc*d^4 - 45*b^4*bb*c^5*cc - 24*a^4*c*cc*d^4 -
          18*a*b^4*c^4*ee + 33*aa*b^4*c^5*ee - 27*a^5*aa*d^4*ee - 27*a^5*bb*d^4*dd + 
          33*b^4*bb*c^5*dd - 72*a^4*c*d^4*dd - 27*a^5*bb*d^4*ee - 27*a^5*cc*d^4*dd + 
          33*b^4*bb*c^5*ee + 33*b^4*c^5*cc*dd + 18*b^3*c^5*d*dd - 72*a^4*c*d^4*ee - 
          27*a^5*cc*d^4*ee + 33*b^4*c^5*cc*ee + 18*b^3*c^5*d*ee + 27*a^5*d^4*dd*ee - 
          72*b^4*c^5*dd*ee + 132*a*aa*b^4*bb*c^4 - 27*a^5*aa*b*bb*d^3 + 132*a*aa*b^4*c^4*cc - 
          39*a*aa*b^3*c^4*d + 45*a^4*aa*b*c*d^3 - 72*a^4*aa*bb*c*d^4 - 27*a^5*aa*b*cc*d^3 - 
          54*a*aa*b^4*c^4*dd + 132*a*b^4*bb*c^4*cc - 27*aa*b^4*bb*c^5*cc - 27*a^5*aa*bb*cc*d^4 - 
          39*a*b^3*bb*c^4*d + 45*a^4*b*bb*c*d^3 - 72*a^4*aa*c*cc*d^4 - 27*a^5*b*bb*cc*d^3 - 
          54*a*aa*b^4*c^4*ee - 54*a*b^4*bb*c^4*dd + 9*aa*b^4*bb*c^5*dd - 3*a^5*aa*bb*d^4*dd - 
          39*a*b^3*c^4*cc*d + 45*a^4*b*c*cc*d^3 + 42*a^4*aa*c*d^4*dd - 72*a^4*bb*c*cc*d^4 - 
          54*a*b^4*bb*c^4*ee - 54*a*b^4*c^4*cc*dd + 9*aa*b^4*bb*c^5*ee + 9*aa*b^4*c^5*cc*dd - 
          3*a^5*aa*bb*d^4*ee - 3*a^5*aa*cc*d^4*dd - 63*a*b^3*c^4*d*dd - 9*a^4*b*c*d^3*dd + 
          42*a^4*aa*c*d^4*ee + 42*a^4*bb*c*d^4*dd - 54*a*b^4*c^4*cc*ee + 9*aa*b^4*c^5*cc*ee - 
          3*a^5*aa*cc*d^4*ee - 3*a^5*bb*cc*d^4*dd + 9*b^4*bb*c^5*cc*dd - 63*a*b^3*c^4*d*ee - 
          9*a^4*b*c*d^3*ee + 42*a^4*bb*c*d^4*ee + 42*a^4*c*cc*d^4*dd + 108*a*b^4*c^4*dd*ee - 
          3*aa*b^4*c^5*dd*ee + 9*a^5*aa*d^4*dd*ee - 3*a^5*bb*cc*d^4*ee + 9*b^4*bb*c^5*cc*ee + 
          42*a^4*c*cc*d^4*ee + 9*a^5*bb*d^4*dd*ee - 3*b^4*bb*c^5*dd*ee - 72*a^4*c*d^4*dd*ee + 
          9*a^5*cc*d^4*dd*ee - 3*b^4*c^5*cc*dd*ee + 63*b^3*c^5*d*dd*ee + 45*a^3*aa^2*b^2*c^2*d^2 + 
          45*a^3*b^2*bb^2*c^2*d^2 + 45*a^3*b^2*c^2*cc^2*d^2 - 54*a^2*b^2*c^3*d^2*dd^2 - 
          54*a^2*b^2*c^3*d^2*ee^2 - 81*a^2*aa*b^4*bb*c^3 - 18*a*aa*b^4*bb^2*c^4 - 
          18*a*aa^2*b^4*bb*c^4 - 9*a^5*aa*b*bb^2*d^3 - 9*a^5*aa^2*b*bb*d^3 - 
          81*a^2*aa*b^4*c^3*cc - 18*a*aa*b^4*c^4*cc^2 - 18*a*aa^2*b^4*c^4*cc - 
          9*a^2*aa*b^3*c^3*d - 21*a^3*aa*b*c^2*d^3 - 57*a^4*aa^2*b*c*d^3 - 
          9*a^5*aa*b*cc^2*d^3 - 9*a^5*aa^2*b*cc*d^3 - 81*a^2*b^4*bb*c^3*cc + 
          12*a*aa^2*b^4*c^4*dd - 18*a*b^4*bb*c^4*cc^2 - 18*a*b^4*bb^2*c^4*cc - 
          9*a^2*b^3*bb*c^3*d - 21*a^3*b*bb*c^2*d^3 - 57*a^4*b*bb^2*c*d^3 - 
          9*a^5*b*bb*cc^2*d^3 - 9*a^5*b*bb^2*cc*d^3 + 12*a*aa^2*b^4*c^4*ee + 
          12*a*b^4*bb^2*c^4*dd - 9*a^2*b^3*c^3*cc*d - 21*a^3*b*c^2*cc*d^3 - 
          57*a^4*b*c*cc^2*d^3 + 12*a^4*aa*c*d^4*dd^2 + 12*a*b^4*bb^2*c^4*ee + 
          12*a*b^4*c^4*cc^2*dd + 54*a*b^2*c^4*d^2*dd - 162*a^2*b*c^3*d^3*dd + 
          45*a^2*b^3*c^3*d*dd + 99*a^3*b*c^2*d^3*dd + 54*a*b^3*c^4*d*dd^2 + 12*a^4*aa*c*d^4*ee^2 + 
          12*a^4*bb*c*d^4*dd^2 + 12*a*b^4*c^4*cc^2*ee + 54*a*b^2*c^4*d^2*ee - 162*a^2*b*c^3*d^3*ee + 
          45*a^2*b^3*c^3*d*ee + 99*a^3*b*c^2*d^3*ee + 54*a*b^3*c^4*d*ee^2 + 12*a^4*bb*c*d^4*ee^2 + 
          12*a^4*c*cc*d^4*dd^2 + 12*a^4*c*cc*d^4*ee^2 + 45*a^3*c^2*d^4*dd*ee - 18*a^4*c*d^4*dd*ee^2 - 
          18*a^4*c*d^4*dd^2*ee - 9*b^3*c^5*d*dd*ee^2 - 9*b^3*c^5*d*dd^2*ee + 9*a^2*aa*b^4*bb^2*c^3 + 
          9*a^2*aa^2*b^4*bb*c^3 + 9*a^2*aa*b^4*c^3*cc^2 + 9*a^2*aa^2*b^4*c^3*cc + 
          99*a^2*aa*b^2*c^3*d^2 - 81*a^3*aa*b^2*c^2*d^2 + 69*a^2*aa^2*b^3*c^3*d - 
          135*a^3*aa^2*b^3*c^2*d + 27*a^4*aa^2*b^2*c*d^2 + 18*a^3*aa^3*b^3*c^2*d - 
          18*a^4*aa^3*b^2*c*d^2 + 9*a^2*b^4*bb*c^3*cc^2 + 9*a^2*b^4*bb^2*c^3*cc + 
          99*a^2*b^2*bb*c^3*d^2 - 81*a^3*b^2*bb*c^2*d^2 + 69*a^2*b^3*bb^2*c^3*d - 
          135*a^3*b^3*bb^2*c^2*d + 27*a^4*b^2*bb^2*c*d^2 + 18*a^3*b^3*bb^3*c^2*d - 
          18*a^4*b^2*bb^3*c*d^2 + 99*a^2*b^2*c^3*cc*d^2 - 81*a^3*b^2*c^2*cc*d^2 + 
          69*a^2*b^3*c^3*cc^2*d - 135*a^3*b^3*c^2*cc^2*d + 27*a^4*b^2*c*cc^2*d^2 + 
          18*a^3*b^3*c^2*cc^3*d - 18*a^4*b^2*c*cc^3*d^2 + 27*a^2*b^2*c^3*d^2*dd - 
          27*a^3*b^2*c^2*d^2*dd + 108*a^2*b*c^3*d^3*dd^2 - 54*a^3*b*c^2*d^3*dd^2 - 
          18*a*b^2*c^4*d^2*dd^3 + 18*a^2*b*c^3*d^3*dd^3 + 27*a^2*b^2*c^3*d^2*ee - 
          27*a^3*b^2*c^2*d^2*ee + 108*a^2*b*c^3*d^3*ee^2 - 54*a^3*b*c^2*d^3*ee^2 - 
          18*a*b^2*c^4*d^2*ee^3 + 18*a^2*b*c^3*d^3*ee^3 + 9*a^3*c^2*d^4*dd*ee^2 + 
          9*a^3*c^2*d^4*dd^2*ee + 297*a^2*aa*b^2*bb*c^3*d^2 - 342*a^3*aa*b^2*bb*c^2*d^2 + 
          45*a^2*aa*b^3*bb^2*c^3*d + 45*a^2*aa^2*b^3*bb*c^3*d - 27*a^3*aa*b^3*bb^2*c^2*d - 
          27*a^3*aa^2*b^3*bb*c^2*d + 27*a^4*aa*b^2*bb^2*c*d^2 + 27*a^4*aa^2*b^2*bb*c*d^2 + 
          297*a^2*aa*b^2*c^3*cc*d^2 - 342*a^3*aa*b^2*c^2*cc*d^2 + 45*a^2*aa*b^3*c^3*cc^2*d + 
          45*a^2*aa^2*b^3*c^3*cc*d - 27*a^3*aa*b^3*c^2*cc^2*d - 27*a^3*aa^2*b^3*c^2*cc*d + 
          27*a^4*aa*b^2*c*cc^2*d^2 + 27*a^4*aa^2*b^2*c*cc*d^2 - 207*a^2*aa*b^2*c^3*d^2*dd + 
          297*a^2*b^2*bb*c^3*cc*d^2 + 189*a^3*aa*b^2*c^2*d^2*dd - 342*a^3*b^2*bb*c^2*cc*d^2 - 
          30*a^2*aa^2*b^3*c^3*d*dd + 45*a^2*b^3*bb*c^3*cc^2*d + 45*a^2*b^3*bb^2*c^3*cc*d - 
          30*a^3*aa*b*c^2*d^3*dd^2 - 27*a^3*b^3*bb*c^2*cc^2*d - 27*a^3*b^3*bb^2*c^2*cc*d + 
          27*a^4*b^2*bb*c*cc^2*d^2 + 27*a^4*b^2*bb^2*c*cc*d^2 - 207*a^2*aa*b^2*c^3*d^2*ee - 
          207*a^2*b^2*bb*c^3*d^2*dd + 189*a^3*aa*b^2*c^2*d^2*ee + 189*a^3*b^2*bb*c^2*d^2*dd - 
          30*a^2*aa^2*b^3*c^3*d*ee - 30*a^2*b^3*bb^2*c^3*d*dd - 30*a^3*aa*b*c^2*d^3*ee^2 - 
          30*a^3*b*bb*c^2*d^3*dd^2 - 207*a^2*b^2*bb*c^3*d^2*ee - 207*a^2*b^2*c^3*cc*d^2*dd + 
          189*a^3*b^2*bb*c^2*d^2*ee + 189*a^3*b^2*c^2*cc*d^2*dd - 30*a^2*b^3*bb^2*c^3*d*ee - 
          30*a^2*b^3*c^3*cc^2*d*dd - 30*a^3*b*bb*c^2*d^3*ee^2 - 30*a^3*b*c^2*cc*d^3*dd^2 - 
          207*a^2*b^2*c^3*cc*d^2*ee + 189*a^3*b^2*c^2*cc*d^2*ee - 30*a^2*b^3*c^3*cc^2*d*ee - 
          30*a^3*b*c^2*cc*d^3*ee^2 + 432*a^2*b^2*c^3*d^2*dd*ee - 243*a^3*b^2*c^2*d^2*dd*ee + 
          27*a*b^2*c^4*d^2*dd*ee^2 + 27*a*b^2*c^4*d^2*dd^2*ee - 27*a^2*b*c^3*d^3*dd*ee^2 - 
          27*a^2*b*c^3*d^3*dd^2*ee + 45*a^3*b*c^2*d^3*dd*ee^2 + 45*a^3*b*c^2*d^3*dd^2*ee + 
          54*a*aa*b^4*bb*c^4*cc - 117*a*aa*b^3*bb*c^4*d + 174*a^4*aa*b*bb*c*d^3 + 
          36*a^5*aa*b*bb*cc*d^3 + 6*a*aa*b^4*bb*c^4*dd - 117*a*aa*b^3*c^4*cc*d + 
          174*a^4*aa*b*c*cc*d^3 + 54*a^4*aa*bb*c*cc*d^4 + 6*a*aa*b^4*bb*c^4*ee + 
          6*a*aa*b^4*c^4*cc*dd + 75*a*aa*b^3*c^4*d*dd - 117*a*b^3*bb*c^4*cc*d - 
          81*a^4*aa*b*c*d^3*dd + 174*a^4*b*bb*c*cc*d^3 - 18*a^4*aa*bb*c*d^4*dd + 
          6*a*aa*b^4*c^4*cc*ee + 6*a*b^4*bb*c^4*cc*dd + 75*a*aa*b^3*c^4*d*ee + 
          75*a*b^3*bb*c^4*d*dd - 81*a^4*aa*b*c*d^3*ee - 81*a^4*b*bb*c*d^3*dd - 
          18*a^4*aa*bb*c*d^4*ee - 18*a^4*aa*c*cc*d^4*dd - 18*a*aa*b^4*c^4*dd*ee + 
          6*a*b^4*bb*c^4*cc*ee + 75*a*b^3*bb*c^4*d*ee + 75*a*b^3*c^4*cc*d*dd - 
          81*a^4*b*bb*c*d^3*ee - 81*a^4*b*c*cc*d^3*dd - 18*a^4*aa*c*cc*d^4*ee - 
          18*a^4*bb*c*cc*d^4*dd - 18*a*b^4*bb*c^4*dd*ee + 75*a*b^3*c^4*cc*d*ee - 
          81*a^4*b*c*cc*d^3*ee + 6*a^4*aa*c*d^4*dd*ee - 18*a^4*bb*c*cc*d^4*ee - 
          18*a*b^4*c^4*cc*dd*ee - 144*a*b^3*c^4*d*dd*ee + 135*a^4*b*c*d^3*dd*ee + 
          6*a^4*bb*c*d^4*dd*ee + 6*a^4*c*cc*d^4*dd*ee - 27*a^3*aa*b^2*bb^2*c^2*d^2 - 
          27*a^3*aa^2*b^2*bb*c^2*d^2 - 27*a^3*aa*b^2*c^2*cc^2*d^2 - 27*a^3*aa^2*b^2*c^2*cc*d^2 + 
          18*a^2*aa*b^2*c^3*d^2*dd^2 + 18*a^3*aa^2*b^2*c^2*d^2*dd - 27*a^3*b^2*bb*c^2*cc^2*d^2 - 
          27*a^3*b^2*bb^2*c^2*cc*d^2 + 18*a^2*aa*b^2*c^3*d^2*ee^2 + 18*a^2*b^2*bb*c^3*d^2*dd^2 + 
          18*a^3*aa^2*b^2*c^2*d^2*ee + 18*a^3*b^2*bb^2*c^2*d^2*dd + 18*a^2*b^2*bb*c^3*d^2*ee^2 + 
          18*a^2*b^2*c^3*cc*d^2*dd^2 + 18*a^3*b^2*bb^2*c^2*d^2*ee + 18*a^3*b^2*c^2*cc^2*d^2*dd + 
          18*a^2*b^2*c^3*cc*d^2*ee^2 + 18*a^3*b^2*c^2*cc^2*d^2*ee - 27*a^2*b^2*c^3*d^2*dd*ee^2 - 
          27*a^2*b^2*c^3*d^2*dd^2*ee - 36*a^2*aa*b^4*bb*c^3*cc - 6*a^2*aa*b^3*bb*c^3*d - 
          63*a^3*aa*b*bb*c^2*d^3 + 135*a^3*aa*b^3*bb*c^2*d - 27*a^4*aa*b^2*bb*c*d^2 - 
          9*a^4*aa*b*bb^2*c*d^3 - 9*a^4*aa^2*b*bb*c*d^3 - 6*a^2*aa*b^3*c^3*cc*d - 
          63*a^3*aa*b*c^2*cc*d^3 + 135*a^3*aa*b^3*c^2*cc*d - 27*a^4*aa*b^2*c*cc*d^2 - 
          9*a^4*aa*b*c*cc^2*d^3 - 9*a^4*aa^2*b*c*cc*d^3 - 27*a^2*aa*b^3*c^3*d*dd - 
          6*a^2*b^3*bb*c^3*cc*d + 57*a^3*aa*b*c^2*d^3*dd - 63*a^3*b*bb*c^2*cc*d^3 + 
          135*a^3*b^3*bb*c^2*cc*d - 27*a^4*b^2*bb*c*cc*d^2 + 6*a*aa*b^3*c^4*d*dd^2 - 
          9*a^4*b*bb*c*cc^2*d^3 - 9*a^4*b*bb^2*c*cc*d^3 + 6*a^4*aa^2*b*c*d^3*dd - 
          27*a^2*aa*b^3*c^3*d*ee - 27*a^2*b^3*bb*c^3*d*dd + 57*a^3*aa*b*c^2*d^3*ee + 
          57*a^3*b*bb*c^2*d^3*dd + 6*a*aa*b^3*c^4*d*ee^2 + 6*a*b^3*bb*c^4*d*dd^2 + 
          6*a^4*b*bb^2*c*d^3*dd + 6*a^4*aa^2*b*c*d^3*ee - 27*a^2*b^3*bb*c^3*d*ee - 
          27*a^2*b^3*c^3*cc*d*dd + 57*a^3*b*bb*c^2*d^3*ee + 57*a^3*b*c^2*cc*d^3*dd + 
          6*a*b^3*bb*c^4*d*ee^2 + 6*a*b^3*c^4*cc*d*dd^2 + 6*a^4*b*c*cc^2*d^3*dd + 
          6*a^4*b*bb^2*c*d^3*ee - 27*a^2*b^3*c^3*cc*d*ee + 57*a^3*b*c^2*cc*d^3*ee + 
          6*a*b^3*c^4*cc*d*ee^2 + 6*a^4*b*c*cc^2*d^3*ee - 81*a*b^2*c^4*d^2*dd*ee - 
          27*a^2*b*c^3*d^3*dd*ee - 27*a^2*b^3*c^3*d*dd*ee - 144*a^3*b*c^2*d^3*dd*ee - 
          9*a*b^3*c^4*d*dd*ee^2 - 9*a*b^3*c^4*d*dd^2*ee - 135*a^2*aa*b^3*bb*c^3*cc*d - 
          135*a^3*aa*b*bb*c^2*cc*d^3 + 108*a^3*aa*b^3*bb*c^2*cc*d - 108*a^4*aa*b^2*bb*c*cc*d^2 - 
          15*a^2*aa*b^3*bb*c^3*d*dd + 45*a^3*aa*b*bb*c^2*d^3*dd - 15*a^2*aa*b^3*bb*c^3*d*ee - 
          15*a^2*aa*b^3*c^3*cc*d*dd + 45*a^3*aa*b*bb*c^2*d^3*ee + 45*a^3*aa*b*c^2*cc*d^3*dd - 
          15*a^2*aa*b^3*c^3*cc*d*ee - 15*a^2*b^3*bb*c^3*cc*d*dd + 45*a^3*aa*b*c^2*cc*d^3*ee + 
          45*a^3*b*bb*c^2*cc*d^3*dd + 45*a^2*aa*b^3*c^3*d*dd*ee - 15*a^2*b^3*bb*c^3*cc*d*ee - 
          15*a^3*aa*b*c^2*d^3*dd*ee + 45*a^3*b*bb*c^2*cc*d^3*ee + 45*a^2*b^3*bb*c^3*d*dd*ee - 
          15*a^3*b*bb*c^2*d^3*dd*ee + 45*a^2*b^3*c^3*cc*d*dd*ee - 15*a^3*b*c^2*cc*d^3*dd*ee + 
          81*a^2*aa*b^2*bb*c^3*cc*d^2 + 81*a^3*aa*b^2*bb*c^2*cc*d^2 - 27*a^2*aa*b^2*bb*c^3*d^2*dd + 
          9*a^3*aa*b^2*bb*c^2*d^2*dd - 27*a^2*aa*b^2*bb*c^3*d^2*ee - 27*a^2*aa*b^2*c^3*cc*d^2*dd + 
          9*a^3*aa*b^2*bb*c^2*d^2*ee + 9*a^3*aa*b^2*c^2*cc*d^2*dd - 27*a^2*aa*b^2*c^3*cc*d^2*ee - 
          27*a^2*b^2*bb*c^3*cc*d^2*dd + 9*a^3*aa*b^2*c^2*cc*d^2*ee + 9*a^3*b^2*bb*c^2*cc*d^2*dd + 
          9*a^2*aa*b^2*c^3*d^2*dd*ee - 27*a^2*b^2*bb*c^3*cc*d^2*ee - 27*a^3*aa*b^2*c^2*d^2*dd*ee + 
          9*a^3*b^2*bb*c^2*cc*d^2*ee + 9*a^2*b^2*bb*c^3*d^2*dd*ee - 27*a^3*b^2*bb*c^2*d^2*dd*ee + 
          9*a^2*b^2*c^3*cc*d^2*dd*ee - 27*a^3*b^2*c^2*cc*d^2*dd*ee + 27*a*aa*b^3*bb*c^4*cc*d + 
          27*a^4*aa*b*bb*c*cc*d^3 - 9*a*aa*b^3*bb*c^4*d*dd + 3*a^4*aa*b*bb*c*d^3*dd - 
          9*a*aa*b^3*bb*c^4*d*ee - 9*a*aa*b^3*c^4*cc*d*dd + 3*a^4*aa*b*bb*c*d^3*ee + 
          3*a^4*aa*b*c*cc*d^3*dd - 9*a*aa*b^3*c^4*cc*d*ee - 9*a*b^3*bb*c^4*cc*d*dd + 
          3*a^4*aa*b*c*cc*d^3*ee + 3*a^4*b*bb*c*cc*d^3*dd + 3*a*aa*b^3*c^4*d*dd*ee - 
          9*a*b^3*bb*c^4*cc*d*ee - 9*a^4*aa*b*c*d^3*dd*ee + 3*a^4*b*bb*c*cc*d^3*ee + 
          3*a*b^3*bb*c^4*d*dd*ee - 9*a^4*b*bb*c*d^3*dd*ee + 3*a*b^3*c^4*cc*d*dd*ee - 
          9*a^4*b*c*cc*d^3*dd*ee)*x^2 + (- 45*a^4*d^5 + 72*b^5*c^4 - 135*a*b^5*c^3 + 
          9*aa*b^5*c^4 - 3*a^4*aa*d^5 + 72*a^4*b*d^4 - 3*a^4*bb*d^5 + 9*b^5*bb*c^4 + 
          84*a^3*c*d^5 - 3*a^4*cc*d^5 + 9*b^5*c^4*cc - 90*b^4*c^4*d - 27*a^4*d^5*dd - 
          9*b^5*c^4*dd - 27*a^4*d^5*ee - 9*b^5*c^4*ee - 21*aa^2*b^5*c^4 - 81*a^4*b^2*d^3 - 
          21*b^5*bb^2*c^4 - 21*b^5*c^4*cc^2 + 60*b^3*c^4*d^2 + 18*a^4*d^5*dd^2 + 18*a^4*d^5*ee^2 + 
          45*a*aa^2*b^5*c^3 - 6*a*aa^3*b^5*c^3 + 6*a^4*aa^2*b*d^4 + 45*a*b^5*bb^2*c^3 - 
          6*a*b^5*bb^3*c^3 - 9*aa*b^5*bb^2*c^4 - 9*aa^2*b^5*bb*c^4 + 6*a^4*b*bb^2*d^4 + 
          45*a*b^5*c^3*cc^2 - 6*a*b^5*c^3*cc^3 - 9*aa*b^5*c^4*cc^2 - 9*aa^2*b^5*c^4*cc - 
          36*a*b^2*c^3*d^3 - 45*a*b^3*c^3*d^2 - 108*a^2*b*c^2*d^4 + 189*a^2*b^4*c^2*d + 
          144*a^3*b^2*c*d^3 + 27*a^3*b^3*c*d^2 + 6*a^4*b*cc^2*d^4 + 6*a^4*aa*d^5*dd^2 + 
          6*aa^2*b^5*c^4*dd - 9*b^5*bb*c^4*cc^2 - 9*b^5*bb^2*c^4*cc + 6*a^4*aa*d^5*ee^2 + 
          6*a^4*bb*d^5*dd^2 + 6*aa^2*b^5*c^4*ee + 6*b^5*bb^2*c^4*dd - 36*a^3*c*d^5*dd^2 - 
          6*a^3*c*d^5*dd^3 + 6*a^4*bb*d^5*ee^2 + 6*a^4*cc*d^5*dd^2 + 6*b^5*bb^2*c^4*ee +
          6*b^5*c^4*cc^2*dd + 18*b^3*c^4*d^2*dd - 36*a^3*c*d^5*ee^2 - 6*a^3*c*d^5*ee^3 +
          6*a^4*cc*d^5*ee^2 + 6*b^5*c^4*cc^2*ee + 18*b^3*c^4*d^2*ee - 9*a^4*d^5*dd*ee^2 -
          9*a^4*d^5*dd^2*ee + 27*a^4*aa^2*b^2*d^3 + 6*a^4*aa^3*b^2*d^3 + 27*a^4*b^2*bb^2*d^3 + 
          6*a^4*b^2*bb^3*d^3 + 315*a^2*b^2*c^2*d^3 - 432*a^2*b^3*c^2*d^2 + 27*a^4*b^2*cc^2*d^3 +
          6*a^4*b^2*cc^3*d^3 - 36*b^3*c^4*d^2*dd^2 + 6*b^3*c^4*d^2*dd^3 - 36*b^3*c^4*d^2*ee^2 + 
          6*b^3*c^4*d^2*ee^3 + 18*a^4*aa*b*d^4 + 30*aa*b^5*bb*c^4 - 9*a^4*aa*bb*d^5 + 18*a^4*b*bb*d^4 +
          30*aa*b^5*c^4*cc - 9*a^4*aa*cc*d^5 + 144*a*b^4*c^3*d - 135*a^3*b*c*d^4 - 30*aa*b^4*c^4*d + 
          18*a^4*b*cc*d^4 - 9*aa*b^5*c^4*dd + 3*a^4*aa*d^5*dd - 9*a^4*bb*cc*d^5 + 30*b^5*bb*c^4*cc - 
          30*b^4*bb*c^4*d + 18*a^4*b*d^4*dd - 9*aa*b^5*c^4*ee + 3*a^4*aa*d^5*ee + 3*a^4*bb*d^5*dd - 
          9*b^5*bb*c^4*dd - 30*b^4*c^4*cc*d + 54*a^3*c*d^5*dd + 18*a^4*b*d^4*ee + 3*a^4*bb*d^5*ee + 
          3*a^4*cc*d^5*dd - 9*b^5*bb*c^4*ee - 9*b^5*c^4*cc*dd + 18*b^4*c^4*d*dd + 54*a^3*c*d^5*ee +
          3*a^4*cc*d^5*ee - 9*b^5*c^4*cc*ee + 18*b^4*c^4*d*ee + 27*b^5*c^4*dd*ee - 45*a*aa*b^5*bb*c^3 +
          84*a^4*aa*b*bb*d^4 - 45*a*aa*b^5*c^3*cc + 27*a*aa*b^4*c^3*d - 33*a^3*aa*b*c*d^4 + 
          84*a^4*aa*b*cc*d^4 - 45*a*b^5*bb*c^3*cc + 27*aa*b^5*bb*c^4*cc + 27*a^4*aa*bb*cc*d^5 + 
          27*a*b^4*bb*c^3*d - 33*a^3*b*bb*c*d^4 - 90*aa*b^4*bb*c^4*d - 54*a^4*aa*b*d^4*dd + 
          84*a^4*b*bb*cc*d^4 + 3*aa*b^5*bb*c^4*dd - 9*a^4*aa*bb*d^5*dd + 27*a*b^4*c^3*cc*d - 
          33*a^3*b*c*cc*d^4 - 90*aa*b^4*c^4*cc*d - 54*a^4*aa*b*d^4*ee - 54*a^4*b*bb*d^4*dd + 
          3*aa*b^5*bb*c^4*ee + 3*aa*b^5*c^4*cc*dd - 9*a^4*aa*bb*d^5*ee - 9*a^4*aa*cc*d^5*dd +
          9*a*b^4*c^3*d*dd - 9*a^3*b*c*d^4*dd + 66*aa*b^4*c^4*d*dd - 90*b^4*bb*c^4*cc*d - 
          54*a^4*b*bb*d^4*ee - 54*a^4*b*cc*d^4*dd + 3*aa*b^5*c^4*cc*ee - 9*a^4*aa*cc*d^5*ee - 
          9*a^4*bb*cc*d^5*dd + 3*b^5*bb*c^4*cc*dd + 9*a*b^4*c^3*d*ee - 9*a^3*b*c*d^4*ee +
          66*aa*b^4*c^4*d*ee + 66*b^4*bb*c^4*d*dd - 54*a^4*b*cc*d^4*ee - 9*aa*b^5*c^4*dd*ee + 
          3*a^4*aa*d^5*dd*ee - 9*a^4*bb*cc*d^5*ee + 3*b^5*bb*c^4*cc*ee + 66*b^4*bb*c^4*d*ee + 
          66*b^4*c^4*cc*d*dd + 54*a^4*b*d^4*dd*ee + 3*a^4*bb*d^5*dd*ee - 9*b^5*bb*c^4*dd*ee + 
          66*b^4*c^4*cc*d*ee + 9*a^3*c*d^5*dd*ee + 3*a^4*cc*d^5*dd*ee - 9*b^5*c^4*cc*dd*ee - 
          144*b^4*c^4*d*dd*ee + 99*a^2*aa^2*b^3*c^2*d^2 + 99*a^2*b^3*bb^2*c^2*d^2 + 
          99*a^2*b^3*c^2*cc^2*d^2 - 90*a^2*b^2*c^2*d^3*dd^2 - 90*a^2*b^2*c^2*d^3*ee^2 + 
          9*a*aa*b^5*bb^2*c^3 + 9*a*aa^2*b^5*bb*c^3 - 27*a^4*aa*b^2*bb*d^3 + 18*a^4*aa*b*bb^2*d^4 + 
          18*a^4*aa^2*b*bb*d^4 + 9*a*aa*b^5*c^3*cc^2 + 9*a*aa^2*b^5*c^3*cc + 21*a*aa*b^3*c^3*d^2 +
          9*a^3*aa*b^2*c*d^3 - 15*a*aa^2*b^4*c^3*d - 27*a^4*aa*b^2*cc*d^3 + 18*a^4*aa*b*cc^2*d^4 + 
          18*a^4*aa^2*b*cc*d^4 + 9*a*b^5*bb*c^3*cc^2 + 9*a*b^5*bb^2*c^3*cc + 21*a*b^3*bb*c^3*d^2 + 
          9*a^3*b^2*bb*c*d^3 - 15*a*b^4*bb^2*c^3*d - 27*a^4*b^2*bb*cc*d^3 + 18*a^4*b*bb*cc^2*d^4 + 
          18*a^4*b*bb^2*cc*d^4 - 12*a^4*aa^2*b*d^4*dd + 21*a*b^3*c^3*cc*d^2 + 9*a^3*b^2*c*cc*d^3 - 
          15*a*b^4*c^3*cc^2*d - 12*a^4*b*bb^2*d^4*dd - 12*a^4*aa^2*b*d^4*ee + 18*a*b^2*c^3*d^3*dd -
          99*a*b^3*c^3*d^2*dd - 90*a^2*b*c^2*d^4*dd - 45*a^3*b^2*c*d^3*dd + 18*a^3*b*c*d^4*dd^2 - 
          12*aa*b^4*c^4*d*dd^2 - 12*a^4*b*bb^2*d^4*ee - 12*a^4*b*cc^2*d^4*dd + 18*a*b^2*c^3*d^3*ee - 
          99*a*b^3*c^3*d^2*ee - 90*a^2*b*c^2*d^4*ee - 45*a^3*b^2*c*d^3*ee + 18*a^3*b*c*d^4*ee^2 - 
          12*aa*b^4*c^4*d*ee^2 - 12*b^4*bb*c^4*d*dd^2 - 12*a^4*b*cc^2*d^4*ee - 12*b^4*bb*c^4*d*ee^2 -
          12*b^4*c^4*cc*d*dd^2 - 12*b^4*c^4*cc*d*ee^2 + 9*a^3*c*d^5*dd*ee^2 + 9*a^3*c*d^5*dd^2*ee +
          63*b^3*c^4*d^2*dd*ee + 18*b^4*c^4*d*dd*ee^2 + 18*b^4*c^4*d*dd^2*ee - 9*a^4*aa*b^2*bb^2*d^3 -
          9*a^4*aa^2*b^2*bb*d^3 + 45*a^2*aa*b^2*c^2*d^3 - 63*a^2*aa*b^3*c^2*d^2 - 
          63*a^2*aa^2*b^4*c^2*d - 69*a^3*aa^2*b^2*c*d^3 - 9*a^3*aa^2*b^3*c*d^2 + 18*a^2*aa^3*b^4*c^2*d -
          18*a^3*aa^3*b^3*c*d^2 - 9*a^4*aa*b^2*cc^2*d^3 - 9*a^4*aa^2*b^2*cc*d^3 + 
          45*a^2*b^2*bb*c^2*d^3 - 63*a^2*b^3*bb*c^2*d^2 - 63*a^2*b^4*bb^2*c^2*d - 
          69*a^3*b^2*bb^2*c*d^3 - 9*a^3*b^3*bb^2*c*d^2 + 18*a^2*b^4*bb^3*c^2*d - 
          18*a^3*b^3*bb^3*c*d^2 - 9*a^4*b^2*bb*cc^2*d^3 - 9*a^4*b^2*bb^2*cc*d^3 + 
          45*a^2*b^2*c^2*cc*d^3 - 63*a^2*b^3*c^2*cc*d^2 - 63*a^2*b^4*c^2*cc^2*d - 
          69*a^3*b^2*c*cc^2*d^3 - 9*a^3*b^3*c*cc^2*d^2 + 18*a^2*b^4*c^2*cc^3*d - 
          18*a^3*b^3*c*cc^3*d^2 + 117*a^2*b^2*c^2*d^3*dd + 27*a^2*b^3*c^2*d^2*dd + 
          36*a*b^2*c^3*d^3*dd^2 + 54*a*b^3*c^3*d^2*dd^2 + 36*a^2*b*c^2*d^4*dd^2 - 
          18*a*b^2*c^3*d^3*dd^3 + 18*a^2*b*c^2*d^4*dd^3 + 117*a^2*b^2*c^2*d^3*ee + 
          27*a^2*b^3*c^2*d^2*ee + 36*a*b^2*c^3*d^3*ee^2 + 54*a*b^3*c^3*d^2*ee^2 + 
          36*a^2*b*c^2*d^4*ee^2 - 18*a*b^2*c^3*d^3*ee^3 + 18*a^2*b*c^2*d^4*ee^3 - 
          9*b^3*c^4*d^2*dd*ee^2 - 9*b^3*c^4*d^2*dd^2*ee + 135*a^2*aa*b^2*bb*c^2*d^3 - 
          234*a^2*aa*b^3*bb*c^2*d^2 - 27*a^2*aa*b^4*bb^2*c^2*d - 27*a^2*aa^2*b^4*bb*c^2*d - 
          45*a^3*aa*b^2*bb^2*c*d^3 + 27*a^3*aa*b^3*bb^2*c*d^2 - 45*a^3*aa^2*b^2*bb*c*d^3 + 
          27*a^3*aa^2*b^3*bb*c*d^2 + 135*a^2*aa*b^2*c^2*cc*d^3 - 234*a^2*aa*b^3*c^2*cc*d^2 - 
          27*a^2*aa*b^4*c^2*cc^2*d - 27*a^2*aa^2*b^4*c^2*cc*d - 45*a^3*aa*b^2*c*cc^2*d^3 + 
          27*a^3*aa*b^3*c*cc^2*d^2 - 45*a^3*aa^2*b^2*c*cc*d^3 + 27*a^3*aa^2*b^3*c*cc*d^2 - 
          81*a^2*aa*b^2*c^2*d^3*dd + 99*a^2*aa*b^3*c^2*d^2*dd + 135*a^2*b^2*bb*c^2*cc*d^3 - 
          234*a^2*b^3*bb*c^2*cc*d^2 + 30*a*aa*b^3*c^3*d^2*dd^2 - 27*a^2*b^4*bb*c^2*cc^2*d - 
          27*a^2*b^4*bb^2*c^2*cc*d + 30*a^3*aa^2*b^2*c*d^3*dd - 45*a^3*b^2*bb*c*cc^2*d^3 - 
          45*a^3*b^2*bb^2*c*cc*d^3 + 27*a^3*b^3*bb*c*cc^2*d^2 + 27*a^3*b^3*bb^2*c*cc*d^2 - 
          81*a^2*aa*b^2*c^2*d^3*ee + 99*a^2*aa*b^3*c^2*d^2*ee - 81*a^2*b^2*bb*c^2*d^3*dd + 
          99*a^2*b^3*bb*c^2*d^2*dd + 30*a*aa*b^3*c^3*d^2*ee^2 + 30*a*b^3*bb*c^3*d^2*dd^2 + 
          30*a^3*aa^2*b^2*c*d^3*ee + 30*a^3*b^2*bb^2*c*d^3*dd - 81*a^2*b^2*bb*c^2*d^3*ee - 
          81*a^2*b^2*c^2*cc*d^3*dd + 99*a^2*b^3*bb*c^2*d^2*ee + 99*a^2*b^3*c^2*cc*d^2*dd + 
          30*a*b^3*bb*c^3*d^2*ee^2 + 30*a*b^3*c^3*cc*d^2*dd^2 + 30*a^3*b^2*c*cc^2*d^3*dd + 
          30*a^3*b^2*bb^2*c*d^3*ee - 81*a^2*b^2*c^2*cc*d^3*ee + 99*a^2*b^3*c^2*cc*d^2*ee + 
          30*a*b^3*c^3*cc*d^2*ee^2 + 30*a^3*b^2*c*cc^2*d^3*ee + 144*a^2*b^2*c^2*d^3*dd*ee - 
          189*a^2*b^3*c^2*d^2*dd*ee + 27*a*b^2*c^3*d^3*dd*ee^2 + 27*a*b^2*c^3*d^3*dd^2*ee - 
          45*a*b^3*c^3*d^2*dd*ee^2 - 45*a*b^3*c^3*d^2*dd^2*ee - 27*a^2*b*c^2*d^4*dd*ee^2 - 
          27*a^2*b*c^2*d^4*dd^2*ee - 36*a*aa*b^5*bb*c^3*cc + 114*a*aa*b^4*bb*c^3*d - 
          99*a^3*aa*b*bb*c*d^4 - 54*a^4*aa*b*bb*cc*d^4 + 114*a*aa*b^4*c^3*cc*d - 
          99*a^3*aa*b*c*cc*d^4 - 6*a^4*aa*b*bb*d^4*dd - 63*a*aa*b^4*c^3*d*dd + 
          114*a*b^4*bb*c^3*cc*d + 69*a^3*aa*b*c*d^4*dd - 99*a^3*b*bb*c*cc*d^4 - 
          54*aa*b^4*bb*c^4*cc*d - 6*a^4*aa*b*bb*d^4*ee - 6*a^4*aa*b*cc*d^4*dd - 
          63*a*aa*b^4*c^3*d*ee - 63*a*b^4*bb*c^3*d*dd + 69*a^3*aa*b*c*d^4*ee + 
          69*a^3*b*bb*c*d^4*dd + 18*aa*b^4*bb*c^4*d*dd - 6*a^4*aa*b*cc*d^4*ee - 
          6*a^4*b*bb*cc*d^4*dd - 63*a*b^4*bb*c^3*d*ee - 63*a*b^4*c^3*cc*d*dd + 
          69*a^3*b*bb*c*d^4*ee + 69*a^3*b*c*cc*d^4*dd + 18*aa*b^4*bb*c^4*d*ee + 
          18*aa*b^4*c^4*cc*d*dd + 18*a^4*aa*b*d^4*dd*ee - 6*a^4*b*bb*cc*d^4*ee - 
          63*a*b^4*c^3*cc*d*ee + 69*a^3*b*c*cc*d^4*ee + 18*aa*b^4*c^4*cc*d*ee +
          18*b^4*bb*c^4*cc*d*dd + 18*a^4*b*bb*d^4*dd*ee + 81*a*b^4*c^3*d*dd*ee - 
          144*a^3*b*c*d^4*dd*ee - 6*aa*b^4*c^4*d*dd*ee + 18*b^4*bb*c^4*cc*d*ee + 
          18*a^4*b*cc*d^4*dd*ee - 6*b^4*bb*c^4*d*dd*ee - 6*b^4*c^4*cc*d*dd*ee +
          27*a^2*aa*b^3*bb^2*c^2*d^2 + 27*a^2*aa^2*b^3*bb*c^2*d^2 + 27*a^2*aa*b^3*c^2*cc^2*d^2 + 
          27*a^2*aa^2*b^3*c^2*cc*d^2 - 18*a^2*aa*b^2*c^2*d^3*dd^2 - 18*a^2*aa^2*b^3*c^2*d^2*dd +
          27*a^2*b^3*bb*c^2*cc^2*d^2 + 27*a^2*b^3*bb^2*c^2*cc*d^2 - 18*a^2*aa*b^2*c^2*d^3*ee^2 - 
          18*a^2*aa^2*b^3*c^2*d^2*ee - 18*a^2*b^2*bb*c^2*d^3*dd^2 - 18*a^2*b^3*bb^2*c^2*d^2*dd - 
          18*a^2*b^2*bb*c^2*d^3*ee^2 - 18*a^2*b^2*c^2*cc*d^3*dd^2 - 18*a^2*b^3*bb^2*c^2*d^2*ee - 
          18*a^2*b^3*c^2*cc^2*d^2*dd - 18*a^2*b^2*c^2*cc*d^3*ee^2 - 18*a^2*b^3*c^2*cc^2*d^2*ee + 
          27*a^2*b^2*c^2*d^3*dd*ee^2 + 27*a^2*b^2*c^2*d^3*dd^2*ee + 63*a*aa*b^3*bb*c^3*d^2 +
          63*a^2*aa*b^4*bb*c^2*d + 6*a^3*aa*b^2*bb*c*d^3 + 9*a^3*aa*b^3*bb*c*d^2 + 
          9*a*aa*b^4*bb^2*c^3*d + 9*a*aa^2*b^4*bb*c^3*d + 36*a^4*aa*b^2*bb*cc*d^3 + 
          63*a*aa*b^3*c^3*cc*d^2 + 63*a^2*aa*b^4*c^2*cc*d + 6*a^3*aa*b^2*c*cc*d^3 + 
          9*a^3*aa*b^3*c*cc*d^2 + 9*a*aa*b^4*c^3*cc^2*d + 9*a*aa^2*b^4*c^3*cc*d - 
          57*a*aa*b^3*c^3*d^2*dd + 63*a*b^3*bb*c^3*cc*d^2 + 63*a^2*b^4*bb*c^2*cc*d +
          27*a^3*aa*b^2*c*d^3*dd + 6*a^3*b^2*bb*c*cc*d^3 + 9*a^3*b^3*bb*c*cc*d^2 - 
          6*a*aa^2*b^4*c^3*d*dd + 9*a*b^4*bb*c^3*cc^2*d + 9*a*b^4*bb^2*c^3*cc*d - 
          6*a^3*aa*b*c*d^4*dd^2 - 57*a*aa*b^3*c^3*d^2*ee - 57*a*b^3*bb*c^3*d^2*dd +
          27*a^3*aa*b^2*c*d^3*ee + 27*a^3*b^2*bb*c*d^3*dd - 6*a*aa^2*b^4*c^3*d*ee -
          6*a*b^4*bb^2*c^3*d*dd - 6*a^3*aa*b*c*d^4*ee^2 - 6*a^3*b*bb*c*d^4*dd^2 -
          57*a*b^3*bb*c^3*d^2*ee - 57*a*b^3*c^3*cc*d^2*dd + 27*a^3*b^2*bb*c*d^3*ee + 
          27*a^3*b^2*c*cc*d^3*dd - 6*a*b^4*bb^2*c^3*d*ee - 6*a*b^4*c^3*cc^2*d*dd -
          6*a^3*b*bb*c*d^4*ee^2 - 6*a^3*b*c*cc*d^4*dd^2 - 57*a*b^3*c^3*cc*d^2*ee +
          27*a^3*b^2*c*cc*d^3*ee - 6*a*b^4*c^3*cc^2*d*ee - 6*a^3*b*c*cc*d^4*ee^2 - 
          117*a*b^2*c^3*d^3*dd*ee + 144*a*b^3*c^3*d^2*dd*ee + 45*a^2*b*c^2*d^4*dd*ee +
          27*a^3*b^2*c*d^3*dd*ee + 9*a^3*b*c*d^4*dd*ee^2 + 9*a^3*b*c*d^4*dd^2*ee + 
          135*a*aa*b^3*bb*c^3*cc*d^2 + 108*a^2*aa*b^4*bb*c^2*cc*d + 135*a^3*aa*b^2*bb*c*cc*d^3 - 
          108*a^3*aa*b^3*bb*c*cc*d^2 - 45*a*aa*b^3*bb*c^3*d^2*dd + 15*a^3*aa*b^2*bb*c*d^3*dd -
          45*a*aa*b^3*bb*c^3*d^2*ee - 45*a*aa*b^3*c^3*cc*d^2*dd + 15*a^3*aa*b^2*bb*c*d^3*ee +
          15*a^3*aa*b^2*c*cc*d^3*dd - 45*a*aa*b^3*c^3*cc*d^2*ee - 45*a*b^3*bb*c^3*cc*d^2*dd + 
          15*a^3*aa*b^2*c*cc*d^3*ee + 15*a^3*b^2*bb*c*cc*d^3*dd + 15*a*aa*b^3*c^3*d^2*dd*ee - 
          45*a*b^3*bb*c^3*cc*d^2*ee - 45*a^3*aa*b^2*c*d^3*dd*ee + 15*a^3*b^2*bb*c*cc*d^3*ee + 
          15*a*b^3*bb*c^3*d^2*dd*ee - 45*a^3*b^2*bb*c*d^3*dd*ee + 15*a*b^3*c^3*cc*d^2*dd*ee - 
          45*a^3*b^2*c*cc*d^3*dd*ee - 81*a^2*aa*b^2*bb*c^2*cc*d^3 - 81*a^2*aa*b^3*bb*c^2*cc*d^2 +
          27*a^2*aa*b^2*bb*c^2*d^3*dd - 9*a^2*aa*b^3*bb*c^2*d^2*dd + 27*a^2*aa*b^2*bb*c^2*d^3*ee + 
          27*a^2*aa*b^2*c^2*cc*d^3*dd - 9*a^2*aa*b^3*bb*c^2*d^2*ee - 9*a^2*aa*b^3*c^2*cc*d^2*dd +
          27*a^2*aa*b^2*c^2*cc*d^3*ee - 9*a^2*aa*b^3*c^2*cc*d^2*ee + 27*a^2*b^2*bb*c^2*cc*d^3*dd - 
          9*a^2*b^3*bb*c^2*cc*d^2*dd - 9*a^2*aa*b^2*c^2*d^3*dd*ee + 27*a^2*aa*b^3*c^2*d^2*dd*ee + 
          27*a^2*b^2*bb*c^2*cc*d^3*ee - 9*a^2*b^3*bb*c^2*cc*d^2*ee - 9*a^2*b^2*bb*c^2*d^3*dd*ee +
          27*a^2*b^3*bb*c^2*d^2*dd*ee - 9*a^2*b^2*c^2*cc*d^3*dd*ee + 27*a^2*b^3*c^2*cc*d^2*dd*ee -
          27*a*aa*b^4*bb*c^3*cc*d - 27*a^3*aa*b*bb*c*cc*d^4 - 3*a*aa*b^4*bb*c^3*d*dd + 
          9*a^3*aa*b*bb*c*d^4*dd - 3*a*aa*b^4*bb*c^3*d*ee - 3*a*aa*b^4*c^3*cc*d*dd + 
          9*a^3*aa*b*bb*c*d^4*ee + 9*a^3*aa*b*c*cc*d^4*dd - 3*a*aa*b^4*c^3*cc*d*ee - 
          3*a*b^4*bb*c^3*cc*d*dd + 9*a^3*aa*b*c*cc*d^4*ee + 9*a^3*b*bb*c*cc*d^4*dd + 
          9*a*aa*b^4*c^3*d*dd*ee - 3*a*b^4*bb*c^3*cc*d*ee - 3*a^3*aa*b*c*d^4*dd*ee + 
          9*a^3*b*bb*c*cc*d^4*ee + 9*a*b^4*bb*c^3*d*dd*ee - 3*a^3*b*bb*c*d^4*dd*ee + 
          9*a*b^4*c^3*cc*d*dd*ee - 3*a^3*b*c*cc*d^4*dd*ee)*x + (16*a^3*d^6 - 27*b^6*c^3 - 
          45*a^3*b*d^5 + 72*b^5*c^3*d + 12*a^3*d^6*dd + 12*a^3*d^6*ee + 9*aa^2*b^6*c^3 - 
          2*aa^3*b^6*c^3 + 36*a^3*b^2*d^4 - 27*a^3*b^3*d^3 + 9*b^6*bb^2*c^3 - 2*b^6*bb^3*c^3 +
          9*b^6*c^3*cc^2 - 2*b^6*c^3*cc^3 + 20*b^3*c^3*d^3 - 45*b^4*c^3*d^2 - 6*a^3*d^6*dd^2 -
          2*a^3*d^6*dd^3 - 6*a^3*d^6*ee^2 - 2*a^3*d^6*ee^3 + 9*a^3*aa*b^2*d^4 + 3*aa*b^6*bb^2*c^3 +
          3*aa^2*b^6*bb*c^3 + 9*a^3*b^2*bb*d^4 + 3*aa*b^6*c^3*cc^2 + 3*aa^2*b^6*c^3*cc - 
          24*a*b^2*c^2*d^4 + 45*a*b^3*c^2*d^3 - 108*a*b^4*c^2*d^2 + 45*a^2*b^2*c*d^4 +
          27*a^2*b^4*c*d^2 - 15*aa*b^4*c^3*d^2 - 21*aa^2*b^5*c^3*d + 9*a^3*b^2*cc*d^4 +
          3*b^6*bb*c^3*cc^2 + 3*b^6*bb^2*c^3*cc - 15*b^4*bb*c^3*d^2 - 21*b^5*bb^2*c^3*d + 
          9*a^3*b^2*d^4*dd + 18*a^3*b*d^5*dd^2 - 15*b^4*c^3*cc*d^2 - 21*b^5*c^3*cc^2*d + 
          9*a^3*b^2*d^4*ee + 18*a^3*b*d^5*ee^2 + 6*b^3*c^3*d^3*dd + 9*b^4*c^3*d^2*dd +
          6*b^3*c^3*d^3*ee + 9*b^4*c^3*d^2*ee + 3*a^3*d^6*dd*ee^2 + 3*a^3*d^6*dd^2*ee + 
          3*a^3*aa^2*b^2*d^4 + 9*a^3*aa^2*b^3*d^3 + 2*a^3*aa^3*b^3*d^3 + 3*a^3*b^2*bb^2*d^4 + 
          9*a^3*b^3*bb^2*d^3 + 2*a^3*b^3*bb^3*d^3 + 3*a^3*b^2*cc^2*d^4 + 9*a^3*b^3*cc^2*d^3 + 
          2*a^3*b^3*cc^3*d^3 - 12*b^3*c^3*d^3*dd^2 + 2*b^3*c^3*d^3*dd^3 - 12*b^3*c^3*d^3*ee^2 + 
          2*b^3*c^3*d^3*ee^3 - 3*a^3*aa*b*d^5 - 9*aa*b^6*bb*c^3 - 3*a^3*b*bb*d^5 - 
          9*aa*b^6*c^3*cc + 27*a*b^5*c^2*d - 12*a^2*b*c*d^5 + 9*aa*b^5*c^3*d - 
          3*a^3*b*cc*d^5 - 9*b^6*bb*c^3*cc + 9*b^5*bb*c^3*d - 27*a^3*b*d^5*dd + 
          9*b^5*c^3*cc*d - 27*a^3*b*d^5*ee - 9*b^5*c^3*d*dd - 9*b^5*c^3*d*ee - 
          3*a^3*d^6*dd*ee - 9*a^3*aa*b*bb*d^5 - 9*a^3*aa*b*cc*d^5 - 12*aa*b^6*bb*c^3*cc + 
          30*aa*b^5*bb*c^3*d + 3*a^3*aa*b*d^5*dd - 9*a^3*b*bb*cc*d^5 + 30*aa*b^5*c^3*cc*d + 
          3*a^3*aa*b*d^5*ee + 3*a^3*b*bb*d^5*dd - 18*a^2*b*c*d^5*dd - 9*aa*b^5*c^3*d*dd + 
          30*b^5*bb*c^3*cc*d + 3*a^3*b*bb*d^5*ee + 3*a^3*b*cc*d^5*dd - 18*a^2*b*c*d^5*ee - 
          9*aa*b^5*c^3*d*ee - 9*b^5*bb*c^3*d*dd + 3*a^3*b*cc*d^5*ee - 9*b^5*bb*c^3*d*ee - 
          9*b^5*c^3*cc*d*dd - 9*b^5*c^3*cc*d*ee + 27*b^5*c^3*d*dd*ee + 42*a^3*aa*b^2*bb*d^4 - 
          9*a^3*aa*b^3*bb*d^3 + 27*a*aa*b^3*c^2*d^3 - 9*a*aa*b^4*c^2*d^2 - 9*a^2*aa*b^2*c*d^4 - 
          9*a^2*aa*b^3*c*d^3 - 9*a*aa^2*b^5*c^2*d + 6*a*aa^3*b^5*c^2*d + 42*a^3*aa*b^2*cc*d^4 - 
          9*a^3*aa*b^3*cc*d^3 + 27*a*b^3*bb*c^2*d^3 - 9*a*b^4*bb*c^2*d^2 - 9*a^2*b^2*bb*c*d^4 - 
          9*a^2*b^3*bb*c*d^3 - 9*a*b^5*bb^2*c^2*d + 6*a*b^5*bb^3*c^2*d - 45*aa*b^4*bb*c^3*d^2 - 
          9*aa*b^5*bb^2*c^3*d - 9*aa^2*b^5*bb*c^3*d - 27*a^3*aa*b^2*d^4*dd + 42*a^3*b^2*bb*cc*d^4 - 
          9*a^3*b^3*bb*cc*d^3 + 6*a^3*aa*b*d^5*dd^2 + 27*a*b^3*c^2*cc*d^3 - 9*a*b^4*c^2*cc*d^2 - 
          9*a^2*b^2*c*cc*d^4 - 9*a^2*b^3*c*cc*d^3 - 9*a*b^5*c^2*cc^2*d + 6*a*b^5*c^2*cc^3*d - 
          45*aa*b^4*c^3*cc*d^2 - 9*aa*b^5*c^3*cc^2*d - 9*aa^2*b^5*c^3*cc*d - 27*a^3*aa*b^2*d^4*ee - 
          27*a^3*b^2*bb*d^4*dd + 6*a^3*aa*b*d^5*ee^2 + 6*a^3*b*bb*d^5*dd^2 - 45*a*b^3*c^2*d^3*dd + 
          27*a*b^4*c^2*d^2*dd + 63*a^2*b^2*c*d^4*dd - 27*a^2*b^3*c*d^3*dd + 6*a^2*b*c*d^5*dd^3 + 
          33*aa*b^4*c^3*d^2*dd - 45*b^4*bb*c^3*cc*d^2 + 6*aa^2*b^5*c^3*d*dd - 9*b^5*bb*c^3*cc^2*d - 
          9*b^5*bb^2*c^3*cc*d - 27*a^3*b^2*bb*d^4*ee - 27*a^3*b^2*cc*d^4*dd + 6*a^3*b*bb*d^5*ee^2 + 
          6*a^3*b*cc*d^5*dd^2 - 45*a*b^3*c^2*d^3*ee + 27*a*b^4*c^2*d^2*ee + 63*a^2*b^2*c*d^4*ee - 
          27*a^2*b^3*c*d^3*ee + 6*a^2*b*c*d^5*ee^3 + 33*aa*b^4*c^3*d^2*ee + 33*b^4*bb*c^3*d^2*dd + 
          6*aa^2*b^5*c^3*d*ee + 6*b^5*bb^2*c^3*d*dd - 27*a^3*b^2*cc*d^4*ee + 6*a^3*b*cc*d^5*ee^2 + 
          33*b^4*bb*c^3*d^2*ee + 33*b^4*c^3*cc*d^2*dd + 6*b^5*bb^2*c^3*d*ee + 6*b^5*c^3*cc^2*d*dd + 
          27*a^3*b^2*d^4*dd*ee - 9*a^3*b*d^5*dd*ee^2 - 9*a^3*b*d^5*dd^2*ee + 33*b^4*c^3*cc*d^2*ee + 
          6*b^5*c^3*cc^2*d*ee + 21*b^3*c^3*d^3*dd*ee - 72*b^4*c^3*d^2*dd*ee + 9*a^3*aa*b^2*bb^2*d^4 - 
          3*a^3*aa*b^3*bb^2*d^3 + 9*a^3*aa^2*b^2*bb*d^4 - 3*a^3*aa^2*b^3*bb*d^3 + 
          45*a*aa^2*b^4*c^2*d^2 - 27*a^2*aa^2*b^3*c*d^3 - 9*a^2*aa^2*b^4*c*d^2 - 
          6*a^2*aa^3*b^4*c*d^2 + 9*a^3*aa*b^2*cc^2*d^4 - 3*a^3*aa*b^3*cc^2*d^3 + 
          9*a^3*aa^2*b^2*cc*d^4 - 3*a^3*aa^2*b^3*cc*d^3 + 45*a*b^4*bb^2*c^2*d^2 -
          27*a^2*b^3*bb^2*c*d^3 - 9*a^2*b^4*bb^2*c*d^2 - 6*a^2*b^4*bb^3*c*d^2 - 
          6*a^3*aa^2*b^2*d^4*dd + 9*a^3*b^2*bb*cc^2*d^4 + 9*a^3*b^2*bb^2*cc*d^4 - 
          3*a^3*b^3*bb*cc^2*d^3 - 3*a^3*b^3*bb^2*cc*d^3 + 45*a*b^4*c^2*cc^2*d^2 - 
          27*a^2*b^3*c*cc^2*d^3 - 9*a^2*b^4*c*cc^2*d^2 - 6*a^2*b^4*c*cc^3*d^2 - 
          6*a^3*aa^2*b^2*d^4*ee - 6*a^3*b^2*bb^2*d^4*dd + 18*a*b^2*c^2*d^4*dd^2 + 
          18*a*b^3*c^2*d^3*dd^2 - 36*a^2*b^2*c*d^4*dd^2 - 6*a*b^2*c^2*d^4*dd^3 - 
          6*aa*b^4*c^3*d^2*dd^2 - 6*a^3*b^2*bb^2*d^4*ee - 6*a^3*b^2*cc^2*d^4*dd + 
          18*a*b^2*c^2*d^4*ee^2 + 18*a*b^3*c^2*d^3*ee^2 - 36*a^2*b^2*c*d^4*ee^2 - 
          6*a*b^2*c^2*d^4*ee^3 - 6*aa*b^4*c^3*d^2*ee^2 - 6*b^4*bb*c^3*d^2*dd^2 - 
          6*a^3*b^2*cc^2*d^4*ee - 6*b^4*bb*c^3*d^2*ee^2 - 6*b^4*c^3*cc*d^2*dd^2 - 
          6*b^4*c^3*cc*d^2*ee^2 - 3*b^3*c^3*d^3*dd*ee^2 - 3*b^3*c^3*d^3*dd^2*ee + 
          9*b^4*c^3*d^2*dd*ee^2 + 9*b^4*c^3*d^2*dd^2*ee + 27*a*aa*b^4*bb^2*c^2*d^2 + 
          27*a*aa^2*b^4*bb*c^2*d^2 - 27*a^2*aa*b^3*bb^2*c*d^3 + 9*a^2*aa*b^4*bb^2*c*d^2 - 
          27*a^2*aa^2*b^3*bb*c*d^3 + 9*a^2*aa^2*b^4*bb*c*d^2 + 27*a*aa*b^4*c^2*cc^2*d^2 + 
          27*a*aa^2*b^4*c^2*cc*d^2 - 27*a^2*aa*b^3*c*cc^2*d^3 + 9*a^2*aa*b^4*c*cc^2*d^2 - 
          27*a^2*aa^2*b^3*c*cc*d^3 + 9*a^2*aa^2*b^4*c*cc*d^2 + 18*a*aa*b^3*c^2*d^3*dd^2 - 
          18*a*aa^2*b^4*c^2*d^2*dd + 27*a*b^4*bb*c^2*cc^2*d^2 + 27*a*b^4*bb^2*c^2*cc*d^2 - 
          18*a^2*aa*b^2*c*d^4*dd^2 + 18*a^2*aa^2*b^3*c*d^3*dd - 27*a^2*b^3*bb*c*cc^2*d^3 - 
          27*a^2*b^3*bb^2*c*cc*d^3 + 9*a^2*b^4*bb*c*cc^2*d^2 + 9*a^2*b^4*bb^2*c*cc*d^2 + 
          18*a*aa*b^3*c^2*d^3*ee^2 - 18*a*aa^2*b^4*c^2*d^2*ee + 18*a*b^3*bb*c^2*d^3*dd^2 - 
          18*a*b^4*bb^2*c^2*d^2*dd - 18*a^2*aa*b^2*c*d^4*ee^2 + 18*a^2*aa^2*b^3*c*d^3*ee - 
          18*a^2*b^2*bb*c*d^4*dd^2 + 18*a^2*b^3*bb^2*c*d^3*dd + 18*a*b^3*bb*c^2*d^3*ee^2 + 
          18*a*b^3*c^2*cc*d^3*dd^2 - 18*a*b^4*bb^2*c^2*d^2*ee - 18*a*b^4*c^2*cc^2*d^2*dd - 
          18*a^2*b^2*bb*c*d^4*ee^2 - 18*a^2*b^2*c*cc*d^4*dd^2 + 18*a^2*b^3*c*cc^2*d^3*dd + 
          18*a^2*b^3*bb^2*c*d^3*ee + 18*a*b^3*c^2*cc*d^3*ee^2 - 18*a*b^4*c^2*cc^2*d^2*ee - 
          18*a^2*b^2*c*cc*d^4*ee^2 + 18*a^2*b^3*c*cc^2*d^3*ee + 9*a*b^2*c^2*d^4*dd*ee^2 + 
          9*a*b^2*c^2*d^4*dd^2*ee - 27*a*b^3*c^2*d^3*dd*ee^2 - 27*a*b^3*c^2*d^3*dd^2*ee + 
          27*a^2*b^2*c*d^4*dd*ee^2 + 27*a^2*b^2*c*d^4*dd^2*ee + 9*a*aa*b^5*bb*c^2*d + 
          27*a^3*aa*b*bb*cc*d^5 + 9*a*aa*b^5*c^2*cc*d - 9*a^3*aa*b*bb*d^5*dd + 
          9*a*b^5*bb*c^2*cc*d + 27*aa*b^5*bb*c^3*cc*d - 9*a^3*aa*b*bb*d^5*ee - 
          9*a^3*aa*b*cc*d^5*dd + 3*aa*b^5*bb*c^3*d*dd - 9*a^3*aa*b*cc*d^5*ee - 
          9*a^3*b*bb*cc*d^5*dd + 3*aa*b^5*bb*c^3*d*ee + 3*aa*b^5*c^3*cc*d*dd + 
          3*a^3*aa*b*d^5*dd*ee - 9*a^3*b*bb*cc*d^5*ee + 3*aa*b^5*c^3*cc*d*ee + 
          3*b^5*bb*c^3*cc*d*dd + 3*a^3*b*bb*d^5*dd*ee + 27*a^2*b*c*d^5*dd*ee - 
          9*aa*b^5*c^3*d*dd*ee + 3*b^5*bb*c^3*cc*d*ee + 3*a^3*b*cc*d^5*dd*ee - 
          9*b^5*bb*c^3*d*dd*ee - 9*b^5*c^3*cc*d*dd*ee + 81*a*aa*b^3*bb*c^2*d^3 -
          18*a*aa*b^4*bb*c^2*d^2 - 27*a^2*aa*b^2*bb*c*d^4 - 54*a^2*aa*b^3*bb*c*d^3 + 
          9*a^2*aa*b^4*bb*c*d^2 - 9*a*aa*b^5*bb^2*c^2*d - 9*a*aa^2*b^5*bb*c^2*d - 
          27*a^3*aa*b^2*bb*cc*d^4 + 12*a^3*aa*b^3*bb*cc*d^3 + 81*a*aa*b^3*c^2*cc*d^3 - 
          18*a*aa*b^4*c^2*cc*d^2 - 27*a^2*aa*b^2*c*cc*d^4 - 54*a^2*aa*b^3*c*cc*d^3 +
          9*a^2*aa*b^4*c*cc*d^2 - 9*a*aa*b^5*c^2*cc^2*d - 9*a*aa^2*b^5*c^2*cc*d - 
          3*a^3*aa*b^2*bb*d^4*dd - 63*a*aa*b^3*c^2*d^3*dd - 9*a*aa*b^4*c^2*d^2*dd + 
          81*a*b^3*bb*c^2*cc*d^3 - 18*a*b^4*bb*c^2*cc*d^2 + 27*a^2*aa*b^2*c*d^4*dd +
          45*a^2*aa*b^3*c*d^3*dd - 27*a^2*b^2*bb*c*cc*d^4 - 54*a^2*b^3*bb*c*cc*d^3 + 
          9*a^2*b^4*bb*c*cc*d^2 - 9*a*b^5*bb*c^2*cc^2*d - 9*a*b^5*bb^2*c^2*cc*d - 
          27*aa*b^4*bb*c^3*cc*d^2 - 3*a^3*aa*b^2*bb*d^4*ee - 3*a^3*aa*b^2*cc*d^4*dd - 
          63*a*aa*b^3*c^2*d^3*ee - 9*a*aa*b^4*c^2*d^2*ee - 63*a*b^3*bb*c^2*d^3*dd - 
          9*a*b^4*bb*c^2*d^2*dd + 27*a^2*aa*b^2*c*d^4*ee + 45*a^2*aa*b^3*c*d^3*ee + 
          27*a^2*b^2*bb*c*d^4*dd + 45*a^2*b^3*bb*c*d^3*dd + 9*aa*b^4*bb*c^3*d^2*dd -
          3*a^3*aa*b^2*cc*d^4*ee - 3*a^3*b^2*bb*cc*d^4*dd - 63*a*b^3*bb*c^2*d^3*ee - 
          63*a*b^3*c^2*cc*d^3*dd - 9*a*b^4*bb*c^2*d^2*ee - 9*a*b^4*c^2*cc*d^2*dd +
          27*a^2*b^2*bb*c*d^4*ee + 27*a^2*b^2*c*cc*d^4*dd + 45*a^2*b^3*bb*c*d^3*ee + 
          45*a^2*b^3*c*cc*d^3*dd + 9*aa*b^4*bb*c^3*d^2*ee + 9*aa*b^4*c^3*cc*d^2*dd + 
          9*a^3*aa*b^2*d^4*dd*ee - 3*a^3*b^2*bb*cc*d^4*ee - 63*a*b^3*c^2*cc*d^3*ee - 
          9*a*b^4*c^2*cc*d^2*ee + 27*a^2*b^2*c*cc*d^4*ee + 45*a^2*b^3*c*cc*d^3*ee + 
          9*aa*b^4*c^3*cc*d^2*ee + 9*b^4*bb*c^3*cc*d^2*dd + 9*a^3*b^2*bb*d^4*dd*ee - 
          45*a*b^2*c^2*d^4*dd*ee + 144*a*b^3*c^2*d^3*dd*ee - 27*a*b^4*c^2*d^2*dd*ee - 
          72*a^2*b^2*c*d^4*dd*ee - 27*a^2*b^3*c*d^3*dd*ee - 9*a^2*b*c*d^5*dd*ee^2 - 
          9*a^2*b*c*d^5*dd^2*ee - 3*aa*b^4*c^3*d^2*dd*ee + 9*b^4*bb*c^3*cc*d^2*ee + 
          9*a^3*b^2*cc*d^4*dd*ee - 3*b^4*bb*c^3*d^2*dd*ee - 3*b^4*c^3*cc*d^2*dd*ee +
          81*a*aa*b^3*bb*c^2*cc*d^3 - 81*a*aa*b^4*bb*c^2*cc*d^2 - 81*a^2*aa*b^2*bb*c*cc*d^4 +
          81*a^2*aa*b^3*bb*c*cc*d^3 - 36*a^2*aa*b^4*bb*c*cc*d^2 - 27*a*aa*b^3*bb*c^2*d^3*dd -
          9*a*aa*b^4*bb*c^2*d^2*dd + 27*a^2*aa*b^2*bb*c*d^4*dd + 9*a^2*aa*b^3*bb*c*d^3*dd - 
          27*a*aa*b^3*bb*c^2*d^3*ee - 27*a*aa*b^3*c^2*cc*d^3*dd - 9*a*aa*b^4*bb*c^2*d^2*ee - 
          9*a*aa*b^4*c^2*cc*d^2*dd + 27*a^2*aa*b^2*bb*c*d^4*ee + 27*a^2*aa*b^2*c*cc*d^4*dd + 
          9*a^2*aa*b^3*bb*c*d^3*ee + 9*a^2*aa*b^3*c*cc*d^3*dd - 27*a*aa*b^3*c^2*cc*d^3*ee - 
          9*a*aa*b^4*c^2*cc*d^2*ee - 27*a*b^3*bb*c^2*cc*d^3*dd - 9*a*b^4*bb*c^2*cc*d^2*dd + 
          27*a^2*aa*b^2*c*cc*d^4*ee + 9*a^2*aa*b^3*c*cc*d^3*ee + 27*a^2*b^2*bb*c*cc*d^4*dd + 
          9*a^2*b^3*bb*c*cc*d^3*dd + 9*a*aa*b^3*c^2*d^3*dd*ee + 27*a*aa*b^4*c^2*d^2*dd*ee - 
          27*a*b^3*bb*c^2*cc*d^3*ee - 9*a*b^4*bb*c^2*cc*d^2*ee - 9*a^2*aa*b^2*c*d^4*dd*ee - 
          27*a^2*aa*b^3*c*d^3*dd*ee + 27*a^2*b^2*bb*c*cc*d^4*ee + 9*a^2*b^3*bb*c*cc*d^3*ee + 
          9*a*b^3*bb*c^2*d^3*dd*ee + 27*a*b^4*bb*c^2*d^2*dd*ee - 9*a^2*b^2*bb*c*d^4*dd*ee - 
          27*a^2*b^3*bb*c*d^3*dd*ee + 9*a*b^3*c^2*cc*d^3*dd*ee + 27*a*b^4*c^2*cc*d^2*dd*ee -
          9*a^2*b^2*c*cc*d^4*dd*ee - 27*a^2*b^3*c*cc*d^3*dd*ee + 36*a*aa*b^5*bb*c^2*cc*d), [x]), 
          3 = poly((36*a*b^2*c^6 - 18*a^2*b^2*c^5 - 18*a^3*b^2*c^4 + 36*a^3*c^4*d^2 - 
          18*a^4*c^3*d^2 - 18*a^5*c^2*d^2 - 54*a^2*aa*b^2*c^5 - 54*a^2*b^2*bb*c^5 - 
          54*a^4*aa*c^3*d^2 + 36*a^2*b^2*c^5*cc - 18*a*b^2*c^6*cc^2 - 54*a^4*bb*c^3*d^2 + 
          36*a^2*b^2*c^5*dd - 18*a*b^2*c^6*dd^2 + 18*a^3*c^4*cc*d^2 + 36*a^4*c^3*cc*d^2 + 
          18*a^3*c^4*d^2*dd + 36*a^4*c^3*d^2*dd - 18*a^3*c^4*cc^2*d^2 - 18*a^3*c^4*d^2*dd^2 +
          18*a*b^2*c^6*cc - 72*a^2*b*c^5*d + 36*a^3*b*c^4*d + 36*a^4*b*c^3*d + 18*a*b^2*c^6*dd + 
          108*a^3*aa*b*c^4*d + 108*a^3*b*bb*c^4*d - 36*a^2*b*c^5*cc*d - 72*a^3*b*c^4*cc*d + 
          18*a*b^2*c^6*cc*dd - 36*a^2*b*c^5*d*dd - 72*a^3*b*c^4*d*dd + 36*a^2*b*c^5*cc^2*d + 
          36*a^2*b*c^5*d*dd^2 + 18*a^3*c^4*cc*d^2*dd - 36*a^2*b*c^5*cc*d*dd)*x^4 + (2*a^6*d^3 +
          20*b^3*c^6 - 33*a*b^3*c^5 - 15*a^5*c*d^3 + 6*b^3*c^6*cc + 6*b^3*c^6*dd - 57*a^2*b^3*c^4 -
          2*a^3*b^3*c^3 + 124*a^3*c^3*d^3 - 39*a^4*c^2*d^3 - 12*b^3*c^6*cc^2 + 2*b^3*c^6*cc^3 - 
          12*b^3*c^6*dd^2 + 2*b^3*c^6*dd^3 - 9*a^2*aa*b^3*c^4 - 9*a^2*b^3*bb*c^4 - 117*a^4*aa*c^2*d^3 +
          6*a^2*b^3*c^4*cc - 6*a*b^3*c^5*cc^2 - 228*a^2*b*c^4*d^2 + 27*a^2*b^2*c^4*d + 
          45*a^3*b*c^3*d^2 + 99*a^3*b^2*c^3*d - 27*a^4*b*c^2*d^2 + 6*a^4*b^2*c^2*d - 
          117*a^4*bb*c^2*d^3 + 6*a^2*b^3*c^4*dd - 6*a*b^3*c^5*dd^2 + 66*a^3*c^3*cc*d^3 + 
          75*a^4*c^2*cc*d^3 + 66*a^3*c^3*d^3*dd + 75*a^4*c^2*d^3*dd - 3*b^3*c^6*cc*dd^2 -
          3*b^3*c^6*cc^2*dd - 60*a^3*c^3*cc^2*d^3 + 6*a^4*c^2*cc^2*d^3 - 2*a^3*c^3*cc^3*d^3 -
          60*a^3*c^3*d^3*dd^2 + 6*a^4*c^2*d^3*dd^2 - 2*a^3*c^3*d^3*dd^3 - 99*a*aa*b^3*c^5 -
          99*a*b^3*bb*c^5 + 9*a^5*aa*c*d^3 + 69*a*b^3*c^5*cc + 84*a*b^2*c^5*d - 6*a^5*b*c*d^2 +
          9*a^5*bb*c*d^3 + 69*a*b^3*c^5*dd - 6*a^5*c*cc*d^3 - 6*a^5*c*d^3*dd + 21*b^3*c^6*cc*dd - 
          27*a*aa*b^3*bb*c^5 + 9*a*aa*b^3*c^5*cc + 9*a*aa*b^3*c^5*dd + 9*a*b^3*bb*c^5*cc + 
          9*a*b^3*bb*c^5*dd + 54*a*b^2*c^5*cc*d - 3*a*b^3*c^5*cc*dd + 54*a*b^2*c^5*d*dd + 
          81*a^2*aa*b^2*c^4*d + 135*a^3*aa*b*c^3*d^2 + 27*a^3*aa*b^2*c^3*d - 27*a^4*aa*b*c^2*d^2 +
          27*a^4*aa*bb*c^2*d^3 + 81*a^2*b^2*bb*c^4*d + 135*a^3*b*bb*c^3*d^2 + 27*a^3*b^2*bb*c^3*d -
          27*a^4*b*bb*c^2*d^2 - 9*a^4*aa*c^2*cc*d^3 - 126*a^2*b*c^4*cc*d^2 - 63*a^2*b^2*c^4*cc*d -
          81*a^3*b*c^3*cc*d^2 - 18*a^3*b^2*c^3*cc*d + 18*a^4*b*c^2*cc*d^2 - 36*a*b^2*c^5*cc^2*d - 
          6*a*b^2*c^5*cc^3*d - 9*a^4*aa*c^2*d^3*dd - 9*a^4*bb*c^2*cc*d^3 - 126*a^2*b*c^4*d^2*dd - 
          63*a^2*b^2*c^4*d*dd - 81*a^3*b*c^3*d^2*dd - 18*a^3*b^2*c^3*d*dd + 18*a^4*b*c^2*d^2*dd -
          36*a*b^2*c^5*d*dd^2 - 6*a*b^2*c^5*d*dd^3 - 9*a^4*bb*c^2*d^3*dd + 51*a^3*c^3*cc*d^3*dd +
          3*a^4*c^2*cc*d^3*dd + 108*a^2*b*c^4*cc^2*d^2 + 18*a^2*b^2*c^4*cc^2*d - 
          18*a^3*b*c^3*cc^2*d^2 + 6*a^2*b*c^4*cc^3*d^2 + 108*a^2*b*c^4*d^2*dd^2 + 
          18*a^2*b^2*c^4*d*dd^2 - 18*a^3*b*c^3*d^2*dd^2 + 6*a^2*b*c^4*d^2*dd^3 + 
          3*a^3*c^3*cc*d^3*dd^2 + 3*a^3*c^3*cc^2*d^3*dd - 9*a^2*b*c^4*cc*d^2*dd^2 - 
          9*a^2*b*c^4*cc^2*d^2*dd + 9*a*b^2*c^5*cc*d*dd + 81*a^2*aa*b^2*bb*c^4*d - 
          81*a^3*aa*b*bb*c^3*d^2 - 27*a^2*aa*b^2*c^4*cc*d + 27*a^3*aa*b*c^3*cc*d^2 - 
          27*a^2*aa*b^2*c^4*d*dd - 27*a^2*b^2*bb*c^4*cc*d + 27*a^3*aa*b*c^3*d^2*dd + 
          27*a^3*b*bb*c^3*cc*d^2 - 27*a^2*b^2*bb*c^4*d*dd + 27*a^3*b*bb*c^3*d^2*dd - 
          81*a^2*b*c^4*cc*d^2*dd + 9*a^2*b^2*c^4*cc*d*dd - 9*a^3*b*c^3*cc*d^2*dd + 
          9*a*b^2*c^5*cc*d*dd^2 + 9*a*b^2*c^5*cc^2*d*dd)*x^3 + (3*a^5*d^4 - 15*b^4*c^5 - 
          60*a*b^4*c^4 - 45*aa*b^4*c^5 + 9*a^5*aa*d^4 + 6*a^5*b*d^3 + 9*a^5*bb*d^4 - 
          45*b^4*bb*c^5 - 24*a^4*c*d^4 - 6*a^5*cc*d^4 + 33*b^4*c^5*cc + 60*b^3*c^5*d -
          6*a^5*d^4*dd + 33*b^4*c^5*dd - 6*a^2*b^4*c^3 + 156*a^3*c^2*d^4 - 6*b^4*c^5*cc^2 - 
          6*b^4*c^5*dd^2 + 36*a*b^2*c^4*d^2 - 252*a^2*b*c^3*d^3 + 69*a^2*b^3*c^3*d - 
          21*a^3*b*c^2*d^3 + 18*a^3*b^3*c^2*d - 18*a^4*b^2*c*d^2 + 90*a^3*c^2*cc*d^4 + 
          12*a^4*c*cc^2*d^4 - 36*b^3*c^5*cc^2*d + 6*b^3*c^5*cc^3*d + 90*a^3*c^2*d^4*dd + 
          12*a^4*c*d^4*dd^2 - 36*b^3*c^5*d*dd^2 + 6*b^3*c^5*d*dd^3 + 99*a^2*b^2*c^3*d^2 + 
          45*a^3*b^2*c^2*d^2 - 72*a^3*c^2*cc^2*d^4 - 6*a^3*c^2*cc^3*d^4 - 72*a^3*c^2*d^4*dd^2 - 
          6*a^3*c^2*d^4*dd^3 - 18*a*aa*b^4*c^4 - 18*a*b^4*bb*c^4 - 27*aa*b^4*bb*c^5 - 
          72*a^4*aa*c*d^4 + 12*a*b^4*c^4*cc + 9*aa*b^4*c^5*cc - 39*a*b^3*c^4*d - 57*a^4*b*c*d^3 - 
          72*a^4*bb*c*d^4 + 12*a*b^4*c^4*dd + 9*aa*b^4*c^5*dd + 9*b^4*bb*c^5*cc + 42*a^4*c*cc*d^4 + 
          9*b^4*bb*c^5*dd + 18*b^3*c^5*cc*d + 42*a^4*c*d^4*dd - 3*b^4*c^5*cc*dd + 18*b^3*c^5*d*dd - 
          117*a*aa*b^3*c^4*d - 9*a^4*aa*b*c*d^3 + 54*a^4*aa*bb*c*d^4 - 117*a*b^3*bb*c^4*d - 
          9*a^4*b*bb*c*d^3 - 18*a^4*aa*c*cc*d^4 + 75*a*b^3*c^4*cc*d + 6*a^4*b*c*cc*d^3 - 
          18*a^4*aa*c*d^4*dd - 18*a^4*bb*c*cc*d^4 + 75*a*b^3*c^4*d*dd + 6*a^4*b*c*d^3*dd - 
          18*a^4*bb*c*d^4*dd + 6*a^4*c*cc*d^4*dd + 63*b^3*c^5*cc*d*dd + 18*a^2*b^2*c^3*cc^2*d^2 + 
          18*a^2*b^2*c^3*d^2*dd^2 + 45*a^2*aa*b^3*c^3*d - 63*a^3*aa*b*c^2*d^3 + 45*a^2*b^3*bb*c^3*d -
          63*a^3*b*bb*c^2*d^3 + 54*a*b^2*c^4*cc*d^2 - 162*a^2*b*c^3*cc*d^3 - 30*a^2*b^3*c^3*cc*d +
          57*a^3*b*c^2*cc*d^3 + 6*a*b^3*c^4*cc^2*d + 54*a*b^2*c^4*d^2*dd - 162*a^2*b*c^3*d^3*dd -
          30*a^2*b^3*c^3*d*dd + 57*a^3*b*c^2*d^3*dd + 6*a*b^3*c^4*d*dd^2 + 45*a^3*c^2*cc*d^4*dd -
          9*b^3*c^5*cc*d*dd^2 - 9*b^3*c^5*cc^2*d*dd + 297*a^2*aa*b^2*c^3*d^2 - 27*a^3*aa*b^2*c^2*d^2 +
          297*a^2*b^2*bb*c^3*d^2 - 27*a^3*b^2*bb*c^2*d^2 - 207*a^2*b^2*c^3*cc*d^2 + 
          18*a^3*b^2*c^2*cc*d^2 + 108*a^2*b*c^3*cc^2*d^3 - 30*a^3*b*c^2*cc^2*d^3 - 
          18*a*b^2*c^4*cc^3*d^2 + 18*a^2*b*c^3*cc^3*d^3 - 207*a^2*b^2*c^3*d^2*dd +
          18*a^3*b^2*c^2*d^2*dd + 108*a^2*b*c^3*d^3*dd^2 - 30*a^3*b*c^2*d^3*dd^2 - 
          18*a*b^2*c^4*d^2*dd^3 + 18*a^2*b*c^3*d^3*dd^3 + 9*a^3*c^2*cc*d^4*dd^2 +
          9*a^3*c^2*cc^2*d^4*dd + 81*a^2*aa*b^2*bb*c^3*d^2 - 27*a^2*aa*b^2*c^3*cc*d^2 -
          27*a^2*aa*b^2*c^3*d^2*dd - 27*a^2*b^2*bb*c^3*cc*d^2 - 27*a^2*b^2*bb*c^3*d^2*dd +
          9*a^2*b^2*c^3*cc*d^2*dd + 27*a*b^2*c^4*cc*d^2*dd^2 + 27*a*b^2*c^4*cc^2*d^2*dd -
          27*a^2*b*c^3*cc*d^3*dd^2 - 27*a^2*b*c^3*cc^2*d^3*dd + 27*a*aa*b^3*bb*c^4*d -
          9*a*aa*b^3*c^4*cc*d - 9*a*aa*b^3*c^4*d*dd - 9*a*b^3*bb*c^4*cc*d - 9*a*b^3*bb*c^4*d*dd + 
          3*a*b^3*c^4*cc*d*dd - 135*a^3*aa*b*bb*c^2*d^3 + 45*a^3*aa*b*c^2*cc*d^3 + 
          45*a^3*aa*b*c^2*d^3*dd + 45*a^3*b*bb*c^2*cc*d^3 + 45*a^3*b*bb*c^2*d^3*dd - 
          81*a*b^2*c^4*cc*d^2*dd - 27*a^2*b*c^3*cc*d^3*dd - 15*a^3*b*c^2*cc*d^3*dd)*x^2 + 
          (- 3*a^4*d^5 - 21*b^5*c^4 - 6*a*b^5*c^3 - 9*aa*b^5*c^4 - 9*a^4*aa*d^5 + 6*a^4*b*d^4 - 
          9*a^4*bb*d^5 - 9*b^5*bb*c^4 + 84*a^3*c*d^5 + 3*a^4*cc*d^5 + 6*b^5*c^4*cc - 30*b^4*c^4*d + 
          3*a^4*d^5*dd + 6*b^5*c^4*dd + 6*a^4*b^2*d^3 + 6*a^4*cc^2*d^5 + 60*b^3*c^4*d^2 + 
          6*a^4*d^5*dd^2 - 36*a*b^2*c^3*d^3 + 21*a*b^3*c^3*d^2 - 108*a^2*b*c^2*d^4 + 
          18*a^2*b^4*c^2*d - 69*a^3*b^2*c*d^3 - 18*a^3*b^3*c*d^2 - 36*a^3*c*cc^2*d^5 - 
          6*a^3*c*cc^3*d^5 + 18*b^3*c^4*cc*d^2 - 12*b^4*c^4*cc^2*d - 36*a^3*c*d^5*dd^2 - 
          6*a^3*c*d^5*dd^3 + 18*b^3*c^4*d^2*dd - 12*b^4*c^4*d*dd^2 + 45*a^2*b^2*c^2*d^3 + 
          99*a^2*b^3*c^2*d^2 - 36*b^3*c^4*cc^2*d^2 + 6*b^3*c^4*cc^3*d^2 - 36*b^3*c^4*d^2*dd^2 + 
          6*b^3*c^4*d^2*dd^3 + 18*a^4*aa*b*d^4 + 27*a^4*aa*bb*d^5 + 18*a^4*b*bb*d^4 - 
          9*a^4*aa*cc*d^5 - 15*a*b^4*c^3*d - 33*a^3*b*c*d^4 - 90*aa*b^4*c^4*d - 12*a^4*b*cc*d^4 - 
          9*a^4*aa*d^5*dd - 9*a^4*bb*cc*d^5 - 90*b^4*bb*c^4*d + 54*a^3*c*cc*d^5 - 12*a^4*b*d^4*dd - 
          9*a^4*bb*d^5*dd + 66*b^4*c^4*cc*d + 54*a^3*c*d^5*dd + 3*a^4*cc*d^5*dd + 66*b^4*c^4*d*dd + 
          9*a*aa*b^4*c^3*d - 99*a^3*aa*b*c*d^4 + 9*a*b^4*bb*c^3*d - 99*a^3*b*bb*c*d^4 - 
          54*aa*b^4*bb*c^4*d - 6*a*b^4*c^3*cc*d + 69*a^3*b*c*cc*d^4 + 18*aa*b^4*c^4*cc*d - 
          6*a*b^4*c^3*d*dd + 69*a^3*b*c*d^4*dd + 18*aa*b^4*c^4*d*dd + 18*b^4*bb*c^4*cc*d + 
          18*b^4*bb*c^4*d*dd + 9*a^3*c*cc*d^5*dd - 6*b^4*c^4*cc*d*dd - 18*a^2*b^2*c^2*cc^2*d^3 - 
          18*a^2*b^2*c^2*d^3*dd^2 + 63*a*aa*b^3*c^3*d^2 - 45*a^3*aa*b^2*c*d^3 + 63*a*b^3*bb*c^3*d^2 - 
          45*a^3*b^2*bb*c*d^3 + 18*a*b^2*c^3*cc*d^3 - 57*a*b^3*c^3*cc*d^2 - 90*a^2*b*c^2*cc*d^4 + 
          30*a^3*b^2*c*cc*d^3 - 6*a^3*b*c*cc^2*d^4 + 18*a*b^2*c^3*d^3*dd - 57*a*b^3*c^3*d^2*dd -
          90*a^2*b*c^2*d^4*dd + 30*a^3*b^2*c*d^3*dd - 6*a^3*b*c*d^4*dd^2 + 9*a^3*c*cc*d^5*dd^2 +
          9*a^3*c*cc^2*d^5*dd + 63*b^3*c^4*cc*d^2*dd + 135*a^2*aa*b^2*c^2*d^3 + 27*a^2*aa*b^3*c^2*d^2 +
          135*a^2*b^2*bb*c^2*d^3 + 27*a^2*b^3*bb*c^2*d^2 - 81*a^2*b^2*c^2*cc*d^3 - 
          18*a^2*b^3*c^2*cc*d^2 + 36*a*b^2*c^3*cc^2*d^3 + 30*a*b^3*c^3*cc^2*d^2 + 
          36*a^2*b*c^2*cc^2*d^4 - 18*a*b^2*c^3*cc^3*d^3 + 18*a^2*b*c^2*cc^3*d^4 - 
          81*a^2*b^2*c^2*d^3*dd - 18*a^2*b^3*c^2*d^2*dd + 36*a*b^2*c^3*d^3*dd^2 + 
          30*a*b^3*c^3*d^2*dd^2 + 36*a^2*b*c^2*d^4*dd^2 - 18*a*b^2*c^3*d^3*dd^3 + 
          18*a^2*b*c^2*d^4*dd^3 - 9*b^3*c^4*cc*d^2*dd^2 - 9*b^3*c^4*cc^2*d^2*dd - 
          81*a^2*aa*b^2*bb*c^2*d^3 + 27*a^2*aa*b^2*c^2*cc*d^3 + 27*a^2*aa*b^2*c^2*d^3*dd +
          27*a^2*b^2*bb*c^2*cc*d^3 + 27*a^2*b^2*bb*c^2*d^3*dd - 9*a^2*b^2*c^2*cc*d^3*dd + 
          27*a*b^2*c^3*cc*d^3*dd^2 + 27*a*b^2*c^3*cc^2*d^3*dd - 27*a^2*b*c^2*cc*d^4*dd^2 -
          27*a^2*b*c^2*cc^2*d^4*dd - 27*a^3*aa*b*bb*c*d^4 + 9*a^3*aa*b*c*cc*d^4 + 
          9*a^3*aa*b*c*d^4*dd + 9*a^3*b*bb*c*cc*d^4 + 9*a^3*b*bb*c*d^4*dd - 
          3*a^3*b*c*cc*d^4*dd + 135*a*aa*b^3*bb*c^3*d^2 - 45*a*aa*b^3*c^3*cc*d^2 - 
          45*a*aa*b^3*c^3*d^2*dd - 45*a*b^3*bb*c^3*cc*d^2 - 45*a*b^3*bb*c^3*d^2*dd - 
          117*a*b^2*c^3*cc*d^3*dd + 15*a*b^3*c^3*cc*d^2*dd + 45*a^2*b*c^2*cc*d^4*dd)*x +
          (16*a^3*d^6 - 2*b^6*c^3 - 3*a^3*b*d^5 + 12*a^3*cc*d^6 - 21*b^5*c^3*d + 12*a^3*d^6*dd +
          3*a^3*b^2*d^4 + 2*a^3*b^3*d^3 - 6*a^3*cc^2*d^6 - 2*a^3*cc^3*d^6 + 20*b^3*c^3*d^3 -
          15*b^4*c^3*d^2 - 6*a^3*d^6*dd^2 - 2*a^3*d^6*dd^3 + 9*a^3*aa*b^2*d^4 + 9*a^3*b^2*bb*d^4 - 
          24*a*b^2*c^2*d^4 + 27*a*b^3*c^2*d^3 + 45*a*b^4*c^2*d^2 - 9*a^2*b^2*c*d^4 -
          27*a^2*b^3*c*d^3 - 6*a^2*b^4*c*d^2 - 45*aa*b^4*c^3*d^2 - 6*a^3*b^2*cc*d^4 + 
          6*a^3*b*cc^2*d^5 - 45*b^4*bb*c^3*d^2 - 6*a^3*b^2*d^4*dd + 6*a^3*b*d^5*dd^2 +
          6*b^3*c^3*cc*d^3 + 33*b^4*c^3*cc*d^2 + 3*a^3*cc*d^6*dd^2 + 3*a^3*cc^2*d^6*dd +
          6*b^3*c^3*d^3*dd + 33*b^4*c^3*d^2*dd - 12*b^3*c^3*cc^2*d^3 - 6*b^4*c^3*cc^2*d^2 + 
          2*b^3*c^3*cc^3*d^3 - 12*b^3*c^3*d^3*dd^2 - 6*b^4*c^3*d^2*dd^2 + 2*b^3*c^3*d^3*dd^3 - 
          9*a^3*aa*b*d^5 - 9*a^3*b*bb*d^5 + 6*a*b^5*c^2*d - 12*a^2*b*c*d^5 - 9*aa*b^5*c^3*d + 
          3*a^3*b*cc*d^5 - 9*b^5*bb*c^3*d + 3*a^3*b*d^5*dd + 6*b^5*c^3*cc*d - 3*a^3*cc*d^6*dd + 
          6*b^5*c^3*d*dd + 27*a^3*aa*b*bb*d^5 - 9*a^3*aa*b*cc*d^5 - 9*a^3*aa*b*d^5*dd - 
          9*a^3*b*bb*cc*d^5 - 18*a^2*b*c*cc*d^5 - 9*a^3*b*bb*d^5*dd - 18*a^2*b*c*d^5*dd + 
          3*a^3*b*cc*d^5*dd + 81*a*aa*b^3*c^2*d^3 + 27*a*aa*b^4*c^2*d^2 - 27*a^2*aa*b^2*c*d^4 - 
          27*a^2*aa*b^3*c*d^3 + 81*a*b^3*bb*c^2*d^3 + 27*a*b^4*bb*c^2*d^2 - 27*a^2*b^2*bb*c*d^4 - 
          27*a^2*b^3*bb*c*d^3 - 27*aa*b^4*bb*c^3*d^2 - 63*a*b^3*c^2*cc*d^3 - 18*a*b^4*c^2*cc*d^2 +
          27*a^2*b^2*c*cc*d^4 + 18*a^2*b^3*c*cc*d^3 + 6*a^2*b*c*cc^3*d^5 + 9*aa*b^4*c^3*cc*d^2 -
          63*a*b^3*c^2*d^3*dd - 18*a*b^4*c^2*d^2*dd + 27*a^2*b^2*c*d^4*dd + 18*a^2*b^3*c*d^3*dd +
          6*a^2*b*c*d^5*dd^3 + 9*aa*b^4*c^3*d^2*dd + 9*b^4*bb*c^3*cc*d^2 + 9*b^4*bb*c^3*d^2*dd + 
          21*b^3*c^3*cc*d^3*dd - 3*b^4*c^3*cc*d^2*dd + 18*a*b^2*c^2*cc^2*d^4 + 18*a*b^3*c^2*cc^2*d^3 -
          18*a^2*b^2*c*cc^2*d^4 - 6*a*b^2*c^2*cc^3*d^4 + 18*a*b^2*c^2*d^4*dd^2 + 18*a*b^3*c^2*d^3*dd^2 -
          18*a^2*b^2*c*d^4*dd^2 - 6*a*b^2*c^2*d^4*dd^3 - 3*b^3*c^3*cc*d^3*dd^2 - 3*b^3*c^3*cc^2*d^3*dd +
          9*a*b^2*c^2*cc*d^4*dd^2 + 9*a*b^2*c^2*cc^2*d^4*dd + 27*a^2*b*c*cc*d^5*dd + 
          81*a*aa*b^3*bb*c^2*d^3 - 81*a^2*aa*b^2*bb*c*d^4 - 27*a*aa*b^3*c^2*cc*d^3 + 
          27*a^2*aa*b^2*c*cc*d^4 - 27*a*aa*b^3*c^2*d^3*dd - 27*a*b^3*bb*c^2*cc*d^3 + 
          27*a^2*aa*b^2*c*d^4*dd + 27*a^2*b^2*bb*c*cc*d^4 - 27*a*b^3*bb*c^2*d^3*dd + 
          27*a^2*b^2*bb*c*d^4*dd - 45*a*b^2*c^2*cc*d^4*dd + 9*a*b^3*c^2*cc*d^3*dd - 
          9*a^2*b^2*c*cc*d^4*dd - 9*a^2*b*c*cc*d^5*dd^2 - 9*a^2*b*c*cc^2*d^5*dd), [x]), 
          2 = poly((36*a*b^2*c^4 - 54*a^4*c*d^2 - 54*a^2*b^2*c^3 + 36*a^3*c^2*d^2 - 
          18*a*b^2*bb^2*c^4 - 18*a*b^2*c^4*cc^2 + 18*a^3*bb*c^2*d^2 + 18*a^3*c^2*cc*d^2 - 
          18*a^3*bb^2*c^2*d^2 - 18*a^3*c^2*cc^2*d^2 + 18*a*b^2*bb*c^4 + 18*a*b^2*c^4*cc -
          72*a^2*b*c^3*d + 108*a^3*b*c^2*d + 18*a*b^2*bb*c^4*cc - 36*a^2*b*bb*c^3*d - 
          36*a^2*b*c^3*cc*d + 36*a^2*b*bb^2*c^3*d + 36*a^2*b*c^3*cc^2*d + 18*a^3*bb*c^2*cc*d^2 - 
          36*a^2*b*bb*c^3*cc*d)*x^2 + (- 9*a^4*d^3 + 20*b^3*c^4 - 99*a*b^3*c^3 + 27*a^4*aa*d^3 - 
          9*a^4*bb*d^3 + 6*b^3*bb*c^4 + 52*a^3*c*d^3 - 9*a^4*cc*d^3 + 6*b^3*c^4*cc - 12*b^3*bb^2*c^4 +
          2*b^3*bb^3*c^4 - 12*b^3*c^4*cc^2 + 2*b^3*c^4*cc^3 - 84*a^2*b*c^2*d^2 + 189*a^2*b^2*c^2*d - 
          24*a^3*bb^2*c*d^3 - 2*a^3*bb^3*c*d^3 - 3*b^3*bb*c^4*cc^2 - 3*b^3*bb^2*c^4*cc - 
          24*a^3*c*cc^2*d^3 - 2*a^3*c*cc^3*d^3 - 27*a*aa*b^3*c^3 + 9*a*b^3*bb*c^3 + 9*a*b^3*c^3*cc + 
          12*a*b^2*c^3*d - 81*a^3*b*c*d^2 + 30*a^3*bb*c*d^3 + 21*b^3*bb*c^4*cc + 30*a^3*c*cc*d^3 -
          81*a^3*aa*b*c*d^2 + 18*a*b^2*bb*c^3*d + 27*a^3*b*bb*c*d^2 + 18*a*b^2*c^3*cc*d + 
          27*a^3*b*c*cc*d^2 + 15*a^3*bb*c*cc*d^3 + 81*a^2*aa*b^2*c^2*d - 54*a^2*b*bb*c^2*d^2 - 
          27*a^2*b^2*bb*c^2*d - 6*a*b^2*bb^3*c^3*d - 54*a^2*b*c^2*cc*d^2 - 27*a^2*b^2*c^2*cc*d - 
          6*a*b^2*c^3*cc^3*d + 3*a^3*bb*c*cc^2*d^3 + 3*a^3*bb^2*c*cc*d^3 + 36*a^2*b*bb^2*c^2*d^2 + 
          6*a^2*b*bb^3*c^2*d^2 + 36*a^2*b*c^2*cc^2*d^2 + 6*a^2*b*c^2*cc^3*d^2 - 
          9*a^2*b*bb*c^2*cc^2*d^2 - 9*a^2*b*bb^2*c^2*cc*d^2 - 27*a*b^2*bb*c^3*cc*d -
          9*a^2*b*bb*c^2*cc*d^2 + 9*a*b^2*bb*c^3*cc^2*d + 9*a*b^2*bb^2*c^3*cc*d)*x + (16*a^3*d^4 - 
          45*b^4*c^3 - 27*aa*b^4*c^3 - 9*a^3*b*d^3 + 12*a^3*bb*d^4 + 9*b^4*bb*c^3 + 12*a^3*cc*d^4 + 
          9*b^4*c^3*cc + 20*b^3*c^3*d - 6*a^3*bb^2*d^4 - 2*a^3*bb^3*d^4 - 6*a^3*cc^2*d^4 - 
          2*a^3*cc^3*d^4 - 24*a*b^2*c^2*d^2 - 27*a^2*b^2*c*d^2 + 3*a^3*bb*cc^2*d^4 + 
          3*a^3*bb^2*cc*d^4 - 12*b^3*bb^2*c^3*d + 2*b^3*bb^3*c^3*d - 12*b^3*c^3*cc^2*d +
          2*b^3*c^3*cc^3*d + 27*a^3*aa*b*d^3 - 9*a^3*b*bb*d^3 + 81*a*b^3*c^2*d - 12*a^2*b*c*d^3 - 
          9*a^3*b*cc*d^3 - 3*a^3*bb*cc*d^4 + 6*b^3*bb*c^3*d + 6*b^3*c^3*cc*d + 81*a*aa*b^3*c^2*d -
          27*a*b^3*bb*c^2*d - 18*a^2*b*bb*c*d^3 - 27*a*b^3*c^2*cc*d - 18*a^2*b*c*cc*d^3 + 
          21*b^3*bb*c^3*cc*d - 81*a^2*aa*b^2*c*d^2 + 27*a^2*b^2*bb*c*d^2 + 6*a^2*b*bb^3*c*d^3 +
          27*a^2*b^2*c*cc*d^2 + 6*a^2*b*c*cc^3*d^3 - 3*b^3*bb*c^3*cc^2*d - 3*b^3*bb^2*c^3*cc*d +
          18*a*b^2*bb^2*c^2*d^2 - 6*a*b^2*bb^3*c^2*d^2 + 18*a*b^2*c^2*cc^2*d^2 - 6*a*b^2*c^2*cc^3*d^2 +
          9*a*b^2*bb*c^2*cc^2*d^2 + 9*a*b^2*bb^2*c^2*cc*d^2 + 27*a^2*b*bb*c*cc*d^3 - 
          45*a*b^2*bb*c^2*cc*d^2 - 9*a^2*b*bb*c*cc^2*d^3 - 9*a^2*b*bb^2*c*cc*d^3), [x]), 
          1 = poly((36*a*b^2*c^4 + 36*a^3*c^2*d^2 - 18*a*aa^2*b^2*c^4 - 18*a*b^2*bb^2*c^4 + 
          18*a^3*aa*c^2*d^2 + 18*a^3*bb*c^2*d^2 - 18*a^3*aa^2*c^2*d^2 - 18*a^3*bb^2*c^2*d^2 + 
          18*a*aa*b^2*c^4 + 18*a*b^2*bb*c^4 - 72*a^2*b*c^3*d + 18*a*aa*b^2*bb*c^4 - 
          36*a^2*aa*b*c^3*d - 36*a^2*b*bb*c^3*d + 36*a^2*aa^2*b*c^3*d + 18*a^3*aa*bb*c^2*d^2 +
          36*a^2*b*bb^2*c^3*d - 36*a^2*aa*b*bb*c^3*d)*x^2 + (27*a^4*d^3 + 20*b^3*c^4 -
          27*a*b^3*c^3 + 6*aa*b^3*c^4 + 6*b^3*bb*c^4 + 52*a^3*c*d^3 - 12*aa^2*b^3*c^4 +
          2*aa^3*b^3*c^4 - 12*b^3*bb^2*c^4 + 2*b^3*bb^3*c^4 - 3*aa*b^3*bb^2*c^4 -
          3*aa^2*b^3*bb*c^4 - 24*a^3*aa^2*c*d^3 - 2*a^3*aa^3*c*d^3 - 84*a^2*b*c^2*d^2 +
          81*a^2*b^2*c^2*d - 24*a^3*bb^2*c*d^3 - 2*a^3*bb^3*c*d^3 + 21*aa*b^3*bb*c^4 + 
          30*a^3*aa*c*d^3 + 12*a*b^2*c^3*d - 81*a^3*b*c*d^2 + 30*a^3*bb*c*d^3 + 18*a*aa*b^2*c^3*d + 
          15*a^3*aa*bb*c*d^3 + 18*a*b^2*bb*c^3*d - 54*a^2*aa*b*c^2*d^2 - 6*a*aa^3*b^2*c^3*d +
          3*a^3*aa*bb^2*c*d^3 + 3*a^3*aa^2*bb*c*d^3 - 54*a^2*b*bb*c^2*d^2 - 6*a*b^2*bb^3*c^3*d + 
          36*a^2*aa^2*b*c^2*d^2 + 6*a^2*aa^3*b*c^2*d^2 + 36*a^2*b*bb^2*c^2*d^2 + 6*a^2*b*bb^3*c^2*d^2 -
          9*a^2*aa*b*bb^2*c^2*d^2 - 9*a^2*aa^2*b*bb*c^2*d^2 - 27*a*aa*b^2*bb*c^3*d - 
          9*a^2*aa*b*bb*c^2*d^2 + 9*a*aa*b^2*bb^2*c^3*d + 9*a*aa^2*b^2*bb*c^3*d)*x - (- 
          16*a^3*d^4 + 27*b^4*c^3 - 12*a^3*aa*d^4 - 27*a^3*b*d^3 - 12*a^3*bb*d^4 - 20*b^3*c^3*d + 
          6*a^3*aa^2*d^4 + 2*a^3*aa^3*d^4 + 6*a^3*bb^2*d^4 + 2*a^3*bb^3*d^4 - 3*a^3*aa*bb^2*d^4 - 
          3*a^3*aa^2*bb*d^4 + 24*a*b^2*c^2*d^2 + 81*a^2*b^2*c*d^2 + 12*aa^2*b^3*c^3*d - 
          2*aa^3*b^3*c^3*d + 12*b^3*bb^2*c^3*d - 2*b^3*bb^3*c^3*d + 3*a^3*aa*bb*d^4 - 
          81*a*b^3*c^2*d + 12*a^2*b*c*d^3 - 6*aa*b^3*c^3*d - 6*b^3*bb*c^3*d + 18*a^2*aa*b*c*d^3 +
          18*a^2*b*bb*c*d^3 - 21*aa*b^3*bb*c^3*d - 6*a^2*aa^3*b*c*d^3 - 6*a^2*b*bb^3*c*d^3 +
          3*aa*b^3*bb^2*c^3*d + 3*aa^2*b^3*bb*c^3*d - 18*a*aa^2*b^2*c^2*d^2 + 6*a*aa^3*b^2*c^2*d^2 -
          18*a*b^2*bb^2*c^2*d^2 + 6*a*b^2*bb^3*c^2*d^2 - 9*a*aa*b^2*bb^2*c^2*d^2 - 
          9*a*aa^2*b^2*bb*c^2*d^2 - 27*a^2*aa*b*bb*c*d^3 + 45*a*aa*b^2*bb*c^2*d^2 + 
          9*a^2*aa*b*bb^2*c*d^3 + 9*a^2*aa^2*b*bb*c*d^3), [x]));
    
    while fundSys = {} and cnt < 4 do 
      /* 
  
        In case the above explicit definitions of 'DenomI1Unknowns','NumerI1Unknowns',
        'DenomI0Unknowns','NumerI0Unknowns' should not be used, use the following for 
        generating these expressions automatically: 
  
          cnt:= cnt+1;
          c2_eqHypergeom:= eqnF2[cnt][1]; 
          c1_eqHypergeom:= eqnF2[cnt][2];
          c0_eqHypergeom:= eqnF2[cnt][3];
          eq0F2:= diff(y(x),x,x,x) + c2_eqHypergeom*diff(y(x),x,x) + c1_eqHypergeom*diff(y(x),x) + 
                  c0_eqHypergeom*y(x);    
          Inv_eq0F2Normal1:= diff(c2_eqHypergeom,x)+(c2_eqHypergeom)^2/3-c1_eqHypergeom; 
          t:= ode::normal(ode::normal(diff(Moebius,x)^2*evalAt(Inv_eq0F2Normal1,x=Moebius)));
          denomI1Unknowns:= poly(denom(t),[x]);
          numerI1Unknowns:= poly(numer(t),[x]);
          Inv_eq0F2Normal0:= (diff(c2_eqHypergeom,x,x)/3-2*(c2_eqHypergeom)^3/27+
                                    (c1_eqHypergeom)*(c2_eqHypergeom)/3-c0_eqHypergeom); 
          t:= ode::normal(ode::normal(diff(Moebius,x)*diff(Moebius,x,x)*evalAt(Inv_eq0F2Normal1,x=Moebius)+
                            diff(Moebius,x)^3*evalAt(Inv_eq0F2Normal0,x=Moebius)));
          denomI0Unknowns:= poly(denom(t),[x]);
          numerI0Unknowns:= poly(numer(t),[x]);
          sys:= {(coeff(denomI1,i) = coeff(denomI1Unknowns,i)) $ i = 0..max(degree(denomI1),degree(denomI1Unknowns)),
                 (coeff(denomI0,i) = coeff(denomI0Unknowns,i)) $ i = 0..max(degree(denomI0),degree(denomI0Unknowns))};
          //-----------------  
          // [1], Step (5)(i)
          //-----------------  
  
          ... rest as below!!!                                                                              */ 
      
      cnt:= cnt+1;
      c2_eqHypergeom:= eqnF2[cnt][1]; 
      c1_eqHypergeom:= eqnF2[cnt][2];
      c0_eqHypergeom:= eqnF2[cnt][3];
      eq0F2:= diff(y(x),x,x,x) + c2_eqHypergeom*diff(y(x),x,x) + c1_eqHypergeom*diff(y(x),x) + 
              c0_eqHypergeom*y(x);    
      denomI1Unknowns:= DenomI1Unknowns[cnt]; 
      numerI1Unknowns:= NumerI1Unknowns[cnt];
      denomI0Unknowns:= DenomI0Unknowns[cnt]; 
      numerI0Unknowns:= NumerI0Unknowns[cnt];
      sys:= {(coeff(denomI1,i) = coeff(denomI1Unknowns,i)) $ i = 0..max(degree(denomI1),degree(denomI1Unknowns)),
             (coeff(denomI0,i) = coeff(denomI0Unknowns,i)) $ i = 0..max(degree(denomI0),degree(denomI0Unknowns))};
      //-----------------  
      // [1], Step (5)(i)
      //-----------------  
      L1:= solve(sys,{a,b,c,d},op(solveOptions),IgnoreAnalyticConstraints);
      // If no solution could be found, then return 'FAIL'. 
      if type(L1) <> DOM_SET or nops(L1) = 0 or (nops(L1) > 0 and type(L1[1]) <> DOM_LIST) then 
        L1:= {};
      end_if;
      /* 
         We apply a heuristic startegy to choose a solution, which is as simple 
         as possible. Currently we use 'length' as a measure for complexity. The 
         function 'length' may be replaced by something more elaborate if this 
         shows up to be important in the future. 
      */
      if L1 <> {} then 
        L1:= [op(L1)];
        tmp:= map(map([op(L1)], elem -> map(elem,op,2/*i.e. rhs*/)),length);
        m:= min(tmp);
        if type(m) = DOM_INT and m > 0 then 
          ind:= contains(tmp,m);
          sol1:= L1[ind];
        else 
          sol1:= L1[1];
        end_if;
        /* 
           SPECIAL HEURISTIC STRATEGY FOR PARAMETRIZED SOLUTIONS. 
  
           Note that in the 2F2-case the system determining the 
           parameters in the Moebius transform in general does 
           not have a unique solution. 
           Hence, we make a special heuristic here and try to 
           insert some simple values like 1, 2, 3, etc. in those 
           cases where the solver returned a parametrized solution 
           'L1'. Note that due to performance issues it is 
           absolutely NO OPTION to try to perform the computations
           below on a parametrized solution 'L2'.  
        */
        if freeIndets(sol1) minus indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a,b,c,d]) <> {} then 
          specVal:= [1,2,3,4];
          j:= 0;
          while j < nops(specVal) do 
            j:= j+1;
            tmp:= map(freeIndets(sol1) minus indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a,b,c,d]),
                      elem -> elem = specVal[j]);
            if traperror((z:= evalAt(sol1,tmp))) = 0 then 
              sol1:= z;
              break;
            end_if;                        
          end_while;  
        end_if;  
        //------------------  
        // [1], Step (5)(ii)
        //------------------  
        sys:= {(coeff(numerI1,i) = coeff(numerI1Unknowns,i)) $ i = 0..max(degree(numerI1),degree(numerI1Unknowns)),
               (coeff(numerI0,i) = coeff(numerI0Unknowns,i)) $ i = 0..max(degree(numerI0),degree(numerI0Unknowns))};
        if traperror((sysSpecial:= evalAt(sys,sol1))) = 0 then 
          //-------------------  
          // [1], Step (5)(iii)
          //-------------------  
          L2:= solve(sysSpecial,indets(eqnF2[cnt]) minus {x},IgnoreAnalyticConstraints);
          if type(L2) <> DOM_SET or nops(L2) = 0 or (nops(L2) > 0 and type(L2[1]) <> DOM_LIST) then 
            delete L1[ind]; // remove 'sol1' 
            L1:= sort(L1, (x,y) -> length(x) < length(y));
            L2:= {};
            for sol1 in L1 do 
              if type(sol1) = DOM_LIST and traperror((sysSpecial:= evalAt(sys,sol1))) = 0 then 
                //-------------------  
                // Still: [1], Step (5)(iii) (trying the other soltuions)
                //-------------------  
                L2:= solve(sysSpecial,indets(eqnF2[cnt]) minus {x},IgnoreAnalyticConstraints);
                if type(L2) <> DOM_SET or nops(L2) = 0 or (nops(L2) > 0 and type(L2[1]) <> DOM_LIST) then 
                  L2:= {};
                else 
                  break;
                end_if;  
              end_if;
            end_for;  
          end_if;
          if L2 <> {} then 
            sol2:= FAIL;
            for i from 1 to nops(L2) do 
              if {op(map(L2[i],elem -> op(elem,2) /*i.e. rhs(elem) */))} <> {0} then 
                sol2:= L2[i];
                break;
              end_if;  
            end_for;  
            if sol2 <> FAIL then 
              fundSys:= map(Sol_eq0F2[cnt],
                            elem -> if traperror((z:= evalAt(elem,sol2))) = 0 then
                                      z;
                                    end_if);  
              if nops(fundSys) < 3 then 
                fundSys_complete:= map(Sol_eq0F2_replacement[cnt],
                            elem -> if traperror((z:= evalAt(elem,sol2))) = 0 then
                                      z;
                                    end_if);  
                fundSys:= fundSys union {fundSys_complete[i] $ i = 1..min(nops(fundSys_complete),
                                                                          3-nops(fundSys))};                    
              end_if;  
            end_if;                      
          end_if;                      
        end_if;  
      end_if;                        
    end_while;  
    if fundSys <> {} then 
      break;
    end_if;
  end_for;
  
  // Proceed only if solutions have been found. What follows is prost-processing.                       
  if fundSys = {} then 
    return(FAIL);
  end_if;
  //------------------------------------------------------------------------------  
  // [1], Step (6)
  //------------------------------------------------------------------------------  
  // Compute 'A_tilde' as in [1], (3.16)
  t:= solvelib::getIdent(Any, indets([eq,y,x,y0,y1,y2,aa,bb,cc,dd,ee,ff,a,b,c]));
  eqI0:= y3-subs(II1,x=t)*y1-subs(II0,x=t)*y0;
  // Establishing transformation rules changing dependent variable. 
  Subs:= [0 $ 3]; 
  Subs[1]:= simplify(1/diff(x^k,x)*diff(y(x),x)); // apply [1], Lemma 2.3
  for i from 2 to 3 do 
    Subs[i]:= simplify(1/diff(x^k,x)*diff(Subs[i-1],x)); // apply [1], Lemma 2.3 
  end_for;
  Subs:= subs(Subs,[diff(y(x),x,x,x) = y3, diff(y(x),x,x) = y2, diff(y(x),x) = y1, y(x) = y0]);
  tmp:= evalAt(eqI0,[y1 = Subs[1], 
                     y2 = Subs[2], 
                     y3 = Subs[3],
                      t = x^k]); // finally transforming the idependent variable 
  A_tilde:= ode::normal(diff(tmp*1/diff(tmp,y3),y2));
  // Compute 'A_bar' as in [1], (3.17)
  eq0F2:= evalAt(eq0F2,sol2);
  MoebiusTrans:= simplify(evalAt((a*t+b)/(c*t+d),sol1));
  // Establishing transformation rules changing dependent variable. 
  Subs:= [0 $ 3]; 
  Subs[1]:= simplify(1/diff(MoebiusTrans,t)*diff(y(t),t)); // apply [1], Lemma 2.3
  for i from 2 to 3 do 
    Subs[i]:= simplify(1/diff(MoebiusTrans,t)*diff(Subs[i-1],t)); // apply [1], Lemma 2.3 
  end_for;
  Subs:= subs(Subs,[diff(y(t),t,t,t) = y3, diff(y(t),t,t) = y2, diff(y(t),t) = y1, y(t) = y0]);
  tmp:= evalAt(eq0F2,[diff(y(x),x) = Subs[1], 
                      diff(y(x),x,x) = Subs[2], 
                      diff(y(x),x,x,x) = Subs[3],
                      x = MoebiusTrans]); // finally transforming the idependent variable 
  A_bar:= ode::normal(diff(tmp*1/diff(tmp,y3),y2));
  //-----------------  
  // [1], Step (6)(i)
  //-----------------  
  if traperror((fundSys:= evalAt(fundSys,x=MoebiusTrans))) <> 0 then 
    return(FAIL)
  end_if; 
  //------------------  
  // [1], Step (6)(ii)
  //------------------  
  fundSys:= map(fundSys,_mult,exp(1/3*int(A_bar,t,intOptions)));
  //-------------------  
  // [1], Step (6)(iii)
  //-------------------  
  fundSys:= evalAt(fundSys,t=x^k); 
  //------------------  
  // [1], Step (6)(iv)
  //------------------  
  fundSys:= map(fundSys,_mult,exp(1/3*int(A_tilde-c2,x,intOptions)));
  // Simplify the solutions to be returned as far as possible.   
  fundSys:= simplify(fundSys,IgnoreAnalyticConstraints);
  /* 
     Remove unnecessary constant factors from the fundamental set of 
     solutions, i.e.: if 'c*f(x)' has been found as a solution, we
     should better return 'f(x)' and remove 'c'. 
  */
  fundSys:= map(fundSys,elem -> if type(elem) = "_mult" then 
                                  _mult(op(select({op(elem)},has,x)))
                                else 
                                  elem
                                end_if);
  return(fundSys);                   
    
end_proc:



