/*  
   
   =========================
   FROBENIUS METHOD FOR ODEs
   =========================
 
   REFERENCES: [1] G.M. Murphy: ."Ordinary differential equations and their solutions", 
                   The Method of Frobenius, Subsection b. on pp. 104-105

   CALL: 'ode::frobenius' tries to compute series expansions of linear ODEs
         at a regular singular point

*/

ode::frobenius:= proc(eq,y,x,ivs,order=ORDER,dir=Real,solveOptions,odeOptions)
  local L,n,l0,l,t,newind,oldind,der,cof,p0,i,r,f,indeq,expo,roots,e,j,scof,sol,
        tmp,N,nb,m,k,scal,frobseriescof,frobscal,soli,C,s,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  // first test if eq is linear homogeneous:
  if (L:=ode::isLODE(eq, y(x), Hlode, solveOptions, odeOptions))=FALSE then
    return(FAIL);
  else
    eq:= L[1]; 
    n:= L[4]; 
    l:= ode::vectorize(eq,y,x,n,solveOptions,odeOptions);
    l0:= [diff(y(x),x$i) $ i=0..n];
  end_if;
  
  // put the singular point x0 at 0 :
  if not iszero(ivs[1]) then
    if type(ivs[1])=stdlib::Infinity then
      t:= genident();
      newind:= map(l0,(e,t,x) -> e=subs(e,x=t),t,x);
      oldind:= map(l0,(e,t,x) -> subs(e,x=t)=e,t,x);
      eq:= subs(eq,newind);
      der[1]:= y(x);
      for i from 2 to n+1 do 
        der[i]:= expand(-diff(der[i-1],x)*x^2,optIgnoreAnalyticConstraints);
      end_for;
      eq:= subs(eq,x=1/x);
      eq:= subs(eq,oldind);
      eq:= subs(eq,{diff(y(x),x$i) = der[i+1] $ i=1..n});
      l:= Type::Linear(expand(eq,optIgnoreAnalyticConstraints),l0);
    else
      l:= map(l,()->expand(subs(args(),x=x+ivs[1]),optIgnoreAnalyticConstraints));
    end_if;
  end_if;
  
  // check if eq is regular singular at 0 and compute its cofficients
  // of its standard form :
  cof:= [];
  p0:= l[n+1]/x^n;
  for i from 1 to n+1 do
    cof:= append(cof,ode::normal(l[i]/x^(i-1)/p0));
    if not iszero(cof[i]) and traperror(taylor(cof[i],x,order,dir))<>0 then
      return(FAIL);
    end_if;
  end_for;
  
  r:= genident("r");
  f:= expand(_plus(_mult(r-i $ i=0..j-2)*cof[j] $ j=1..n+1),optIgnoreAnalyticConstraints);
  indeq:= subs(f,x=0); // indicial equation
  expo:= solvelib::discreteSolve(indeq, r, Multiple);
  if expo={} then 
    return(FAIL);
  end_if;
  
  // order the roots of the indicial equation in a standard form:
  roots:= [];
  for e in expo do
    for j from 1 to nops(roots) do
      if testtype(e[1]-roots[j][1], Type::Integer) then
        roots:= subsop(roots, j=[op(roots[j]), e[1]]); 
        break;
      end_if;
    end_for;
    if nops(roots)=0 or nops(roots)<j then
      roots:= append(roots,[e[1] $ e[2]]);
    end_if;
  end_for;
  roots:= sort(map(roots,sort,(a,b)->bool(b<a)),(a,b)->bool(nops(a)<nops(b)));
 
  //------------------------------ Subroutines ----------------------------------
  
  // Compute the coefficients of the series as in Ince, Chapter 16.11, p.396-397 
  frobseriescof:= proc(eq,r,x,order)
    local f, cof, i, mat, k, j;
  begin
    f:= series(eq,x,order,dir);
    f:= array(0..order-1, [coeff(f,x,i) $ i=0..order-1]);
    cof:= array(0..order-1, 0=1, 1=-f[1]/subs(f[0], r=r+1));
    for i from 2 to order-1 do
      mat:= matrix(i, i);
      for k from 1 to i do 
        mat[1,k]:= subs(f[k], r=r+i-k);
      end_for;
      for j from 2 to i do
        for k from j-1 to i do
          mat[j,k]:= subs(f[k-j+1], r=r+i-k);
        end_for;
      end_for;
      cof[i]:= (-1)^i*linalg::det(mat)/_mult(subs(eq,{x=0,r=r+k}) $ k=1..i);
    end_for;
    
    return(cof);
  end_proc:
    
  frobscal:= proc(eq,x,scal,order)
    local i,j;
  begin
    if iszero(scal) then
      return(eq);
    elif type(eq)="_mult" then
      i:= 0;
      repeat
        i:= i+1
      until type(eq[i])=Series::Puiseux or i>=nops(eq) end;
      if i<=nops(eq) then
        return(hold(_mult)(eq[j] $ j=1..i-1, series(eq[i]/scal, x, order,dir),
                    _mult(eq[j] $ j=i+1..nops(eq))));
      else
        return(eq);
      end_if:
    elif type(eq)=Series::Puiseux then
      return(series(eq/scal,x,order,dir));
    else
      return(eq);
    end_if:
  end_proc:
    
  //-------------------- End of subroutines --------------------
    
  scof:= frobseriescof(f,r,x,order);
  sol:= [];
  l:= 1;
  for i in roots do
    cof:= _mult(subs(f,{x=0,r=r+m}) $ m=1..i[1]-i[nops(i)]);
    tmp:= map(series(_plus(cof*scof[m]*x^m $ m=0..order-1)+x^order, x,order,dir),
              ode::normal);
    j:= 0;
    while coeff(tmp,x,j)=0 and j<order do 
      j:=j+1;
    end_while;
    if j<order then 
      scal:= coeff(tmp,x,j);
    else 
      scal:= 0; 
    end_if:
    tmp:= x^r*expr(tmp);
    N:= null();
    for j in i do
      if j=N then 
        next;
      end_if;
      N:= j;
      nb:= nops(map(i, (x,j)-> if x=j then 1 else null() end_if, j));
      sol:= append(sol,frobscal(series(subs(tmp, r=j),x,order,dir), x,
                                ode::normal(subs(scal, r=j)), order));
      for k from 2 to nb do
        tmp:= diff(tmp, r);
        sol:= append(sol,frobscal(series(subs(tmp, r=j),x,order,dir), x,
                                  ode::normal(subs(scal, r=j)),order));
      end_for:
      l:= l+nb;
      if l<=n then 
        tmp:= diff(tmp,r);
      end_if;
    end_for;
  end_for;
  
  // if initial conditions are given :
  if nops(ivs)>1 then
    soli:= [];
    for s in sol do
      if op(s,3)>=0 then
        eq:= {};
        C:= genident("C");
        for i from 2 to nops(ivs) do
          eq:= eq union {C*limit(lmonomial(diff(s,x$i-2)),x=0, dir)-ivs[i]};
        end_for:
        if eq={0} then
          soli:= append(soli, C*s)
        elif not has(eq,infinity) and not has(eq,undefined) then
          eq:= solvelib::discreteSolve(eq,C);
          if eq<>{} then
            soli:= append(soli,subs(C*s,C=op(eq)));
          end_if:
        end_if:
      end_if:
    end_for:
  else
    soli:= sol;
  end_if:
  
  // translate to be at the original point :
  if type(ivs[1])=stdlib::Infinity then
    return(subs({op(soli)}, x=1/x));
  else
    return(subs({op(soli)}, x=x-ivs[1]));
  end_if;
  
end_proc:

