/* 
   ode::normalize(Ly, y, x, n)
  
   PARAMETER:
    ++ Ly, y(x)   :DOM_EXPR
    ++ y, x       :DOM_IDENT
    ++ n          :DOM_INT

   DESCRIPTION:
    ++ returns Ly normalized, i.e. the leading coefficient is 1.

   EXAMPLE: 
    
    >> ode::normalize(-diff(y(x),x,x)/x+y(x)/4/x^3-diff(y(x),x)/x^2,y,x,2);

                                     y(x)   diff(y(x), x)
                  diff(y(x), x, x) - ---- + -------------
                                        2         x
                                     4 x
*/

ode::normalize:= proc(eq, y, x, n, solveOptions={},odeOptions={})
  local p;
begin
  p:= ode::ode2poly(eq, y, x, n, solveOptions,odeOptions);
  return(expr(multcoeffs(p, 1/expr(coeff(p, diff(y(x),x$n), 1)))));
end_proc:

  
/*
   ode::removeDenominator(Ly, y, x, n)

   PARAMETER:
    ++ Ly, y(x)   :DOM_EXPR
    ++ y, x       :DOM_IDENT
    ++ n          :DOM_INT

   DESCRIPTION: 
    ++ returns 'Ly' multiplied by the common denominator of a 'll' its 
       cofficients (the coefficient are assumed to be rational functions 
       in 'x') 

   EXAMPLE: 

    >> Ly:= 1/x*diff(y(x),x,x,x) + 1/x^2*diff(y(x),x) + 1/(x+1)*y(x):
    >> ode::removeDenominator(Ly,y,x,3,{},{})
         2
        x  y(x) + x diff(y(x), x, x, x) + x diff(y(x), x) + diff(y(x), x) +
     
            2
           x  diff(y(x), x, x, x)

*/
    
ode::removeDenominator:= proc(eq, y, x, n, solveOptions={},odeOptions={})
  local p;
begin
  p:= ode::ode2poly(eq, y, x, n, solveOptions,odeOptions);
  // Note: I have tried to introduce 'Expand=FALSE' here, but it breaks 
  //       particular test examples with an out of memory. Do not change 
  //       the line below with levity!
  p:= numer(expr(mapcoeffs(p,ode::normal,Rationalize=None)));
  return(p);
end_proc:
  

/* 
   ode::normal(args)
  
   PARAMETER:
    ++ args   :   arguments as accepted by 'normal'

   DESCRIPTION:
    ++ function is just a wrapper for 'normal' explicitly using the option 'Expand=TRUE'. 

   EXAMPLE: 
    
    >> ode::normal(1/(x+1)^2);

                                     y(x)   diff(y(x), x)
                  diff(y(x), x, x) - ---- + -------------
                                        2         x
                                     4 x
*/
  
ode::normal:= () -> if not contains({args()},Expand=FALSE) then 
                      normal(args(),Expand=TRUE) 
                    else 
                      normal(args()) 
                    end_if;
                    


