/* 

  DESCRIPTION: 

    This file contains functions for dealing with
    linear ordinary (homogeneous) differential equations.
    NOTE: almost all functions do not check arguments! 

  PARAMETERS: 
    - used Parameter:
      ++ Ly, y(x),L :DOM_EXPR
      ++ a_i, v, mp :DOM_EXPR
      ++ y, x, u,   :DOM_IDENT
      ++ n          :DOM_INT
      ++ R,F        :DOM_DOMAIN
      ++ p          :DOM_POLY
      ++ Ly         = ordinary linear homogeneous differential equation 
      ++ y(x)       = the operator (function) of Ly  
      ++ y          = name of the operator of Ly
      ++ x          = independant variable (of the operator y)
      ++ n          = order of Ly
      ++ a_i,v      = function in x 
      ++ R          = differential ring
      ++ F          = factorial domain
      ++ p          = (minimal) polynomial in u over rational functions in x
      ++ mp         = (minimal) polynomial expression in u over rational 
      ++              functions in x
      ++ u          = dependant variable of p (or mp)
      ++ T          = name of differential operator corresponding to Ly.
      ++ L          = polynomial expression in T, representing a differential operator
  
  FUNCTIONS: 

    - ode::mkODE([a_i], y, x)
       ++ returns a linear homogenous differential equation Ly in y(x), 
       ++ where the last element of the list [a_i] corresponds to
       ++ the leading coefficient of Ly.

    - ode::getOrder(Ly, y(x))
       ++ returns the order of Ly in y(x), even if Ly is not linear.

    - ode::lodoExpr(Ly, y(x), T)
      ode::lodoExpr(Ly, y, x, n, T)
      ode::lodoExpr([a_i], T)
       ++ returns a linear ordinary differential operator expression in T out 
       ++ of either a list of coefficients [a_i] of a differential equation or a
       ++ linear homogeneous differential equation Ly in y(x). 

    - ode::lodeExpr(L, T, y(x))
      ode::lodeExpr(L, T, y, x, n)
       ++ returns the differential equation in y(x) associated to the operator L
       ++ in the variable T.

    - ode::ode2poly(Ly, y, x, n)
       ++ converts Ly into a polynomial in the derivations of y(x).

    - ode::normalize(Ly, y, x, n)
       ++ returns Ly normalized, i.e. the leading coefficient is 1.

    - ode::derivationOfRootOf(p, u, x, n)
       ++ returns the n-th derivative of u, where u=RootOf(p, u) using 
       ++ p(u)=0. The result diff(u, x$n) is a polynomial in u over the
       ++ rational functions in x of degree lower than degree(p).

    - ode::applyRootOf(Ly, y, x, n, v)
       ++ applies v to Ly, if v contains RootOf else returns an error.

    - ode::evalOde(Ly, y(x)=v)
       ++ evaluates Ly with y(x)=v. 

    - ode::printMinPol(p, u, d, m, < F >)
       ++ returns a reasonable output of a computed minimal polynomial p in u,
       ++ i.e. the coefficients are in factored form in it's numerator and 
       ++ denominator:
       ++         mp(u) = u^(d*m)+c_(m-1)*u^(d*(m-1))+ ... + c_0
       ++ Thereby the parameter d gives the sparsity of p and 
       ++ the parameter m the number of it's coefficients. 
  

  REFERENCES: 

    - Walter, W. (1990). Gewhnliche Differentialgleichungen. 4th Edition.
      Berlin, Heidelberg, New York: Springer.
    - Comtet, L. (1964). Calcul pratique des coefficients de Taylor d'une 
      fonction alg`ebrique. Einseignement Math. 10, 267-270.
    - Salvy, B., Zimmermann, P. (1994). Gfun: A Maple Package for the 
      Manipulation of Generating and Holonomic Functions in One Variable. 
      toms, Vol.20 No.2, 163-177.
   
  EXAMPLES: 

    >> eq := 4*x^2*diff(y(x), x$2)+4*x*diff(y(x),x)-y(x);
    >> ode::vectorize(eq,y,x,2);   
                                               2
                                  [-1, 4 x, 4 x ]
    
    >> ode::mkODE(%, y, x);
    
                                                2
                - y(x) + 4 x diff(y(x), x) + 4 x  diff(y(x), x, x)
*/

ode::mkODE:= proc(coeff, y, x, solveOptions={}, odeOptions={})
  local i;
begin
  return(_plus( coeff[i+1]*diff(y(x), x$i) $ i=0..nops(coeff)-1 ));
end_proc:


ode::getOrder:= proc(eq, z, solveOptions={}, odeOptions={})
begin 
  return(max(-infinity, 
             op(map(ode::odeIndets(eq, {op(z,0)}, solveOptions, odeOptions), 
                    nops)))
         -1);
end_proc: 


// computes diff(RootOf(p,y),x$n) as a polynomial in y 
ode::derivationOfRootOf:=
proc(p,y,x,n,solveOptions,odeOptions)
  local py,g,res;
  option remember;
begin 
  if n=0 then
    y
  elif n=1 then
    py:=diff(p,y);
    g:=gcdex(p,py,y); // g[1]=g[2]*p+g[3]*py
    collect(divide(-diff(p,x)*ode::normal(g[3], Expand = FALSE)/g[1], p,[y],Rem),[y],ode::normal)
  else
    res:= ode::derivationOfRootOf(p,y,x,n-1,solveOptions,odeOptions);
    res:= diff(res,x)+diff(res,y)*ode::derivationOfRootOf(p,y,x,1,solveOptions,odeOptions);
    return(collect(divide(res,p,[y],Rem),[y],ode::normal));
  end_if
end_proc:


ode::applyRootOf:=
proc(eq, y, x, n, v, solveOptions, odeOptions)
  local r, p, u, i, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  v := context(v);
  if hastype(v, RootOf) then
    r := op(misc::subExpressions(v, RootOf))
  else
    error("there is no RootOf(...) to apply")
  end_if;
  p := expr(op(r, 1));
  u := op(r, 2);
/*
    eq := eval( subs(eq, y(x)=subs(v, r=u(x))) );
    subs(eq, [diff(u(x),x$i)=ode::derivationOfRootOf(p,u,x,i) $ i=0..n]); 
    simplify(numer(eval(%)))
*/
  eq := subs(eq, y(x)=subs(v, r=u(x)),EvalChanges);
  eval(simplify
       (numer(
              subs(eq,
                   [diff(u(x),x$i)=
                   ode::derivationOfRootOf(p,u,x,i,solveOptions,odeOptions) 
                   $ i=0..n]
                   )
              )
        ), optIgnoreAnalyticConstraints)
      end_proc:
      

      
ode::evalOde:=
proc(eq, subeq, solveOptions={}, odeOptions={})
  local res,r,s,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  res:= ode::isLODE(eq, op(subeq, 1), Lode, solveOptions, odeOptions);
  if nops(res)=1 then
    return(error("not an ordinary linear differential "))
  elif iszero(traperror((r:=ode::applyRootOf(res,op(subeq, 2),solveOptions,odeOptions)))) then
    return(r);
  else
    r:= subs(eq,subeq,EvalChanges);
    s:= simplify(numer(r),optIgnoreAnalyticConstraints);
    if iszero(s) then
      return(s);
    else
      return(ode::normal(s/simplify(denom(r),optIgnoreAnalyticConstraints))):
    end_if;
  end_if;
  
end_proc:


ode::printMinPol:= proc(mp, u, d, m, solveOptions, odeOptions)
  local l, ci, i, F;
begin
  userinfo(10, "compute a pretty representation of the minimal polynomial");
  //if args(0)=5 then
  //  F:=args(5);
  //  if F::hasProp(Dom::BaseDomain)=FAIL or
  //    not(F::hasProp(Cat::FactorialDomain)) then
  //    error("illegal coefficient ring")
  //  end_if
  //else
  F:=Expr;
  //end_if;
  mp := poly(mp, [u], F);
  l := [u^(d*m)];
  for i from m-1 downto 1 do
    ci := coeff(mp, d*i);
    if not iszero(ci) then
      l:= append(l, hold(_mult)(Factored::convert_to(factor(ode::normal(ci)),DOM_EXPR), u^(d*i)))
    end_if
  end_for;
  
  return(hold(_plus)(op(l), Factored::convert_to(factor(ode::normal(coeff(mp,0))),DOM_EXPR)));
end_proc:




