/*
    ode::vspace(f, m, z)

    DETAILS: 

      f: a squarefree polynomial
      m: integer
      z: independent variable of the ODE
  
      computes the solutions of the differential equation with constant
      coefficients

      a_n * diff(y(z), y$n) + ... + a_1 * diff(y(z), y) + a_0 = 0
  
         where 
  
      f^m = a_n*y^n + ... + a_0

*/

ode::vspace:= proc(f: DOM_POLY, m:DOM_INT, z, solveOptions, odeOptions)
local l, createSols, realFundamentalSystem;
begin
  // ------------------------- local methdos --------------------------------
  //
  // If both 'alpha' and 'conjugate(alpha)' are roots of 'f', we prefer a
  // "real" fundamental system over one containing 'exp(alpha*z)', 
  // 'exp(conjugate(alpha)*z)'
  realFundamentalSystem:= proc(S: DOM_SET, solveOptions, odeOptions)
    local a, b, result;
  begin
    result:= [];
    while nops(S) > 0 do
      b:= conjugate((a:= op(S, 1)));
      if a=b or not contains(S, b) then
        // cannot create pair of conjugates: degenerate case
        S:= S minus {a};
        result:= result.[exp(a*z)];
      else
        S:= S minus {a, b};
        result:= result.[exp(Re(a)*z)*cos(Im(a)*z),exp(Re(a)*z)*sin(Im(a)*z)];
      end_if;
    end_while;
    
    return(result);
  end_proc;
  
  
  createSols:=  proc(S: Type::Set, solveOptions,odeOptions)
    local i: DOM_INT, s;
  begin
    if type(S) = piecewise then
      S:= piecewise::disregardPoints(S);
    end_if;
    case type(S)
      of RootOf do
        // expand to set
        S:= RootOf::convert_to(S, DOM_SET)
        // fall through
      of DOM_SET do
        {(s*z^i $i=0..m-1) $ s in realFundamentalSystem(S,solveOptions,odeOptions)};
        break;
      of "_union" do
        map(S, createSols, solveOptions,odeOptions);
        break
      otherwise
        // we would not be able to handle ImageSets; the following might be
        // better:
        // _union(Dom::ImageSet(exp(`#x`)*z^i, `#x`, S) $i=0..m-1)
        // but how can we express a linear combination \sum_t C_t * t,
        // with t ranging over that set ?!
        ode::odeWarning("Solutions may be lost"):
        return({})
    end_case
  end_proc;
    // ------------------------- main --------------------------------
      
  l:= solve(f, op(f, [2, 1]), MaxDegree = 5, op(solveOptions));
  
  return(createSols(l,solveOptions,odeOptions));
end_proc:

