/*
   =========================
   METHODS FOR SOLVING CHINI 
   =========================
 
   REFERENCES: [1] E. Kamke: "Differentialgleichungen", I.55, p303. 

   DETAILS: Applicable to equations of the form:
 
     (eq) y' = f(x) y^n + g(x) y + h(x)

   Only a simple case is implemented: denote z=(h/f)^(1/n), if (z'-gz)/h
   is a constant alpha not depending on x then a solution of (eq) is
   z*u(x), where int(1/(u^n-alpha*u+1),u)+C=int((f/h)^(1/n)*h,x). 

*/

ode::chini:= proc(eq, y, x, solveOptions, odeOptions)
  local yp,P,n,i,f,g,h,aux,alpha,hf,sing,intOptions,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  yp:= genident();
  eq:= subs(eq,diff(y(x),x)=yp,y(x)=y, EvalChanges);
  userinfo(2,"trying to recognize an integrable Chini equation");
  if testtype(eq,Type::PolyExpr(yp)) then
    if degree((eq:= poly(eq,[yp])))=1 then
      if testtype((P:= coeff(eq,0)/coeff(eq,1)),Type::PolyExpr(y)) then
        P:= poly(-P, [y]);
        n:= degree(P);
        if not iszero(n) and contains({{0},{}},{coeff(P,y,i) $ i=2..n-1}) then
          f:= combine(coeff(P,y,n),optIgnoreAnalyticConstraints);
          // f cannot be zero due to the fact that it is the leading
          // coefficient of P 
          g:= combine(coeff(P,y,1),optIgnoreAnalyticConstraints);
          h:= combine(coeff(P,y,0),optIgnoreAnalyticConstraints);
          if not iszero(h) then 
            aux:= ode::normal(1/n/f/h^2*(f*diff(h,x)-diff(f,x)*h-n*f*h*g),Expand=FALSE);
            alpha:= ode::normal(aux^n*h/f);
            if not has(alpha,x) then
              userinfo(2,"Integrable Chini equation");
              hf:= factor(combine(h/f,optIgnoreAnalyticConstraints));
              hf:= coerce(hf, DOM_LIST);
              hf:= _mult(hf[1]^(1/n), hf[2*i]^(hf[2*i+1]/n) $ i=1..nops(hf) div 2);
              if iszero(alpha) then
                eq:= int(1/(y^n+1),y,intOptions);
                eq:= eq - (genident("C")+int(1/hf*h,x,intOptions));
                sing:= solve(y^n+1,y,op(solveOptions))
              else
                alpha:=ode::normal(expand(hf*aux, optIgnoreAnalyticConstraints));
                // special case implemented directly here, because
                // int may return correct antiderivatives in a large
                // number of forms and the solve command below does
                // not handle most of them well.
                if n=2 then
                  if alpha=2 then
                    eq:= -1/(y - 1);
                  elif alpha=-2 then
                    eq:= -1/(y + 1);
                  else
                    eq:= 2*arctan((2*y-alpha)/sqrt(4-alpha^2))/sqrt(4-alpha^2);
                  end_if;
                else
                  eq:= int(1/(y^n-alpha*y+1),y,intOptions);
                end_if;
                eq:= eq + 1/alpha*genident("C")-int(h/hf,x,intOptions);
                sing:= solve(y^n-alpha*y+1,y,op(solveOptions))
              end_if;
              eq:= solve(eq,y,op(solveOptions)) union sing;
              return(ode::mapsol(eq, _mult, hf, solveOptions, odeOptions));
            end_if;
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;
  if contains(odeOptions, Type = Chini) then 
    ode::odeWarning("cannot detect Chini equation");
  end_if;  
  
  return(FAIL);
end_proc:

