/*  
   =========================
   METHODS FOR CLAIRAUT ODES
   =========================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 50, pp. 216

   DETAILS:  

     Input: eq is a first-order ODE in y(x).
     Output: a solution or FAIL.
  
   EXAMPLE: ode::solve((x*diff(y(x),x)-y(x))^2-diff(y(x),x)^2-1,y(x));

*/

ode::clairaut :=
proc(eq,y,x,solveOptions,odeOptions) 
  local eq0, f, yp, sp, Z, Y, res, optIgnoreAnalyticConstraints, tmp;
begin 
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  res:= FAIL;
  if type(eq) = "_equal" then 
    eq:= op(eq,1)-op(eq,2); 
  end_if;
  yp:= genident();
  Z:= genident();
  eq0:= eq; 
  if traperror((eq:= expand(subs(eq,diff(y(x),x)=yp,y(x)=x*yp-Z,EvalChanges),
                            optIgnoreAnalyticConstraints))) <> 0 then 
    return(FAIL);
  end_if;  
  if not has(eq,{x,y}) and type(eq) = "_plus" then
    f:= select(eq,has,Z);
    // The following test is too restrictive : for ex. yy'-xy'^2+1=0
    // is a Clairaut equation but fails the test
    if not has(f,yp) then
      userinfo(1,"Clairaut's equation");
      res:= ode::clairautEq(subs(f,Z=x),subs(f-eq,yp=x),x,solveOptions,odeOptions);
    else // we make this alternative test:
      if type(f) <> "_mult" then 
        f:= hold(_mult)(f);
      end_if;
      sp:= split(f,has,Z);
      if not has(sp[1],yp) then
        userinfo(1,"Clairaut's equation");
        res:= ode::clairautEq(subs(f/sp[2],Z=x),
                              expand(subs(f/sp[2]-eq/sp[2],yp=x),optIgnoreAnalyticConstraints),x,
                              solveOptions,odeOptions);
      end_if
    end_if
  end_if;
  if res = FAIL and ode::order(eq0,{y},solveOptions,odeOptions) = 1 then
    eq:= subs(eq0,diff(y(x),x)=yp,y(x)=Z,EvalChanges);
    if testtype((Y:= solvelib::discreteSolve(eq,Z,op(solveOptions))),DOM_SET) then
      Y:= map(Y, () -> if not has((tmp:= ode::normal(-args(1)+x*yp)), x) then
                         op(ode::clairautEq(x,subs(tmp,yp=x,EvalChanges),x,solveOptions,odeOptions));
                       else 
                         null();
                       end_if);
      if Y<>{} then
        userinfo(1,"Clairaut's equation");
        res:= Y;
      end_if;
    end_if;
  end_if;
  if contains(odeOptions, Type = Clairaut) and res = FAIL then 
    ode::odeWarning("cannot detect Clairaut equation"); 
  end_if;  
  return(res);
end_proc:

/* The method solves f(x*y'-y) = g(y'). A general solution is given by 
   f(xC-y)=g(C) where C is a constant. 

    EXAMPLE: ode::clairautEq(x^2,x^2-1,x);
*/

ode::clairautEq :=
proc(_f_,_g_,x,solveOptions,odeOptions) // f and g are expressions in x 
  local C,gensol,eq,partsol,Partsol,p,y,yp,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  if not has(_f_, x) then
    return(FAIL)
  end_if;
  y:= genident();
  yp:= genident();
  userinfo(3,"f=",_f_,"g=",_g_);
  // Search for a general solution 
  C:= genident("C");
  gensol:= solvelib::discreteSolve(subs(_f_,x=x*C-y,EvalChanges)=subs(_g_,x=C,EvalChanges),y,
                                   op(solveOptions));
  if gensol=FAIL then
    userinfo(3,"finding general solution failed while solving ".
               expr2text(subs(_f_,x=x*C-y,EvalChanges)=subs(_g_,x=C,EvalChanges))." in ".expr2text(y));
    return(FAIL);
  else
    userinfo(3,"general solution is",gensol);
  end_if;
  // Search for a singular solution 
  eq:= subs(diff(_f_,x),x=x*yp-y,EvalChanges)*x-subs(diff(_g_,x),x=yp,EvalChanges);
  partsol:= solvelib::discreteSolve(eq,yp,op(solveOptions));
   //partsol:=solvelib::discreteSolve(eq,yp,MaxDegree=ode::maxdegree);
  userinfo(3,"singular solution for y' is",partsol);
  if type(partsol)="solve" or has(partsol, solve) or partsol=FAIL then
    partsol:= {};
    userinfo(3,"particular solution cannot be found!");         
    ode::odeWarning("particular solution of Clairaut ODE cannot be found");
  end_if;
  Partsol:= {};
  if partsol<>{} then
    for p in partsol do
      eq:= ode::normal(subs(_f_,x=x*p-y,EvalChanges)-subs(_g_,x=p,EvalChanges));
      p:= solvelib::discreteSolve(eq,y,op(solveOptions));
      //partsol:=solvelib::discreteSolve(eq,y,MaxDegree=lev);
      if p<>FAIL then
        Partsol:= Partsol union p
      end_if; // ugly poor work around
    end_for;  
  end_if;
  userinfo(3,"particular solution is",Partsol);
  return(gensol union Partsol);
end_proc:

