/*  
   ==============================================
   METHODS FOR ODES USING CONTACT TRANSFORMATIONS
   ==============================================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 53, pp. 227.

   DETAILS: 

     Input: eq is a first-order ODE in y(x).
     Output: a solution or FAIL.

   EXAMPLES: 

     >> ode::contact(2*y(x)*diff(y(x),x)^2-2*x*diff(y(x),x)-y(x),y,x);

*/

ode::contact:= proc(eq,y,x,solveOptions,odeOptions)
  local n,sol,EQ,m, P, X, Y;
begin
  delete P,X,Y;
  P:= genident();
  X:= genident();
  Y:= genident();
  userinfo(2,"trying contact transformation");
  eq:= subs(eq,diff(y(x),x)=X);
  n:= degree(eq,[X]); // degree of initial equation in y' 
  eq:= subs(eq,y(x)=P*X-Y,x=P);
  m:= degree(eq,[P]);
  if m=FAIL or n=FAIL then 
    return(FAIL) 
  end_if;
  if m<n then // to avoid loops
    userinfo(1,"contact transformation works");
    EQ:= subs(eq,Y=Y(X),P=diff(Y(X),X));
    userinfo(2,"solving new equation",EQ);
    sol:= genident("C")*ode::firstord(EQ,Y,X,solveOptions,odeOptions); 
    // firstord returns either FAIL or a solution
    if sol<>FAIL then
      userinfo(2,"solution of new equation is",sol);
      sol:= subs({Y-sol,eq},P=x,Y=x*X-y);
      userinfo(2,"after back contact transformation, we get",sol);
      if (sol:= ode::eliminate(op(sol),X,solveOptions,odeOptions)) <> FAIL then
        userinfo(2,"after elimination of X, we get",sol);
        sol:= solve(sol,y,op(solveOptions));
        return(sol)
      end_if
    else
      userinfo(1,"contact transformation failed");
    end_if;
  end_if;
  
  return(FAIL);
end_proc:


// eliminates x between f and g 
ode::eliminate:= proc(f,g,x,solveOptions,odeOptions)
begin
  f:= ode::topoly(f,x,solveOptions,odeOptions);
  if f <> FAIL then
    g:= ode::topoly(g,x,solveOptions,odeOptions);
    if g <> FAIL then
      return(polylib::resultant(f,g,x))
    end_if
  end_if;
  
  return(FAIL);
end_proc:

/* tries to convert expression f into a polynomial in x
  (this can add some other solution like in resultant) */
  
ode::topoly:= proc(f,x,solveOptions,odeOptions)
  local k,good,g;
begin
  if testtype(f,Type::PolyExpr(x)) then
    return(f);
  elif type(f)="_plus" then
    good:= select(f,testtype,Type::PolyExpr(x));
    f:= f-good;
    if type(f)="_mult" then
      g:= select(f,testtype,Type::PolyExpr(x));
      assert(not iszero(g));
      f:= f/g;
      if type(f)="_power" then
        if type((k:=op(f,2)))=DOM_RAT then // good+g*bad^k=0
          return((-good)^op(k,2)+g^op(k,2)*f^op(k,2))
        end_if
      end_if
    end_if
  end_if;
  
  return(FAIL);
end_proc:

