/* 

   ==============================
   REDUCTION METHOD BY D'ALEMBERT
   ==============================

   REFERENCES: [1] W. Walter: Gewhnliche Differentialgleichungen. 6th Edition.
                   Berlin, Heidelberg, New York: Springer, pp. 141.

   DESCRIPTION: This file contains the reduction method of d'Alembert for
                ordinary linear (homogeneous) differential equations.
 
   PARAMETERS: 

      ++ Ly, y(x), v:DOM_EXPR
      ++ y, x       :DOM_IDENT
      ++ n          :DOM_INT
      ++ Ly    = ordinary linear homogeneous differential equation 
      ++ y(x)  = the operator (function) of Ly  
      ++ y     = name of the operator of Ly
      ++ x     = dependent variable of the operator y
      ++ n     = order of Ly
      ++ v     = special nonzero solution of Ly

   METHODS:
 
      - ode::dAlembert(Ly, y(x), v)
           ++ returns the reduced equation of an equation Ly of arbitrary order, i.e.
           ++ if u is a solution of the reduced equation then v*int(u) is
           ++ a solution of Ly.
     
      - ode::secondOrderReduction(Ly, y, x, v)
           ++ returns the reduced first order equation of a normal form
           ++ second order equation Ly, i.e. the leading coefficient must be 1.
           ++ NOTE: there is no test of arguments.
     
      - ode::generalReduction(Ly, y, x, n, v, < Normal >)
           ++ returns the reduced equation of an equation Ly of arbitrary order.
           ++ When the option Normal is used, the leading coefficient of the 
           ++ reduced equation will be normalized (i.e. is 1).
           ++ NOTE: there is no test of arguments.

  EXAMPLES: 

      >> wy := diff(y(x), x, x)*(-3) + x^(-2)*y(x)*6;
      
      >> ode::dAlembert(wy, y(x), x^2);    

*/

ode::dAlembert:= proc(eq,z,v,solveOptions={},odeOptions={})
  begin
    return(ode::generalReduction(ode::isLODE(eq, z, Hlode, solveOptions, odeOptions),
                                 v, Normal, solveOptions, odeOptions));
  end_proc:
    
ode::secondOrderReduction:= proc(eq, y, x, v, solveOptions, odeOptions)
  local a;
  begin 
    a:= ode::vectorize(eq,y,x,2,solveOptions,odeOptions);
    if a[3] <> 1 then 
      error("equation must be in normal form");
    end_if;
    return(diff(y(x),x)+ode::normal(a[2] + 2*diff(v,x)/v, Expand=FALSE)*y(x));
  end_proc:

ode::generalReduction:= proc(eq, y, x, n, v, o=" ", solveOptions, odeOptions) // Option: Normal 
  local a, dv, i, j, b;
  begin
    a:= ode::vectorize(eq, y, x, n, solveOptions, odeOptions);
    dv:= [ diff(v, x$i) $ i=0..n ];
    b:= [_plus(binomial(i,j)*a[i+1]*dv[i-j+1] $ i=j..n) $ j=1..n];
    if o=Normal then 
      b:= map(map(b, _mult, 1/v), ode::normal, Expand=FALSE) 
    end_if;
    return(ode::mkODE(b, y, x, solveOptions,odeOptions));
  end_proc:

/*
ode::reduceOrder:=
  proc(eq,z,v)
    local ind,param,sol,e;
  begin
    ind:=indets(eq) union indets(z) union indets(v);
    sol:=ode::reduction(eq,z,v);
    param:=[op(indets(sol) minus ind)];
    if type(sol)=DOM_SET then
      sol:=op(sol);
      {coeff(sol,param,e,1) $ e in param}
    else {v}
    end_if;
  end_proc:
*/


/* ode::reduction(eq,yofx,v) returns a complete solution of eq,
  v being a particular solution */
 
ode::reduction:= proc(eq, yofx, v, solveOptions, odeOptions) 
  local s,n,C,ind,lC,ss,i,j,intOptions;
begin
   intOptions:= null();            
   if has(solveOptions, IgnoreSpecialCases) then 
     intOptions:= intOptions,IgnoreSpecialCases;
   end_if;
   if has(solveOptions, IgnoreAnalyticConstraints) then   
     intOptions:= intOptions,IgnoreAnalyticConstraints;
   end_if;   
   n:= ode::order(eq,{op(yofx,0)},solveOptions,odeOptions);
   ind:= indets(eq) minus {PI, EULER, CATALAN};
   eq:= ode::dAlembert(eq, yofx, v, solveOptions, odeOptions); // reduced equation 
   s:= ode::solve_linear(ode::vectorize(eq,op(yofx,0),op(yofx),n,solveOptions,odeOptions),
                         op(yofx),n-1,solveOptions, odeOptions);
   lC:=indets(s) minus ind;
   /*  
      I modifyed the previous code since ode::solve_linear does not return a
      basis of solutions of the form {y_1,y_2,...,y_(n-1)} but in the form
      {C_1*y_1+C_2*y_2+...+C_(n-1)y_(n-1)} where the C_i are the constants of
      integration.
   */
   if type(s)=DOM_SET and nops(lC)=n-1 then
     ss:={};
     for i from 1 to nops(lC) do
       ss:= ss union subs(s, [lC[i]=1, lC[j]=0 $ j=1..nops(lC)])
     end_for;
     C:= genident("C");
     return({C*v+_plus(v*lC[i]*eval(int(ss[i], op(yofx), intOptions)) $ i=1..nops(lC))});
   else
     return(FAIL);
   end_if
   
end_proc:

    
