/*

  =====================================
  METHODS FOR ODES EQUIDIMENSIONAL IN X 
  =====================================

  REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 59 pp. 250

    ode::equidimensional_in_x(eq,y,x,n) solves the nth order ODE eq=0 for y(x)
    if it is equidimensional in x 
    Input: eq is an expression, y,x are variables, n is the order
    Output: FAIL or a list of solutions

  EXAMPLES: 

    >> ode::equidimensional_in_x(x*diff(y(x),x,x)-2*y(x)*diff(y(x),x),y,x,2);
    >> ode::equidimensional_in_x(diff(y(x),x$2)-2/x*diff(y(x),x)+3/x^2*y(x)-2*x+1,y,x,2);
*/

/* 
  ok=TRUE when the equation is known to be equidimensional in x,
  for example when called from ode::scale_invariant 
*/
ode::equidimensional_in_x :=
proc(eq,_0_y,_0_x,n,ok=FALSE,solveOptions,odeOptions) 
  local eq2,i,l, X, a, t, intterm, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;  // delete X,a,t;
  X:= genident();
  a:= genident();
  t:= genident();
  if ok<>TRUE then
    // first check if there are no expressions of the form int(f(x),x) :
    intterm:= misc::subExpressions(eq, "int");
    if not has(map(intterm, ()->bool(op(args(1),2)=_0_x)),TRUE) then
      // first substitute x by a*X 
      eq2:= subs(eq,(diff(_0_y(_0_x),_0_x$i)=diff(_0_y(X),X$i)/a^i) $ i=1..n,
                 _0_y(_0_x)=_0_y(X),_0_x=a*X, X=_0_x);
      ok:= not has(ode::normal(eq2/eq),{_0_x,_0_y});
    end_if;
  end_if;
  if ok<>TRUE then
    return(FAIL);
  else 
    // equidimensional in x
    userinfo(1,"equidimensional in x equation");
    /* 
       change x to exp(t), and exp(t) to 1 because it should disappear
	     (autonomous equation) 
    */
    l:= diff(_0_y(_0_x),_0_x$i) = ode::equidimx_change(_0_y,t,i,solveOptions,odeOptions) $ i=1..n, _0_y(_0_x)=_0_y(t),_0_x=1;
    eq:= subs(eq,l);
    eq:= numer(eq);
    userinfo(2,"new equation to solve is",eq);
    /* 
       ode::depth:=ode::depth+1;
       eq:=ode::solve_eq(eq,_0_y,t);
       sysassign(ode::depth,ode::depth-1); 
    */
    eq:= ode::autonomousEq(eq,_0_y,t,n,solveOptions,odeOptions);
    if has(eq,FAIL) then
      userinfo(1,"equidimensional in x method failed"); 
      return(FAIL);
    end_if;
    return(map(subs(eq,t=ln(_0_x)), x -> if testtype(x, Type::Arithmetical) and not
                                            (has(x, solve)) 
                                         then 
                                           ode::normal(x) 
                                         else 
                                           x 
                                         end_if)
          );
  end_if;
end_proc:

/*
   returns the value of diff(y(x),x$n) in terms of diff(Y(t),x$i) where
   y(x) = y(exp(t)) = Y(t) 
*/
ode::equidimx_change :=
proc(_0_y,t,n,solveOptions,odeOptions)
  local i;
  option remember;
begin
   // exp(-n*t)*_plus(ode::euler_coeff(n,i)*diff(_0_y(t),t$i)$i=1..n) 
   // remove the exp(-n*t) factor because it will be replaced by 1 
  _plus(ode::euler_coeff(n,i,solveOptions,odeOptions)*diff(_0_y(t),t$i)$i=1..n)
end_proc:

/* 
  ======================
  METHODS FOR EULER ODES
  ======================

  REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 61 pp. 256

  Euler equations are linear ODEs of the form
     
     a[0] x^n y^(n)+a[1] x^{n-1} y^(n-1) + ... + a[n-1] x y' + a[n] y = 0.

  They represent a special sub-class of equidimensional-in-x equations,
  which are solved by the transformation x = exp(t). 

    ode::euler(l,z,n) solves the equation 

     l[1]*y(z)+l[2]*diff(y(z),z)+...+l[n+1]*diff(y(z),z$n)=0

  EXAMPLES: 
 
    >> ode::euler([2,-2*x,x^2],x,2);
    >> ode::euler([8/x^4,8/x^3,-4/x^2,0,1],x,4);
*/

ode::euler :=
proc(l,x,n,solveOptions,odeOptions) local L,i,m;
begin
  L:= [_plus(l[m+1]*ode::euler_coeff(m,i,solveOptions,odeOptions)/x^m$m=i..n)$i=0..n];
  L:= map(L,ode::normal);
  if has(L,x) then
    return(FAIL);
  else
    userinfo(1,"Euler's equation");
    L:= ode::constant(L,x,n,solveOptions,odeOptions); // constant coefficient equation
    return(subs(L,x=ln(x),EvalChanges));
  end_if;
end_proc:

/* 
  returns the coefficient of diff(Y(t),t$i)/exp(n*t) in diff(y(x),x$n)
  for the change y(x) = y(exp(t)) = Y(t), cf table 49 p. 194 in Zwillinger.
  we should have 0<=i<=n 
*/

ode::euler_coeff :=
proc(n,i,solveOptions,odeOptions)
  option remember;
begin
  if n=i then
    return(1);
  elif i=0 then
    return(0);
  else
    return(ode::euler_coeff(n-1,i-1,solveOptions,odeOptions)-
           (n-1)*ode::euler_coeff(n-1,i,solveOptions,odeOptions));
  end_if;
end_proc:

