/* 
   ============================
   METHODS FOR HOMOGENEOUS ODES
   ============================

   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 73 pp. 297

   DETAILS: 

    ode::homogeneous(eq,y,x) tries to solve the ODE eq with respect to y(x)
    using the method of homogeneous equations. It returns either a set of
    solutions or FAIL. 

    Applicable to: first order ODEs of the form y'=P(x,y)/Q(x,y)
    where P and Q are homogeneous in x and y of the same degree.


   EXAMPLES: 

      >> ode::homogeneous(diff(y(x),x)*(x^4-2*x*y(x)^3)-2*x^3*y(x)+y(x)^4,y,x);

    The following comment holds only for the older version of [1]. The edition
    we now use is fixed (corresponds to eq. (73.2) in the edition we now use). 
 
      There is a typo in Zwillinger's example: in (61.5), the right hand side 
      should be (2*v-v^4)/(1-2*v^3), or the initial equation should be 
      y'=(x^3y+y^4)/(x^4-2xy^3).
*/

ode::homogeneous:= proc(eq,y,x,solveOptions,odeOptions) 
  local v, yp, res, optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  yp:= genident();
  userinfo(2, "trying to recognize an homogeneous equation");
  eq:= subs(eq, diff(y(x), x) = yp);
  eq:= ode::normal(eq,Expand=FALSE);
  if testtype(eq, Type::PolyExpr(yp)) then
    if degree(eq, [yp], yp) = 1 then
      eq:= -coeff(eq,[yp],yp,0)/coeff(eq,[yp],yp,1);
       v:= genident("v"); // to avoid name space problems
      eq:= ode::normal(subs(eq,y(x)=v*x,EvalChanges));
      if not has(eq,x) then // homogeneous
        userinfo(1,"homogeneous equation");
        // eqq:=genident("C")* expand(exp(combine(int(1/(eq-v),v),ln,optIgnoreAnalyticConstraints)))-x;
        // res1:=solvelib::discreteSolve(eqq,v);
        res:= ode::separate(x*diff(v(x),x)+v(x)-subs(eq,v=v(x)),v,x,1,solveOptions,{});
        if res <> FAIL and not hastype(res, "solve") then
          return(ode::mapsol(res, _mult, x, solveOptions, {}));
        else
          return(FAIL);
        end_if;
      end_if;
    end_if
  end_if;
  if contains(odeOptions, Type = Homogeneous) then 
    ode::odeWarning("cannot detect homogeneous ordinary differential equation");
  end_if;  
  return(FAIL);
end_proc:


