
/* 

   REFERENCES: [1] E. Kamke: "Differentialgleichungen", p. 19
               [2] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 73, pp. 297  

   DETAILS: 

     ode::homogeneousC(eq,y,x) tries to solve the ODE eq with respect to y(x)
     using the method of homogeneous C equations (the letter C comes from the 
     classification of homogeneous ODEs by Kamke on p. 19-20). It returns 
     either a set of solutions or FAIL.
  
     Applicable to: first order ODEs of the form:
  
          (1)    y' = f((a1*x+b1*y+c1)/(a2*x+b2*y+c2))
  
     where f is any function and D = a1*b2 - a2*b1 <> 0.
  
     For further details see: [1] eq. (73.4) and [2].

   METHOD: when doing the change of variables 

       X = x - (b1*c2 - c1*b2)/D
       Y = y - (-a1*c2 + c1*a2)/D

     it reduces to solve the homogeneous equation:

          (2)    Y' = f((a1*X+b1*Y)/(a2*X+b2*Y))

     If an ODE y'=F(x,y) is of the form (1) then we have
     
           F_x/F_y = (D*y + a1*c2 - c1*a2)/(-D*x + b1*c2 - b2*c1)

     where F_x (resp. F_y) is the partial derivative of F in 
     x (resp. y).


   NOTE:  (i) The formulas above have been verified. They are o.k. 
         (ii) The success of this method depends on the success of 
              'ode::homogeneous' and the success of 'ode::homogeneous'
              depends on the success of 'ode::separate'. In case that 
              'ode:separate' does not return an explicit solution, no 
              result can be returned!   

*/

ode::homogeneousC:= proc(eq,y,x,solveOptions, odeOptions) 
  local yp,eqx,eqy,R,num,den,delta,A,B,aux;
begin
  yp:= genident();
  userinfo(2, "trying to recognize a homogeneous equation of type C");
  eq:= subs(eq, diff(y(x), x) = yp, y(x)=y);
  if testtype(eq, Type::PolyExpr(yp)) then
    if degree(eq, [yp], yp) = 1 then
      eq:= ode::normal(-coeff(eq,[yp],yp,0)/coeff(eq,[yp],yp,1),Expand=FALSE);
      eqx:= diff(eq,x); 
      eqy:= diff(eq,y);
      if eqx <> 0 and eqy <> 0 then
        R:= ode::normal(eqx/eqy,List);
        if testtype((num:=R[1]), Type::PolyExpr(y)) and
           testtype((den:=R[2]), Type::PolyExpr(x)) and
           not has(num,x) and not has(den, y) then
           if degree(num, [y], y) = 1 and degree(den, [x], x) = 1 then
             if (delta:= coeff(num, [y], y, 1))+coeff(den, [x], x, 1) = 0 then
               userinfo(1,"type C homogeneous equation");
               A:= coeff(den, [x], x, 0)/delta;
               B:= -coeff(num, [y], y, 0)/delta;
               eq:= subs(eq, {x=x+A, y=y+B});
               aux:= ode::homogeneous(diff(y(x),x)-subs(eq, y=y(x)), y, x,
                                      solveOptions,odeOptions);
               if aux <> FAIL then
                 userinfo(1,"homogeneous C worked");
                 aux:= subs(aux, x=x-A,EvalChanges);
                 return(ode::mapsol(aux, _plus,  B, solveOptions, odeOptions))
               else
                 userinfo(1,"homogeneous C failed");
               end_if;
             end_if;   
           end_if;        
        end_if;
      end_if;
    end_if;
  end_if;
  
  return(FAIL);
end_proc:

      