
/* 
   REFERENCES: [1] E. Kamke: "Differentialgleichungen", p. 20
               [2] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 73, pp. 297  

   DETAILS: 

    ode::homogeneousD(eq,y,x) tries to solve the ODE eq with respect to y(x)
    using the method of homogeneous D equations (the letter D comes from the
    classification of homogeneous ODEs by Kamke on p.19-20). It returns either
    a set of solutions or FAIL.

    Applicable to: first order ODEs of the form :
    
         (1)    y' = y/x + f(x)*g(y/x)

    Reference: [1]

   METHOD: 

    By the change of variable y = x*u(x) in the ODE (1) we get the
    separable equation : x*u'(x) = f(x)g(u).

   EXAMPLES: 

     >> ode::solve(x*(x^2+y(x)^2)*diff(y(x),x)-(x^2+x^4+y(x)^2)*y(x),y(x));
*/

ode::homogeneousD:= proc(eq,y,x,solveOptions,odeOptions) 
  local yp,u,aux;
begin
  yp:= genident();
  userinfo(2, "trying to recognize an homogeneous equation of type D");
  eq:= subs(eq, diff(y(x), x) = yp);
  if testtype(eq, Type::PolyExpr(yp)) then
    if degree(eq, [yp], yp) = 1 then
      eq:= ode::normal(-coeff(eq,[yp],yp,0)/coeff(eq,[yp],yp,1),Expand=FALSE);
      eq:= ode::normal(eq-y(x)/x,Expand=FALSE);
      u:= genident("u");
      eq:= ode::normal(subs(eq,y(x) = u(x)*x,EvalChanges));
      aux:= ode::separate(x*diff(u(x),x)-eq,u,x,1,solveOptions,odeOptions);
      if aux <> FAIL then
        userinfo(1,"homogeneous D worked");
        return(ode::mapsol(aux, _mult, x, solveOptions, odeOptions));
      end_if;
    end_if;
  end_if;
  
  return(FAIL);
end_proc:
      