/* 
   =========================
   METHODS FOR LAGRANGE ODES
   =========================

   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 81 pp. 328
 
   DETAILS: 

     ode::lagrange(eq,y,x) tries to solve the first-order ODE eq=0 wrt y(x)
     trying to recognize a Lagrange equation, i.e. an equation of the form

                            y = x*F(y') + G(y')
   
     If F=1, then we have a Clairaut's equation.


   EXAMPLES: 

     >> ode::solve(y(x)-2*x*diff(y(x),x)+a*diff(y(x),x)^3,y(x));

     >> ode::solve(y(x)-2*x*diff(y(x),x)+diff(y(x),x)^2,y(x));

     >> ode::solve(diff(y(x),x)^4-3*(1-x)*diff(y(x),x)^2+3*(1-2*y(x))*diff(y(x),x)+3*x, y(x));
*/

ode::lagrange:= proc(eq,y,x,solveOptions,odeOptions)
  local p,F,G,eqinit,eq1,eq2,r,rr,sing,C,eqC,solC,X,optIgnoreAnalyticConstraints,
        regular:DOM_PROC;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;  userinfo(2,"trying to recognize a Lagrange equation");
  p:= genident("p");// to avoid name space pb
  eqinit:= eq;
  eq:= subs(eq,diff(y(x),x)=p,y(x)=y);
  if testtype(eq,Type::PolyExpr(y)) then
    if degree((eq:= poly(eq,[y])))=1 then
      eq:= -coeff(eq,0)/coeff(eq,1); // should match x*F(y')+G(y')
      if testtype(eq,Type::PolyExpr(x)) then
        if degree((eq:= poly(eq,[x])))=1 then
          userinfo(1,"Lagrange equation");
          F:= ode::normal(coeff(eq,1),Expand=FALSE);
          G:= ode::normal(coeff(eq,0),Expand=FALSE);
          // If F=p then we can't divide by (p-F) and the result is given
          // by ode::clairaut(eq,y,x)
          if F = p then
            return(ode::clairaut(eqinit,y,x,solveOptions,{}))
          else
            X:= genident("X");
            eq:= diff(X(p),p)-X(p)*diff(F,p)/(p-F)-diff(G,p)/(p-F); // eq. (81.4) in [1]
            userinfo(2,"new equation is",eq);
            sysassign(ode::depth,ode::depth+1);
            eq:= ode::solve_eq(eq,X,p,{},solveOptions,{}); // eq. (81.5) in [1]
            sysassign(ode::depth,ode::depth-1);
            eq:= piecewise::disregardPoints(eq);
            if has(eq,FAIL) then
              userinfo(1,"Lagrange method failed");
              return(FAIL)
            end_if;
            if testtype((sing:=solvelib::discreteSolve(p-F,p,MaxDegree=3,
                                                       op(solveOptions))),
                        DOM_SET) then
              sing:=map(sing, int, x)
            else
              sing:={}
            end_if;
            // -----------------------------------------------------            
            // local method 'sing' for computing singular solutions
            // -----------------------------------------------------            
            sing:=map(sing,
                      proc()
                      begin
                        C:=genident("C");
                        eqC:=subs(eqinit,y(x)=args(1)+C);
                        if traperror((eqC:=eval(eqC)))=0 then
                          eqC:= ode::normal(eqC,Expand=FALSE);
                          if (solC:=
                              solvelib::discreteSolve(eqC,C,op(solveOptions)))<>{} then
                            return(args(1)+op(solC))
                          end_if
                        end_if;
                      null()
                      end_proc);
            // ------------------------------------------------------            
            // local method 'regular' for computing regular solutions
            // ------------------------------------------------------            
            regular:= proc(eq)
              local tmp;  
            begin
              if eq = {} then
                return({})
              end_if;
              if hastype(eq, "ln") and hastype(eq, "exp") then 
                eq:= simplify(eq, optIgnoreAnalyticConstraints);
              end_if;
              if type(eq) <> DOM_SET then 
                return(FAIL);
              end_if;  
              eq1:= map(eq, proc() begin args(1)-x end_proc);
              eq2:= map(map(eq1,numer),ode::eliminate,numer(y-x*F-G),p,solveOptions,odeOptions);
              // p=y' should not appear any more now
              if has(eq2,FAIL) or has(eq2,p) or not has(eq2,{x,y}) then
                if has(eq1, FAIL) then
                  userinfo(1,"Lagrange method failed");
                  return(FAIL)
                else
                  eq1:=select(eq1, has, p);
                  r:=solve(op(eq1), p, op(solveOptions));
                  if testtype(r, DOM_SET) then
                    userinfo(1,"Lagrange method worked");
                    tmp:= {expand(x*subs(F, p=rr)+subs(G, p=rr),optIgnoreAnalyticConstraints) $ rr in r};
                    return(simplify(tmp,optIgnoreAnalyticConstraints) union sing)                      
                  elif not has(r, hold(solve)) then
                    userinfo(1,"Lagrange method worked, parametric solution");
                    // Note by Kai: Here 'r' itself might already be an 'ImageSet'
                    //              itself. It seems like this is intended, so I
                    //              currently do not change this. In general, it
                    //              does not make so much sense to generate 
                    //              ImageSets of ImageSets, especially when one 
                    //              has to solve an initial value problem with that
                    //              general.
                    return(Dom::ImageSet(expand(x*F+ G,optIgnoreAnalyticConstraints), p, r));
                  else
                    return(FAIL)
                  end_if:
                end_if;
              else
                eq2:=map(eq2, proc()
                              begin
                                solvelib::discreteSolve
                                (expand(
                                        args(1)/content(args(1),[y]),optIgnoreAnalyticConstraints),
                                 y,MaxDegree=2,
                                 op(solveOptions))
                              end_proc
                         );
                if contains(eq2, FAIL) then
                  return(FAIL)
                end_if;
                userinfo(1,"Lagrange method worked");
                return(op(eq2) union sing)
              end_if
            end_proc;
            
            // END of local procedures

            eq:= regular(eq);  
            //eq:= piecewise::extmap(eq, regular);
            if has(eq, FAIL) then
              return(FAIL);
            else
              return(eq);
            end_if
          end_if
        end_if
      end_if
    end_if
  end_if;
  if contains(odeOptions, Type = Lagrange) then 
    ode::odeWarning("cannot detect Lagrange equation");
  end_if;  
  return(FAIL);
end_proc:

// eliminates x between f and g 
// This procedure is also defined in 'contact.mu'.
ode::eliminate :=
proc(f,g,x,solveOptions,odeOptions)
begin
  f:= ode::topoly(f,x,solveOptions,odeOptions);
  if f<>FAIL then
    g:= ode::topoly(g,x,solveOptions,odeOptions);
    if g<>FAIL then
      return(polylib::resultant(f,g,x))
    end_if
  end_if;
  return(FAIL);
end_proc:

/* tries to convert expression f into a polynomial in x
  (this can add some other solution like in resultant) */
// This procedure is also defined in 'contact.mu'.
ode::topoly :=
proc(f,x,solveOptions,odeOptions)
  local k,good,g;
begin
  if testtype(f,Type::PolyExpr(x)) then
    return(f)
  elif type(f)="_plus" then
    good:=select(f,testtype,Type::PolyExpr(x));
    f:= f-good;
    if type(f)="_mult" then
      g:=select(f,testtype,Type::PolyExpr(x));
      f:= f/g;
      if type(f)="_power" then
        if type((k:=op(f,2)))=DOM_RAT then // good+g*bad^k=0
          return((-good)^op(k,2)+g^op(k,2)*f^op(k,2))
        end_if
      end_if
    end_if
  end_if;
  return(FAIL);
end_proc:

