/* 
   =====================================
   METHODS FOR SCALE INVARIANT EQUATIONS
   =====================================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 88, pp. 361.

   DETAILS: 

     ode::scale_invariant(eq,y,x,n) tries to solve the n-th order ODE eq = 0 
     with the method of scale invariant equations (see [1]). An equation is 
     scale invariant if its form is unchanged by the transformation 
         
                            x -> a*x, y -> a^p*y.

   OUTPUT: either FAIL or a set of solutions

   EXAMPLES:

     >> ode::scale_invariant(x^2*diff(y(x),x,x)+3*x*diff(y(x),x)-1/y(x)^3/x^4,
                             y,x,2);

     >> ode::scale_invariant(x^5*y(x)^3*diff(y(x),x)*3+x^6*y(x)^3*diff(y(x),x,x)-1,
                             y,x,2);

     >> ode::scale_invariant(x^2*diff(y(x),x,x)+3*x*diff(y(x),x)+2*y(x)-1/y(x)^3/x^4,
                             y,x,2);
*/

ode::scale_invariant:= proc(eq,_0_y,_0_x,n,solveOptions,odeOptions) 
  local i,l,s,u,X,Y,p,a,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  X:= genident("x");
  Y:= genident("y");
  a:= genident();
  if type(eq)<>"_plus" then
    return(FAIL);
  end_if;
  // Build-up transformation x -> a*x, y -> a^p*y and take care to 
  // transform the derivatives of 'y' appropriately. 
  l:= diff(_0_y(_0_x),_0_x$i) = a^(`#p`-i)*diff(Y(X),X$i) $ i=1..n,
                   _0_y(_0_x) = a^`#p`*Y(X), 
                         _0_x = a*X, 
                            Y = _0_y, 
                            X = _0_x;
  s:= {map(op(eq), proc() begin 
                     return(subs(args(1),l)/args(1))
                   end_proc)};
  s:= map(s,combine,optIgnoreAnalyticConstraints);
  s:= map(s, () -> (if type(args(1))="_power" and op(args(1),1)=a then
                      return(op(args(1),2));
                    elif args(1)=1 then
                      return(0);
                    elif args(1)=a then
                      return(1);
                    else
                      return(FAIL);
                    end_if));
   if has(s,FAIL) then
     return(FAIL);
   end_if;
   case nops(s)
     of 1 do
       s:= {0};
       break; // any value of p works 
    of 2 do
      s:= solvelib::discreteSolve(op(s,1)=op(s,2),`#p`,op(solveOptions));
      break;
    otherwise
      return(FAIL);
  end_case;
  if type(s)="solve" or nops(s)=0 then
    return(FAIL);
  end_if;
  p:= op(s,1);
  userinfo(1,"scale invariant equation");
  u:= genident("u");
  eq:= numer(subs(eq,_0_y(_0_x)=_0_x^p*u(_0_x),EvalChanges));
  userinfo(2,"new equation is",eq);
  // the equation is now equidimensional in x 
  eq:= ode::equidimensional_in_x(eq,u,_0_x,n,TRUE,solveOptions,odeOptions);
  if eq=FAIL then
    userinfo(1,"scale invariant method failed"); 
    return(FAIL);
  else
    userinfo(1,"scale invariant method worked");
    return(ode::mapsol(eq,_mult,_0_x^p,solveOptions,odeOptions));
  end_if;
  
end_proc:

