/* 
   ===============================
   METHODS FOR ODES SOLVABLE FOR X
   ===============================
 
   REFERENCE: [1] D. Zwillinger: "Handbook of Differential Equations", 
                  Section 91, pp. 370.

   DETAILS: 

   ode::solvable_for_x(eq,y,x) tries to solve the first-order ODE eq=0 wrt y(x)
   using the method of "solvable in x" equations. 

   Output: either FAIL or a set of solutions.

   EXAMPLE:
     >> ode::solvable_for_x(2*x*diff(y(x),x)+y(x)*diff(y(x),x)^2-y(x),y,x);

*/

ode::solvable_for_x:= proc(eq,y,x,solveOptions,odeOptions)
  local p,xx,sol,l,soll,foundUnresolvedIntegral;
begin 
  userinfo(2,"trying to recognize an equation solvable for x");
  p:= genident("p");
  eq:= subs(eq,diff(y(x),x)=p,y(x)=y,EvalChanges);
  // Note, I use MaxDegree only, since otherwise solve needs too much time
  sol:= solvelib::discreteSolve(eq,x,MaxDegree=1,op(solveOptions));
   if not ode::isResultOfSolveOK(sol,"NoRootOf",solveOptions,odeOptions) then
     return(FAIL);
   end_if;
   if nops(sol)=0 or sol={0} or not has(sol, y) then
     return(FAIL);
   end_if;
   userinfo(1,"equation solvable for x");
   xx:= subs(op(sol,1),p=p(y),EvalChanges);
   sol:= numer(diff(xx,y)-1/p(y),Expand=TRUE);
   sysassign(ode::depth,ode::depth+1);
   sol:= ode::solve_eq(sol,p,y,{},solveOptions,odeOptions);
   sysassign(ode::depth,ode::depth-1);
   // We currently do not allow implicit solutions 
   if type(sol) <> DOM_SET or contains(sol, FAIL) then 
     userinfo(1,"solvable for x method failed");
     return(FAIL);
   end_if; 
   sol:=map(sol,
            proc(solution)
            begin
              expr(factor(ode::normal(subs(eq,p=solution),Expand=FALSE,List)[1]))
            end_proc
            );     
   sol:= map(sol,solvelib::discreteSolve,y,op(solveOptions));
//   sol:= select(sol,elem -> if not hastype(elem,"solve") then TRUE else FALSE end_if);
   foundUnresolvedIntegral:= FALSE;
   misc::maprec(sol,{"int"} = proc(elem) 
                              local i;
                            begin
                              //if has(op(elem,1),[diff(y(x),x$i) $ i=0..n]) then 
                              if has(op(elem,1),y) then 
                                foundUnresolvedIntegral:= misc::breakmap();
                              end_if;
                              elem;
                            end_proc);                         
   if foundUnresolvedIntegral then 
     return(FAIL);
   end_if;   
   
   if not(has(sol, FAIL)) then
     userinfo(1,"solvable for x method worked");
     soll:= {};
     for l in sol do
       if type(l) = "_union" then
         l:= map({op(l)}, ()-> (if type(args(1))="solve" then 
                                  return(args(1));
                                else 
                                  return(op(args(1)));
                                end_if));
       end_if;
       soll:= soll union l;
     end_for;
     return(soll);
   else
     return(FAIL);
   end_if;
   
end_proc:

