// Basic structure for plot backends
//
// performs actions like traversing the tree, handling inheritance
// and standard animations.

alias(Out = plot::GenericOutput):

Out := newDomain("plot::GenericOutput"):
Out::interface := {}:
Out::create_dom:=hold(Out):

  //  locate ColorFunction, LineColorFunction, FillColorFunction,
  //         Function, XFunction, YFunction, ZFunction in attrib
  //  and transform them into procedures
  //  used by plot::GenericOutput::new and plot::Hatch::MuPlotML
Out::makeFuncs :=
proc(attrib : DOM_TABLE,
     param = FAIL) : DOM_TABLE
  local vars, makeColorFunc, makeFunc, i, specfuncs, noSpecfuncs;
begin
  vars := null():
  
  noSpecfuncs := {hold(_index)};
  
  // functions with float attributes are replaced by their
  // float attributes
  specfuncs := indets(attrib, All) minus indets(attrib);
  specfuncs := select(specfuncs,
                      x -> _lazy_and(domtype(eval(x)) = DOM_FUNC_ENV,
                                     not contains(noSpecfuncs, x), 
                                     (eval(x))::float <> FAIL)):
  specfuncs := map(specfuncs, x -> x=(eval(x))::float):

  for i in plot::varNames do
    if contains(attrib, i) then
      vars := vars, attrib[i]
    end_if;
  end_for;
    
    // makeColorFunc
  makeColorFunc :=
  proc(attrName, param)
    local colFunc, l;
    option escape;
  begin
    l := attrib[attrName];
    if hastype(l, "int") then
      // use numeric::intsolve to plot symbolic integrals
      l := misc::maprec(l,
                        (y -> (testtype(y, "int") and not
                               testtype(op(y, 2), "_equal"))) =
                        (z -> (numeric::intsolve(z);
                               subsop(`#111`(op(z,2)), 0= %))))
    end_if;
    if hastype(l, piecewise) then
      // use numeric::piecewise instead
      l := misc::maprec(l,
                        (y -> testtype(y, piecewise)) =
                        (z -> (numeric::piecewise::convert(z))))
    end_if;
    if hastype(l, Series::Puiseux) or hastype(l, Series::gseries) then
      // use the corresponding expr instead
      l := misc::maprec(l,
                        (y -> testtype(y, Series::Puiseux) or
                              testtype(y, Series::gseries)) =
                        (z -> (expr(z))))
    end_if;
    if domtype(l) = DOM_LIST then
      colFunc := plot::unapply(subs(map(l, float), [op(specfuncs)], Unsimplified),
                               vars);
    elif param <> FAIL then
      colFunc := () -> float(l(args(), param));
    else
      colFunc := float@l;
    end_if;
      attrib[attrName] := colFunc;
  end_proc;
  
  // makeFunc
  makeFunc :=
  proc(attrName, param)
    local Func, f;
    option escape;
  begin
    f := attrib[attrName];
    if hastype(f, "int") then
      // use numeric::intsolve to plot symbolic integrals
      f := misc::maprec(f,
                        (y -> (testtype(y, "int") and not
                               testtype(op(y, 2), "_equal"))) =
                        (z -> (numeric::intsolve(z);
                               subsop(`#111`(op(z,2)), 0= %))))
    end_if;
    if hastype(f, piecewise) then
      // use numeric::piecewise instead
      f := misc::maprec(f,
                        (y -> testtype(y, piecewise)) =
                        (z -> (numeric::piecewise::convert(z))))
    end_if;
    if hastype(f, Series::Puiseux) or hastype(f, Series::gseries) then
      // use the corresponding expr instead
      f := misc::maprec(f,
                        (y -> testtype(y, Series::Puiseux) or
                              testtype(y, Series::gseries)) =
                        (z -> (expr(z))))
    end_if;
    // avoid problem with id and parameter insertion
    if f = id then f := proc(x) name id; begin x end end:
    if not testtype(f, Type::Function) then
      Func := plot::unapply(subs(float(f), [op(specfuncs)], Unsimplified), vars);
      Func := subsop(Func, 6=attrName);
    elif param <> FAIL then
      Func := () -> float(f(args(), param));
      Func := subsop(Func, 6=attrName);
    else
      Func := float@f;
    end_if;
    attrib[attrName] := Func;
  end_proc;

  for i in plot::colorFunctions do
    if contains(attrib, i) then
      makeColorFunc(i, param);
    end_if;
  end_for;
  for i in plot::functions do
    if contains(attrib, i) then
      makeFunc(i, param);
    end_if;
  end_for;

  attrib
end:

/* fixAttributes removes animation attributes and attributes which
   are not allowed for targetDom.

   Used in structs like XRotate before calling MuPlotML::new with
   option Raw.
*/
Out::fixAttributes :=
proc(attributes : DOM_TABLE, targetDom : DOM_DOMAIN)
  local allowed;
begin
  delete attributes[ParameterValue], attributes[TimeValue],
         attributes[ParameterName], attributes[ParameterBegin],
         attributes[ParameterEnd], attributes[Frames],
         attributes[TimeBegin], attributes[TimeEnd]:

  // remove all attributees which are not allowed for targetDom
  allowed := _union(op(plot::legalAttributes[targetDom::objType]),
                    select(map({op(targetDom)}, rhs),
                           x -> domtype(x)=plot::StyleSheetEntry));
  
  attributes := select(attributes, v -> contains(allowed, op(v, 1)));

  attributes
end_proc:

  // compute the common ViewingBox of a set of ViewingBoxes
  // and merge the time ranges in s with the current timeRange
Out::commonViewingBox :=
proc(s : DOM_SET, affectVB = TRUE : DOM_BOOL, timeRange = null() : "_range")
  local list, VB, dim, TR;
begin
  s := map(s, x -> if domtype(x) <> DOM_TABLE then
                     table("ViewingBox"=x,
                           "TimeRange"=null())
                   else
                     x
                   end);
  case nops(s)
    of 0 do
      VB := null();
      TR := timeRange;
      break;
    of 1 do
      VB := op(s)["ViewingBox"];
      TR := op(s)["TimeRange"];
      if TR = null() then
        TR := timeRange
      elif timeRange <> null() then
        TR := min(op(TR, 1), op(timeRange, 1))..
              max(op(TR, 2), op(timeRange, 2))
      else
        // TR has a value, timeRange is null(); nothing to do
      end_if;
      break;
    otherwise
      // compute the common viewing box
      list := map([op(s)], _index, "ViewingBox");
      VB := list[1];
      
      if nops(list) > 1 then
        dim := nops(VB);
        if dim = 2 then
          VB := [min(map(op(list), op, [1, 1]))..
                 max(map(op(list), op, [1, 2])),
                 min(map(op(list), op, [2, 1]))..
                 max(map(op(list), op, [2, 2]))]
        else
          VB := [min(map(op(list), op, [1, 1]))..
                 max(map(op(list), op, [1, 2])),
                 min(map(op(list), op, [2, 1]))..
                 max(map(op(list), op, [2, 2])),
                 min(map(op(list), op, [3, 1]))..
                 max(map(op(list), op, [3, 2]))]
        end_if;
      end_if;
      
      // compute the common  time range
      list := map([op(s)], _index, "TimeRange").[timeRange];
      if nops(list) <= 1 then
        TR := op(list)
      else
        TR := min(map(op(list), op, 1)).. max(map(op(list), op, 2))
      end_if;
  end_case;
  if affectVB = TRUE then
    table("ViewingBox"=map(VB, float),
          "TimeRange" =map(TR, float));
  else
    table("ViewingBox"=null(),
          "TimeRange" =map(TR, float));
  end_if;
end_proc:  /* commonViewingBox  */

// this procedure prints style sheets
Out::printStyleSheets :=
proc(current, inherited, dimensions)
  // uses SceneAttrib, CoordAttrib, ObjAttrib set by splitAttributes
  local allChildren, styleSheets, rep, op1, splitAttributes,
        i, j, updateInherited, global, perObject, dummy;
begin    
  // split the given attributes into those belonging to the current
  // object type, Scene attributes, CoordinateSystem attributes and
  // object attributes
  splitAttributes :=
  proc(attrib)
    local i, idx, value;
  begin
    delete attrib[Name], attrib[AutoViewingBox];

    styleSheets["Scene2dStyle"] := table();
    styleSheets["Scene3dStyle"] := table();
    styleSheets["CoordinateSystem2dStyle"] := table();
    styleSheets["CoordinateSystem3dStyle"] := table();
    styleSheets["Default"] := table();
   
    for i in attrib do
      idx := op(i, 1);
      value := op(i, 2);
      if contains(plot::legalAttributes["Scene2d"][5], idx) then
        styleSheets["Scene2dStyle"][idx] := value;
      end_if;
      if contains(plot::legalAttributes["Scene3d"][5], idx) then
        styleSheets["Scene3dStyle"][idx] := value;
      end_if;
      if contains(plot::legalAttributes["CoordinateSystem2d"][5], idx) then
        styleSheets["CoordinateSystem2dStyle"][idx] := value;
      end_if;
      if contains(plot::legalAttributes["CoordinateSystem3d"][5], idx) then
        styleSheets["CoordinateSystem3dStyle"][idx] := value;
      end_if;
      if plot::attributes[idx][4] minus
        {"Canvas", "Scene2d", "Scene3d", "CoordinateSystem2d",
         "CoordinateSystem3d"} <> {} then
        styleSheets["Default"][idx] := value;
      end_if;
    end_for;
  end_proc:

  // domain of the internal representation
  rep := extop(current, 1);
  allChildren := rep::structs minus {current::dom::objType};
  
  if current::dom::objType = "Canvas" then
    // explicitly add Camara style sheet for 3D plots
    if allChildren intersect plot::all3dObjects <> {} then
      allChildren := allChildren union {"Camera"};
    end_if;
    // get all available styleSeets
    // and select only inheritable attributes
    styleSheets :=
    table(map(allChildren,
              x -> (x =
                    select((slot(plot, x))::styleSheet,
                           y -> plot::attributes[op(y, 1)][1] =
                           "Inherited"))));

    // check style sheet entries for validity
    for i in allChildren do
      if contains(styleSheets, i) then
        styleSheets[i] := plot::checkOptions(i, [op(styleSheets[i])]);
      end_if;
    end_for;
    
    // remove all defaults from styleSheets for which the
    // user has explicitly given other values (they are in rep::styleEntries)
    for i in select(map({op(rep::styleEntries)}, op, 1),
                    testtype, DOM_IDENT) do
      for j in allChildren do
        if contains(styleSheets[j], i) then
          delete styleSheets[j][i]
        end_if;
      end_for;
    end_for;
    // now look for qualified values in rep::styleEntries and put them in
    // the style sheets
    for i in select(map({op(rep::styleEntries)}, op, 1),
                    testtype, plot::StyleSheetEntry) do
      for j in allChildren do
        if i::dom::objType(i) = j then
          styleSheets[j][i::dom::attribute(i)] := rep::styleEntries[i];
        end_if;
      end_for;
    end_for;
    splitAttributes(plot::getInherited());
  else
    styleSheets := table(map(allChildren, x -> (x = table())));
    styleSheets["Scene2dStyle"] := table();
    styleSheets["Scene3dStyle"] := table();
    styleSheets["CoordinateSystem2dStyle"] := table();
    styleSheets["CoordinateSystem3dStyle"] := table();
  end_if;

  [global, perObject, dummy] := split(rep::styleEntries,
                                      x -> testtype(op(x, 1), DOM_IDENT));


  // now add entries of rep::styleEntries
  for i in global do
    op1 := op(i, 1);
    if plot::attributes[op1][4] intersect {"Scene2d", "Scene3d"} <> {} then
      if plot::attributes[op1][4] intersect {"Scene2d"} <> {} then
        // put Scene2d-Attributes to Scene2dStyle
        styleSheets["Scene2dStyle"][op1] := op(i, 2);
      end_if;
      if plot::attributes[op1][4] intersect {"Scene3d"} <> {} then
        // put Scene3d-Attributes to Scene3dStyle
        styleSheets["Scene3dStyle"][op1] := op(i, 2);
      end_if;
    elif plot::attributes[op1][4] intersect
      {"CoordinateSystem2d", "CoordinateSystem3d"} <> {} then
      if plot::attributes[op1][4] intersect
        {"CoordinateSystem2d"} <> {} then
        // put CoordinateSystem2d-Attributes to CoordinateSystem2dStyle
        styleSheets["CoordinateSystem2dStyle"][op1] := op(i, 2);
      end_if;
      if plot::attributes[op1][4] intersect
        {"CoordinateSystem3d"} <> {} then
        // put CoordinateSystem3d-Attributes to CoordinateSystem3dStyle
        styleSheets["CoordinateSystem3dStyle"][op1] := op(i, 2);
      end_if;
    else
      // rest are global object defaults
      // put them in all object stylesheets
      for j in styleSheets do
        if contains({"Scene2dStyle", "CoordinateSystem2dStyle",
                     "Scene3dStyle", "CoordinateSystem3dStyle",
                     "Scene2d", "Scene3d",
                     "CoordinateSystem2d", "CoordinateSystem3d"},
                    op(j, 1)) = TRUE then
          next;
        end_if;
        // check if the attribute is legal for the given
        // stylesheet
        if op(j,1) = "Default" or
          contains(plot::legalAttributes[op(j,1)][3], op1) then
          styleSheets[op(j, 1)][op1] := op(i, 2);
        end_if;
      end_for;
    end_if;
  end_for;

  for i in perObject do
    op1 := op(i, 1);
    assert(domtype(op1) = plot::StyleSheetEntry);
    styleSheets[op1::dom::objType(op1)][op1::dom::attribute(op1)] := op(i,2);
  end_for;
 
  // for each style sheet put the corresponding values into inherited
  updateInherited := styleSheets;
  delete updateInherited["Default"];
  delete updateInherited["Scene2dStyle"];
  delete updateInherited["Scene3dStyle"];
  delete updateInherited["CoordinateSystem2dStyle"];
  delete updateInherited["CoordinateSystem3dStyle"];
  
  for i in {op(updateInherited)} do
    if contains(inherited, op(i,1)) then
      // merge in
      inherited[op(i, 1)] := table(inherited[op(i, 1)], op(i,2));
    else
      // set explicitly
      inherited[op(i, 1)] := op(i,2);
    end_if;
  end_for;

  if contains(styleSheets, "Scene2d") then
    styleSheets["Scene2dStyle"] :=
      table(styleSheets["Scene2dStyle"],
            styleSheets["Scene2d"]);
    delete styleSheets["Scene2d"];
  end_if;
  if contains(styleSheets, "Scene3d") then
    styleSheets["Scene3dStyle"] :=
      table(styleSheets["Scene3dStyle"],
            styleSheets["Scene3d"]);
    delete styleSheets["Scene3d"];
  end_if;
  if contains(styleSheets, "CoordinateSystem2d") then
    styleSheets["CoordinateSystem2dStyle"] :=
      table(styleSheets["CoordinateSystem2dStyle"],
            styleSheets["CoordinateSystem2d"]);
    delete styleSheets["CoordinateSystem2d"];
  end_if;
  if contains(styleSheets, "CoordinateSystem3d") then
    styleSheets["CoordinateSystem3dStyle"] :=
      table(styleSheets["CoordinateSystem3dStyle"],
            styleSheets["CoordinateSystem3d"]);
    delete styleSheets["CoordinateSystem3d"];
  end_if;
  delete styleSheets["Group2d"];
  delete styleSheets["Group3d"];

  dom::printDefaultStyles(current, styleSheets, dimensions);

  inherited;
end_proc:
  

// apply a transformation on a list of ViewingBoxes
Out::transformViewingBoxList :=
proc(trMat, trShift, VBList)
  local transformViewingBox, dim;
begin
  dim := sqrt(nops(trMat));
  
  // transformation on one ViewingBox
  transformViewingBox :=
  proc(VB)
    local i, VB_, x, y, z;
  begin
    VB_  := VB["ViewingBox"];
    if VB_ <> null() then
      // Vec := trMat*Vec+trShift;
      if dim=2 then
        VB_ := [trMat[1]*x+trMat[2]*y,
                trMat[3]*x+trMat[4]*y]
             $ x in VB_[1]
             $ y in VB_[2];
      else
        VB_ := [trMat[1]*x+trMat[2]*y+trMat[3]*z,
                trMat[4]*x+trMat[5]*y+trMat[6]*z,
                trMat[7]*x+trMat[8]*y+trMat[9]*z]
             $ x in VB_[1]
             $ y in VB_[2]
             $ z in VB_[3];
      end_if;
      table(VB, "ViewingBox"=[min(map(VB_, op, i))+trShift[i]..
                              max(map(VB_, op, i))+trShift[i] $ i=1..dim])
    else
      table(VB, "ViewingBox"=null())
    end_if
  end_proc:
  
  {op(map(VBList, transformViewingBox))};
end_proc:

/**
 *   plot::GenericOutput::new :  write scene
 *
 *   Arguments :
 *       obj    -- a plot primitive
 *       attrib -- table with inherited attributes
 *       Raw    -- do not print Obj2d/3d and Img2d/3d elements (optional)
 *
 *   Return value : TRUE  -- print successful
 *                  FALSE -- an error occurred
 */
Out::new :=
proc(obj, inherited = FAIL, raw=FAIL)
  local attrib, optionals, nonExprAttrib, exprAttrib, extraAttrib,
        doSort, frames, imgName, objName, parBegin, parEnd,
        parName, param, t, timeBegin, timeEnd, timeVal,
        titlepos, transf, trOpt, trMatrix, invTrMatrix, trShift,
        viewingbox, result, opts, i, structDefs, styleEntries,
        transformation, transformations, autoVB, VBexpr, VBEntries,
        titleVB, timeRange, VB, attr, Xdelta, Ydelta, Zdelta,
        scaling, scalingRatio, dimensions;
begin
  timeRange := null();
  
  // obj has to be a plot object
  assert(type(obj) = "plotObject");

  structDefs := {}:
  if domtype(obj) = plot::Canvas then
    // xml header
    dom::header();
    
    // collect inspector definitions for library plot primitives
    for i in (extop(obj, 1))::structs do
      if contains(plot::inspector::structTable, i) then
        structDefs := structDefs union {plot::inspector::structTable[i]};
      end_if;
    end_for;

    // get dimensions of objects in this plot
    dimensions := map({op(obj)}, x -> x::dom::dimension);
  else
    dimensions := {obj::dom::dimension}
  end;

  // get inherited attributes, unless provided by our caller
  if inherited = FAIL then
    inherited := plot::getInherited();
  end_if;
  
  // get inheritable attributes of this object 
  attrib := plot::getInherited(obj);

  if contains(inherited, obj::dom::objType) then
    styleEntries := inherited[obj::dom::objType];
  else
    styleEntries := table();
  end_if;
  
  // merge inherited with settings in attrib and style entries
  inherited := table(inherited, styleEntries, attrib, (extop(obj, 1))::styleEntries);

  // get optional attributes for obj from attribute table
  // combined with optional attributes given in obj
  optionals := plot::getOptionals(obj);

  // split optionals into real XML-attributes and expressions
  // printed as XML attributes
  nonExprAttrib := table(optionals[1], attrib); 
  exprAttrib :=    optionals[2]; // printed as elements
  extraAttrib :=   optionals[3]; // not printed

  extraAttrib := table(map([op(extraAttrib)],
                     x -> if expr2text(op(x,1))[-1] = "_" then
                            text2expr(expr2text(op(x,1))[1..-2])=op(x,2)
                          else
                            x
                          end_if));
  
  // attributes given as 2. argument to doPlotStatic methods
  optionals := table(inherited,
                     nonExprAttrib,
                     // convert into expressions
                     map(exprAttrib, plot::ExprAttribute::getExpr),
                     extraAttrib);

  // if exprAttrib contains a viewingbox, then extract it here,
  // because it is printed after the object children.
  // The object can only be a CoordinateSystem !!!
  viewingbox := table();
  autoVB := 0;
  if contains({"CoordinateSystem2d", "CoordinateSystem3d"},
              obj::dom::objType) then
    viewingbox[ViewingBoxXMin] := exprAttrib[ViewingBoxXMin];
    delete exprAttrib[ViewingBoxXMin];
    viewingbox[ViewingBoxXMax] := exprAttrib[ViewingBoxXMax];
    delete exprAttrib[ViewingBoxXMax];
    viewingbox[ViewingBoxYMin] := exprAttrib[ViewingBoxYMin];
    delete exprAttrib[ViewingBoxYMin];
    viewingbox[ViewingBoxYMax] := exprAttrib[ViewingBoxYMax];
    delete exprAttrib[ViewingBoxYMax];
    if contains(exprAttrib, ViewingBoxZMin) then
      // 3D
      viewingbox[ViewingBoxZMin] := exprAttrib[ViewingBoxZMin];
      delete exprAttrib[ViewingBoxZMin];
      viewingbox[ViewingBoxZMax] := exprAttrib[ViewingBoxZMax];
      delete exprAttrib[ViewingBoxZMax];
    end_if;
    VBexpr := map(viewingbox, plot::ExprAttribute::getExpr);
    VBEntries := [ViewingBoxXMin, ViewingBoxXMax,
                  ViewingBoxYMin, ViewingBoxYMax,
                  ViewingBoxZMin, ViewingBoxZMax];
    for i from 1 to 6 do
      if contains(VBexpr, VBEntries[i]) and
        VBexpr[VBEntries[i]] = Automatic then
        autoVB := autoVB + 2^(i-1)
      end_if;
    end_for;
    nonExprAttrib:= table(nonExprAttrib, AutoViewingBox = autoVB);
  end_if;
  inherited := table(inherited, map(viewingbox, plot::ExprAttribute::getExpr));

  if obj::dom::hidden <> {} then
    // remove hidden slots from exprAttrib
    // used to remove Matrix{23} from Scale, Translate
    for attr in obj::dom::hidden do
      delete exprAttrib[attr];
    end_for;
  end_if;
  
  transformation :=
  proc()
  begin
    inherited[ViewingBoxXMin] := trOpt[ViewingBoxXMin];
    inherited[ViewingBoxXMax] := trOpt[ViewingBoxXMax];
    inherited[ViewingBoxYMin] := trOpt[ViewingBoxYMin];
    inherited[ViewingBoxYMax] := trOpt[ViewingBoxYMax];
    if contains(plot::allTransformations2d, obj::dom::objType) then
      trMatrix := map(trOpt[Matrix2d], float);
      trShift := (float(trOpt[ShiftX]), float(trOpt[ShiftY]));
      transf :=
      ("Shift" = trShift,
       "Row1"  = (trMatrix[1], trMatrix[2]),
       "Row2"  = (trMatrix[3], trMatrix[4]));
    else
      // 3D-Transformation
      inherited[ViewingBoxZMin] := trOpt[ViewingBoxZMin];
      inherited[ViewingBoxZMax] := trOpt[ViewingBoxZMax];
      trMatrix := map(trOpt[Matrix3d], float);
      trShift := (float(trOpt[ShiftX]),
                  float(trOpt[ShiftY]),
                  float(trOpt[ShiftZ]));
      transf :=
      ("Shift" = trShift,
       "Row1"  = (trMatrix[1], trMatrix[2], trMatrix[3]),
       "Row2"  = (trMatrix[4], trMatrix[5], trMatrix[6]),
       "Row3"  = (trMatrix[7], trMatrix[8], trMatrix[9]));
       // and now apply the transformation on the ViewingBox
      if contains(inherited, ViewingBoxZMin) and
         inherited[ViewingBoxZMin] <> Automatic then
        inherited[ViewingBoxZMin] := inherited[ViewingBoxZMin] -
                                   op(trShift, 3);
      end_if;
      if contains(inherited, ViewingBoxZMax) and
         inherited[ViewingBoxZMax] <> Automatic then
        inherited[ViewingBoxZMax] := inherited[ViewingBoxZMax] -
                                   op(trShift, 3);
      end_if;
    end_if;
          // and now apply the transformation on the ViewingBox
          // which is given to the children;  Function2d reacts
          // on this ViewingBox
    if contains(inherited, ViewingBoxXMin) and
       inherited[ViewingBoxXMin] <> Automatic then
      inherited[ViewingBoxXMin] := inherited[ViewingBoxXMin] -
                                 op(trShift, 1);
    end_if;
    if contains(inherited, ViewingBoxXMax) and
       inherited[ViewingBoxXMax] <> Automatic then
      inherited[ViewingBoxXMax] := inherited[ViewingBoxXMax] -
                                 op(trShift, 1);
    end_if;
    if contains(inherited, ViewingBoxYMin) and
       inherited[ViewingBoxYMin] <> Automatic then
      inherited[ViewingBoxYMin] := inherited[ViewingBoxYMin] -
                                 op(trShift, 2);
    end_if;
    if contains(inherited, ViewingBoxYMax) and
       inherited[ViewingBoxYMax] <> Automatic then
      inherited[ViewingBoxYMax] := inherited[ViewingBoxYMax] -
                                 op(trShift, 2);
    end_if;
        // and now apply the transformation matrix
    invTrMatrix := [FAIL, FAIL];
    if contains(plot::allTransformations2d, obj::dom::objType) then
      if contains(inherited, ViewingBoxXMin) and
         inherited[ViewingBoxXMin] <> Automatic and
         contains(inherited, ViewingBoxXMax) and
         inherited[ViewingBoxXMax] <> Automatic and
         contains(inherited, ViewingBoxYMin) and
         inherited[ViewingBoxYMin] <> Automatic and
         contains(inherited, ViewingBoxYMax) and
         inherited[ViewingBoxYMax] <> Automatic then
            // all corners are given
        invTrMatrix := numeric::matlinsolve(
            matrix([[op(trMatrix, 1..2)], [op(trMatrix, 3..4)]]),
            matrix([[inherited[ViewingBoxXMin],inherited[ViewingBoxXMax]],
                    [inherited[ViewingBoxYMin],inherited[ViewingBoxYMax]]])
                                           );
        if invTrMatrix[1] <> FAIL then
          inherited[ViewingBoxXMin] := min(op(matrix::row(invTrMatrix[1], 1)));
          inherited[ViewingBoxXMax] := max(op(matrix::row(invTrMatrix[1], 1)));
          inherited[ViewingBoxYMin] := min(op(matrix::row(invTrMatrix[1], 2)));
          inherited[ViewingBoxYMax] := max(op(matrix::row(invTrMatrix[1], 2)));
        end_if;
      end_if;
      if invTrMatrix[1] = FAIL and trMatrix <> [1.0, 0.0, 0.0, 1.0] then
        inherited[ViewingBoxXMin] := Automatic;
        inherited[ViewingBoxXMax] := Automatic;
        inherited[ViewingBoxYMin] := Automatic;
        inherited[ViewingBoxYMax] := Automatic;
      end_if
    else
          // same transformation for 3D
      if contains(inherited, ViewingBoxXMin) and
         inherited[ViewingBoxXMin] <> Automatic and
         contains(inherited, ViewingBoxXMax) and
         inherited[ViewingBoxXMax] <> Automatic and
         contains(inherited, ViewingBoxYMin) and
         inherited[ViewingBoxYMin] <> Automatic and
         contains(inherited, ViewingBoxYMax) and
         inherited[ViewingBoxYMax] <> Automatic and
         contains(inherited, ViewingBoxZMin) and
         inherited[ViewingBoxZMin] <> Automatic and
         contains(inherited, ViewingBoxZMax) and
         inherited[ViewingBoxZMax] <> Automatic then
            // all corners are given
        invTrMatrix := numeric::matlinsolve(
            matrix([[inherited[ViewingBoxXMin],inherited[ViewingBoxXMax]],
                    [inherited[ViewingBoxYMin],inherited[ViewingBoxYMax]],
                    [inherited[ViewingBoxZMin],inherited[ViewingBoxZMax]]]),
                                            matrix([[op(trMatrix, 1..3)],
                                                    [op(trMatrix, 4..6)],
                                                    [op(trMatrix, 7..9)]]));
        if invTrMatrix[1] <> FAIL then
          inherited[ViewingBoxXMin] := min(op(matrix::row(invTrMatrix[1], 1)));
          inherited[ViewingBoxXMax] := max(op(matrix::row(invTrMatrix[1], 1)));
          inherited[ViewingBoxYMin] := min(op(matrix::row(invTrMatrix[1], 2)));
          inherited[ViewingBoxYMax] := max(op(matrix::row(invTrMatrix[1], 2)));
          inherited[ViewingBoxZMin] := min(op(matrix::row(invTrMatrix[1], 3)));
          inherited[ViewingBoxZMax] := max(op(matrix::row(invTrMatrix[1], 3)));
        end_if;
      end_if;
      if invTrMatrix[1] = FAIL and trMatrix <> [1.0, 0.0, 0.0,
                                                0.0, 1.0, 0.0,
                                                0.0, 0.0, 1.0] then
        inherited[ViewingBoxXMin] := Automatic;
        inherited[ViewingBoxXMax] := Automatic;
        inherited[ViewingBoxYMin] := Automatic;
        inherited[ViewingBoxYMax] := Automatic;
        inherited[ViewingBoxZMin] := Automatic;
        inherited[ViewingBoxZMax] := Automatic;
      end_if
    end_if;
 end_proc;

  //--------------------------------------------------------------
  // the values in exprAttrib must have a certain order
  // given by the DTD.  This order is only necessary for
  // the XML validator, not for VcamNG.  This code may be
  // commented out in the final version.
  doSort :=
  proc()
    local sortProc, order;
  begin
    order := ["Header", "Footer",
              "XMin", "XMax", "YMin", "YMax", "ZMin", "ZMax",
              "AxesOriginX", "AxesOriginY", "AxesOriginZ",
              "XAxisTitle", "YAxisTitle", "ZAxisTitle",
              "XTicksAnchor", "XTicksDistance",
              "YTicksAnchor", "YTicksDistance",
              "ZTicksAnchor", "ZTicksDistance",
              "XTicksAt", "YTicksAt", "ZTicksAt",
              "XAxisTextOrientation", "YAxisTextOrientation",
              "ZAxisTextOrientation",
              "Expr", "ExprSeq","ColorFunc","Prop",
              "Points2d", "Points3d", "Matrix2d", "Matrix3d",
              "TextOrientation",
              "ParameterName", "ParameterBegin", "ParameterEnd",
              "Title", "TitlePositionX", "TitlePositionY", "TitlePositionZ",
              "LegendText"];
    sortProc :=
    proc(x, y)
    begin
        contains(order, op(extop(op(x, 2), 2), 1)) <
        contains(order, op(extop(op(y, 2), 2), 1))
      end_proc;
    exprAttrib := map(sort([op(exprAttrib)], sortProc), op, 2);
  end_proc;
  // comment out this line when order of Expr attributes is not needed
  doSort();
  //--------------------------------------------------------------

  //--------------------------------------------------------------
  //              FROM HERE ON WE PRINT THE OBJECT
  //--------------------------------------------------------------

  // if obj can be animated and animation attributes are
  // available, then animate
  if obj::dom::animable = TRUE then
    // obj is a graphical primitive, transtormation,
    // Camera, ClippingBox or Light

    // option Raw must not be set for transformations
    assert((
            if contains(plot::allTransformations, obj::dom::objType) then
              raw = FAIL
            else
              TRUE
            end_if
            ));

    if contains(plot::allTransformations2d, obj::dom::objType) then
      // types of Objects and Images: 
      objName := "Transform2d";
      imgName := "Tr2d";
    elif contains(plot::allTransformations3d, obj::dom::objType) then
      // types of Objects and Images: 
      objName := "Transform3d";
      imgName := "Tr3d";
    else
      // type of Objects: Obj2d or Obj3d
      objName := "Obj".obj::dom::dimension."d";
      // type of Images: Img2d or Img3d
      imgName := "Img".obj::dom::dimension."d";
    end_if;
      
    titlepos := null();
    if contains(optionals, TitlePositionX) = TRUE then
      titlepos := "TitlePositionX"=optionals[TitlePositionX];
    end_if;
    if contains(optionals, TitlePositionY) = TRUE then
      titlepos := (titlepos,
                   "TitlePositionY"=optionals[TitlePositionY]);
    end_if;
    if contains(optionals, TitlePositionZ) = TRUE then
      titlepos := (titlepos,
                   "TitlePositionZ"=optionals[TitlePositionZ]);
    end_if;

    if contains(optionals, ParameterBegin) and
      contains(optionals, ParameterEnd) then
      //--------------------------------------------------------------
      //               we are doing an animation
      //--------------------------------------------------------------
      
      if not contains(optionals, ParameterName) then
        optionals[ParameterName] := solvelib::getIdent(Any,
          indets(obj) union indets(optionals) union indets(inherited));
      end_if;
      
      // calculate parameter value for each time stamp,
      // substitute it in all expression options (optionals)
      // and call obj::dom::doPlotStatic for each time stamp
      parName  := optionals[ParameterName];
      parBegin := optionals[ParameterBegin];
      parEnd   := optionals[ParameterEnd];

      frames   := inherited[Frames] - 1;
      
      timeBegin := optionals[TimeBegin];
      timeEnd   := optionals[TimeEnd];
      timeRange := timeBegin..timeEnd;
      
      if raw <> Raw then
        dom::startAnimatedObject(obj, objName, nonExprAttrib, exprAttrib);
      end_if;
        
      //--------------------------------------------------------------
      //            begin animation loop over frames
      //--------------------------------------------------------------
      result := {};
      transformations := [NIL $ frames+1];

      for t from 0 to frames do
        if iszero(frames) then
          param := float(parBegin);
          timeVal := float(timeBegin);
        else
          param := float(((frames - t)*parBegin+t*parEnd)/frames);
          timeVal := float(((frames - t)*timeBegin+t*timeEnd)/frames);
        end_if;

        optionals[ParameterValue] := param;
        optionals[TimeValue] := timeVal;

        if contains(plot::allTransformations, obj::dom::objType) then
          trOpt := table(map(subs(optionals, parName=param), eval),
                         ParameterName=parName);
          transformation();
          // no way to transmit a correct ViewingBox in animations
          inherited[ViewingBoxXMin] := Automatic;
          inherited[ViewingBoxXMax] := Automatic;
          inherited[ViewingBoxYMin] := Automatic;
          inherited[ViewingBoxYMax] := Automatic;
          if contains(plot::allTransformations3d, obj::dom::objType) then
            inherited[ViewingBoxZMin] := Automatic;
            inherited[ViewingBoxZMax] := Automatic;
          end_if;

          transformations[t+1] := [trMatrix, [trShift]];
        else
          transf := null()
        end_if;
        
        if raw <> Raw then
          if titlepos <> null() then
            // merge title position into viewingbox
            if param <> FAIL then
              titleVB := map([map(subs(titlepos, parName=param), float)],
                              x ->  op(x, 2)..op(x,2));
            else
              titleVB := map([map(titlepos, float)],
                              x ->  op(x, 2)..op(x,2));
            end_if;
            if nops(op(titleVB)) = obj::dom::dimension then
              result := result union
                        {table("ViewingBox"=titleVB,
                               "TimeRange"=null())};
            end_if;
          end_if;
          
          dom::startAnimationFrame(obj, imgName,
            nonExprAttrib, 
            subs(exprAttrib, parName=param),
            optionals, table(transf), FALSE);
        end_if;

        if not contains(plot::allTransformations, obj::dom::objType) then
          opts := optionals;
          if param = FAIL or
            traperror((opts := table(map(subs(optionals, parName=param),
                                            eval),
                                        ParameterName=parName))) = 0 then
              // if an error occurs during parameter substitution, the image is empty
            opts := dom::makeFuncs(opts, param);
            result := result union
            // TODO TODO TODO
            // during the transition phase, fall back to MuPlotML.
            // in the final version, look for slot(obj::dom, dom::specialOutputName)
            // and use that if it exists.
                      {if obj::dom::doPlotStatic <> FAIL then
                        obj::dom::doPlotStatic(dom, obj, opts, inherited)
                       else
                        obj::dom::MuPlotML(obj, opts, inherited)
                       end_if};
          end_if;
        end_if;

        if raw <> Raw then
          dom::endAnimationFrame(obj, imgName,
            nonExprAttrib, 
            subs(exprAttrib, parName=param),
            optionals, table(transf), FALSE);
        end_if;
      end_for;  /* loop over frames in animation */
      //--------------------------------------------------------------
      //            end animation loop over frames
      //--------------------------------------------------------------
      
      if contains(plot::allTransformations, obj::dom::objType) then
        // print children
        result := dom::commonViewingBox(result union
                                   {op(map((extop(obj, 1))::children,
                                           dom, inherited))},
                                             inherited[AffectViewingBox]);
        // apply all transformation images on this combined
        // viewingbox of all children
        result := map({op(transformations)},
                      x -> op(dom::transformViewingBoxList(op(x,1), op(x,2),
                                                           [result])));
      end_if;
      
      // result is a set of ViewingBoxes, compute the common ViewingBox
      result := dom::commonViewingBox(result,
                                      inherited[AffectViewingBox],
                                      timeRange);
      
      if raw <> Raw then
        dom::endAnimatedObject(obj, objName, exprAttrib, nonExprAttrib);
      end_if;
    else
      //--------------------------------------------------------------
      //               no animation, just one Img.
      //--------------------------------------------------------------
      if contains(plot::allTransformations, obj::dom::objType) then
        trOpt := optionals;
        transformation();
      else
        transf := null()
      end_if;
      result := {};
      
      // no animation, just one Img.
      if raw <> Raw then
        if titlepos <> null() then
          // merge title position into viewingbox
          titleVB := table("ViewingBox"=map([map(titlepos, float)],
                                            x ->  op(x, 2)..op(x,2)),
                           "TimeRange"=null());
          if nops(titleVB["ViewingBox"]) = obj::dom::dimension then
            result := result union {titleVB};
          end_if;
        end_if;
        dom::startStaticObject(obj, objName, nonExprAttrib,
          exprAttrib);
        dom::startAnimationFrame(obj, imgName,
          nonExprAttrib, exprAttrib,
          optionals, table(transf), TRUE);
      end_if;

      if obj::TimeBegin <> FAIL and obj::TimeEnd <> FAIL then
        timeRange := obj::TimeBegin .. obj::TimeEnd;
      end_if;
      
      if not contains(plot::allTransformations, obj::dom::objType) then
        optionals := dom::makeFuncs(optionals);
        result := result union
            // TODO TODO TODO
            // during the transition phase, fall back to MuPlotML.
            // in the final version, look for slot(obj::dom, dom::specialOutputName)
            // and use that if it exists.
                      {if obj::dom::doPlotStatic <> FAIL then
                        obj::dom::doPlotStatic(dom, obj, optionals, inherited)
                       else
                        obj::dom::MuPlotML(obj, optionals, inherited)
                       end_if};
        result := dom::commonViewingBox(result,
                                        inherited[AffectViewingBox],
                                        timeRange);
      end_if;

      if raw <> Raw then
        dom::endAnimationFrame(obj, imgName,
          nonExprAttrib, exprAttrib,
          optionals, table(transf), TRUE);
   
        if contains(plot::allTransformations, obj::dom::objType) then
          // print children
          result := dom::transformViewingBoxList(trMatrix, [trShift],
                                             map((extop(obj, 1))::children,
                                                 dom::new, inherited));
          result := dom::commonViewingBox(result,
                                          inherited[AffectViewingBox],
                                          timeRange);
        end_if;

        dom::endStaticObject(obj, objName, nonExprAttrib,
          exprAttrib);
      end_if;  /* printing a single image */
    end_if     /* potentially animable object */
  else
    //----------------------------------------------------------------
    // standard output for other object types, like
    // Canvas, Scene2d/3d, CoordinateSystem2d/3d, Group2d/3d, View2/3d
    //----------------------------------------------------------------
    
    // print element begin and XML attributes
    dom::startSpecialObject(obj, nonExprAttrib, exprAttrib, structDefs);

    if not contains({"View2d", "View3d"}, obj::dom::objType) then
      inherited := dom::printStyleSheets(obj, inherited, dimensions);
    end_if;
    
    // print children
    result := map((extop(obj, 1))::children, dom::new, inherited);

    // e.g. a group with VisibleFromTo 
    if contains({"Group2d", "Group3d"}, obj::dom::objType) and
      obj::TimeBegin <> FAIL and obj::TimeEnd <> FAIL then
      timeRange := obj::TimeBegin .. obj::TimeEnd;
    end_if;
      
    // compute the common viewing box
    result := dom::commonViewingBox({op(result)},
                                    inherited[AffectViewingBox],
                                    timeRange);

    if contains({"CoordinateSystem2d", "CoordinateSystem3d"},
                obj::dom::objType) then
      if result <> null() and domtype(result["ViewingBox"]) = DOM_LIST then
        VB := map(result["ViewingBox"], map, float);
        VB := table(ViewingBoxXMin=op(VB, [1,1]),
                    ViewingBoxXMax=op(VB, [1,2]),
                    ViewingBoxYMin=op(VB, [2,1]),
                    ViewingBoxYMax=op(VB, [2,2]),
                    if obj::dom::dimension = 3 then
                      ViewingBoxZMin=op(VB, [3,1]),
                      ViewingBoxZMax=op(VB, [3,2])
                    else
                      null()
                    end)
      else 
        VB := null();
      end_if;
      // replace all Automatic values in viewingbox by the
      // values in result
      viewingbox := select(viewingbox, x -> plot::ExprAttribute::getExpr(op(x,2))<>Automatic);
      VB := table(VB, map(viewingbox, plot::ExprAttribute::getExpr));

      // get deltas for all directions
      if contains(VB, ViewingBoxXMin) and
        contains(VB, ViewingBoxXMax) then
        Xdelta := abs(float(VB[ViewingBoxXMax] - VB[ViewingBoxXMin]));
        if Xdelta = 0.0 or
           Xdelta < 1e-8 and abs(float(VB[ViewingBoxXMax])) >= 1e-8 then
          // if delta is nearly 0.0, but not around 0 increase by 1.0
          VB[ViewingBoxXMin] := VB[ViewingBoxXMin] - 1.0;
          VB[ViewingBoxXMax] := VB[ViewingBoxXMax] + 1.0;
          Xdelta := 2.0
        end_if;
      else
        VB[ViewingBoxXMin] := -1.0;
        VB[ViewingBoxXMax] :=  1.0;
        Xdelta := 2.0
      end_if;
      if contains(VB, ViewingBoxYMin) and
        contains(VB, ViewingBoxYMax) then
        Ydelta := abs(float(VB[ViewingBoxYMax] - VB[ViewingBoxYMin]));
        if Ydelta = 0.0 or
           Ydelta < 1e-8 and abs(float(VB[ViewingBoxYMax])) >= 1e-8 then
          // if delta is nearly 0.0, but not around 0 increase by 1.0
          VB[ViewingBoxYMin] := VB[ViewingBoxYMin] - 1.0;
          VB[ViewingBoxYMax] := VB[ViewingBoxYMax] + 1.0;
          Ydelta := 2.0
        end_if;
      else
        VB[ViewingBoxYMin] := -1.0;
        VB[ViewingBoxYMax] :=  1.0;
        Ydelta := 2.0
      end_if;
      if contains(VB, ViewingBoxZMin) and
        contains(VB, ViewingBoxZMax) then
        Zdelta := abs(float(VB[ViewingBoxZMax] - VB[ViewingBoxZMin]));
        if Zdelta = 0.0 or
           Zdelta < 1e-8 and abs(float(VB[ViewingBoxZMax])) >= 1e-8 then
          // if delta is nearly 0.0, but not around 0 increase by 1.0
          VB[ViewingBoxZMin] := VB[ViewingBoxZMin] - 1.0;
          VB[ViewingBoxZMax] := VB[ViewingBoxZMax] + 1.0;
          Zdelta := 2.0
        end_if;
      else
        if obj::dom::dimension = 3 then
          VB[ViewingBoxZMin] := -1.0;
          VB[ViewingBoxZMax] :=  1.0;
          Zdelta := 2.0
        else
          Zdelta := FAIL
        end_if;
      end_if;

      dom::printViewingBox(obj, VB);

      // use deltas to compute a value for Scaling
      scaling := "Constrained";
      scalingRatio := plot::scalingRatio();

      if obj::dom::dimension = 2 then
        Xdelta := Xdelta/inherited[Width];
        Ydelta := Ydelta/inherited[Height];
      else
        Xdelta := Xdelta/inherited[Width];
        Ydelta := Ydelta/inherited[Width];
        Zdelta := Zdelta/inherited[Height];
      end_if;
      
      if Xdelta <> FAIL then
        if Ydelta <> FAIL and
          ( Xdelta/Ydelta > scalingRatio or
           Ydelta/Xdelta > scalingRatio) then
          scaling := "Unconstrained"
        end_if;
        if Zdelta <> FAIL and
          (Xdelta/Zdelta > scalingRatio or
           Zdelta/Xdelta > scalingRatio) then
          scaling := "Unconstrained"
        end_if;
      end_if;
      if Ydelta <> FAIL then
        if Zdelta <> FAIL and
          (Ydelta/Zdelta > scalingRatio or
           Zdelta/Ydelta > scalingRatio) then
          scaling := "Unconstrained"
        end_if;
      end_if;

      dom::printHint("Scaling"=scaling);
    end_if;  /*  CoordinateSystem stuff */ 
    
    // print animation range, if available
    if obj::dom::objType = "Canvas" then
      if result["TimeRange"] <> null() then
        assert(type(result["TimeRange"]) = "_range");
        dom::printTimeRange(float(result["TimeRange"]));
      end_if;
    end_if;
    
    // print end tag
    dom::endSpecialObject(obj, nonExprAttrib, exprAttrib, structDefs);
  end_if;
  result
end_proc:

