/**
 *   MuPlotML:  Domain which writes MuPAD plot primitives into a
 *              file;  the output is XML code conform with the
 *              muplotml_{23}d.dtd 
 */

alias(STL = plot::STL):

plot::STL := newDomain("plot::STL"):

// poor man's inheritance
map([op(plot::GenericOutput)],
  eq -> if not contains(STL, op(eq, 1)) then
          slot(STL, op(eq, 1), prog::bless(op(eq, 2), STL))
        end_if):

plot::STL::create_dom:=hold(plot::STL):
plot::STL::OutputFile := 0:
plot::STL::format := Text:
plot::STL::append := FALSE:
plot::STL::outputbox := FAIL:
plot::STL::scaling := Unconstrained:
plot::STL::triangles := []:
plot::STL::appendData := []:
plot::STL::oldConstraints := []:

plot::STL::initialize := proc()
begin
  sysassign(plot::STL::OutputFile, 0);
  sysassign(plot::STL::format, Text);
  sysassign(plot::STL::append, FALSE);
  sysassign(plot::STL::outputbox, FAIL);
  sysassign(plot::STL::scaling, Unconstrained);
  sysassign(plot::STL::triangles, []);
  sysassign(plot::STL::oldConstraints, [RD_INF, RD_NINF, RD_INF, RD_NINF, RD_INF, RD_NINF]);
end_proc:

plot::STL::setScaling := proc(outputbox, scaling)
begin
  sysassign(plot::STL::outputbox, outputbox);
  sysassign(plot::STL::scaling, scaling);
end_proc:

plot::STL::setOutputFile := proc(file : DOM_INT, format=Text, append=FALSE) : DOM_BOOL
begin
  sysassign(plot::STL::format, format);
  sysassign(plot::STL::append, append);
  Pref::floatFormat("f"):
  sysassign(plot::STL::OutputFile, file);
  if format<>Binary and not append then
    fprint(Unquoted, plot::STL::OutputFile, "solid MuPADtoSTL"):
  end_if;
  TRUE;
end_proc:

/**
 *   plot::STL::openOutputFile :  this function opens an output file.
 *
 *     All following calls of MuPlotML::new will write into this file
 *     until it is closed with MuPlotML::closeOutputFile.
 *
 *   Argument :     file : DOM_STRING  - the name of the file 
 *                  If the string is empty an temporary file is opened.

 *   Return value : TRUE   iff the file was successfully opened
 *                  FALSE  iff the file could not be opened
 */
plot::STL::openOutputFile :=
proc(filename : DOM_STRING, format=Text, append=FALSE) : DOM_BOOL
  local n,
    oldheader, oldntri, oldfacetdata,
    oldx_min, oldx_max, oldy_min, oldy_max, oldz_min, oldz_max,
    i, tmp;
  save solid, vertex, endsolid;
begin
  oldx_min:= RD_INF: oldx_max:= RD_NINF:
  oldy_min:= RD_INF: oldy_max:= RD_NINF:
  oldz_min:= RD_INF: oldz_max:= RD_NINF:

  sysassign(plot::STL::format, format);
  sysassign(plot::STL::append, append);
  Pref::floatFormat("f"):
  if format = Binary then
    if append then
      n:= fopen(filename, Read, Raw):
      oldheader:= readbytes(n, 80, Byte, LittleEndian):
      oldntri:= readbytes(n, 1, Word, LittleEndian)[1]:
      if plot::STL::outputbox = FAIL then
          oldfacetdata:= readbytes(n, Byte, LittleEndian);
          assert(nops(oldfacetdata) = (12*4 + 2)*oldntri);
      else
          oldfacetdata:= [0 $ oldntri];
          for i from 1 to oldntri do
            oldfacetdata[i]:= readbytes(n, 12, Float, LittleEndian):
            tmp:= oldfacetdata[i];
            oldx_min:= min(oldx_min, tmp[4], tmp[7], tmp[10]);
            oldx_max:= max(oldx_max, tmp[4], tmp[7], tmp[10]);
            oldy_min:= min(oldy_min, tmp[5], tmp[8], tmp[11]);
            oldy_max:= max(oldy_max, tmp[5], tmp[8], tmp[11]);
            oldz_min:= min(oldz_min, tmp[6], tmp[9], tmp[12]);
            oldz_max:= max(oldz_max, tmp[6], tmp[9], tmp[12]);
            readbytes(n, 2, Byte, LittleEndian); // 2 unused bytes = end of facet
          end_for:
          assert(nops(oldfacetdata) = oldntri);
      end_if;
      sysassign(plot::STL::appendData, [oldheader, oldntri, oldfacetdata]);
      fclose(n);
    end_if;
    n:= fopen(filename, Write, Raw)
  else // format = Text
    delete solid, vertex, endsolid;
    if append then
        oldfacetdata:= import::readdata(filename);
        if plot::STL::outputbox <> FAIL then
          for i from 1 to nops(oldfacetdata) do
            tmp:= oldfacetdata[i];
            if has(tmp, vertex) then
              oldx_min:= min(oldx_min, tmp[2]);
              oldx_max:= max(oldx_max, tmp[2]);
              oldy_min:= min(oldy_min, tmp[3]);
              oldy_max:= max(oldy_max, tmp[3]);
              oldz_min:= min(oldz_min, tmp[4]);
              oldz_max:= max(oldz_max, tmp[4]);
            end_if:
          end_for:
        end_if;
        sysassign(plot::STL::appendData, [oldfacetdata]);
    end_if;
    n:= fopen(filename, Write, Text):
  end_if;
  if n = FAIL then
    error("cannot open file ".filename);
  end_if:
  sysassign(plot::STL::OutputFile, n);
  if format<>Binary and not append then
    fprint(Unquoted, plot::STL::OutputFile, "solid MuPADtoSTL"):
  end_if;
  sysassign(plot::STL::oldConstraints, [oldx_min, oldx_max, oldy_min, oldy_max, oldz_min, oldz_max]);
  TRUE;
end:

plot::STL::flushOutputFile:=proc()
  local i, tri, x_min, x_max, y_min, y_max, z_min, z_max,
    oldx_min, oldx_max, oldy_min, oldy_max, oldz_min, oldz_max,
    scale, shift, x0, x1, y0, y1, z0, z1,
    fitToOutputbox, fitNormal,
    normal, pt1, pt2, pt3,
    oldheader, oldntri, oldfacetdata, oldTris, olddata,
    header, objectnr;
  save DIGITS, endsolid, solid;
begin
  objectnr := 1;
  delete endsolid, solid;
  [oldx_min, oldx_max, oldy_min, oldy_max, oldz_min, oldz_max] := plot::STL::oldConstraints;
    fitToOutputbox:= proc(pt)
    local i;
    begin
        [shift[i] + scale[i]*pt[i] $ i = 1..3];
    end_proc;

    fitNormal:= proc(normal)
    local i;
    begin
      if iszero(_plus(normal[i]^2 $ i = 1..3)) then return([0.0,0.0,0.0]); end_if;
      normal:= [normal[i]/scale[i] $ i = 1..3];
      map(normal, _divide, _plus(normal[i]^2 $ i = 1..3)^(1/2));
    end_proc;

  tri := plot::STL::triangles;
  if nops(tri)=0 then return(); end_if;
  if plot::STL::outputbox<>FAIL then
    x_min := min(oldx_min,
                  tri[i][2][1] $ i = 1..nops(tri),
                  tri[i][3][1] $ i = 1..nops(tri),
                  tri[i][4][1] $ i = 1..nops(tri));
    x_max := max(oldx_max,
                  tri[i][2][1] $ i = 1..nops(tri),
                  tri[i][3][1] $ i = 1..nops(tri),
                  tri[i][4][1] $ i = 1..nops(tri));
    y_min := min(oldy_min,
                  tri[i][2][2] $ i = 1..nops(tri),
                  tri[i][3][2] $ i = 1..nops(tri),
                  tri[i][4][2] $ i = 1..nops(tri));
    y_max := max(oldy_max,
                  tri[i][2][2] $ i = 1..nops(tri),
                  tri[i][3][2] $ i = 1..nops(tri),
                  tri[i][4][2] $ i = 1..nops(tri));
    z_min := min(oldz_min,
                  tri[i][2][3] $ i = 1..nops(tri),
                  tri[i][3][3] $ i = 1..nops(tri),
                  tri[i][4][3] $ i = 1..nops(tri));
    z_max := max(oldz_max,
                  tri[i][2][3] $ i = 1..nops(tri),
                  tri[i][3][3] $ i = 1..nops(tri),
                  tri[i][4][3] $ i = 1..nops(tri));
    if x_min= RD_INF then x_min:= 0: end_if;
    if x_max= RD_NINF then x_max:= 1: end_if;
    if y_min= RD_INF then y_min:= 0: end_if;
    if y_max= RD_NINF then y_max:= 1: end_if;
    if z_min= RD_INF then z_min:= 0: end_if;
    if z_max= RD_NINF then z_max:= 1: end_if;

    [x0, x1] := float([op(plot::STL::outputbox[1])]);
    [y0, y1] := float([op(plot::STL::outputbox[2])]);
    [z0, z1] := float([op(plot::STL::outputbox[3])]);

    if iszero(x_max - x_min) then
       scale[1]:= 1;
    else 
       scale[1]:= (x1 - x0)/(x_max - x_min);
    end_if;
    if iszero(y_max - y_min) then
       scale[2]:= 1;
    else 
       scale[2]:= (y1 - y0)/(y_max - y_min);
    end_if;
    if iszero(z_max - z_min) then
       scale[3]:= 1;
    else
       scale[3]:= (z1 - z0)/(z_max - z_min);
    end_if;

    if plot::STL::scaling = Constrained then
       scale[1]:= min(scale[1], scale[2], scale[3]);
       scale[2]:= scale[1];
       scale[3]:= scale[1];
    end_if:

    shift[1]:= (x0 + x1 - scale[1]*(x_min + x_max))/2;
    shift[2]:= (y0 + y1 - scale[2]*(y_min + y_max))/2;
    shift[3]:= (z0 + z1 - scale[3]*(z_min + z_max))/2;
  else
    fitNormal := id;
    fitToOutputbox := id;
  end_if;

  if plot::STL::format = Binary then
    if plot::STL::append then
      [oldheader, oldntri, oldfacetdata] := plot::STL::appendData;
      writebytes(plot::STL::OutputFile, oldheader, Byte, LittleEndian):
      // replace the new number of facets
      writebytes(plot::STL::OutputFile, [oldntri + nops(tri)], Word, LittleEndian):
      if nops(oldfacetdata) > 0 then
          if plot::STL::outputbox=FAIL then
            writebytes(plot::STL::OutputFile, oldfacetdata, Byte, LittleEndian);
          else
            for i from 1 to oldntri do
                assert(nops(oldfacetdata[i]) = 12);
                normal:= fitNormal(oldfacetdata[i][1..3]);
                pt1:= fitToOutputbox(oldfacetdata[i][4..6]);
                pt2:= fitToOutputbox(oldfacetdata[i][7..9]);
                pt3:= fitToOutputbox(oldfacetdata[i][10..12]);
                writebytes(plot::STL::OutputFile, normal.pt1.pt2.pt3, Float, LittleEndian);
                writebytes(plot::STL::OutputFile, [0, 0], Byte, LittleEndian); // 2 unused bytes = end of facet
            end_for:
          end_if;
      end_if;
    else // not append
      header:= numlib::toAscii("MuPADtoSTL".expr2text(objectnr));
      if nops(header) > 80 then
          header:= header[1..80];
      else
          header:= header . [0 $ 80 - nops(header)];
      end_if;
      assert(nops(header) = 80);
      writebytes(plot::STL::OutputFile, header, Byte, LittleEndian):    // the header
      writebytes(plot::STL::OutputFile, [nops(tri)], Word, LittleEndian): // the number of facets
    end_if;
  else // Format = Text
    if plot::STL::append then
      i:= 1;
      [olddata] := plot::STL::appendData;
      oldTris := [];
      repeat
        assert(i+7 <= nops(olddata));
        if has(olddata[i], solid) then
          fprint(Unquoted, plot::STL::OutputFile, "solid MuPADtoSTL".expr2text(objectnr)): // begin new solid
          i:= i + 1;
        end_if;
        normal:= olddata[i][3..5];
        pt1:= olddata[i+2][2..4]:
        pt2:= olddata[i+3][2..4]:
        pt3:= olddata[i+4][2..4]:
        oldTris := oldTris.[[normal, pt1, pt2, pt3]];
        i:= i + 7;
        if has(olddata[i], endsolid) then
          i:= i + 1;
        end_if;
      until i > nops(olddata) end_repeat:
      tri := oldTris.tri;
    end_if;
  end_if;

  for i from 1 to nops(tri) do
      /* 10 Schutzziffern fr Skalierung */
      DIGITS:= DIGITS + 10:
      [normal, pt1, pt2, pt3]:= tri[i];
      normal:= fitNormal(normal);
      pt1:= fitToOutputbox(pt1):
      pt2:= fitToOutputbox(pt2):
      pt3:= fitToOutputbox(pt3):
      DIGITS:= DIGITS - 10:
      plot::STL::writeTri(normal, pt1, pt2, pt3);
  end_for;
end_proc:

/**
 *   plot::STL::closeOutputFile :  closes the file opened by
 *                                plot::STL::openOutputFile.
 *
 *   Argument :      -
 *
 *   Return value :  (0   -- (which is the new value of MuPlotML::OutputFile))
 *                   Testwise: the name of the just closed file.
 */
plot::STL::closeOutputFile :=
proc()
begin
  if plot::STL::format <> Binary then
    fprint(Unquoted, plot::STL::OutputFile, "endsolid MuPADtoSTL"):
  end_if;
  fclose(plot::STL::OutputFile);
end:

//-------------------------------------------------
/***   Utilities   ***/


/////////////////////////////////////////////////////////////////////////////
// Utils for 'new' MuPlotML (kd, 20.6.02)

//-------------------------------------------------

//-------------------------------------------------
plot::STL::startAnimatedObject :=
proc(obj, objName, nonExprAttrib, exprAttrib)
begin
  error("Animated objects are not supported");
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::endAnimatedObject := (obj, objName, nonExprAttrib, exprAttrib) -> null():
//-------------------------------------------------

//-------------------------------------------------
plot::STL::startStaticObject :=
proc(obj, objName, nonExprAttrib, exprAttrib)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::endStaticObject :=
proc(obj, objName, nonExprAttrib, exprAttrib)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::startSpecialObject :=
proc(obj, nonExprAttrib, exprAttrib, structDefs)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::endSpecialObject :=
proc(obj, nonExprAttrib, exprAttrib, structDefs)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::printViewingBox :=
proc(obj, VB)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::startAnimationFrame :=
proc(obj, imgName, nonExprAttrib, exprAttrib, optionals, transf=table(), static=FALSE)
begin
//   error("Animated objects are not supported");
end_proc:
//-------------------------------------------------

//-------------------------------------------------
plot::STL::endAnimationFrame :=
proc(obj, imgName, nonExprAttrib, exprAttrib, optionals, transf=table(), static=FALSE)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// Low level output functions

// 2D
  
plot::STL::writeArc2d :=
proc(attributes, explicit_attribs, centerX, centerY, axisX, axisY,
     startAngle=0.0, endAngle=float(2*PI))
begin
end_proc:

plot::STL::writeArrow2d :=
proc(attributes, explicit_attribs, fx, fy, tx, ty)
begin
end_proc:

plot::STL::writeColorArray2d :=
proc(attributes, explicit_attribs, data, xmesh, ymesh, xmin, xmax, ymin, ymax)
begin
end_proc:

plot::STL::writeDensityArray2d :=
proc(attributes, explicit_attribs, data:Type::Union(DOM_ARRAY, DOM_HFARRAY), xmin, xmax, ymin, ymax)
begin
end_proc:

plot::STL::writeField2d :=
proc(attributes, explicit_attribs, points, colorfunction,
      umin, vmin, stepU, stepV, umesh, vmesh, ops=1..2)
begin
end_proc:

plot::STL::writeLine2d :=
proc(attributes, explicit_attribs, fromX, fromY, toX, toY)
begin
end_proc:

plot::STL::writePoint2d :=
proc(attributes, explicit_attribs, posX, posY)
begin
end_proc:

plot::STL::writePoints2d :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), xyops=1..2)
begin
end_proc:

plot::STL::writePoly2d :=
proc(attributes, explicit_attribs, points, linecolorfunction=(()->null()), xyops=1..2, tops=FAIL)
begin
end_proc:

plot::STL::writeText2d :=
proc(attributes, explicit_attribs, posX, posY, text, textRotation=0.0)
begin
end_proc:

plot::STL::writeVerticalAsymptote :=
proc(attributes, explicit_attribs, x)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// 3D

plot::STL::writeArrow3d :=
proc(attributes, explicit_attribs, fx, fy, fz, tx, ty, tz)
begin
end_proc:

plot::STL::writeBox :=
proc(attributes, explicit_attribs, xmin, xmax, ymin, ymax, zmin, zmax)
begin
  plot::STL::writeTriangles(attributes, explicit_attribs,
    [[xmin, ymin, zmin], [xmax, ymin, zmin], [xmin, ymin, zmax],
    [xmax,ymin,zmin], [xmax,ymin,zmax], [xmin,ymin,zmax],
    [xmax,ymin,zmax], [xmax,ymax,zmax], [xmin,ymin,zmax],
    [xmax,ymax,zmax], [xmin,ymax,zmax], [xmin,ymin,zmax],
    [xmin, ymin, zmax], [xmin, ymax, zmin], [xmin, ymin, zmin],
    [xmin, ymin, zmax], [xmin, ymax, zmax], [xmin, ymax, zmin],
    [xmin, ymax, zmin], [xmax, ymax, zmax], [xmax, ymax, zmin],
    [xmin, ymax, zmin], [xmin, ymax, zmax], [xmax, ymax, zmax],
    [xmin, ymax, zmin], [xmax, ymin, zmin], [xmin, ymin, zmin],
    [xmin, ymax, zmin], [xmax, ymax, zmin], [xmax, ymin, zmin],
    [xmax, ymin, zmin], [xmax, ymax, zmax], [xmax, ymin, zmax],
    [xmax, ymin, zmin], [xmax, ymax, zmin], [xmax, ymax, zmax]
    ],
    ()->null(), ()->null(), 1..3);
end_proc:

plot::STL::writeCircle3d :=
proc(attributes, explicit_attribs, centerX, centerY, centerZ, normalX, normalY, normalZ, radius)
  local p1, p2, p3, p4, Base, umesh, points, x1, y1, x2, y2, u;
begin
  Base := matrix([centerX, centerY, centerZ]);
  if iszero(normalX) and iszero(normalY) then
    p1 := matrix([1,0,0]);
    p2 := matrix([0,1,0]);
  else
    if iszero(normalY) then
      p1 := linalg::crossProduct(matrix([normalX, normalY, normalZ]), matrix([0,1,0]));
    else
      p1 := linalg::crossProduct(matrix([normalX, normalY, normalZ]), matrix([1,0,0]));
    end_if;
    p2 := linalg::crossProduct(matrix([normalX, normalY, normalZ]), p1);
    p1 := linalg::normalize(p1);
    p2 := linalg::normalize(p2);
  end_if;
  umesh := 25;
  points := [];
  for u from 1 to umesh do
    x1 := float(cos((u-1)/umesh*2.0*PI));
    y1 := float(sin((u-1)/umesh*2.0*PI));
    x2 := float(cos(u/umesh*2.0*PI));
    y2 := float(sin(u/umesh*2.0*PI));
    p3 := x1*p1+y1*p2;
    p4 := x2*p1+y2*p2;
    points := points.[
      radius*p4+Base, Base, radius*p3+Base
      ];
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, ()->null(), ()->null(),
    1..3);
end_proc:

plot::STL::writeCone :=
proc(attributes, explicit_attribs, BaseX, BaseY, BaseZ,
    TopX, TopY, TopZ, BaseRadius, TopRadius)
  local u, umesh,
    xDiff, yDiff, zDiff,
    x1, y1,
    x2, y2,
    p1, p2, p3, p4,
    Base, Top,
    points;
begin
  xDiff := TopX-BaseX;
  yDiff := TopY-BaseY;
  zDiff := TopZ-BaseZ;
  Base := matrix([BaseX, BaseY, BaseZ]);
  Top := matrix([TopX, TopY, TopZ]);
  if iszero(xDiff) and iszero(yDiff) then
    p1 := matrix([1,0,0]);
    p2 := matrix([0,1,0]);
  else
    if iszero(yDiff) then
      p1 := linalg::crossProduct(Top-Base, matrix([0,1,0]));
    else
      p1 := linalg::crossProduct(Top-Base, matrix([1,0,0]));
    end_if;
    p2 := linalg::crossProduct(Top-Base, p1);
    p1 := linalg::normalize(p1);
    p2 := linalg::normalize(p2);
  end_if;
  umesh := 25;
  points := [];
  for u from 1 to umesh do
    x1 := float(cos((u-1)/umesh*2.0*PI));
    y1 := float(sin((u-1)/umesh*2.0*PI));
    x2 := float(cos(u/umesh*2.0*PI));
    y2 := float(sin(u/umesh*2.0*PI));
    p3 := x1*p1+y1*p2;
    p4 := x2*p1+y2*p2;
    points := points.[
      BaseRadius*p4+Base, BaseRadius*p3+Base, TopRadius*p3+Top,
      TopRadius*p3+Top, TopRadius*p4+Top, BaseRadius*p4+Base
      ];
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, ()->null(), ()->null(),
    1..3);
end_proc:

plot::STL::writeEllipsoid :=
proc(attributes, explicit_attribs, CenterX, CenterY, CenterZ,
    SemiAxisX, SemiAxisY, SemiAxisZ)
  local u, v,
    points, mesh;
begin
  points := [];
  for u from 0 to 25 do
    for v from 0 to 25 do
      mesh[u, v] := float([cos(u/25*2*PI)*sin(v/25*PI), sin(u/25*2*PI)*sin(v/25*PI), cos(v/25*PI)]);
    end_for;
  end_for;
  for u from 1 to 25 do
    for v from 1 to 25 do
      points := points.[
        mesh[u-1, v-1], mesh[u, v-1], mesh[u-1, v],
        mesh[u, v-1], mesh[u, v], mesh[u-1, v]
        ];
    end_for;
  end_for;
  points := map(points,(X)->([CenterX+X[1]*SemiAxisX, CenterY+X[2]*SemiAxisY, CenterZ+X[3]*SemiAxisZ]));
  plot::STL::writeTriangles(attributes, explicit_attribs, points, ()->null(), ()->null(), 1..3);
end_proc:

plot::STL::writeField3d :=
proc(attributes, explicit_attribs, points,
      xmin, ymin, zmin, stepX, stepY, stepZ, xmesh, ymesh, zmesh,
      colorfunction=(()->null()), xyops=1..3, uvops=FAIL)
begin
end_proc:

plot::STL::writeLine3d :=
proc(attributes, explicit_attribs, fromX, fromY, fromZ, toX, toY, toZ)
begin
end_proc:

plot::STL::writeLines3d :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), xyops=1..3, tops=FAIL)
begin
end_proc:

plot::STL::writeLineLoop3d :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), xyops=1..3, tops=FAIL)
begin
end_proc:

plot::STL::writePoint3d :=
proc(attributes, explicit_attribs, posX, posY, posZ)
begin
end_proc:

plot::STL::writePlane := /* TODO, bei viewingbox in den Attributes */
proc(attributes, explicit_attribs, posX, posY, posZ, normalX, normalY, normalZ)
begin
end_proc:

plot::STL::writeRectangularMesh :=
proc(attributes, explicit_attribs, data, linecolorfunction, fillcolorfunction,
  umesh, vmesh, usubmesh, vsubmesh, usexylinesvisible=FALSE, xyops=3..5, normalops=FAIL, uvops=FAIL)
local u, v,
  points;
begin
  points := [];
  for u from 1 to umesh-1 do
    for v from 1 to vmesh-1 do
      points := points.[data[u+(v-1)*umesh], data[u+(v)*umesh], data[1+u+(v-1)*umesh],
        data[1+u+(v-1)*umesh], data[u+(v)*umesh], data[1+u+(v)*umesh]];
    end_for;
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, linecolorfunction, fillcolorfunction,
    xyops, normalops, uvops);
end_proc:
//-------------------------------------------------

plot::STL::writeText3d :=
proc(attributes, explicit_attribs, posX, posY, posZ, baseDirX, baseDirY, baseDirZ,
  faceDirX, faceDirY, faceDirZ, text)
begin
end_proc:

plot::STL::writeTriangles :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=3..5, normalops=FAIL, uvops=FAIL)
local i, normal, pt1, pt2, pt3,
  a1, a2, a3, b1, b2, b3, nn;
begin
  for i from 1 to nops(data) step 3 do
    [normal, pt1, pt2, pt3]:= [[0,0,0], [op(data[i], xyops)], [op(data[i+1], xyops)], [op(data[i+2], xyops)]];
    if normalops<>FAIL then
      normal := [op(data[i], normalops)];
    else
      /* Keine normalen vorhanden? Kein Prob, wir bauen welche */
      a1 := pt2[1]-pt1[1];
      a2 := pt2[2]-pt1[2];
      a3 := pt2[3]-pt1[3];
      b1 := pt3[1]-pt1[1];
      b2 := pt3[2]-pt1[2];
      b3 := pt3[3]-pt1[3];
      normal := [a2*b3-a3*b2, a3*b1-a1*b3, a1*b2-a2*b1];
      nn := sqrt(normal[1]^2 + normal[2]^2 + normal[3]^2);
      if not iszero(nn) then normal:=map(normal, X->X/nn); end_if;
    end_if;
/*    if outputbox <> NIL then
      pt1:= fitToOutputbox(pt1):
      pt2:= fitToOutputbox(pt2):
      pt3:= fitToOutputbox(pt3):
    end_if:*/
/*    if format = Binary then
      writebytes(n, normal.pt1.pt2.pt3, Float, byteorder);
      writebytes(n, [0, 0], Byte, byteorder); // 2 unused bytes = end of facet
    else*/
      if plot::STL::format<>Binary and plot::STL::outputbox=FAIL then
        plot::STL::writeTri(normal, pt1, pt2, pt3);
      else
        sysassign(plot::STL::triangles, plot::STL::triangles.[[normal, pt1, pt2, pt3]]);
      end_if;
//     end_if;
  end_for;
end_proc:

plot::STL::writeTri:=proc(normal, pt1, pt2, pt3)
begin
  if plot::STL::format = Binary then
    writebytes(plot::STL::OutputFile, normal.pt1.pt2.pt3, Float, LittleEndian);
    writebytes(plot::STL::OutputFile, [0, 0], Byte, LittleEndian); // 2 unused bytes = end of facet
  else
    fprint(Unquoted, plot::STL::OutputFile, " facet normal ",
            normal[1], " ", normal[2], " ", normal[3]);
    fprint(Unquoted, plot::STL::OutputFile, "  outer loop "):
    fprint(Unquoted, plot::STL::OutputFile, "   vertex ", pt1[1], " ", pt1[2], " ", pt1[3]):
    fprint(Unquoted, plot::STL::OutputFile, "   vertex ", pt2[1], " ", pt2[2], " ", pt2[3]):
    fprint(Unquoted, plot::STL::OutputFile, "   vertex ", pt3[1], " ", pt3[2], " ", pt3[3]):
    fprint(Unquoted, plot::STL::OutputFile, "  endloop "):
    fprint(Unquoted, plot::STL::OutputFile, " endfacet "):
  end_if;
end_proc:
//-------------------------------------------------

plot::STL::writeQuadStrip :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()), xyops=1..3,
    normalops=FAIL, uvops=FAIL)
  local points, i;
begin
  points := [];
  for i from 1 to nops(data)-3 step 2 do
    points := points.[data[i], data[i+1], data[i+2], data[i+1], data[i+2], data[i+3]];
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, linecolorfunction, fillcolorfunction,
    xyops, normalops, uvops);
end_proc:

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
plot::STL::writeTriangleFan :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=3..5, normalops=FAIL, uvops=FAIL)
  local points, i, base;
begin
  points := [];
  base := data[1];
  for i from 2 to nops(data)-1 do
    points := points.[base, data[i], data[i+1]];
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, linecolorfunction, fillcolorfunction,
    xyops, normalops, uvops);
end_proc:

plot::STL::writeTriangleStrip :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
    xyops=3..5, normalops=FAIL, uvops=FAIL)
  local points, i;
begin
  points := [];
  for i from 1 to nops(data)-2 do
    points := points.[data[i], data[i+1], data[i+2]];
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, linecolorfunction, fillcolorfunction,
    xyops, normalops, uvops);
end_proc:

plot::STL::writeSphere :=
proc(attributes, explicit_attribs, centerX, centerY, centerZ, radius)
  local u, v,
    points, mesh;
begin
  points := [];
  for u from 0 to 25 do
    for v from 0 to 25 do
      mesh[u, v] := float([cos(u/25*2*PI)*sin(v/25*PI), sin(u/25*2*PI)*sin(v/25*PI), cos(v/25*PI)]);
    end_for;
  end_for;
  for u from 1 to 25 do
    for v from 1 to 25 do
      points := points.[
        mesh[u-1, v-1], mesh[u, v-1], mesh[u-1, v],
        mesh[u, v-1], mesh[u, v], mesh[u-1, v]
        ];
    end_for;
  end_for;
  points := map(points,(X)->([centerX+X[1]*radius, centerY+X[2]*radius, centerZ+X[3]*radius]));
  plot::STL::writeTriangles(attributes, explicit_attribs, points, ()->null(), ()->null(), 1..3);
end_proc:

plot::STL::writeQuads :=
proc(attributes, explicit_attribs, data, linecolorfunction=(()->null()), fillcolorfunction=(()->null()),
  xyops=1..3, normalops=FAIL, uvops=FAIL)
local i,
  points;
begin
  points := [];
  for i from 1 to nops(data) step 4 do
    points := points.[data[i], data[i+1], data[i+2], data[i], data[i+3], data[i+2]];
  end_for;
  plot::STL::writeTriangles(attributes, explicit_attribs, points, linecolorfunction, fillcolorfunction,
    xyops, normalops, uvops);
end_proc:
//-------------------------------------------------

plot::STL::writeMeshContours :=
proc(attributes, explicit_attribs, data, ops, contdir, contours,
     linecolorfunction, _min, _max)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
//  writeGridContours: plot contour lines
// for a rectangular mesh
// attributes: A table as given into doPlotStatic or MuPlotML
// data = a list of points
// umesh, vmesh = dimension of data
// ops = a range indicating in which position in the points in data x, y, z are found
// contdir = the index in the points where the direction is found
//    for which to plot contour lines
// contours = a list of values for where to write contour lines
//    or the list [Automatic, n] with n being the number of lines.
// linecolorfunction: Function mapping a point to a color. If no function
//   is desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
// _min, _max = viewing box extension in the direction currently considered.  
plot::STL::writeGridContours :=
proc(attributes, explicit_attribs, data, umesh, vmesh, ops, contdir, contours,
     linecolorfunction, _min, _max)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
plot::STL::writePoly3d :=
proc(attributes, explicit_attribs, data,
  linecolorfunction=(()->null()), ops=3..5)
begin
end_proc:
//-------------------------------------------------

//-------------------------------------------------
// attributes: A table as given into doPlotStatic or MuPlotML
// data = [[u1, v1, x1, y1, z1], [u2, v2, x2, y2, z2], ...]
//     or [[u1, v1, x1, y1, z1, nx1, ny1, nz1], ...] (with normals)
// linecolorfunction, fillcolorfunction: Functions. If no functions
//   are desired, pass () -> null(). Otherwise, they'll be called with
//   u, v, x, y, z.
plot::STL::writePoints3d :=
proc(attributes, explicit_attribs, data,
  colorfunction=(()->null()), ops=3..5)
begin
end_proc:
//-------------------------------------------------
