//      

/* -----------------------------------------------------------
    Arc2d -- The graphical primitive for 2D arcs

    Syntax:
    Arc2d(radius<, center><, angleRange><, op1, op2, ...>)

    radius                : expression; radius of circle
                        OR  list of 2 expressions; 2 semi-axes of ellipse
    center                : list of 2 expressions
    angleRange            : range of 2 expressions
    op1, op2, ...         : options of the form 'option = value'

    Example:
    >> plot::Arc2d(2)

    >> plot::Arc2d(2, [1, 2])

    >> plot::Arc2d([2, 3], [1, 2])

    >> plot::Arc2d(2, -PI/4..PI/3)

    >> plot::Arc2d(2, [1, 2], -PI/4..PI/3)

    >> plot::Arc2d([2, 3], [1, 2], -PI/4..PI/3)

    >> Arc:= [1, 1], [0, 0], PI/4..PI/2, Filled, Closed, FillPattern=Solid:
       plot(plot::Arc2d(Arc, Angle=a+0,      a=0..2*PI, FillColor=RGB::Red),
            plot::Arc2d(Arc, Angle=a+1/2*PI, a=0..2*PI, FillColor=RGB::Green),
            plot::Arc2d(Arc, Angle=a+PI,     a=0..2*PI, FillColor=RGB::Yellow),
            plot::Arc2d(Arc, Angle=a+3/2*PI, a=0..2*PI, FillColor=RGB::Blue))

----------------------------------------------------------------*/ 

plot::createPlotDomain("Arc2d",
                       "graphical primitive for 2D arcs",
                       2,
       [CenterX, CenterY, Center, SemiAxisX, SemiAxisY, SemiAxes,
        Angle,
        [AngleBegin, ["Optional", 0],
         ["Definition", "Expr", FAIL, "Beginning of the angle", FALSE]],
        [AngleEnd,   ["Optional", 2*PI],
         ["Definition", "Expr", FAIL, "End of the angle", FALSE]],
        // library interface for AngleBegin, AngleEnd
        [AngleRange,   ["Library", NIL,
                        plot::libRangeOfOptExpr("Angle"),
                        plot::readRangeOfOptExpr("Angle")],
         [[AngleBegin..AngleEnd]]],
        LineColor, LineWidth, LineStyle, LinesVisible, AntiAliased, Closed,
        Filled, FillColor, FillPattern]
):
//-----------------------------------------------------------------------------------
plot::Arc2d::styleSheet := table(
      CenterX = 0,
      CenterY = 0,
      SemiAxisX = 1,
      SemiAxisY = 1,
      Angle   = 0,
      AngleBegin = 0,
      AngleEnd   = PI/2,
      Filled = FALSE
):
plot::Arc2d::hints := {Scaling = Constrained}:
plot::Arc2d::setPrimaryColor(LineColor):
//-----------------------------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//-----------------------------------------------------------------------------------
plot::Arc2d::new:=
  proc()
    local object, other, i, L;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;

    if testargs() then
      if nops(other) > 3 then
        error("unexpected argument: ".expr2text(other[4]))
      end_if;

      if nops(other) > 0 then
        if not testtype(other[1], Type::Arithmetical) and
          not testtype(other[1],
                       Type::ListOf(Type::Arithmetical, 2, 2)) and

          not (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
          error("1st argument: expecting an expression (the radius of a spherical arc) ".
                 "or a list of 2 expressions (the semi-axes of an elliptical arc)")
        end_if;

        for i from 2 to nops(other) do
          L:= op(other,i);
          if not testtype(L, Type::ListOf(Type::Arithmetical, 2, 2)) and
            not testtype(L, "_range")  and
            not L::dom::hasProp(Cat::Matrix)=TRUE then
            error(output::ordinal(i)." argument: expecting a list of 2 expressions or a range")
          end_if;
        end_for;
      end_if;
    end_if;
    
    if nops(other) <> 0 then
      if testtype(other[1], Type::ListOf(Type::Arithmetical, 2, 2)) then
        object::SemiAxisX := other[1][1];
        object::SemiAxisY := other[1][2];
      elif testtype(other[1], Type::Arithmetical) then
        object::SemiAxisX := other[1];
        object::SemiAxisY := other[1];
      else
        object::SemiAxes := other[1]
      end_if;

      if nops(other) > 1 then
        for i in [op(other, 2..nops(other))] do
          if testtype(i, Type::ListOf(Type::Arithmetical, 2, 2)) or
            i::dom::hasProp(Cat::Matrix)=TRUE then
            object::Center := i;
          elif  testtype(i, "_range") then
            object::AngleRange := i;
          end_if;
        end_for;
      end_if;      
    end_if;
      
    // semantically check for validity
    dom::checkObject(object);
  end_proc:

//-----------------------------------------------------------------------------------
plot::Arc2d::print :=
  obj -> if obj::AngleRange = FAIL then
           hold(plot::Arc2d)(obj::SemiAxes, obj::Center):
         else
           hold(plot::Arc2d)(obj::SemiAxes, obj::Center, obj::AngleRange):
         end_if:
//-----------------------------------------------------------------------------------
plot::Arc2d::doPlotStatic:=
  proc(out, obj, attrib)
    local centerX, centerY, axisX, axisY,
          angleA, angleB, angle;
  begin
    centerX:= float(attrib[CenterX]);
    centerY:= float(attrib[CenterY]);
    axisX  := float(attrib[SemiAxisX]);
    axisY  := float(attrib[SemiAxisY]);
    angle  := float(attrib[Angle]);
    angleA := float(attrib[AngleBegin]);
    angleB := float(attrib[AngleEnd]);

    angleA:= angleA + angle;
    angleB:= angleB + angle;

    return(out::writeArc2d(attrib, table(), centerX, centerY, axisX, axisY,
       angleA, angleB));
  end_proc:
//-----------------------------------------------------------------------------------
