/* --------------------------------------------------------------------
plot::Bars3d - 3D histogram plot of matrix data

Calls:  plot::Bars3d(data, <x = xmin .. xmax, y = ymin .. ymax>, 
                     Ground = g, XGap = xgap, YGap = ygap, BarStyle = style)

Parameters:
   data       - a list of lists, an array, or a matrix
   x          - hint for XAxisTitle: identifier or indexed identifier
   xmin..xmax - x range. 
                If not specified, xmin = 0, xmax = number of data columns
   y          - hint for YAxisTitle: identifier or indexed identifier
   ymin..ymax - y range
                If not specified, ymin = 0, ymax = number of data rows
   g          - real value (the bars extend from z = g to z = data value
   xgap,ygap  - real values between 0 and 1.
                Only have an effect for Style = Boxes. With xgap = 0,
                there is no gap between the boxes. With xgap = 0.5, 
                the gap between the boxes is as wide as the boxes.
   style      - Boxes, Lines, Points, LinesPoints
-------------------------------------------------------------------- */

plot::createPlotDomain("Bars3d",
		       "graphical primitive for 3D bar plots",
		       3,
             [// declare the attributes visible in the object inspector
		        LinesVisible, LineStyle, LineWidth, LineColor,
              PointSize, PointStyle,
              Filled, 
          //  FillColor, // no FillColor (it does not trigger a replot)
              Color,     // the lib infra structure lets Color set Colors = [Color]
              Colors, 
              XName, XMin, XMax,
              YName, YMin, YMax, 
              [XGap, ["Optional", NIL],
               ["Style", "Expr", FAIL,
                "Gap between bars in x direction.", TRUE]],
              [YGap, ["Optional", NIL],
			      ["Style", "Expr", FAIL,
   		       "Gap between bars in y direction.", TRUE]],
              [Gap, ["Library", NIL,
                     plot::libListOfExpr([Gap, plot::elementExpr,
                                          [XGap, YGap]]),
                     plot::readListOfExpr([Gap, [XGap, YGap]])],
               [[[XGap, YGap]]]
              ], 
              [Ground, ["Optional", NIL],
	            ["Definition", "Expr", FAIL,
			       "The base value.", TRUE]
              ],
              [Data, ["Mandatory", NIL],
		         ["Definition", "ExprSeq", FAIL,
			      "Data to plot.", TRUE]
              ],
			     [BarStyle, ["Optional", Boxes],
			      [["Style", "Surface"], "Prop", 
			       {Boxes, Lines, Points, LinesPoints},
			       "Type of bars.", TRUE]
              ]
		       ]):
//----------------------------------------------------------------
plot::Bars3d::styleSheet := table(LinesVisible = TRUE,
                                  LineColor = RGB::Black.[0.25],
                                  Colors = RGB::ColorList[1..20],
                                  XGap = 0, YGap = 0,
                                  Ground = 0,
                                  BarStyle = Boxes
                                 ):
//----------------------------------------------------------------
plot::Bars3d::hints := {XAxisTitle, YAxisTitle, Axes = None}:
//----------------------------------------------------------------
plot::Bars3d::print := object -> hold(plot::Bars3d)("...",
     object::XName = object::XMin .. object::XMax,
     object::YName = object::YMin .. object::YMax
):
//----------------------------------------------------------------
// On input, 'Data' may be a list of lists, an array or a matrix.
// Internally, it is stored as a list of lists. Convert the input.
//----------------------------------------------------------------
plot::Bars3d::changeNotifier := proc(object, eq)
local attributename, data, s, i, j;
begin
    attributename:= op(eq, 1);
    if attributename = "Data" then
       data := op(eq, 2);
       if data::dom::hasProp(Cat::Matrix)=TRUE then
          data:= expr(data);
       end_if;
       case domtype(data)
       of DOM_LIST do // input via a list of lists
          s:= {op(data)}:
          if map(s, domtype) <> {DOM_LIST} then
             error("expecting a list of lists of real values");
          end_if;
          s:= map(s, nops):
          if nops(s) <> 1 then
             error("all sublists (rows of the raster image) must have the same length");
          end_if:
          if object::XMin = FAIL then
             object::dom::extslot(object, "XMin", 0):
          end_if;
          if object::XMax = FAIL then
             object::dom::extslot(object, "XMax", nops(data[1]));
          end_if;
          if object::YMin = FAIL then
             object::dom::extslot(object, "YMin", 0):
          end_if;
          if object::YMax = FAIL then
             object::dom::extslot(object, "YMax", nops(data));
          end_if;
           // Data are the 'official' data seen in the inspector
           object::dom::extslot(object, "Data", data);
           break;
       //-------------------------------------------------------------------
       of DOM_ARRAY do
          if op(data, [0, 1]) <> 2 then
             error("First argument: expecting a two dimensional array ".
                   "of RGB or RGBa values")
          end_if;
          if object::XMin = FAIL then
             object::dom::extslot(object, "XMin", op(data, [0, 3, 1]) - 1);
          end_if;
          if object::XMax = FAIL then
             object::dom::extslot(object, "XMax", op(data, [0, 3, 2]));
          end_if;
          if object::YMin = FAIL then
             object::dom::extslot(object, "YMin", op(data, [0, 2, 1]) - 1);
          end_if;
          if object::YMax = FAIL then
             object::dom::extslot(object, "YMax", op(data, [0, 2, 2]));
          end_if;

          object::dom::extslot(object, "Data", float(
             [[data[i,j] $ j = op(data, [0, 3, 1]) .. op(data, [0, 3, 2])]
                         $ i = op(data, [0, 2, 1]) .. op(data, [0, 2, 2])]));
          break;
       //-------------------------------------------------------------------
       otherwise
            return("Expecting a list of lists or an array of real values");
       end_case;
       return(FALSE); // notify extslot that the assignment was already
                      // done inside the changeNotifier method
    else
       // any attributes other than Data are OK.
       // Let extop do the assignment
       return(TRUE);
    end_if;
end_proc:
//----------------------------------------------------------------
plot::Bars3d::new := proc()
local object, other;
begin
  object := dom::checkArgs([["X"], ["Y"]], args());
  if (object::XRange = FAIL and object::YRange <> FAIL)  or
     (object::YRange = FAIL and object::XRange <> FAIL)  then
     error("expecting ranges 'x = xmin .. xmax' and 'y = ymin .. ymax'");
  end_if;
  other:= object::other;
  if nops(other) <> 0 then
     if nops(other) > 1 then
        error("Unknown attribute: ".expr2text(other[2]));
     end_if;
     if object::XName <> FAIL then
        if object::XAxisTitle = FAIL then
           plot::setHint(object, XAxisTitle = expr2text(object::XName));
        end_if:
     else
        object::XName := hold(x); // Ouch
     end_if;
     if object::YName <> FAIL then
        if object::YAxisTitle = FAIL then
           plot::setHint(object, YAxisTitle = expr2text(object::YName));
        end_if:
     else
        object::YName := hold(y); // Ouch
     end_if;
     if (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
        other[1]:= expr(other[1]);
     end_if;
     case domtype(other[1])
     of DOM_LIST do
     of DOM_ARRAY do
        // changeNotifier does further type checking
        plot::Bars3d::changeNotifier(object, "Data" = other[1]);
        break;
     otherwise
       error("expecting a list of list, an array or a matrix of real data");
     end_case;
  end_if;
  //--------------------------------
  // check consistency of the object
  //--------------------------------
  dom::checkObject(object);
end_proc:
//----------------------------------------------------------------
plot::Bars3d::doPlotStatic := proc(out, object, attributes, inherited)
local xmesh, ymesh, ground, data, i, j, xgap, ygap, style, 
      color, colors, ncolors,
      x, xmin, xmax, dx,
      y, ymin, ymax, dy,
      z, zmin, zmax; 
begin
    //-------------------
    // get the attributes
    //-------------------
    xmin := float(attributes[XMin]);
    xmax := float(attributes[XMax]);
    ymin := float(attributes[YMin]);
    ymax := float(attributes[YMax]);
    //-----------------
    // get the data
    //-----------------
    data:= float(attributes[Data]);
    xmesh:= nops(data[1]): // number of data columns
    ymesh:= nops(data):    // number of data rows
    if iszero(xmesh) or iszero(ymesh) then
       return(); // empty plot
    end_if;
    //-----------------
    // get the colors
    //-----------------
    if contains(attributes, Colors) then
       colors:= attributes[Colors]:
    else
       // there is a default for 'Colors' in the styleSheet,
       // so this should not happen:
       error("expecting a value of the attribute 'Colors'"):
    end_if;
    ncolors:= nops(colors);
    //-----------------------------
    // determine the z range (needed
    // for the viewing box)
    //-----------------------------
    zmin:= min(map(data, op)):
    zmax:= max(map(data, op)):
    //-----------------------------
    // determine the 'BarStyle'
    //-----------------------------
    style:= attributes[BarStyle]:
    //-----------------------------
    // determine the 'ground level'
    //-----------------------------
    ground:= float(attributes[Ground]);
    if domtype(ground) <> DOM_FLOAT then
    // ground:= zmin:
       ground:= 0:
    end_if;
    //-----------------------------
    // correct the the viewing box
    //-----------------------------
    if style <> Points then
       zmin:= min(zmin, ground):
       zmax:= max(zmax, ground):
    end_if;
    //-----------------------------
    // determine the gaps
    //-----------------------------
    if style = Boxes then
       xgap:= float(attributes[XGap]);
       if domtype(xgap) = DOM_FLOAT then
            xgap := max(0, min(1, xgap)); // clip to [0, 1]
       else xgap := 0; // default value 
       end_if;
       ygap:= float(attributes[YGap]);
       if domtype(ygap) = DOM_FLOAT then
            ygap := max(0, min(1, ygap)); // clip to [0, 1]
       else ygap := 0; // default value 
       end_if;
    end_if;
    //-------------------------------------------------------
    // create the 'bars' (= 3D boxes), 'lines', 'points' etc.
    //-------------------------------------------------------
    case style 
    of Boxes do
       dx:= (1 - xgap)*(xmax - xmin)/xmesh/2:
       dy:= (1 - ygap)*(ymax - ymin)/ymesh/2:
       for i from 1 to ymesh do 
         y:= ymin + (i - 1/2)*(ymax - ymin)/ymesh:
         color:= colors[((i - 1) mod ncolors) + 1]:
         for j from 1 to xmesh do 
             x:= xmin + (j - 1/2)*(xmax - xmin)/xmesh:
             z:= data[i][j]:
             if domtype(z) <> DOM_FLOAT then
                error("expecting real numerical data. Got: ".expr2text(data[i][j]));
             end_if:
             out::writeBox(attributes, table("FillColor"=color),
               x - dx, x + dx, y - dy, y + dy, min(z, ground), max(z, ground));
         end_for;
       end_for:
       break;
    of Lines do
       for i from 1 to ymesh do 
         y:= ymin + (i - 1/2)*(ymax - ymin)/ymesh:
         color:= colors[((i - 1) mod ncolors) + 1]:
         for j from 1 to xmesh do 
             x:= xmin + (j - 1/2)*(xmax - xmin)/xmesh:
             z:= data[i][j]:
             if domtype(z) <> DOM_FLOAT then
                error("expecting real numerical data. Got: ".expr2text(data[i][j]));
             end_if:
             out::writePoly3d(attributes, table("LineColor" = color),
               [[x, y, x, y, min(z, ground)], [x, y, x, y, max(z, ground)]]);
         end_for;
       end_for:
       break;
    of Points do
       for i from 1 to ymesh do 
         y:= ymin + (i - 1/2)*(ymax - ymin)/ymesh:
         color:= colors[((i - 1) mod ncolors) + 1]:
         for j from 1 to xmesh do 
             x:= xmin + (j - 1/2)*(xmax - xmin)/xmesh:
             z:= data[i][j]:
             if domtype(z) <> DOM_FLOAT then
                error("expecting real numerical data. Got: ".expr2text(data[i][j]));
             end_if:
             out::writePoints3d(attributes, table("PointsVisible"=TRUE, "PointColor" = color),
               [[x, y, x, y, z]]);
         end_for;
       end_for:
       break;
    of LinesPoints do
       for i from 1 to ymesh do
         y:= ymin + (i - 1/2)*(ymax - ymin)/ymesh:
         color:= colors[((i - 1) mod ncolors) + 1]:
         for j from 1 to xmesh do
             x:= xmin + (j - 1/2)*(xmax - xmin)/xmesh:
             z:= data[i][j]:
             if domtype(z) <> DOM_FLOAT then
                error("expecting real numerical data. Got: ".expr2text(data[i][j]));
             end_if:
             // the lines
             out::writePoly3d(attributes, table("LineColor" = color),
               [[x, y, x, y, min(z, ground)], [x, y, x, y, max(z, ground)]]);
             // the points
             out::writePoints3d(attributes, table("PointsVisible"=TRUE, "PointColor" = color),
               [[x, y, x, y, z]]);
         end_for;
       end_for:
       break;
    otherwise
       error("unknown BarStyle = ".expr2text(style));
    end_case;
    //-----------------------
    // return the viewing box
    //-----------------------
    return([xmin .. xmax, ymin .. ymax, zmin .. zmax]);
end_proc:
//----------------------------------------------------------------------------
