/* -------------------------------------------------------------------
TODOs: 
  - What happened to Merge?
  - No type checking for NotchWidth
  - Hardly any type checking in general
  - What the heck is "TitleOrientation" doing in
      [DrawMode, ["Optional", Vertical],
           ["Style", "Prop", "TitleOrientation",
           "Orientation of bars/boxes", TRUE]],
    

  The function boxplot gives a plot summarizing the data.
  Box plots are quite useful to compare similar data sets. 
  For example, one can quickly compare the monthly mid-day temperature 
  over the period of one year by producing one boxplot per month, 
  and displaying them side by side.
   
   A boxplot contains a rectangle with the following properties: 
   - a lower bound showing the first quartile,  
   - an upper bound showing the third quartile;  
   - a central line showing the median,        
    - Two lines extending from the central box of maximal length 3/2 the
      interquartile range but not extending past the range of the data; 
   - Outliers, points that lie outside the extent of the previous elements. 

    Call: boxplot(data,<options>)

    data: List of Lists, Matrix or Array(one- or two-dimensional)

    Options: 
      DrawMode = Vertical/Horizontal (Default=Vertical)
      Notched  = TRUE/FALSE
                 Using the parameter format=notched will create a box plot 
                 with one additional feature. The sides of the box are indented, 
                 or notched, at the median line. This notch has a maximal width 
                 of 3.14 times the interquartile range divided by the square 
                 root of the total weight of the data. The notch is constrained 
                 inside the first and third quartiles.
                               
              -> Notched-box plots can be used to determine if two random samples 
                 were drawn from the same population. Overlapping notches
                 indicate that the data sets have the same distribution.
                 However, overlapping notches are no \emph{rigorous} criterion 
                 that the data sets are indeed identically distributed.

      LineWidth = DOM_INT (Default=1)
      Merge     = Merges two or three Plots into one diagram, while one list is plotted 
                  against the others in 2- or 3 dimensional diagrams.
      Colors    = DOM_LIST of Type Color (Default=RGB::ColorList)
      BoxCenters= A list of real values which specify 
                  the x-centers for each of the boxes. 
      BoxWidths  = A list of values which specify the widths for each of the boxes. 
*/

//--------------------------------------
//TODO: Stat::sample Column Indices
//--------------------------------------

plot::createPlotDomain("Boxplot",
                       "graphical primitive for boxplots",
                       2,  // Dimension
                       [LineStyle, LineWidth, LinesVisible, AntiAliased,
                        Filled, FillPattern,

                       /*

                       Mandatory: boxplot(data) 
                         data    = List of Lists, Matrix or Array(one- or two-dimensional)

                       Optional:
                        DrawMode = Vertical/Horizontal (Default=Vertical)
                        Notched  = TRUE/FALSE/Float (Default=false)
                        
                        Merge    = TRUE/FALSE (Only when providing more than one list)
                        Colors   = DOM_LIST of Type Color (Default=RGB::ColorList)
                        BoxWidths   = list of expressions. 
                        */
                                                 
                        Colors,  Color,

                        [Averaged, ["Optional", TRUE],
                             ["Definition", "Prop", "Bool",
                              "Use averaged quantiles", TRUE]],

                        [Data, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                          "Data to plot.", TRUE]],

                        [DrawMode, ["Optional", Vertical],
                             ["Style", "Prop", "TitleOrientation",
                             "Orientation of bars/boxes", TRUE]],

                        [Notched, ["Optional", FALSE],
                             ["Style", "Prop", "Bool",
                             "Draws notched indents", TRUE]],

                        [NotchWidth, ["Optional", 0.2],
                         ["Style", "Expr", FAIL,
                          "Width of notches", TRUE]],
                         
                        [BoxWidths, ["Optional", NIL],
                             ["Style", "ExprSeq", FAIL,
                             "List of widths for the boxes", TRUE]],
                        [BoxCenters, ["Optional", NIL],
                             ["Style", "ExprSeq", FAIL,
                             "List of centers for the boxes", TRUE]]
                         ]):

//----------------------------------------------------------------
plot::Boxplot::styleSheet := table(BoxWidths = [0.8],
                                   BoxCenters = [1],
                                   AntiAliased = FALSE,
                                   Averaged = TRUE,
                                   Colors = RGB::ColorList[1..10]):
plot::Boxplot::hints:= {Axes = Frame}:
//----------------------------------------------------------------
plot::Boxplot::new:=
  proc()
    local object, other, check;
  begin
    object := dom::checkArgs([], args());
    
    other := object::other;
    if nops(other) > 0 then
      if testtype(other[1], Type::ListOf(Type::Arithmetical)) then
        other := [other];
      end_if;
      check := dom::changeNotifier(object, "Data"=other[1]);
      if domtype(check) = DOM_STRING then
        error(check);
      end_if;
    elif object::Data <> FAIL then
      check := dom::changeNotifier(object, "Data"=object::Data);
      if domtype(check) = DOM_STRING then
        error(check);
      end_if;      
    end_if;
    
    dom::checkObject(object);
  end_proc:
//----------------------------------------------------------------
plot::Boxplot::print:= 
  proc(b)
  begin
    if _plus(op(map(b::Data, nops))) < 5 then
      hold(plot::Boxplot)(b::Data);
    else
      "plot::Boxplot(...)";
    end_if;
  end_proc:
//----------------------------------------------------------------
plot::Boxplot::changeNotifier :=
  proc(obj, eq)
    local slotName, newval, l, columnIndices, dim, i, j, numberOfColumns;
  begin
    [slotName, newval] := [op(eq)];
    case slotName
      of "Data" do
        l := newval;
        if not (testtype(l, Type::ListOf(DOM_LIST)) or
                domtype(l) = DOM_ARRAY or
                domtype(l) = stats::sample or
                domtype(l) = DOM_LIST  or
                l::dom::hasProp(Cat::Matrix) = TRUE) then
          return("data expected as a list, a list of lists, an array, ".
                 "a matrix, or a stats::sample");
        end_if;

        //--------------------------------------------
        // process boxplot(List1, options): 
        // convert to boxplot([List1], options)
        //--------------------------------------------
        if domtype(l) = DOM_LIST and
           domtype(expr(l[1]))<>DOM_LIST then 
          l:=[l];
        end_if;

        //--------------------------------------------
        // Matrix, Array --> List of Lists :
        //--------------------------------------------
        if l::dom::hasProp( Cat::Matrix ) = TRUE then
          l:= expr(l): // Matrix -> Array, which will be converted next
        end_if;
   
        //--------------------------------------------
        // process data array: convert to list of lists
        // Warning: transpose the array (we need a list
        // columns, not a list of rows)!
        //--------------------------------------------
        if domtype(l) = DOM_ARRAY then
          // Array -> List of Lists
          dim := op(l, [0,1]): // Dimension of Array
          if dim = 2 then
            l:=[ [l[j,i] $ j = op(l, [0,2,1])..op(l, [0,2,2])] 
                         $ i = op(l, [0,3,1])..op(l, [0,3,2])];
          elif dim = 1 then 
            // 1-dimensionales Array in eine ListOfList
            l:= [ [op(l)] ]:
          else
            return("Attribute \"Data\": the data array must be one or two-dimensional");
          end_if;
        end_if;
   
        // ---------------------------
        // process statistical samples
        // ---------------------------
        
        if domtype(l) = stats::sample then
          columnIndices:= []:
   
          // check if column indices are provided by the user:
          if nops(obj::other) > 1 and domtype(obj::other[2]) = DOM_LIST 
            then
            // the column indices are specified by a list
            columnIndices:= obj::other[2];
            numberOfColumns:= nops(columnIndices);         
          else
            i:= 1; // initialize: if args(0) = 1, the
                   // following loop will not be entered
            for i from 2 to nops(obj::other) do
              if domtype(obj::other[i]) = DOM_INT then
                columnIndices:= append(columnIndices, obj::other[i]);
              else
                break;
              end_if;
            end_for;
            numberOfColumns:= nops(columnIndices);
            
         end_if;
          if numberOfColumns = 0 then
            // No columns were specified by the
            // user. Extract *all* columns of l.
            // Check the number of elements in the
            // first row extop(l, 1)[1]:
            numberOfColumns:= nops(extop(l, 1)[1]);
            columnIndices:= [i $ i = 1 .. numberOfColumns];
          end_if;
          l:= [stats::getdata(
               TRUE,           // double check the data
               "all_data",     // accept all sorts of numerical data
                               // as well as strings
               numberOfColumns,// number of data columns to be read
               l,              // the sample
               columnIndices   // the indices of the columns to be read
              )];
          // a string returned by stats::getdata indicates an error:
          if domtype(l) = DOM_STRING then 
            return(l);
          end_if;
      
           // check for string columns in l:
          for i from 1 to nops(l) do
            if domtype(l[i][1]) = DOM_STRING then
              l[i]:= NIL;
            end_if;
          end_for:
          if has(l, NIL) then 
            l:= subs(l, NIL = null);
          end_if;
          if nops(l) = 0 then
            return("No numerical data found in attribute \"Data\"");
          end_if;
        end_if; // of domtype(l) = stats::sample
   
        //---------------------------------------------
        // ---> Now: the data l is a list of lists ----
        //---------------------------------------------
        l := float(l);
        
        // for the inspector:
        dom::extslot(obj, "Data", l);
        
        return(FALSE); // we've done all there is to do
        break;
    end_case;
    TRUE;
  end_proc:
//--------------------------------------------------------------------
plot::Boxplot::doPlotStatic :=
  proc(out, b, attributes, inherited)
    local l, swap, line, xmin, xmax, ymin, ymax, colors, color, points,
          widths, width, centers, center, range, x, x25, x50, x75,
          big, small, outRange, i, notch, notchIndent, notchIndentU,
          notchUp, notchDown, temp, c, w, averaged;
  begin
    xmin := RD_INF;
    xmax := RD_NINF;
    ymin := RD_INF;
    ymax := RD_NINF;
    
    if attributes[DrawMode] = Horizontal then
      swap := (x, y) -> (y, x);
    else
      swap := (x, y) -> (x, y);
    end_if;

    if attributes[Averaged] = TRUE then
       averaged:= Averaged;
    else
       averaged:= null();
    end_if:
    
    line := proc(x1, y1, x2, y2)
            begin
              out::writePoly2d(attributes, table("LineColor"=color), [[swap(x1,y1)],[swap(x2,y2)]]):
              xmin := min(xmin, x1, x2);
              xmax := max(xmax, x1, x2);
              ymin := min(ymin, y1, y2);
              ymax := max(ymax, y1, y2);
            end_proc:
    
    if attributes[Notched] = TRUE then
      notchIndent := attributes[NotchWidth];
    else
      notchIndent := 0;
    end_if;
    
    colors  := attributes[Colors];
    if contains(attributes, BoxCenters) then
         centers:= float(attributes[BoxCenters]);
    else centers:= []; // default values are inserted below
    end_if:
    if contains(attributes, BoxWidths) then
         widths:= float(attributes[BoxWidths]);
    else widths:= []; // default values are inserted below
    end_if;
    
    // these should be superfluous
    if map({op(centers)}, domtype) minus {DOM_FLOAT} <> {} then
      error("non-numerical or complex value in attribute \"BoxCenters\"");
    end_if;
    if map({op(widths)}, domtype) minus {DOM_FLOAT} <> {} then
      error("non-numerical or complex value in attribute \"BoxWidths\"");
    end_if;
    
    l := attributes[Data];
    if (_union(map({op(l[i])},domtype)$i=1..nops(l)) minus {DOM_FLOAT}) <> {} then 
       return("Some data elements are not numerical in attribute \"Data\"");
    end_if;  
    
    //---------------------------------
    // set default centers if necessary
    //---------------------------------
    c:= nops(centers):
    if c = 0 then
       centers := [i $ i = 1..nops(l)];
    elif c < nops(l) then 
       centers := centers . [centers[c] + i $ i = 1..nops(l)-c];
    end_if;
    //--------------------------------
    // set default widths if necessary
    //--------------------------------
    w:= nops(widths):
    if w = 0 then
      widths := [0.8 $ nops(l)];
    elif w < nops(l) then
      widths := widths . [0.8 $ i = 1..nops(l)-w];
    end_if;
    
    //--------------------------------
    // render the boxes
    //--------------------------------
    for i from 1 to nops(l) do
      
      x:=stats::empiricalQuantile(l[i]);
      /* -------------------
      Stephan Huckemann: March 2006
      For symmetry reasons use averaged quantiles, i.e.
      median of, say, [1,2,3,4,5,6,7,8,9,10,11,12] should be 13/2, the two
      quartiles should be 7/2 and 19/2
      ------------------- */
      x25:=x(0.25, averaged);
      x50:=x(0.50, averaged);
      x75:=x(0.75, averaged);
      outRange:=[];
      range:=1.5*(x75-x25);
      // Select Points which lie outside of the range
      outRange:=select(l[i], y -> (y > x75 + range) or (y < x25 - range)); 
      
      // Select Points which lie outside of our box, but within the range
      temp  := select(l[i],(y-> (y > x25-range) and (y < x25)));
      small := min(op(temp), x25);

      temp:= select(l[i],(y -> (y > x75) and (y < x75+ range )));
      big := max(op(temp), x75);
      
      if attributes[Notched]=TRUE then
         notch:=float(PI*((range/1.5)/(nops(l[i]))^(1/2)));
      else
         notch := 0;
      end_if;
      
      notchUp:= x50+(notch/2);
      notchDown:=x50-(notch/2);
       
      if notchUp>x75 then notchUp:=x75 end_if; 
      if notchDown<x25 then notchDown:=x25 end_if; 
      
      center := centers[i];
      width  := widths[i]/2;
      color  := colors[1+modp(i-1, nops(colors))];
      notchIndentU := 2*notchIndent*width;
      
      /// plot a box
      // bar for lower bound of range
      line(center-width/2, small,
           center+width/2, small);
      // bar for upper bound of range
      line(center-width/2, big,
           center+width/2, big);
      // lines from bottom & top to the box
      line(center, small,
           center, x25);
      line(center, x75,
           center, big);
      // main box
      points := [[center+width, x25], [center+width, notchDown], [center+width-notchIndentU, x50], [center+width, notchUp],
        [center+width, x75], [center-width, x75], [center-width, notchUp], [center-width+notchIndentU, x50],
        [center-width, notchDown], [center-width, x25]];
      xmin := min(xmin, op(map(points, op, 1)));
      ymin := min(ymin, op(map(points, op, 2)));
      xmax := max(xmax, op(map(points, op, 1)));
      ymax := max(ymax, op(map(points, op, 2)));
      points := map(points, X->[swap(op(X))]);
      out::writePoly2d(attributes, table("Closed"=TRUE,
                                "FillColor"=color,
                                "LineColor"=color ), points):

      // and a line across
      line(center-width+notchIndentU, x50,
           center+width-notchIndentU, x50);
      
      // Outliers
      if nops(outRange)>0 then
        points := [];
        for temp in outRange do
          points := points.[[swap(center, temp)]];
        end_for;
        out::writePoly2d(attributes, table("Closed" = FALSE,
                                  "LinesVisible" = FALSE,
                                  "PointsVisible" = TRUE,
                                  "LineColor" = color,
                                  "Filled" = FALSE), points):
        ymin := min(ymin, op(outRange));
        ymax := max(ymax, op(outRange));
      end_if;
    end_for;
    
    if xmax = RD_NINF then // nothing plotted?
      return(null());
    else
      return([swap(xmin..xmax, ymin..ymax)]);
    end_if;
  end_proc:
