/* -----------------------------------------------------------
plot::Function2d -- The graphical primitive for 2D function plots

Call(s): Function2d(f(x), 
                   <x = XMin .. XMax>,
                   <a = AMin .. AMax>,
                   <Color = c>,
                   <Mesh = n>,
                   <Submesh = m>,
                   <AdaptiveMesh = l>
                   <DiscontinuitySearch = TRUE/FALSE>
                   <opt1, opt2, ...>)
Parameters:
    f(x)      : the function: arithmetical expression (depending on x)
                and, possibly, on an animation parameter
    x         : the variable of the function: DOM_IDENT or indexed identifier
    XMin, XMax: numerical real values or univariate expressions
                of the animation  parameter
    a         : the animation parameter: DOM_IDENT or indexed identifier
    AMin, AMax: numerical real values
    c         : an RGB value, an RGBa value or a procedure
    n         : the number of mesh points: an integer > 0
    m         : the number of 'inner mesh points': an integer >= 0
    l         : the number of recursive refinements in the
                adaptive mode: an integer >= 0

Options:
    opt1, opt2, ... : general 2d plot options

Examples:
    >> plot::Function2d(sin(t), t = 0..2*PI )
    >> plot::Function2d(sin(t), t = 0..2*PI, Color = RGB::Blue )
----------------------------------------------------------------------*/
plot::createPlotDomain("Function2d",
                  "graphical primitive for 2D function graphs"):

plot::Function2d::styleSheet := table(LegendEntry = TRUE,
                                      AdaptiveMesh = 2,
                                      XMesh=121, 
                                      XMin=-5, 
                                      XMax=5):
plot::Function2d::hints := {XAxisTitle, YAxisTitle}:
plot::Function2d::setHint:= proc(object)
  begin
    if object::XName <> FAIL and
       object::XAxisTitle = FAIL then
       plot::setHint(object, XAxisTitle = expr2text(object::XName)):
    end_if:
    if object::YName <> FAIL and
       object::YAxisTitle = FAIL then
       plot::setHint(object, YAxisTitle = expr2text(object::YName)):
    end_if;
end_proc:
//------------------------------------------------
// Functionality of the new method:
// a) Check the input parameters (generic attributes via
//    the utility dom::checkArgs and special arguments for
//    this object via special code in the 'new' method.
// b) If appropriate, add a suitable legend text via 
//    slot assignments if no legend text is provided by 
//    the user.
// c) Call dom::checkObject
//------------------------------------------------
plot::Function2d::new:= proc()
local object, other, xx;
begin
    // 1st step:   call dom::checkArgs to filter out all generic arguments 
    //             that are allowed for arbitrary primitives/objects. Let
    //             'object' be the return value of dom::checkArgs
    // object: a domain element of plot::Function2d
    // (see model.mu). The special input parameters for this
    // object are in object::other.
    object := dom::checkArgs(["X"], args());

    // process the special attributes in object::other. In the
    // case of Function2d, this is just the list [f(x)]
    other := object::other;

    if nops(other) > 1 then
      error("unexpected argument: ".expr2text(other[2]));
    end_if;

    // Convert the input parameters special to this plot object into
    // the generic form "AttributeName = AttributeValue" as definied in
    // the DTD. In this case, the form must be Function = f(x), where
    // 'Function' is a (protected) attribute name.

    if nops(other) > 0 then
      object::Function:= other[1];
    end_if;

    //---------------------------------------
    // Arrgrr: request by the school fraction.
    //---------------------------------------
    if object::XName = FAIL then
       xx:= numeric::indets(object::Function);
       if nops(xx) = 0 then
           object::XName:= hold(x);
       elif nops(xx) = 1 then
           object::XName:= op(xx);
       else error("cannot figure out the name of the independent variable");
       end_if;
    end_if:
    //---------------------------------------
    // end of Arrgrr
    //---------------------------------------

    dom::setHint(object):
    dom::checkObject(object);
end_proc:

//--------------------------------------------
plot::Function2d::print :=
  obj -> hold(plot::Function2d)(obj::Function, obj::XName=obj::XRange):

//--------------------------------------------------------------------
//--------------------------------------------------------------------
// The method is called from the parent objects.
// Here, the computation of the data is invoked.
//--------------------------------------------------------------------
plot::Function2d::doPlotStatic:= proc(out, object, attributes, inheritedAttributes,
                                  flag=FALSE)
// object:     the object that was created by the new method above.
//             Usually, we do not need it here.
// attributes: a table of all attributes (inherited attributes as well as
//             those created from the mandatory parameters in the new method).
//             Zusaetzlich sind enthalten:
//             attributes[TimeValue] (float)
//             attributes[ParameterValue] (float)
//             However, hints are not included in 'attributes'.
// inheritedAttributes: 
//             the method will be called with 3 arguments.
//             However, you do no need to process the 3rd
//             argument 'inheritedAttributes': just ignore it.
// flag:       Boolean flag which is TRUE when called from Hatch
local mesh, submesh, adaptivemesh, discontinuitysearch,
      f, x, xmin, xmax, ymin, ymax, colorfunction,
      branch, branches, i, viewingbox, singularities,
      _min, _max, _range, clipFactor, clipBranch;
begin
    mesh:=         attributes[XMesh];
    submesh:=      attributes[XSubmesh];
    adaptivemesh:= attributes[AdaptiveMesh];
    discontinuitysearch:=
           attributes[DiscontinuitySearch];
    f   := attributes[Function]; // an expression
    x   := attributes[XName]:    
    xmin:= attributes[XMin]:
    xmax:= attributes[XMax]:
    if contains(attributes, ViewingBoxXMin) and
      attributes[ViewingBoxXMin] <> Automatic then
      if xmin = Automatic then
         xmin:= attributes[ViewingBoxXMin]:
      else
         xmin:= max(xmin, attributes[ViewingBoxXMin]):
      end_if;
    end_if;
    if contains(attributes, ViewingBoxXMax) and
      attributes[ViewingBoxXMax] <> Automatic then
      if xmax = Automatic then
         xmax:= attributes[ViewingBoxXMax]: 
      else
         xmax:= min(xmax, attributes[ViewingBoxXMax]): 
      end_if;
    end_if;
    if contains(attributes, ViewingBoxYMin) then
         ymin:= attributes[ViewingBoxYMin]: 
    else ymin:= Automatic;  
    end_if;
    if contains(attributes, ViewingBoxYMax) then
         ymax:= attributes[ViewingBoxYMax]: 
    else ymax:= Automatic;
    end_if;
    if contains(attributes, LineColorFunction) then 
      colorfunction := attributes[LineColorFunction];
    else colorfunction := () -> null();
    end_if;
    //-------------------------------------------------------------
    // Now, let the library compute the graphical data:
    //-------------------------------------------------------------
    // Set the parameters:
    //-------------------------------------------------------------
    [xmin, xmax, ymin, ymax]:= float([xmin, xmax, ymin, ymax]);
    mesh:= 1 + (mesh - 1)*(submesh + 1);
    if mesh <= 1 then
       warning("expecting a number of mesh points >= 2, got: ".
               expr2text(mesh).
               ". Setting the number of mesh points to 2.");
       mesh:= 2:
    end_if:
    //-------------------------------------------------------------
    // Logarithmic modifications:
    // The adaptive mechanism as well as the automatic viewing
    // box in the utitility routine plot::functionEval assumes
    // a cartesian frame. If other coordinate types are specified,
    // we modify the definition of the function such that it
    // corresponds to the (cartesian) graphical output. Later,
    // we undo the modifications. After undoing, the differences
    // between the cartesian and the logarihmic case are:
    // * The numerical mesh points are not placed equidistantly,
    //   but logarithmically
    // * The automatic clipping (automatic viewing box) is
    //   somewhat different
    //-------------------------------------------------------------
    if attributes[CoordinateType] in {LogLin, LogLog} then
       if (domtype(xmin) = DOM_FLOAT and xmin <= 0) then
          error("expecting a positive X range for a logarithmic X axis. ".
                "Got Xmin = ".expr2text(xmin));
       end_if:
       if (domtype(xmax) = DOM_FLOAT and xmax <= 0) then
          error("expecting a positive X range for a logarithmic X axis. ".
                "Got Xmax = ".expr2text(xmax));
       end_if:
       xmin:= ln(xmin);
       xmax:= ln(xmax);
       f:= f@exp: 
    end_if;
    if attributes[CoordinateType] in {LinLog, LogLog} then
       if (domtype(ymin) = DOM_FLOAT and ymin <= 0) then
          error("expecting a positive Y range for a logarithmic Y axis. ".
                "Got Ymin = ".expr2text(ymin));
       end_if:
       if (domtype(ymax) = DOM_FLOAT and ymax <= 0) then
          error("expecting a positive Y range for a logarithmic Y axis. ".
                "Got Ymax = ".expr2text(ymax));
       end_if:
       if ymin <> Automatic then
          ymin:= ln(ymin);
       end_if;
       if ymax <> Automatic then
          ymax:= ln(ymax);
       end_if;
       f:= ln@f: 
    end_if;
    //-------------------------------------------------------------
    // call the plot::functionEval utility
    //-------------------------------------------------------------
    [viewingbox, branches, singularities]:=
        plot::functionEval(f, x = xmin .. xmax, ymin..ymax, mesh,
                           AdaptiveMesh = adaptivemesh,
                           DiscontinuitySearch = discontinuitysearch):
    //-------------------------------------------------------------
    // We need to undo the logarithmic modifications
    // before writing the data to the XML file. 
    //-------------------------------------------------------------
    if attributes[CoordinateType] in {LogLin, LogLog} then
       // x -> exp(x)
       if viewingbox <> NIL then
          viewingbox:= [map(viewingbox[1], exp), viewingbox[2]];
       end_if;
       branches:= map(branches, map, pt -> [exp(pt[1]), pt[2]]);
       singularities:= map(singularities, exp);
    end_if;
    if attributes[CoordinateType] in {LinLog, LogLog} then
       // y -> exp(y)
       if viewingbox <> NIL then
          viewingbox:= [viewingbox[1], map(viewingbox[2], exp)];
       end_if;
       branches:= map(branches, map, pt -> [pt[1], exp(pt[2])]);
    end_if;
    // ------------------------------------------------------------
    // Special: clip out points that are really far away.
    // This seems necessary for Windoof, where plotting 
    // raises an error if there are points way out of the
    // viewing area (even if they are clipped).
    // ------------------------------------------------------------
    if viewingbox <> NIL then
      clipFactor:= 1000:
      // Do not just eliminate the points outside the
      // viewing box, because the next point might be
      // well inside the viewing box and the connetcing
      // line seqment would be missing
      //   clipBranch:= y -> (_min < y[2] and y[2] < _max);
      // Instead, project the point back to the boundary
      // of the viewing box
      clipBranch:= proc(y)
                   begin
                     if y[2] < _min then y[2]:= _min; end_if;
                     if y[2] > _max then y[2]:= _max; end_if;
                     y;
                   end_proc:
      _min:= op(viewingbox, [2, 1]);
      _max:= op(viewingbox, [2, 2]);
      _range:= _max - _min;
      if iszero(_range) then  
        _range:= _plus(op(viewingbox, [i, 2]) - op(viewingbox, [i, 1]) $ i=1..2);
         if iszero(_range) then
            _range:= 1;
         end_if;
      end_if;
      _min:= _min - clipFactor*_range;
      _max:= _max + clipFactor*_range;
      // branches:= map(branches, select, clipBranch);
      branches:= map(branches, map, clipBranch);
    end_if:
    //-----------------------------------------
    // When Hatch calls:
    //-----------------------------------------
    if flag = TRUE then // The method was called by the Hatch object.
       // The singularities returned by plot::functionEval is a set
       // {x1, x2, ...} of x-values of the singularities of the function.
       return([viewingbox, branches, singularities])
    end_if;
    //-----------------------------------------
    // write the graphical data to the XML file
    //-----------------------------------------
    // We have to create the low-level primitives
    // For a Function2d, it is one or more Poly2d (a polygon line).
    for branch in branches do
      // branch = [ [x1, y1], [x2, y2], ...]:
      // Each branch of the function plot is to be a Poly2d polygon

      // Begin of Poly2d definition that represents a branch of the
      // function graph.
      if nops(branch) > 1 then
        out::writePoly2d(attributes, table("Filled"=FALSE, "Closed"=FALSE), branch, colorfunction):
      else // the branch has only 1 point. Make sure it is visible
        out::writePoly2d(attributes, table("Filled"=FALSE, "Closed"=FALSE, "PointsVisible" = TRUE), branch, colorfunction):
      end_if;
    end_for: // for branch in branches
    //-------------------------------------------------------------
    // *Always* write the positions of the singularities into the
    // XML file. The renderer reacts to VerticalAymptotesVisible.
    //-------------------------------------------------------------
    for x in singularities do
      out::writeVerticalAsymptote(attributes, table(), x);
    end_for:
    //-------------------------------------------------------------
    // Finally, return the viewing box
    //-------------------------------------------------------------
    if viewingbox = NIL then
         return();
    end_if;
    return(viewingbox);
  end_proc:
