/* ----------------------------------------------------------------------------
plot::Hexahedron -- graphical primitive of a hexahedron

Call(s):
    plot::Hexahedron( <args, ...> )

Parameters:
    -

Options:
    attr, ...   : plot options, i.e., equations Attribute = value, where
                  Attribute is one of the attributes listed below or a hint.

Related Domains:
    plot::Tetrahedron, plot::Octahedron, plot::Icosahedron, plot::Dodecahedron,
    plot::Box, plot::Cone, plot::Cylinder, plot::Ellipsoid, plot::Sphere, plot::Rotate3d,
    plot::Scale3d, plot::Translate3d

Example:
    >> plot( plot::Hexahedron() );

Further Reading:
    Refer to Tetrahedron.mu

---------------------------------------------------------------------------- */

//-----------------------------------------------------------------------------
plot::createPlotDomain("Hexahedron",
                       "graphical primitive for hexahedrons",
                       3,
   [
    CenterX, CenterY, CenterZ, Center, Radius,
    LinesVisible, LineStyle, LineWidth,
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::Hexahedron::styleSheet:= table(
      LinesVisible  = TRUE,
      LineColor     = RGB::Black.[0.25],
      Radius        = 1,
      CenterX       = 0,
      CenterY       = 0,
      CenterZ       = 0,
      LineColorDirectionY=0):

//-----------------------------------------------------------------------------
plot::Hexahedron::hints:= {
      Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::Hexahedron::new:= proc()
local  object, other;
option escape;
begin
    // filter out all generic arguments
    object:= dom::checkArgs([],args());

    // process the special attributes in object::other
    other:= object::other;
    if nops(other) > 0 then
      error("unexpected argument: ".expr2text(other[1]));
    end_if;

    // check all attributes
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::Hexahedron::print:= obj -> "plot::Hexahedron(...)":

//-----------------------------------------------------------------------------
plot::Hexahedron::doPlotStatic:= proc(out, object, attributes, inheritedAttributes)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local
    cx, cy, cz, r,
    quads,
    fillcolorfunction,
    linecolorfunction, haslinecolorfunction,
    tn, i,
    entries;

begin
    // check for fill-color function
    if contains(attributes, FillColorFunction) then
      fillcolorfunction:= float@(attributes[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;

    cx:= float(attributes[CenterX]);
    cy:= float(attributes[CenterY]);
    cz:= float(attributes[CenterZ]);
    r := float(attributes[Radius]);

    quads:= [
      [cx-r,cy-r,cz+r], [cx+r,cy-r,cz+r], [cx+r,cy+r,cz+r], [cx-r,cy+r,cz+r],
      [cx-r,cy-r,cz-r], [cx+r,cy-r,cz-r], [cx+r,cy-r,cz+r], [cx-r,cy-r,cz+r],
      [cx+r,cy-r,cz-r], [cx+r,cy+r,cz-r], [cx+r,cy+r,cz+r], [cx+r,cy-r,cz+r],
      [cx-r,cy+r,cz-r], [cx+r,cy+r,cz-r], [cx+r,cy+r,cz+r], [cx-r,cy+r,cz+r],
      [cx-r,cy-r,cz-r], [cx-r,cy+r,cz-r], [cx-r,cy+r,cz+r], [cx-r,cy-r,cz+r],
      [cx-r,cy-r,cz-r], [cx+r,cy-r,cz-r], [cx+r,cy+r,cz-r], [cx-r,cy+r,cz-r]
    ];

    quads := zip([([ (i-1 div 4)+1])$i=1..nops(quads)], quads, _concat);
    out::writeQuads(attributes, table("MeshVisible"   = FALSE,
                              "PointsVisible" = FALSE,
                              "HasLineColors" = FALSE), quads, ()->null(), fillcolorfunction, 2..4 );
    // start to write triangle mesh to XML stream
    // write quad lines to XML stream
    for tn from 4 to nops(quads) step 4 do
      out::writePoly3d(attributes, table("Filled" = FALSE, "Closed" = TRUE),
        [op(quads,(tn-3)..(tn))], linecolorfunction, 2..4);
    end_for;


    // finally, return the viewing box
    entries:= (i) -> map(quads,op,i);
    return( [min(entries(2))..max(entries(2)),
             min(entries(3))..max(entries(3)),
             min(entries(4))..max(entries(4))] );
  end_proc:
//------------------------------------------------------------------------