// 

// implicit3d, based on 
// Luiz Velho, Luiz Henrique de Figueriredo, and Jonas Gomes
// A Unified Approach for Hierarchical
// Adaptive Tesselation of Surfaces
//                      
// ACM Transaction on Graphics, Vol. 18, No. 4, Oct. 1999, pp.329-360                   

// and
// Yutaka Ohtake and Alexander G. Belyaev
// Dual/Primal Mesh Optimization
// for Polygonized Implicit Surfaces
//
// AM SM'02, June 17-21 2002


plot::createPlotDomain("Implicit3d",
                       "graphical primitive for 3D implicit functions"):
//-----------------------------------------------------------------------------------
plot::Implicit3d::styleSheet := table(Contours = [0],
                                      LegendEntry = TRUE,
                                      MeshVisible = FALSE,
                                      LineColor = RGB::Black.[0.15],
                                      XMesh = 11,
                                      YMesh = 11,
                                      ZMesh = 11,
                                      XContours = [Automatic, 15],
                                      YContours = [Automatic, 15],
                                      ZContours = [Automatic, 15],
                                      LineColorDirectionY=0):
//-----------------------------------------------------------------------------------
plot::Implicit3d::new :=
  proc()
    local object, fn;
  begin
    object := dom::checkArgs(["X", "Y", "Z"], args());
    
    fn := object::other;
    
    if nops(fn) > 1 then
      error("unexpected argument ".expr2text(fn[2]));
    end_if;
    
    if nops(fn) = 1 then
      if type(fn[1])="_equal" then
        object::Function := subsop(fn[1],0=hold(_subtract));
      else
        object::Function := fn[1];
      end_if;
    end_if;
    
    // semantically check for validity
    dom::checkObject(object);
  end_proc:
//-----------------------------------------------------------------------------------
plot::Implicit3d::print :=
  obj -> hold(plot::Implicit3d)(obj::Function,
                                obj::XName=obj::XRange,
                                obj::YName=obj::YRange,
                                obj::ZName=obj::ZRange):
//-----------------------------------------------------------------------------------
plot::Implicit3d::doPlotStatic :=
  proc(out, object, attributes, inheritedAttributes)
    local t, data, component, points, normals, triangles, f, df,
          linecolorfunction, fillcolorfunction, contour, c, key,
            tmpPoints;
  begin
    if float(attributes[XMin]) > float(attributes[XMax]) or
       float(attributes[YMin]) > float(attributes[YMax]) or
       float(attributes[ZMin]) > float(attributes[ZMax]) then
      return(null());
    end_if;
    
    if contains(attributes, LineColorFunction) then
      linecolorfunction := float@(attributes[LineColorFunction]);
    else
      linecolorfunction := null();
    end_if;
    if contains(attributes, FillColorFunction) then
      fillcolorfunction := float@(attributes[FillColorFunction]);
    else
      fillcolorfunction := null();
    end_if;
    
    if contains(attributes, ParameterValue) then
      key := attributes[ParameterValue];
    else
      key := 0;
    end_if;
    
    data := dom::_calc_mesh(object, attributes);
    
    for contour in data do
      c := op(contour, 1);
      for component in op(contour, 2) do
        [points, triangles, normals, f, df] := component;
        tmpPoints := [];
        for t in triangles do
          tmpPoints := tmpPoints.[[[linecolorfunction(op(points[t[1]]),
                                                 op(normals[t[1]]), c)],
                               [fillcolorfunction(op(points[t[1]]),
                                                 op(normals[t[1]]), c)], op(points[t[1]]), op(normals[t[1]])],
            [[linecolorfunction(op(points[t[1]]),
                                                 op(normals[t[1]]), c)],
                               [fillcolorfunction(op(points[t[1]]),
                                                 op(normals[t[1]]), c)], op(points[t[2]]), op(normals[t[2]])],
            [[linecolorfunction(op(points[t[1]]),
                                                 op(normals[t[1]]), c)],
                               [fillcolorfunction(op(points[t[1]]),
                                                 op(normals[t[1]]), c)], op(points[t[3]]), op(normals[t[3]])]];
        end_for;
        out::writeTriangles(attributes, table("HasLineColors" = bool(attributes[LineColorType] = Functional)),
          tmpPoints, A->op(A), (A,B)->op(B));
	if nops(triangles) > 0 then
	  triangles := _concat(op(map(triangles,
				      t -> [points[t[1]].normals[t[1]],
					    points[t[2]].normals[t[2]],
					    points[t[3]].normals[t[3]]])));
	  out::writeMeshContours(attributes, table(), triangles, 1..3, 1,
				  attributes[XContours],
				  p -> linecolorfunction(op(p), c),
				  attributes[XMin], attributes[XMax]);
	  out::writeMeshContours(attributes, table(), triangles, 1..3, 2,
				  attributes[YContours],
				  p -> linecolorfunction(op(p), c),
				  attributes[YMin], attributes[YMax]);
	  out::writeMeshContours(attributes, table(), triangles, 1..3, 3,
				  attributes[ZContours],
				  p -> linecolorfunction(op(p), c),
				  attributes[ZMin], attributes[ZMax]);
	end_if;
      end_for;
    end_for;
    
    [attributes[XMin]..attributes[XMax],
     attributes[YMin]..attributes[YMax],
     attributes[ZMin]..attributes[ZMax]];
  end_proc:

plot::Implicit3d::_calc_mesh :=
  proc(imp: plot::Implicit3d, atts: DOM_TABLE)
    local f, points, triangles, fn, result,
          initial_mesh, normals, df, xiv, yiv, ziv, minlength,
          results, c;
  begin
    fn  := atts[Function];
    xiv := atts[XMin] ... atts[XMax];
    yiv := atts[YMin] ... atts[YMax];
    ziv := atts[ZMin] ... atts[ZMax];
    
    // minimum length of a triangle side
    minlength := 1e-1*min(op(xiv, 2)-op(xiv, 1),
                          op(yiv, 2)-op(yiv, 1),
                          op(ziv, 2)-op(ziv, 1));

    initial_mesh   := [atts[XMesh], atts[YMesh], atts[ZMesh]];
    
    results := table();
    if not contains(atts, Contours) then
      atts[Contours] := [0];
    end_if;
    for c in atts[Contours] do
      result := [];
      f := fn-c;
      if traperror((df := map([1,2,3], x->D([x], f)))) <> 0 or
         has(df, [FAIL, hold(diff), hold(D)]) or
         has(map(df, op, 4), [FAIL, hold(diff), hold(D)]) then
           // could not get symbolic derivative
           // this has to be a list of three almost identical functions,
           // because it will be called as df[1](x,y,z) etc. below.
        df := subs([proc(x,y,z)
                      local fxyz, h;
                    begin
                      fxyz := `#f`(x,y,z);
                      if iszero(fxyz) then
                        h := (frandom()-1/2)/1e20;
                      else
                        h := fxyz;
                      end_if;
                      (`#f`(x+h, y, z)-fxyz)/h;
                    end_proc,
                    proc(x,y,z)
                      local fxyz, h;
                    begin
                      fxyz := `#f`(x,y,z);
                      if iszero(fxyz) then
                        h := (frandom()-1/2)/1e20;
                      else
                        h := fxyz;
                      end_if;
                      (`#f`(x, y+h, z)-fxyz)/h;
                    end_proc,
                    proc(x,y,z)
                      local fxyz, h;
                    begin
                      fxyz := `#f`(x,y,z);
                      if iszero(fxyz) then
                        h := (frandom()-1/2)/1e20;
                      else
                        h := fxyz;
                      end_if;
                      (`#f`(x, y, z+h)-fxyz)/h;
                    end_proc], `#f`=f);
      end_if;
      
      userinfo(2, "Calculating initial mesh");
        // guess an initial mesh.  Low resolution is acceptable.
      [points, normals, triangles] := dom::_guess_impl3d(f, df, xiv, yiv, ziv,
                                                         initial_mesh);
      
      if atts[AdaptiveMesh] > 0 then
//      adaptive := atts[AdaptiveMesh];
//      while adaptive > 3 do
//        atts[AdaptiveMesh] := 3;
//        [points, triangles] :=
//        dom::_adaptive_subdiv(f, df, xiv, yiv, ziv,
//                              points, triangles, atts);
//        atts[AdaptiveMesh] := 1;
//        [points, triangles] :=
//        dom::_ohtake_curvature_resampling(f, df, xiv, yiv, ziv,
//                                          points, triangles, atts);
//        adaptive := adaptive - 3;
//      end_while;
//      atts[AdaptiveMesh] := adaptive;
        [points, triangles] :=
        dom::_adaptive_subdiv(f, df, xiv, yiv, ziv,
                              points, triangles, atts);
        
        normals := dom::_normals(f, df, points,
                                 op(xiv,2)-op(xiv,1),
                                 op(yiv,2)-op(yiv,1),
                                 op(ziv,2)-op(ziv,1));
      end_if;
      result := result.[[points, triangles, normals, f, df]];
      results[c] := result;
    end_for;
    
    results;
  end_proc:

// plot::Implicit3d::optimize :=
//   proc(imp1, opttypes)
//     local points, triangles, normals, f, df,
//           xiv, yiv, ziv, options, data, i, opttype;
//   begin
//     error("Implementation not updated yet!");
//     
//     if args(0) < 2  then
//       error("Expecting a list of optimization types.");
//     end_if;
//     
//     if testtype(opttypes, DOM_IDENT) then
//       opttypes := [opttypes];
//     end_if;
//     
//     if imp1::meshdata = FAIL then
//       dom::_calc_mesh(new(dom, imp1));
//     end_if;
//     
//     data := imp1::meshdata;
//     [xiv, yiv, ziv] := imp1::ranges;
//     options := imp1::options;
//     
//     for i from 1 to nops(data) do
//       [points, triangles, normals, f, df] := data[i];
//       
//       for opttype in opttypes do
//         case opttype
//           of hold(Springs) do
//             [points, triangles] := 
//             dom::_spring_mass_improvement(f, df, 
//                                           xiv, yiv, ziv,
//                                           points, triangles, 
//                                           options);
//             break;
//           of hold(Resample) do
//             [points, triangles] :=
//             dom::_adaptive_subdiv(f, df, 
//                                   xiv, yiv, ziv,
//                                   points, triangles, 
//                                   options);
//             break;
//           of hold(DoubleDual) do
//             [points, triangles] := 
//             dom::_ohtake_curvature_resampling(f, df,
//                                               xiv, yiv, ziv,
//                                               points, triangles, 
//                                               options);
//             break;
//           otherwise
//             warning("Unknown optimization type ".opttype);
//         end_case;
//       end_for;
//
//       normals := map(points, p->df(op(p)));
//       data[i] := [points, triangles, normals, f, df];
//     end_for;
//     imp1::meshdata := data;
//     null();
//   end_proc:
  

plot::Implicit3d::_adaptive_subdiv :=
  proc(f, df, xiv, yiv, ziv, points, triangles, options)
    local edges, find_point, edges_this_tr, i, revert_edge,
          add_edge, add_triangle, orient_edges, subdivide,
          simplify_subdiv, split_triangle, subdiv_depth,
          minbend, minlength, minlength2;
  begin
    subdiv_depth   := options[AdaptiveMesh];
    minbend        := tan::float(10/180*PI/2)^2/2;

    minlength      := 1e-2*min(op(xiv, 2) - op(xiv, 1),
                               op(yiv, 2) - op(yiv, 1),
                               op(ziv, 2) - op(ziv, 1));
    minlength2     := minlength^2;
    
    userinfo(3, "Got ".expr2text(nops(points)).
             " points and ".expr2text(nops(triangles))." triangles");
    edges := { op(map(triangles, t->(sort([t[1], t[2]]),
                                     sort([t[2], t[3]]),
                                     sort([t[3], t[1]])))) };
    edges := [op(edges)];
                 
    userinfo(3, "Found ".expr2text(nops(edges))." edges");
    // renumber triangles: point to edges instead of vertices
    triangles := map(triangles, t->[contains(edges, sort([t[1],t[2]])),
                                    contains(edges, sort([t[2],t[3]])),
                                    contains(edges, sort([t[3],t[1]]))]);
    
    // which edges need subdivision?
    subdivide := proc(l, r, levels)
                   local m1, m2, distlr2, bend, lt, rt;
                 begin
                   if levels <= 0 then
                     return(FAIL);
                   end_if;
                   
                   m1 := [(l[1]+r[1])/2,
                          (l[2]+r[2])/2,
                          (l[3]+r[3])/2];
                   if (distlr2 := 
                       (l[1]-r[1])^2+(l[2]-r[2])^2+(l[3]-r[3])^2) < minlength2
                     then
                     return(FAIL);
                   end_if;
                   
                   m2 := dom::_newton3d(f, df, op(m1), xiv, yiv, ziv)[1..3];
                   
                   if m2 = l or m2 = r then
                     return(FAIL);
                   end_if;                         
                   
                   bend := _plus(op(zip(m1, m2, (a,b)->(a-b)^2)))/distlr2;
                   
                   /*
                   if bend < minbend then
                     return(FAIL);
                   end_if;
                   */
                   
                   lt := subdivide(l, m2, levels-1);
                   rt := subdivide(m2, r, levels-1);
                   
                   if lt <> FAIL then bend := max(bend, lt[2]); end;
                   if rt <> FAIL then bend := max(bend, rt[2]); end;
                   
                   return([m2, bend, levels-1, distlr2, lt, rt])
                 end_proc:
    
    simplify_subdiv := proc(tree)
                       begin
                         if tree=FAIL then
                           return(FAIL);
                         end_if;
                         if tree[2] < minbend then
                           return(FAIL);
                         end_if;
                         if op(tree, 5) <> FAIL then
                           if op(tree, [5,2]) < minbend then
                             tree[5] := FAIL;
                           else
                             tree[5] := simplify_subdiv(tree[5]);
                           end_if;
                         end_if;
                         if op(tree, 6) <> FAIL then
                           if op(tree, [6,2]) < minbend then
                             tree[6] := FAIL;
                           else
                             tree[6] := simplify_subdiv(tree[6]);
                           end_if;
                         end_if;
                         tree;
                       end_proc:
    
    edges := map(edges, e->[points[e[1]], points[e[2]],
                            simplify_subdiv(subdivide(points[e[1]],
                                                      points[e[2]], 
                                                      subdiv_depth))]);
    
    points := []; // will be rebuilt later
    userinfo(10, "Subdivision done.  length(edges)=".
             expr2text(length(edges)));
    
    //////////////////////////////////////////////////
    // simplical subdivision
    //
    // Now, triangles are represented by entries of
    // 'triangles', which contain three indices
    // into 'edges', with incident points.
    // edges are split, constructing new triangles.
    // Unfortunately, this means we append new lists
    // of edges and triangles...
    edges := table((i=edges[i])$i=1..nops(edges));
    triangles := table((i=triangles[i])$i=1..nops(triangles));
    
    //---- utility ---------------------------------
    // Revert an edge. This includes reverting all
    // subtrees. This routine only gets the tree.
    //----------------------------------------------
    revert_edge := proc(e)
                   begin
                     if e = FAIL then return(FAIL); end_if;
                     [e[5], e[6]] := [revert_edge(e[6]), revert_edge(e[5])];
                     e;
                   end_proc:
  
    //---- utility ---------------------------------
    // Add a new edge to the table "edges". Returns the new index.
    //----------------------------------------------
    add_edge := proc(e)
                begin
                  edges[nops(edges)+1] := e;
                  return(nops(edges));
                end_proc:
    
    //---- utility ---------------------------------
    // Add a new triangle to the table "triangles"
    //----------------------------------------------
    add_triangle := proc(e1, e2, e3)
                    begin
                      assert(nops({edges[e1][1],
                                   edges[e1][2],
                                   edges[e2][1],
                                   edges[e2][2],
                                   edges[e3][1],
                                   edges[e3][2]})=3);
                      triangles[nops(triangles)+1] := [e1, e2, e3];
                    end_proc:
    
    //---- utility ---------------------------------
    // orient the edges of a triangle to form a path
    //----------------------------------------------
    orient_edges := proc(e1, e2, e3)
                    begin
                      if e1[2] <> e2[1] and
                         e1[2] <> e2[2] then
                        [e1[1], e1[2]] := [e1[2], e1[1]];
                        e1[3] := revert_edge(e1[3]);
                      end_if;
  
                      if e2[2] <> e3[1] and
                         e2[2] <> e3[2] then
                        [e2[1], e2[2]] := [e2[2], e2[1]];
                        e2[3] := revert_edge(e2[3]);
                      end_if;
  
                      if e3[2] <> e1[1] and
                         e3[2] <> e1[2] then
                        [e3[1], e3[2]] := [e3[2], e3[1]];
                        e3[3] := revert_edge(e3[3]);
                      end_if;
  
                      [e1, e2, e3];
                    end_proc:
   
    
    
    //---- utility ---------------------------------
    // Consider the i-th triangle.
    //     * split it into 2, 3, or 4 triangles
    //     * remove it by triangles[i]:= FAIL
    //     * add the new smaller triangles
    //----------------------------------------------
    split_triangle := proc(i, edges_to_split)
                        local edge_indices, edge_values, e, split_this,
                              corner, split_at, j;
                        name plot::Implicit3d::split_triangle;
                      begin
                        userinfo(20, "Splitting triangle ".expr2text(i));
                        // including new edges, we might need to consider
                        // twelve edges.  The list is made long enough
                        // right away:
                        edge_indices := triangles[i].[FAIL$9];
                        triangles[i] := FAIL;
  
                        edge_values := [edges[edge_indices[1]],
                                        edges[edge_indices[2]],
                                        edges[edge_indices[3]],
                                        FAIL$9];
  
                        [edge_values[1],
                         edge_values[2],
                         edge_values[3]] := orient_edges(edge_values[1],
                                                         edge_values[2],
                                                         edge_values[3]);
  
                        case nops(edges_to_split)
                          of 1 do
                            e := edges_to_split[1];
                            split_this := edge_values[e];
                            assert(op(split_this, [3]) <> FAIL); // subtree must exist!
                            // the edge is not deleted from edges, because there is
                            // another triangle using it.
                            edge_indices[e] := FAIL;
                            corner := edge_values[(modp(e, 3))+1][2];
                            split_at := op(split_this, [3,1]);
                            edge_values[4] := [split_this[1], split_at,
                                               op(split_this, [3,5])];
                            edge_indices[4]:= add_edge(edge_values[4]);
                            
                            edge_values[5] := [split_at, split_this[2],
                                               op(split_this, [3,6])];
                            edge_indices[5]:= add_edge(edge_values[5]);
                            
                            edge_values[6] := [corner, split_at,
                                               simplify_subdiv
                                               (subdivide(corner, split_at,
                                                          op(split_this, [3,3])))];
                            edge_indices[6] := add_edge(edge_values[6]);
  
                            add_triangle(edge_indices[modp(e+1, 3)+1],
                                         edge_indices[4],
                                         edge_indices[6]);
                            add_triangle(edge_indices[modp(e, 3)+1],
                                         edge_indices[6],
                                         edge_indices[5]);
                            break;
                          of 2 do
                            e := op({1,2,3} minus edges_to_split);
                            
                            // selection strategy: split longest edge first
                            if edge_values[modp(e, 3)+1][3][4] >
                               edge_values[modp(e+1, 3)+1][3][4] then
                              corner := edge_values[e][1];
                              split_this := edge_values[modp(e, 3)+1];
                              assert(op(split_this, [3]) <> FAIL); // subtree must exist!
                              split_at := op(split_this, [3,1]);
                              edge_values[4]  := [split_this[1], split_at,
                                                  op(split_this, [3,5])];
                              edge_indices[4] := add_edge(edge_values[4]);
  
                              edge_values[5]  := [split_at, split_this[2],
                                                  op(split_this, [3,6])];
                              edge_indices[5] := add_edge(edge_values[5]);
  
                              edge_values[6]   := [corner, split_at,
                                                   simplify_subdiv
                                                   (subdivide(corner, split_at,
                                                              op(split_this, [3,3])))];
                              edge_indices[6]   := add_edge(edge_values[6]);
  
                              // split the second side
                              corner := edge_values[4][2];
                              split_this := edge_values[modp(e+1, 3)+1];
                              assert(op(split_this, [3]) <> FAIL); // subtree must exist!
                              split_at := op(split_this, [3,1]);
                              edge_values[7]  := [split_this[1], split_at,
                                                  op(split_this, [3,5])];
                              edge_indices[7] := add_edge(edge_values[7]);
                              
                              edge_values[8]  := [split_at, split_this[2],
                                                  op(split_this, [3,6])];
                              edge_indices[8] := add_edge(edge_values[8]);
                              
                              edge_values[9]  := [corner, split_at,
                                                  simplify_subdiv
                                                  (subdivide(corner, split_at,
                                                             op(split_this, [3,3])))];
                              edge_indices[9] := add_edge(edge_values[9]);
                              
                              add_triangle(edge_indices[e],
                                           edge_indices[4],
                                           edge_indices[6]);
                              add_triangle(edge_indices[5],
                                           edge_indices[7],
                                           edge_indices[9]);
                              add_triangle(edge_indices[9],
                                           edge_indices[8],
                                           edge_indices[6]);
                            else
                              // same code as above, just the indices are different.
                              corner := edge_values[e][2];
                              split_this := edge_values[modp(e+1, 3)+1];
                              assert(op(split_this, [3]) <> FAIL); // subtree must exist!
                              split_at := op(split_this, [3,1]);
                              edge_values[4]  := [split_this[1], split_at,
                                                  op(split_this, [3,5])];
                              edge_indices[4] := add_edge(edge_values[4]);
                              
                              edge_values[5]  := [split_at, split_this[2],
                                                  op(split_this, [3,6])];
                              edge_indices[5] := add_edge(edge_values[5]);
                              
                              edge_values[6]  := [corner, split_at,
                                                  simplify_subdiv
                                                  (subdivide(corner, split_at,
                                                             op(split_this, [3,3])))];
                              edge_indices[6]  := add_edge(edge_values[6]);
  
                              // split the second side
                              corner := edge_values[modp(e+1, 3)+1][3][1];
                              split_this := edge_values[modp(e, 3)+1];
                              assert(op(split_this, [3]) <> FAIL); // subtree must exist!
                              split_at := op(split_this, [3,1]);
                              edge_values[7]  := [split_this[1], split_at,
                                                  op(split_this, [3,5])];
                              edge_indices[7] := add_edge(edge_values[7]);
                              
                              edge_values[8]  := [split_at, split_this[2],
                                                  op(split_this, [3,6])];
                              edge_indices[8] := add_edge(edge_values[8]);
                              
                              edge_values[9]  := [corner, split_at,
                                                  simplify_subdiv
                                                  (subdivide(corner, split_at,
                                                             op(split_this, [3,3])))];
                              edge_indices[9] := add_edge(edge_values[9]);
  
                              add_triangle(edge_indices[e],
                                           edge_indices[6],
                                           edge_indices[5]);
                              add_triangle(edge_indices[6],
                                           edge_indices[7],
                                           edge_indices[9]);
                              add_triangle(edge_indices[9],
                                           edge_indices[8],
                                           edge_indices[4]);
                            end_if;
                            break;
  
                          of 3 do
                            for j from 1 to 3 do
                              split_at := op(edge_values[j], [3,1]);
                              edge_values[3+2*j-1] := [op(edge_values[j], 1),
                                                       split_at,
                                                       op(edge_values[j], [3,5])];
                              edge_indices[3+2*j-1]:= add_edge(edge_values[3+2*j-1]);
                              
                              edge_values[3+2*j]:= [split_at, op(edge_values[j], 2),
                                                    op(edge_values[j], [3,6])];
                              edge_indices[3+2*j]:= add_edge(edge_values[3+2*j]);
                              
                              corner := op(edge_values[modp(j, 3)+1], [3,1]);
                              edge_values[9+j] := [split_at, corner,
                                                   simplify_subdiv
                                                   (subdivide(split_at, corner,
                                                              op(edge_values[j],
                                                                 [3,3])))];
                              edge_indices[9+j] := add_edge(edge_values[9+j]);
                            end_for;
  
  // //////////////////////////////////////////////////////////////////////
  // // debugging
  //                             assert(edge_values[4][2] = edge_values[5][1]);
  //                             assert(edge_values[4][2] = edge_values[10][1]);
  //                             assert(edge_values[4][2] = edge_values[12][2]);
  //                             assert(edge_values[5][2] = edge_values[6][1]);
  //                             assert(edge_values[6][2] = edge_values[7][1]);
  //                             assert(edge_values[6][2] = edge_values[11][1]);
  //                             assert(edge_values[6][2] = edge_values[10][2]);
  //                             assert(edge_values[7][2] = edge_values[8][1]);
  //                             assert(edge_values[8][2] = edge_values[9][1]);
  //                             assert(edge_values[8][2] = edge_values[12][1]);
  //                             assert(edge_values[8][2] = edge_values[11][2]);
  //                             assert(edge_values[9][2] = edge_values[4][1]);
  //                             assert(edge_values[4][2][1]
  //                                    in edge_values[4][1][1] ... edge_values[5][2][1]);
  //                             assert(edge_values[4][2][2]
  //                                    in edge_values[4][1][2] ... edge_values[5][2][2]);
  //                             assert(edge_values[6][2][1]
  //                                    in edge_values[6][1][1] ... edge_values[7][2][1]);
  //                             assert(edge_values[6][2][2]
  //                                    in edge_values[6][1][2] ... edge_values[7][2][2]);
  //                             assert(edge_values[8][2][1]
  //                                    in edge_values[8][1][1] ... edge_values[9][2][1]);
  //                             assert(edge_values[8][2][2]
  //                                    in edge_values[8][1][2] ... edge_values[9][2][2]);
  // // debugging
  // //////////////////////////////////////////////////////////////////////
                            
                            add_triangle(edge_indices[4],
                                         edge_indices[12],
                                         edge_indices[9]);
                            add_triangle(edge_indices[5],
                                         edge_indices[6],
                                         edge_indices[10]);
                            add_triangle(edge_indices[7],
                                         edge_indices[8],
                                         edge_indices[11]);
                            add_triangle(edge_indices[10],
                                         edge_indices[11],
                                         edge_indices[12]);
                            break;
                          otherwise
                            error("Now that's weird!");
                        end_case;
                        assert(testtype(triangles,
                                        Type::TableOfEntry(Type::Union
                                                           (Type::ListOf(DOM_INT, 3, 3),
                                                            DOM_FAIL))));
                      end_proc:
    
    // Now actually *do* the subdivision
    i := 1;
    while i<nops(triangles) do
      if triangles[i] = FAIL then
        i := i+1;
        next;
      end_if;
      
      // classify triangles[i] by looking at its edges
      edges_this_tr := zip([edges$3], triangles[i], _index);
      
      assert({op(map(edges_this_tr, domtype))} = {DOM_LIST});
      assert({op(map(edges_this_tr, e->domtype(e[3])))} 
             minus {DOM_LIST, DOM_FAIL} = {});
      
      case map(edges_this_tr, domtype@op, 3)
      // ----------------------------------------------
      // triangles that do not need subdivision
      // ----------------------------------------------
        of [DOM_FAIL, DOM_FAIL, DOM_FAIL] do
              // probably flat
              // TODO: Heuristic check for flatness, see p.343 of Velho
          break;
      // ----------------------------------------------
      // triangles with one side that needs subdivision
      // ----------------------------------------------
        of [DOM_LIST, DOM_FAIL, DOM_FAIL] do split_triangle(i, {1}); break;
        of [DOM_FAIL, DOM_LIST, DOM_FAIL] do split_triangle(i, {2}); break;
        of [DOM_FAIL, DOM_FAIL, DOM_LIST] do split_triangle(i, {3}); break;
      // ----------------------------------------------
      // triangle with two sides that need subdivision
      // ----------------------------------------------
        of [DOM_LIST, DOM_LIST, DOM_FAIL] do split_triangle(i, {1, 2}); break;
        of [DOM_LIST, DOM_FAIL, DOM_LIST] do split_triangle(i, {1, 3}); break;
        of [DOM_FAIL, DOM_LIST, DOM_LIST] do split_triangle(i, {2, 3}); break;
      // ----------------------------------------------
      // triangle with three sides that need subdivision
      // ----------------------------------------------
        of [DOM_LIST, DOM_LIST, DOM_LIST] do split_triangle(i, {1, 2, 3}); break;
      // ----------------------------------------------
        otherwise:
          warning("unknown case -- triangle left alone");
          break;
      end_case;
      
      i := i+1;
    end_while;

        //////////////////////////////////////////////////
        // done.  Transform the data structures.
    triangles := select(map([op(triangles)],
                            rhs),
                        `<>`, FAIL);
    
    points := table();
    find_point := proc(p) local i;
                  begin
                    if domtype((i:=points[p])) = DOM_INT then
                      return(i);
                    else points[p] := nops(points)+1;
                      return(points[p]);
                    end_if;
                  end_proc;
    triangles := map(triangles,
                     proc(t)
                       name edges_to_points;
                     begin
                       [find_point(op({edges[t[3]][1],
                                       edges[t[3]][2]}
                                      intersect
                                      {edges[t[1]][1],
                                       edges[t[1]][2]})),
                        find_point(op({edges[t[1]][1],
                                       edges[t[1]][2]}
                                      intersect
                                      {edges[t[2]][1],
                                       edges[t[2]][2]})),
                        find_point(op({edges[t[2]][1],
                                       edges[t[2]][2]}
                                      intersect
                                      {edges[t[3]][1],
                                       edges[t[3]][2]}))];
                     end_proc);
    points := revert(points);
    points := [points[i]$i=1..nops(points)];
    [points, triangles];
  end_proc:

//////////////////////////////////////////////////
// smoothing an existing mesh.  
// This method implements a spring-mass system,
// as suggested in
// Angela Rsch, Matthias Ruhl and Dietmar Saupe
// Interactive Visualization of Implicit Surfaces
// with Singularities
// Eurographics 1997
plot::Implicit3d::_spring_mass_improvement :=
  proc(f, df, xiv, yiv, ziv, points, triangles, options)
    local neighbors, boundary_points, i, j, t, velocities,
          p, n, k, normals, ts, maxvel, normOfNormal;
  begin
    maxvel := 0.2*min(op(xiv,2)-op(xiv,1),
                      op(yiv,2)-op(yiv,1),
                      op(ziv,2)-op(ziv,1));
    
    neighbors := [{} $ nops(points)];
    for t in triangles do
      ts := {op(t)};
      map(t, i->(neighbors[i] := neighbors[i] union ts));
    end_for;
    
    // TODO: More general border point detection!
    // e.g., try to find triangles "all around"    
    boundary_points := [select([$1..nops(points)],
                               i->has(points[i],
                                      [op(xiv)])),
                        select([$1..nops(points)],
                               i->has(points[i],
                                      [op(yiv)])),
                        select([$1..nops(points)],
                               i->has(points[i],
                                      [op(ziv)]))];
    
    velocities:=[[1,1,1]$nops(points)];
    p := 1;
    
    i := 1;
  normals := dom::_normals(f, df, points,
                           op(xiv,2)-op(xiv,1),
                           op(yiv,2)-op(yiv,1),
                           op(ziv,2)-op(ziv,1));
    while p > 1e-3 and i < 4 do
      
  proc() name innen2; 
  begin
      for p from 1 to nops(points) do
        velocities[p] := [0$3];
        for n in neighbors[p] do
          k := 0.2*(1-_plus((normals[p][j]*normals[n][j])$j=1..3));
          assert(specfunc::abs(k)<1);
          for j from 1 to 3 do
            velocities[p][j] := velocities[p][j] 
                               + k*(points[n][j]-points[p][j]);
          end_for;
        end_for;
      end_for;
      map(boundary_points[1], t->(velocities[t][1] := 0));
      map(boundary_points[2], t->(velocities[t][2] := 0));
      map(boundary_points[3], t->(velocities[t][3] := 0));
    
    p := max(op(map(velocities, max@op)));
    userinfo(10, "max velocity: ".expr2text(p));
    if p>maxvel then
      velocities := map(velocities,
                        v->eval(map(v, _divide, p/maxvel)));
    end_if;
  end_proc():
      
  proc() name innen3; 
    begin
      for p from 1 to nops(points) do
        points[p] := dom::_newton3d(f, df, 
                                    points[p][1]+velocities[p][1],
                                    points[p][2]+velocities[p][2],
                                    points[p][3]+velocities[p][3],
                                    xiv, yiv, ziv);
        normals[p] := points[p][5];
        points[p] := points[p][1..3];
        normOfNormal := specfunc::sqrt(normals[p][1]^2
                                       +normals[p][2]^2
                                       +normals[p][3]^2);
        if not iszero(normOfNormal) then
          normals[p] := [normals[p][1]/normOfNormal,
                         normals[p][2]/normOfNormal,
                         normals[p][3]/normOfNormal];
        end_if;
      end_for;
      i := i+1;
    end_proc():
      p := max(op(map(velocities, max@op)));
    end_while;

    [points, triangles];
  end_proc:

//////////////////////////////////////////////////
// another type of refinement: Dual/Primal Mesh optimization
// this consists of three different methods, implemented as
// different functions

////////////////////
// first method: Curvature-weighted vertex resampling
plot::Implicit3d::_ohtake_curvature_resampling :=
  proc(f, df, xiv, yiv, ziv, points, triangles, options)
    local i, j, dual_points, newpoints, t, pass, boundary_points;
  begin
    // TODO: More general border point detection!
    // e.g., try to find triangles "all around"
    boundary_points := select([$1..nops(points)],
                              i->has(points[i],
                                     [op(xiv), op(yiv), op(ziv)]));
    for pass from 1 to 3 do
      // center points of triangles
      dual_points := map(triangles,
                         t->zip(zip(points[t[1]], points[t[2]], `+`),
                                points[t[3]], (a,b)->(a+b)/3));
      
      dual_points := map(dual_points,
                         p->dom::_newton3d(f, df, op(p), xiv, yiv, ziv)[1..3]);
      
      userinfo(10, "optimized dual mesh");
      
      // averaging over retrofitted centroids
      newpoints := [{}$nops(points)];
      for i from 1 to nops(triangles) do
        t := triangles[i];
        for j from 1 to 3 do
          newpoints[t[j]] := newpoints[t[j]] union {dual_points[i]};
        end_for;
      end_for;

      // arithmetical mean, no wheights for now.
      newpoints := map(newpoints,
                       s -> [(_plus(op(map(s, op, i)))/nops(s))$i=1..3]);
      for t in boundary_points do
        newpoints[t] := points[t];
      end_for;
      points := newpoints;
      userinfo(10, "optimized primal mesh");
    end_for;
    [points, triangles];
  end_proc:

//////////////////////////////////////////////////////////////////////
// first version of the initial solution: as described in
// Ulises Cervantes-Pimental
// Adaptive Polygonization of Implicit Surfaces
plot::Implicit3d::_guess_impl3d :=
  proc (ff, df, xiv, yiv, ziv, meshsize=[3,3,3])
    local values, xx, yy, zz, points, triangles, coords, index_of,
          signs, cell, add_triangle, indices, t, triangle_action,
          do_func, normals, signs_here, n2, n3;
  begin
    coords := (xx, yy, zz) -> [op(xiv, 1)+(float(xx)-1)/(meshsize[1]-1)*(op(xiv, 2)-op(xiv, 1)),
                               op(yiv, 1)+(float(yy)-1)/(meshsize[2]-1)*(op(yiv, 2)-op(yiv, 1)),
                               op(ziv, 1)+(float(zz)-1)/(meshsize[3]-1)*(op(ziv, 2)-op(ziv, 1))]:
    
    // evaluate ff at an equidistant mesh
    if traperror((values:=[[[eval(ff(op(coords(xx, yy, zz))))
                             $zz=1..meshsize[3]]
                            $yy=1..meshsize[2]]
                           $xx=1..meshsize[1]])) <> 0 then
      values:=[[[if traperror((t := eval(ff(op(coords(xx, yy, zz)))))) = 0 then
                   t
                 else
                   undefined
                 end_if
                 $zz=1..meshsize[3]]
                $yy=1..meshsize[2]]
               $xx=1..meshsize[1]];
    end_if;
    
    signs := map(values, map, map, specfunc::sign);
    
    points := [];
    triangles := {};
    // insert into list of points if not there yet, return index
    index_of := proc(el)
                  local ind;
                begin
                  ind := contains(points, el);
                  if ind > 0 then
                    ind;
                  else
                    points := points . [el];
                    nops(points);
                  end_if;
                end_proc:

    add_triangle := proc(e1, e2, e3)
                      local inds;
                    begin
                      inds := map([e1, e2, e3], index_of@sort);
                      triangles := triangles union {inds};
                    end_proc:
    
    triangle_action := 
    table([1, -1, -1, -1] = 
          (() -> add_triangle([indices[cell[1]], indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [1, 0, -1, -1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [-1, 1, -1, -1] =
          (() -> add_triangle([indices[cell[2]], indices[cell[1]]],
                              [indices[cell[2]], indices[cell[3]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [0, 1, -1, -1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]], indices[cell[3]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [1, 1, -1, -1] =
          (() -> (add_triangle([indices[cell[1]], indices[cell[4]]],
                               [indices[cell[1]], indices[cell[3]]],
                               [indices[cell[2]], indices[cell[4]]]);
                  add_triangle([indices[cell[2]], indices[cell[4]]],
                               [indices[cell[2]], indices[cell[3]]],
                               [indices[cell[1]], indices[cell[3]]]);
                 )),
          [1, -1, 0, -1] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [0, 0, 0, -1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]]],
                              [indices[cell[3]]])),
          [1, 0, 0, -1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [-1, 1, 0, -1] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [0, 1, 0, -1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[3]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [1, 1, 0, -1] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [-1, -1, 1, -1] =
          (() -> add_triangle([indices[cell[3]], indices[cell[1]]],
                              [indices[cell[3]], indices[cell[2]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [0, -1, 1, -1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]], indices[cell[3]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [1, -1, 1, -1] =
          (() -> (add_triangle([indices[cell[1]], indices[cell[2]]],
                               [indices[cell[1]], indices[cell[4]]],
                               [indices[cell[2]], indices[cell[3]]]);
                  add_triangle([indices[cell[2]], indices[cell[3]]],
                               [indices[cell[4]], indices[cell[3]]],
                               [indices[cell[1]], indices[cell[4]]]);
                 )),
          [-1, 0, 1, -1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [0, 0, 1, -1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [1, 0, 1, -1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[4]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [-1, 1, 1, -1] =
          (() -> (add_triangle([indices[cell[1]], indices[cell[2]]],
                               [indices[cell[1]], indices[cell[3]]],
                               [indices[cell[2]], indices[cell[4]]]);
                  add_triangle([indices[cell[4]], indices[cell[2]]],
                               [indices[cell[4]], indices[cell[3]]],
                               [indices[cell[1]], indices[cell[3]]]);
                 )),
          [0, 1, 1, -1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]], indices[cell[4]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [1, 1, 1, -1] =
          (() -> add_triangle([indices[cell[4]], indices[cell[2]]],
                              [indices[cell[4]], indices[cell[3]]],
                              [indices[cell[4]], indices[cell[1]]])),
          [1, -1, -1, 0] =
          (() -> add_triangle([indices[cell[4]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]])),
          [0, 0, -1, 0] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]]],
                              [indices[cell[4]]])),
          [1, 0, -1, 0] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[4]]],
                              [indices[cell[1]], indices[cell[3]]])),
          [-1, 1, -1, 0] =
          (() -> add_triangle([indices[cell[4]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[2]], indices[cell[3]]])),
          [0, 1, -1, 0] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[4]]],
                              [indices[cell[2]], indices[cell[3]]])),
          [1, 1, -1, 0] =
          (() -> add_triangle([indices[cell[4]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[2]], indices[cell[3]]])),
          [0, -1, 0, 0] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[3]]],
                              [indices[cell[4]]])),
          [1, -1, 0, 0] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[4]]],
                              [indices[cell[1]], indices[cell[2]]])),
          [-1, 0, 0, 0] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[3]]],
                              [indices[cell[4]]])),
          [0, 0, 0, 0] =
          (() -> (add_triangle([indices[cell[1]]],
                               [indices[cell[2]]],
                               [indices[cell[3]]]);
                  add_triangle([indices[cell[1]]],
                               [indices[cell[2]]],
                               [indices[cell[4]]]);
                  add_triangle([indices[cell[1]]],
                               [indices[cell[3]]],
                               [indices[cell[4]]]);
                  add_triangle([indices[cell[2]]],
                               [indices[cell[3]]],
                               [indices[cell[4]]]);
                 )),
          [1, 0, 0, 0] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[3]]],
                              [indices[cell[4]]])),
          [-1, 1, 0, 0] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[4]]],
                              [indices[cell[1]], indices[cell[2]]])),
          [0, 1, 0, 0] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[3]]],
                              [indices[cell[4]]])),
          [-1, -1, 1, 0] =
          (() -> add_triangle([indices[cell[4]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[2]], indices[cell[3]]])),
          [0, -1, 1, 0] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[4]]],
                              [indices[cell[2]], indices[cell[3]]])),
          [1, -1, 1, 0] =
          (() -> add_triangle([indices[cell[4]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[2]], indices[cell[3]]])),
          [-1, 0, 1, 0] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[4]]])),
          [0, 0, 1, 0] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]]],
                              [indices[cell[4]]])),
          [-1, 1, 1, 0] =
          (() -> add_triangle([indices[cell[1]], indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[4]]])),
          [-1, -1, -1, 1] =
          (() -> add_triangle([indices[cell[4]], indices[cell[1]]],
                              [indices[cell[4]], indices[cell[2]]],
                              [indices[cell[4]], indices[cell[3]]])),
          [0, -1, -1, 1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]], indices[cell[4]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [1, -1, -1, 1] =
          (() -> (add_triangle([indices[cell[1]], indices[cell[2]]],
                               [indices[cell[1]], indices[cell[3]]],
                               [indices[cell[2]], indices[cell[4]]]);
                  add_triangle([indices[cell[4]], indices[cell[2]]],
                               [indices[cell[4]], indices[cell[3]]],
                               [indices[cell[1]], indices[cell[3]]]);
                 )),
          [-1, 0, -1, 1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[4]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [0, 0, -1, 1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [1, 0, -1, 1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [-1, 1, -1, 1] =
          (() -> (add_triangle([indices[cell[1]], indices[cell[2]]],
                               [indices[cell[1]], indices[cell[4]]],
                               [indices[cell[2]], indices[cell[3]]]);
                  add_triangle([indices[cell[3]], indices[cell[2]]],
                               [indices[cell[3]], indices[cell[4]]],
                               [indices[cell[1]], indices[cell[4]]]);
                 )),
          [0, 1, -1, 1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]], indices[cell[3]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [1, 1, -1, 1] =
          (() -> add_triangle([indices[cell[3]], indices[cell[2]]],
                              [indices[cell[3]], indices[cell[1]]],
                              [indices[cell[3]], indices[cell[4]]])),
          [-1, -1, 0, 1] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [0, -1, 0, 1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[3]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [1, -1, 0, 1] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [-1, 0, 0, 1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [0, 0, 0, 1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]]],
                              [indices[cell[3]]])),
          [-1, 1, 0, 1] =
          (() -> add_triangle([indices[cell[3]]],
                              [indices[cell[1]], indices[cell[2]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [-1, -1, 1, 1] =
          (() -> (add_triangle([indices[cell[1]], indices[cell[3]]],
                               [indices[cell[1]], indices[cell[4]]],
                               [indices[cell[2]], indices[cell[3]]]);
                  add_triangle([indices[cell[2]], indices[cell[3]]],
                               [indices[cell[2]], indices[cell[4]]],
                               [indices[cell[1]], indices[cell[4]]]);
                 )),
          [0, -1, 1, 1] =
          (() -> add_triangle([indices[cell[1]]],
                              [indices[cell[2]], indices[cell[3]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [1, -1, 1, 1] =
          (() -> add_triangle([indices[cell[2]], indices[cell[1]]],
                              [indices[cell[2]], indices[cell[3]]],
                              [indices[cell[2]], indices[cell[4]]])),
          [-1, 0, 1, 1] =
          (() -> add_triangle([indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]])),
          [-1, 1, 1, 1] =
          (() -> add_triangle([indices[cell[1]], indices[cell[2]]],
                              [indices[cell[1]], indices[cell[3]]],
                              [indices[cell[1]], indices[cell[4]]]))
         );
    
    
    // START OF EXPENSIVE PART
    // iterate over all cubes
    for xx from 1 to meshsize[1]-1 do
      for yy from 1 to meshsize[2]-1 do
        for zz from 1 to meshsize[3]-1 do
          indices := [[xx, yy, zz],
                      [xx, yy, zz+1],
                      [xx, yy+1, zz],
                      [xx, yy+1, zz+1],
                      [xx+1, yy, zz],
                      [xx+1, yy, zz+1],
                      [xx+1, yy+1, zz],
                      [xx+1, yy+1, zz+1]];
          signs_here := map(indices, i -> op(signs, i));
          if {op(signs_here)} in {{1}, {-1}} then
            next;
          end_if;
          
          
           // arbitrary choice of Coxeter-Freudenthal-decomposition -- parameterizable?
          for cell in [[1,5,6,8],
                       [1,5,7,8],
                       [1,2,6,8],
                       [1,2,4,8],
                       [1,3,7,8],
                       [1,3,4,8]] do
            do_func := triangle_action[[signs_here[cell[1]],
                                        signs_here[cell[2]],
                                        signs_here[cell[3]],
                                        signs_here[cell[4]]]];
            if testtype(do_func, DOM_PROC) then
              do_func();
            end_if;
          end_for;
        end_for;
      end_for;
    end_for; // iteration over all cubes
    // END OF EXPENSIVE PART
    
    // now, find the points more precisely
    points := map(points,
                  proc(p)
                    local p1, p2, v1, v2, t, p3;
                  begin
                    if nops(p) = 1 then
                      coords(op(p[1]));
                    else
                      p1 := p[1];
                      p2 := p[2];
                      v1 := op(values, p1);
                      v2 := op(values, p2);
                      assert(sign(v1) <> sign(v2));
                      p1 := coords(op(p1));
                      p2 := coords(op(p2));
                      
//                      t := -v2/(v1-v2);
                      t := numeric::realroot(ff(`#t`*p1[1] + (1-`#t`)*p2[1],
                                                `#t`*p1[2] + (1-`#t`)*p2[2],
                                                `#t`*p1[3] + (1-`#t`)*p2[3]),
                                             `#t` = 0..1);
                      if t = FAIL then
                        t := numeric::realroot(ff(`#t`*p1[1] + (1-`#t`)*p2[1],
                                                  `#t`*p1[2] + (1-`#t`)*p2[2],
                                                  `#t`*p1[3] + (1-`#t`)*p2[3]),
                                               `#t` = -10^(1-DIGITS)..1+10^(1-DIGITS));
                        if t = FAIL then
                          error("roundoff problems");
                        end_if;
                      end_if;
                      p3 := zip(p1, p2,
                                (a,b)->t*a+(1-t)*b);
//                      p3 := dom::_newton3d(ff, df, op(p3), xiv, yiv, ziv);
                    end_if;
                  end_proc);
//    
//    normals := map(points, op, 5);
//    points := map(points, _index, 1..3);
    
    normals := dom::_normals(ff, df, points,
                             op(xiv,2)-op(xiv,1),
                             op(yiv,2)-op(yiv,1),
                             op(ziv,2)-op(ziv,1));
    
    triangles := [op(triangles)];
    // weed out degenerate triangles
    triangles := select(triangles, t->nops({op(map(t, p->points[p]))})=3);
    
    // fix orientation
    triangles := map(triangles,
                     proc(t)
                       local p1, p2, p3, n1;
                     begin
                       p1 := points[t[1]];
                       p2 := points[t[2]];
                       p3 := points[t[3]];
                       n1 := normals[t[1]];
                       n2 := normals[t[2]];
                       n3 := normals[t[3]];
                       /*
                         if the angle between the inner product of
                         (p1-p2) and (p1-p3) and the normal of point p1
                         is larger than PI/2, the triangle must be re-oriented
                       */
                       if (p1[1]-p2[1])*(p1[2]-p3[2])*n1[3] +
                          (p1[2]-p2[2])*(p1[3]-p3[3])*n1[1] +
                          (p1[3]-p2[3])*(p1[1]-p3[1])*n1[2] -
                          (p1[1]-p2[1])*(p1[3]-p3[3])*n1[2] -
                          (p1[2]-p2[2])*(p1[1]-p3[1])*n1[3] -
                          (p1[3]-p2[3])*(p1[2]-p3[2])*n1[1] < 0 then
                         [t[1], t[2]] := [t[2], t[1]];
                       end_if;
                       t;
                     end_proc);
    
    [points, normals, triangles];
  end_proc:

////////////////////////////////////////////////////////////
// utility: find normal vectors
plot::Implicit3d::_normals :=
  proc(f, df, points, wx, wy, wz)
    local normals, t, r, v;
  begin
    r := frandom(176512);
    if traperror((normals := map(points, df@op))) > 0 then
      normals := map(points, p -> if traperror((v:=df(op(p)))) > 0 
                                   then [0.0$3]
                                   else v;
                                  end_if);
    end_if;
//    while (t := contains(normals, FAIL)) > 0 do
//      normals[t] := df(op(points[t]));
//    end_while;
    normals := float(normals);
    // make sure no zero vectors are in here
    while (t := contains(normals, [0.0, 0.0, 0.0])) > 0 do
      while normals[t] = [0.0$3] do
        points[t] := [points[t][1] + wx * 1e-4 * r(),
                      points[t][2] + wy * 1e-4 * r(),
                      points[t][3] + wz * 1e-4 * r()];
        normals[t] := if traperror((v := float(df(op(points[t]))))) > 0
                        then [0.0$3];
                        else v;
                      end_if;
      end_while;
    end_while;
    // normalize the vectors
    map(normals, proc(n)
                   local nn;
                 begin
                   nn := specfunc::sqrt(n[1]^2+n[2]^2+n[3]^2);
                   [n[1]/nn, n[2]/nn, n[3]/nn];
                 end_proc);
  end_proc:

////////////////////////////////////////////////////////////
// utility: newton iteration for f:R_^3 -> R_.

alias(perturb(x, h) =
      ((userinfo(20,"Perturbing...");
        while traperror((x := x + h/(100*(stdlib::frandom()-0.5))))=260
          do end_while;))):

plot::Implicit3d::_newton3d:=
  proc (f, df, x, y, z, xiv, yiv, ziv, nmax=10, eps=10.0^(5-DIGITS))
    local n, dfxyz, fxyz, h, dfsquare, hf,
          xmin, xmax, ymin, ymax, zmin, zmax;
  begin
    n := 0;
    h := 1e-5;
    [xmin, xmax] := [op(xiv)];
    [ymin, ymax] := [op(yiv)];
    [zmin, zmax] := [op(ziv)];
    if traperror
       ((
         fxyz := eval(f(x,y,z));
         dfxyz := eval([df[1](x,y,z),
                        df[2](x,y,z),
                        df[3](x,y,z)]);
         dfsquare := dfxyz[1]^2+dfxyz[2]^2+dfxyz[3]^2;
         while n<nmax and specfunc::abs(fxyz) > eps*dfsquare do
           if specfunc::abs(dfsquare/fxyz)<1e-15 then
          // rang(df) not maximal. perturb.
             perturb(x, h);
             perturb(y, h);
             perturb(z, h);
           else
             hf := fxyz/dfsquare;
             [x, y, z] := [x-hf*dfxyz[1], y-hf*dfxyz[2], z-hf*dfxyz[3]]:
           end_if;
           if x<xmin then x := xmin; end_if;
           if x>xmax then x := xmax; end_if;
           if y<ymin then y := ymin; end_if;
           if y>ymax then y := ymax; end_if;
           if z<zmin then z := zmin; end_if;
           if z>zmax then z := zmax; end_if;
           fxyz := eval(f(x,y,z));
           dfxyz := eval([df[1](x,y,z),
                          df[2](x,y,z),
                          df[3](x,y,z)]);
           dfsquare := dfxyz[1]^2+dfxyz[2]^2+dfxyz[3]^2;
      n := n+1;
    end_while;)) <> 0 then
      userinfo(10,
               "Got an error in newton iteration, trying with traperror()");
      return(dom::_newton3d_traperror(args()));
    end_if;
    userinfo(10..12, "Newton took ".expr2text(n)." steps");
                  
    [x,y,z, fxyz, dfxyz];
  end_proc:

plot::Implicit3d::_newton3d_traperror:=
  proc (f, df, x, y, z, xiv, yiv, ziv, nmax=10, eps=10.0^(5-DIGITS))
    local n, dfxyz, fxyz, h, dfsquare, hf,
          xmin, xmax, ymin, ymax, zmin, zmax;
  begin
    n := 0;
    h := 1e-5;
    [xmin, xmax] := [op(xiv)];
    [ymin, ymax] := [op(yiv)];
    [zmin, zmax] := [op(ziv)];
    while traperror((fxyz := eval(f(x,y,z)))) <> 0 or
          traperror((dfxyz := eval([df[1](x,y,z),
                                    df[2](x,y,z),
                                    df[3](x,y,z)]))) <> 0 do
      perturb(x, h);
      perturb(y, h);
      perturb(z, h);
    end_while;
    dfsquare := DOM_INTERVAL::center(DOM_INTERVAL::abs(hull(dfxyz[1]))^2+
				     DOM_INTERVAL::abs(hull(dfxyz[2]))^2+
				     DOM_INTERVAL::abs(hull(dfxyz[3]))^2);
    while n<nmax and specfunc::abs(fxyz) > eps*dfsquare do
      if specfunc::abs(dfsquare/fxyz)<1e-15 then
          // rang(df) not maximal. perturb.
        perturb(x, h);
        perturb(y, h);
        perturb(z, h);
      else
        hf := fxyz/dfsquare;
	[x, y, z] := [Re(x-hf*dfxyz[1]),
		      Re(y-hf*dfxyz[2]),
		      Re(z-hf*dfxyz[3])]:
      end_if;
      if x<xmin then x := xmin; end_if;
      if x>xmax then x := xmax; end_if;
      if y<ymin then y := ymin; end_if;
      if y>ymax then y := ymax; end_if;
      if z<zmin then z := zmin; end_if;
      if z>zmax then z := zmax; end_if;
    while traperror((fxyz := f(x,y,z);
                     dfxyz := eval([df[1](x,y,z),
                                    df[2](x,y,z),
                                    df[3](x,y,z)]))) <> 0 do
        perturb(x, h);
        perturb(y, h);
        perturb(z, h);
        if x<xmin then x := xmin; end_if;
        if x>xmax then x := xmax; end_if;
        if y<ymin then y := ymin; end_if;
        if y>ymax then y := ymax; end_if;
        if z<zmin then z := zmin; end_if;
        if z>zmax then z := zmax; end_if;
      end_while;
      dfsquare := DOM_INTERVAL::center(DOM_INTERVAL::abs(hull(dfxyz[1]))^2+
				       DOM_INTERVAL::abs(hull(dfxyz[2]))^2+
				       DOM_INTERVAL::abs(hull(dfxyz[3]))^2);
      n := n+1;
    end_while;
    userinfo(10..12, "Newton took ".expr2text(n)." steps");
    [x,y,z, fxyz, dfxyz];
  end_proc:

null():

