/* -----------------------------------------------------------
    plot::Listplot -- plot of discrete data values

    Syntax:
    plot::Listplot([y1, y2, ...] <x = xmin .. xmax>)
    plot::Listplot([[x1, y1], [x2, y2], ..])

    y1, y2, ..   : data values: 
                   real numerical values or expressions
                   of the animation parameter
    x            : the horizontal variable: an identifier or 
                   an indexed identifier
    xmin, xmax   : the range for horizontal variable
    x1, x2, ..   : horizontal coordinates of the data points:
                   real numerical values or expressions
                   of the animation parameter

    Example:
    >> plot::Listplot([1, 2, 3, 2, 1])
    >> plot::Listplot([1, 2, 3, 2, 1], x = 10..20)
    >> plot::Listplot([[1, 2], [3, 2],
                       [5, 3], [5.2, 1]],
                       InterpolationStyle = Cubic)
----------------------------------------------------------------*/ 
plot::createPlotDomain("Listplot",
                       "graphical primitive for discrete data",
                       2,  // Dimension
                       [[Data, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                          "Data to plot.", TRUE]],
                        XName, XMin, XMax, XSubmesh,
                        Submesh,
                        [InterpolationStyle, ["Optional", Linear],
                         ["Style", "Prop", {Linear, Cubic},
                          "Linear or cubic interpolation?", TRUE
                         ]
                        ],
                        PointSize, PointStyle, PointsVisible, PointColor,
                        LinesVisible, LineWidth, LineStyle, AntiAliased,
                        LineColor, LineColor2, LineColorFunction, LineColorType,
                        LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
                        LineColorDirection, FillColorDirection
                       ]):
//-----------------------------------------------------------------
plot::Listplot::styleSheet:= table(
     PointsVisible = TRUE,
     LinesVisible = TRUE,
     InterpolationStyle = Linear,
     XSubmesh = 6, // only relevant for spline interpolation
     PointColor = RGB::Black
):
//-----------------------------------------------------------------
//------------------------------------------------
// TODO: wirklich XName  per Hint als Achsentitle?
plot::Listplot::hints:= {XAxisTitle, Axes = Boxed}:
plot::Listplot::setHint:=
  object -> if object::XName <> FAIL and object::XAxisTitle = FAIL then
              plot::setHint(object, XAxisTitle = expr2text(object::XName)):
            end_if:

//-----------------------------------------------------------------
plot::Listplot::new :=
  proc()
    local object, other, n, tmp;
  begin
    object := dom::checkArgs([["X"]], args());
    other := object::other;
    if nops(other) > 0 then
      if nops(other) > 1 then
          error("data expected as a list, a list of lists, an array, ".
                 "or a matrix");
      end_if;
      dom::changeNotifier(object, "Data" = other[1]);
    end_if;

    if object::Data <> FAIL and
     ( nops(object::Data) = 0 
       or
       domtype(object::Data[1]) = DOM_LIST
     ) then
       // The data are [[x1, y1], [x2, y2], ..]
       // In this case, interpret x = xmin .. xmax as a = amin .. amax
       if object::ParameterName = FAIL and object::XName <> FAIL then
          tmp:= object::XName:
          dom::extslot(object, "XName", FAIL);
          dom::extslot(object, "ParameterName", tmp);
          tmp:= object::XMin:
          dom::extslot(object, "XMin", FAIL);
          dom::extslot(object, "ParameterBegin", tmp);
          tmp:= object::XMax:
          dom::extslot(object, "XMax", FAIL);
          dom::extslot(object, "ParameterEnd", tmp);
       end_if;
    end_if;
    //---------------------------------------------
    // Set XMin .. XMax. They must be set, or the
    // consistency check via dom::checkObject fails
    //---------------------------------------------
    if object::XName = FAIL then
       if object::ParameterName <> hold(x) then
          object::XName:= hold(x); // ouch! 
       else 
          object::XName:= `#x`; // ouch! // do not set XName to hold(x),
                                         // because this might clash with
                                         // the ParameterName
       end_if;
    end_if;
    n := nops(object::Data);
    if object::XMin = FAIL then
       object::dom::extslot(object, "XMin", 1):
    end_if;

    if object::XMax = FAIL then
       object::dom::extslot(object, "XMax", n);
    end_if;

    // with XMax = 0 and XMin = 1 no axes
    // are plotted, so we need to set
    // XMin = 0 = XMax to get axes:
    if n = 0 then
       object::dom::extslot(object, "XMin", n):
    end_if:
    //------------------
    // set hints
    //------------------
    dom::setHint(object):
    //------------------
    // consistency check
    //------------------
//  dom::checkObject(object);
    object;
end_proc:
//-----------------------------------------------------------------
//----------------------------------------------------------------
plot::Listplot::changeNotifier :=
  proc(object, eq)
    local slotName, newval, l, dim, i, j;
  begin
    [slotName, newval] := [op(eq)];
    case slotName
      of "Data" do
        l := newval;
        if not (testtype(l, Type::ListOf(DOM_LIST)) or
                domtype(l) = DOM_LIST  or
                domtype(l) = DOM_ARRAY or
                l::dom::hasProp(Cat::Matrix) = TRUE) then
          return("data expected as a list, a list of lists, an array, ".
                 "or a matrix");
        end_if;

        //--------------------------------------------
        // Matrix, Array --> list of lists :
        //--------------------------------------------
        if l::dom::hasProp( Cat::Matrix ) = TRUE then
          l:= expr(l): // Matrix -> Array, which will be converted next
        end_if;

        //-----------------------------------------------------
        // process data array: convert to list or list of lists
        //-----------------------------------------------------
        if domtype(l) = DOM_ARRAY then
           // Array -> List of Lists
           dim := op(l, [0,1]): // Dimension of Array
           if dim = 2 then
              if op(l, [0, 2, 2]) > op(l, [0, 2, 1]) and
                 op(l, [0, 3, 2]) > op(l, [0, 3, 1]) then
                 l:=[ [l[i,j] $ j = op(l, [0,3,1])..op(l, [0,3,1]) + 1]
                              $ i = op(l, [0,2,1])..op(l, [0,2,2])];
              else
                 l:= [op(l)];
              end_if;
           elif dim = 1 then
              // 1-dimensionaler Array in eine List
              l:= [op(l)]:
           else
               error("1st argument: the data array must be one or two-dimensional")
           end_if;
           if nops(object::other) > 1 then
              error("unexpected argument: ".expr2text(object::other[2]));
           end_if:
        end_if;

        //--------------------------------------------------
        // Now, the data l is a list or a list of lists ----
        //--------------------------------------------------
        l := float(l);

        if nops(l) > 0 and type(l[1]) = DOM_LIST then
           // check that all sublists have two entries
           if {op(map(l, nops))} <> {2} then
             error("the data must be either a list of y-values [y1, y2, ...] ".
                   "or a list of coordinate pairs [[x1, y1], [x2, y2], ...]"):
           end_if;
        end_if;

        dom::extslot(object, "Data", l);

        return(FALSE); // we've done all there is to do
        break;
    end_case;
    TRUE;
  end_proc:
//-----------------------------------------------------------------

plot::Listplot::print :=
  object -> if nops(object::Data) <= 5 then
               hold(plot::Listplot)(object::Data):
            else
               hold(plot::Listplot)("..."):
            end_if:
//-----------------------------------------------------------------

plot::Listplot::doPlotStatic:=
  proc(out, object, attributes, inherited)
    local xvalues, yvalues, tvalues,
          xmin, xmax, ymin, ymax, 
          xscale, // yscale,
          X, Y, x, y, t, 
          m, n, i, j, k, dt, 
          interpolyX, interpolyY,
          linecolFunc, style, submesh,
          points;
  begin
    yvalues := float(attributes[Data]);
    n := nops(yvalues);    // number of data elements
    if n = 0 then
       xmin:= attributes[XMin];
       xmax:= attributes[XMax];
       return([xmin..xmax, float(0)..float(0)]);
    end_if:
    m := nops(yvalues[1]); // number of rows, should be 1 or 2
                           // For m = 1, the data = [y1, y2, ...] consist
                           // of y-values.
                           // For m = 2, the data = [[x1,y1], [x2, y2], ..]
                           // consist of coordinate pairs
    if m = 1 then
       xmin:= attributes[XMin];
       xmax:= attributes[XMax];
       if n = 1 then
         xvalues:= [float(xmin)];
      // yvalues is defined
       else
         xvalues:= float([xmin + (i - 1)/(n - 1)*(xmax - xmin) $ i = 1..n]):
      // yvalues is defined
       end_if;
       xmin:= float(xmin):
       xmax:= float(xmax):
       ymin:= min(op(yvalues));
       ymax:= max(op(yvalues));
    else
       xvalues:= map(yvalues, op, 1);
       xmin:= min(op(xvalues));
       xmax:= max(op(xvalues));
       yvalues:= map(yvalues, op, 2);
       ymin:= min(op(yvalues));
       ymax:= max(op(yvalues));
    end_if;

    //----------------------------------------------
    // switch between linear and cubic interpolation
    //----------------------------------------------
    style:= attributes[InterpolationStyle];

    if contains(attributes, XSubmesh) and style = Cubic then
      submesh := attributes[XSubmesh]
    else
      submesh:= 0:
    end_if;
    if iszero(xmax - xmin) or iszero(ymax - ymin) then
      submesh:= 0; // Spline interpolation could cause problems
    end_if;

    //---------------------------------------------------------
    // obtain x and y values in the different modes via X and Y
    //---------------------------------------------------------
    if style = Cubic and submesh > 0 then // generate spline functions
       // create curve parameter values
       // The following is a heuristic compromise between
       // a `function interpolation' y = spline(x)
       // and a `curve interpolation' x = spline(t), y = spline(t)
       xscale:=      (xmax - xmin)^2:
//     yscale:= 10^3*(ymax - ymin)^2:
       tvalues[1]:= 0:
       for i from 2 to nops(yvalues) do
         /* experimental ---------------------------
           tvalues[i]:= tvalues[i-1] + 0.1/n +
                            ( (xvalues[i] - xvalues[i-1])^2/xscale 
                             +(yvalues[i] - yvalues[i-1])^2/yscale 
                            )^(1/2):
         ---------------------------------*/
           if specfunc::abs(xvalues[i] - xvalues[i-1]) < xscale/10^DIGITS then
              tvalues[i]:= tvalues[i-1] + 1.0/n
           else
              tvalues[i]:= tvalues[i-1] + specfunc::abs(xvalues[i] - xvalues[i-1])/xscale;
           end_if;
       end_for:
/* ----- a simple alternative --- 
       // The following choice is ideal for equidistant x values.
       // However, if the x values are far from equidistant, the 
       // following parameter values may produce a heavily 
       // oscillating x = spline(t) leading to loops in the
       // spline curve (x(t), y(t)).
       tvalues:= [i $ i=1..n]:
------- end of alternative   ---- */
       interpolyX:= numeric::cubicSpline([tvalues[j], xvalues[j]] $ j=1..n, NoWarning);
       interpolyY:= numeric::cubicSpline([tvalues[j], yvalues[j]] $ j=1..n, NoWarning);
       X:= t -> interpolyX(t);
       Y:= t -> interpolyY(t);
     else
       X:= i -> xvalues[i];
       Y:= i -> yvalues[i]:
    end_if;

    //-------------------------------------------------------
    // take care of the color function
    //-------------------------------------------------------
    if contains(attributes, LineColorFunction) then
      linecolFunc := attributes[LineColorFunction]
    else
      linecolFunc := null():
    end_if;

    //---------------------------
    // initialize the viewing box
    //---------------------------
    xmin:=  infinity;
    xmax:= -infinity;
    ymin:=  infinity;
    ymax:= -infinity;

    //-----------------------------------------------
    // draw the lines, but not the points (the points
    // may have a different color)
    //-----------------------------------------------
    points := [];
    if style = Cubic and submesh > 0 then // spline interpolation
       for j from 1 to n - 1 do
         t:= tvalues[j];
         dt:= (tvalues[j+1] - tvalues[j])/(submesh + 1):
         for k from 0 to submesh do
            [x, y]:= [X(t + k*dt), Y(t + k*dt)];
            points := points.[[x, y]];
            if x > xmax then xmax:= x; end_if;
            if x < xmin then xmin:= x; end_if;
            if y > ymax then ymax:= y; end_if;
            if y < ymin then ymin:= y; end_if;
         end_for:
       end_for:
       // draw the very last point
       [x, y]:= [X(tvalues[n]), Y(tvalues[n])];
       points := points.[[x, y]];
       if x > xmax then xmax:= x; end_if;
       if x < xmin then xmin:= x; end_if;
       if y > ymax then ymax:= y; end_if;
       if y < ymin then ymin:= y; end_if;
    else // linear interpolation
       for j from 1 to n do
            [x, y]:= [X(j), Y(j)];
            points := points.[[x, y]];
            if x > xmax then xmax:= x; end_if;
            if x < xmin then xmin:= x; end_if;
            if y > ymax then ymax:= y; end_if;
            if y < ymin then ymin:= y; end_if;
       end_for:
    end_if;
    out::writePoly2d(attributes, table("PointsVisible" = FALSE, "Filled" = FALSE), points, linecolFunc);
 
    //----------------------------------------------
    // draw the mesh points separately (use Pts2d to
    // make the points react to PointColor)
    //----------------------------------------------
    points := [];
    for j from 1 to n do
       points := points.[[xvalues[j], yvalues[j]]];
    end_for;
    out::writePoints2d(attributes, table(), points);

    //-----------------------
    // Return the viewing box
    //-----------------------
    [xmin..xmax, ymin..ymax]
  end_proc:
