// 
// Lsys -- the domain of Lindenmeyer systems

/*
Example: (dragon curve)
>> l:= plot::Lsys(PI/2,"L","L"="L+R+","R"="-L-R","L"=Line,"R"=Line):
>> l::Generations := 10:
>> plot(l)
*/


plot::createPlotDomain("Lsys",
                       "graphical primitive for Lindenmeyer systems",
                       2,  // Dimension
                       [LineStyle, LineWidth, LinesVisible, LineColor,
                        AntiAliased, Color,
                        [RotationAngle, ["Mandatory", NIL],
                         ["Definition", "Expr", FAIL,
                          "Rotation angle for Left and Right commands.", TRUE]],
                        [StartRule, ["Mandatory", NIL],
                         ["Definition", "Prop", "String",
                          "The starting rule.", TRUE]],
                        [IterationRules, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                          "Translation rules from one generation to the next.", FALSE]],
                        [TurtleRules, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                          "Rules translating a generation to Turtle commands.", FALSE]],
                        [Generations, ["Optional", 5],
                         ["Definition", "Expr", FAIL, // "Prop", "NonNegInt",
                          "Number of generations (iterations).", TRUE]],
                        [StepLength, ["Optional", 1.0],
                         ["Definition", "Expr", FAIL, //"Prop", "PosFloat",
                          "Length of lines (coordinate system units).", TRUE]]
                         ]):

//------------------------------------------------------------------------
plot::Lsys::styleSheet:= table(AntiAliased=FALSE):
//------------------------------------------------------------------------
// Hints for parents:
plot::Lsys::hints := {Scaling = Constrained}:
//------------------------------------------------------------------------


/*
new -- create new L-system

new(deg,start,rule...)

deg   - turtle degree (real number)
start - starting rule (non-empty string)
rule  - rule of the form '<lhs>=<rhs>'

The <lhs> must be a string of length 1. The <rhs> may be a string,
a turtle primitive or a color value. A turtle primitive is one of the
identifiers 'Line', 'Move', 'Left', 'Right', 'Push' or 'Pop'. A color value
is a list with 3 floats in the range 0..1 which define the
red-, green- and blue value of the color.
*/

plot::Lsys::new:=
  proc()
    local object, other, i, iterrules, transrules;
  begin
    object := dom::checkArgs([], args());

    // give hint that we don't want to see axes
    object::AxesVisible := FALSE;
  
    other := object::other;
    
    if nops(other) > 0 then
      object::RotationAngle := other[1];
    end_if;
    if nops(other) > 1 then
      object::StartRule := other[2];
    end_if;
    iterrules := [];
    transrules := [];
    if object::IterationRules <> FAIL then
      iterrules := object::IterationRules;
    end_if;
    if object::TurtleRules <> FAIL then
      transrules := object::TurtleRules;
    end_if;
    for i from 3 to nops(other) do
      if op(other[i], 0) <> hold(_equal) or
         domtype(lhs(other[i])) <> DOM_STRING then
        error("unknown option ".expr2text(other[i]));
      end_if;
      if domtype(rhs(other[i])) = DOM_STRING then
        iterrules := iterrules . [other[i]];
      elif rhs(other[i]) in {Line, Move, Left, Right, Push, Pop, Noop} or
           testtype(rhs(other[i]), Type::ListOf(Type::Real, 3, 4)) then
        transrules := transrules . [other[i]];
      else
        error("unknown option ".expr2text(other[i]));
      end_if;
    end_for;
    
    if iterrules = [] then error("No iteration rules given"); end;
    
    object::IterationRules := iterrules;
    object::TurtleRules := transrules;
    
    dom::checkObject(object);
  end_proc:


plot::Lsys::convert_to := 
  proc(l : plot::Lsys, T, attribs, cleaned_attribs)
    local iterrules, transrules, generations,
          state, i, angle, steplength;
  begin
    // we only know how to convert to a turtle
    if T <> plot::Turtle then
      return(FAIL);
    end_if;
    
    if args(0) < 3 then
      state       := l::StartRule;
      iterrules   := l::IterationRules;
      generations := round(l::Generations);
      angle       := l::RotationAngle;
      transrules  := l::TurtleRules;
      steplength  := l::StepLength;
      
      cleaned_attribs := table();
      for i in plot::Turtle::knownAttributes do
        if slot(l, expr2text(i)) <> FAIL then
          cleaned_attribs[i] := slot(l, expr2text(i));
        end_if;
      end_for;
    else
      state       := attribs[StartRule];
      iterrules   := attribs[IterationRules];
      generations := round(attribs[Generations]);
      angle       := attribs[RotationAngle];
      transrules  := attribs[TurtleRules];
      steplength  := attribs[StepLength];
    end_if;
    
    state := [state[i]$i=1..length(state)];
    
    iterrules := map(iterrules,
                     r -> lhs(r) = (op(r, 2)[i]$i=1..length(op(r, 2))));
    
    if domtype(generations) <> DOM_INT then
      generations := plot::getDefault(Generations);
    end_if;
    if steplength = FAIL then
      steplength := plot::getDefault(StepLength);
    end_if;
    steplength := expr(steplength);
    for i from 1 to generations do
      state := subs(state, iterrules);
    end_for;
    
    // append default drawing rules.  They are only used if no other
    // has been specified.
    transrules := transrules.["F" = Line, "f" = Move, 
                              "[" = Push, "]" = Pop,
                              "+" = Left, "-" = Right];
    
    // add parameters, change to proper Turtle commands
    transrules := subs(transrules,
                       [Line = Forward(steplength),
                        Move = (Up, Forward(steplength), Down),
                        Left = Left(angle), Right = Right(angle)]);
    transrules := map(transrules, r -> if testtype(rhs(r), 
                                                   Type::ListOf(Type::Real, 3, 4)) then
                                         lhs(r) = LineColor(rhs(r));
                                       else
                                         r
                                       end_if);
    
    plot::Turtle(subs(state, transrules), op(cleaned_attribs));
  end_proc:


// print -- return expression to print instead of L-system l
plot::Lsys::print:= 
  proc(l:plot::Lsys)
  begin
    if l::Generations <> FAIL then
      hold(plot::Lsys)(l::RotationAngle,
                       l::StartRule,
                       IterationRules = l::IterationRules,
                       TurtleRules = l::TurtleRules,
                       Generations = l::Generations);
    else
      hold(plot::Lsys)(l::RotationAngle,
                       l::StartRule,
                       IterationRules = l::IterationRules,
                       TurtleRules = l::TurtleRules);
    end_if;
  end_proc:

// MuPlotML -- execute L-system l and return turtle path
plot::Lsys::doPlotStatic :=
  proc(out, l, attributes, inheritedAttributes)
    local cleaned_attribs;
  begin
    cleaned_attribs := out::fixAttributes(attributes, plot::Turtle);
    out(dom::convert_to(l, plot::Turtle, attributes, cleaned_attribs),
                   inheritedAttributes, Raw):
  end_proc:

// end of file
