// 

/*
 plot::Ode3d -- depict the numerical solution of an ODE.
 
Intro:  the ODE  dY/dt = f(t,Y), Y(t0)=Y0 is solved
        by numeric::odesolve. A mesh of sample points 
        Y0=Y(t0), Y1=Y(t1), .. is generated. 

        These sample points (or components of these points)
        can be plotted as curves or points
       
Calls:
  plot::Ode3d(f, [t0,t1,..], Y0, 
            [G1 <, Style = style1> <, Color = c1> ],
            [G2 <, Style = style2> <, Color = c2> ],
            ...
            <method,> <Stepsize = h,> <RelativeError = tol,> 
            <AbsoluteError = atol,>
            <PointSize = p>, <LineWidth = w>, <Submesh = n>, ...)
  plot::Ode3d([t0,t1,..], f, Y0, 
            [G1 <, Style = style1> <, Color = c1> ],
            [G2 <, Style = style2> <, Color = c2> ],
            ...
            <method,> <Stepsize = h,> <RelativeError = rtol,> 
            <AbsoluteError = atol,>
            <PointSize = p>, <LineWidth = w>, <Submesh = n>, ...)

Parameters:
    f          - a procedure (t, Y) -> f(t,Y) representing
                 the rhs of the ODE dY/dt = f(t, Y)
    t0, t1, .. - sample points for the "time" parameter: real numerical values
    Y0         - the initial condition: a list or a matrix of of numerical values
    G1, G2, .. - the 'generators of plot data':
                 procedures (t, Y) -> F(t, Y) providing
                 the numerical data that are to be plotted
                 (see the details below).
                 These functions must accept a float t
                 and a list Y of floats. The list Y
                 represents the solutions vector Y(t)
                 of the ODE.
                 The functions G1, G2, .. must produce lists or matrices
                 with numerical values. All lists must be
                 of length 3.
    style1, style2, ..
               - the style in which the solution curve is plotted:
                 either 'Points' (points only),
                 or 'Lines' (curve with straight line segments)
                 or 'Splines' (curve via cubic spline interpolation),
                 or '[Lines, Points]' or '[Lines, Splines]' (do both)
    c1, c2, .. - RGB colors
    method            - see numeric::odesolve
    Stepsize = h      - see numeric::odesolve
    RelativeError = rtol - see numeric::odesolve
    AbsoluteError = atol - see numeric::odesolve

    Submesh = n- The option Submesh = n only has an effect for 
                 generators with Style = Splines.
                 A spline curve is displayed by straight line seqments 
                 connecting n additonal points between two points of
                 the time mesh of the numerical solution. 

Details:
   - A mesh of sample points Y0=Y(t0), Y1=Y(t1), .. is
     generated via numeric::odesolve. 
     The functions G1, G2 etc. are called at these points,
     each generating a data sequence 
     (G1(t0, Y0), G1(t1, Y1), ...),
     (G2(t0, Y0), G2(t1, Y1), ...).
     Each of these sequences is regarded as a sequence
     of mesh points in R^3 
     Each sequence is regarded a curve, which is plotted in the
     specified style and the specified color.
     E.g.:
        G1 := (t, Y) -> [t, Y[1], Y[2]]    --> x-axis == t
                                               y-axis == Y[1]
                                               z-axis == Y[2]
        G2 := (t, Y) -> [Y[1], Y[2], Y[3]] --> x-axis == Y[1]
                                               y-axis == Y[2]
                                               z-axis == Y[3]
        G2 := (t, Y) -> [t, h(Y), 1]       --> x-axis == t
                                               y-axis == some function h(Y)
                                               z-axis == constant

   - For a spline interpolation, the mesh points t0, t1, ... must
     be monotonically increasing.
     For Style = Points or Style = Lines the mesh points can be
     arbitrary.


An example animating lots of things:

plot::Ode3d((t, Y) -> [t*Y[1], (1.5+sin(a))*Y[1]^2], // animated ODE
            [Automatic, a, a+8, 1/(a+1)], // animated nr. of points
            [2, 2+sin(a)/2], // animated initial cond.
            a=0..2*PI):
 
 */
 
 plot::createPlotDomain("Ode3d",
                       "numerical solution of an ordinary differential equation",
                        3,  // Dimension
                        [Function,
                         [InitialConditions, ["Mandatory", NIL],
                          ["Definition", "ExprSeq", FAIL,
                           "Initial conditions of the ODE.", TRUE]],
                         [TimeMesh, ["Mandatory", NIL],
                          ["Calculation", "ExprSeq", FAIL,
                           "Time parameter values to solve the ode at.", TRUE]],
                         [ODEMethod, ["Optional", DOPRI78],
                          ["Calculation", "Expr", // "Prop",
                           FAIL,
                         /*
                           {EULER1,    RKF43,      xRKF43,
                            RKF34,     xRKF34,     RK4,
                            RKF54a,    RKF54b,     DOPRI54,
                            xDOPRI54,  CK54,       xRKF54a,
                            xRKF54b,   xCK54,      RKF45a,
                            RKF45b,    DOPRI45,    CK45,
                            xRKF45a,   xRKF45b,    xDOPRI45,
                            xCK45,     BUTCHER6,   RKF87,
                            xRKF87,    RKF78,      xRKF78,
                            DOPRI65, xDOPRI65, DOPRI56, xDOPRI56,
                            DOPRI87, xDOPRI87, DOPRI78, xDOPRI78,
                            GAUSS(s) 
                           },
                           */
                           "Numerical evaluation scheme.", TRUE]],
                         [Stepsize, ["Optional", NIL],
                          ["Calculation", "Expr", FAIL,
                           "Manual step size control.", TRUE]],
                         [RelativeError, ["Optional", NIL],
                          ["Calculation", "Expr", FAIL,
                           "Tolerance for relative errors.", TRUE]],
                         [AbsoluteError, ["Optional", NIL],
                          ["Calculation", "Expr", FAIL,
                           "Tolerance for absolute errors.", TRUE]],
                         //  UMesh, Mesh,
                         USubmesh,  // needed, otherwise Submesh is not accepted
                         Submesh,
                         [Projectors, ["Mandatory", NIL],
                          ["Style", "ExprSeq", FAIL,
                           "Transformation from raw data to doordinates.", TRUE]],
                         LineStyle, LineWidth,
                         LinesVisible,
                         Colors, 
//  LineColorType,  LineColor2, LineColorFunction,
                         PointStyle, PointSize, PointsVisible
                        ]):
   
//----------------------------------------------------------------
plot::Ode3d::styleSheet := table(USubmesh = 4,
                                 Submesh = 4,
                                 Colors = RGB::ColorList[1..20],
                                 PointsVisible = TRUE
                                ):
//----------------------------------------------------------------
// this is the default, since there is no FillColor
// plot::Ode3d::setPrimaryColor(LineColor):
//----------------------------------------------------------------

plot::Ode3d::new :=
proc()
  local object, other, opt, i, str;
begin
  object := dom::checkArgs([], args());
  other := object::other;
  
  i := 0;
  for opt in other do
    if has(opt,{EULER1,    RKF43,      xRKF43,
                RKF34,     xRKF34,     RK4,
                RKF54a,    RKF54b,     DOPRI54,
                xDOPRI54,  CK54,       xRKF54a,
                xRKF54b,   xCK54,      RKF45a,
                RKF45b,    DOPRI45,    CK45,
                xRKF45a,   xRKF45b,    xDOPRI45,
                xCK45,     BUTCHER6,   RKF87,
                xRKF87,    RKF78,      xRKF78,
                DOPRI65, xDOPRI65, DOPRI56, xDOPRI56,
                DOPRI87, xDOPRI87, DOPRI78, xDOPRI78,
                GAUSS
               }) then
      object::ODEMethod := opt;
      next;
    end_if;

    if opt::dom::hasProp(Cat::Matrix) = TRUE then
       // opt must be the initial condition.
       // Allow matrix input, but convert to a list
       // for speed:
       opt:= [op(opt)];
    end_if:
    
    if domtype(opt) = DOM_LIST then
      case i
        of 0 do // first list is time values
          i := 1;
          object::TimeMesh := opt;
          break;
        of 1 do // second list is Y0
          i := 2;
          object::InitialConditions := opt;
          break;
        otherwise
          if object::Projectors = FAIL then
            object::Projectors := [opt];
          else
            object::Projectors := object::Projectors . [opt];
          end_if;
      end_case;
      next;
    end_if;
    
    if object::Function <> FAIL then
      error("unexpected parameter: ".expr2text(opt));
    end_if;
    object::Function := opt;
  end_for;
  
  // default projections?  What colors should they use?
  // and are these the proper ones or should [Y[1], Y[2]]
  // be used for nops(object::InitialConditions)=2?
  if object::Projectors = FAIL then
    object::Projectors := [([subs((t, Y) -> [t, Y[`#i`], Y[`#j`]],
                                  `#i`=2*i-1, `#j`=2*i),
                             Style=[Splines, Points]])
			   $ i = 1..floor(nops(object::InitialConditions)/2)];
    if nops(object::InitialConditions) mod 2 = 1 then
      // instead of ignoring the last coordinate, use z=0 for it:
      object::Projectors := object::Projectors.
			  [[(t, Y) -> [t, Y[-1], 0],
			    Style = [Splines, Points]]];
    end_if;
  end_if;
  
  // check projectors
  str := dom::changeNotifier(object, "Projectors"=object::Projectors);
  if str::dom = DOM_STRING then
    error(str);
  end_if;
  
  dom::checkObject(object);
end_proc:

plot::Ode3d::print :=
obj -> hold(plot::Ode3d)(obj::Function, obj::TimeMesh, obj::InitialConditions,
                         op(obj::Projectors)):

plot::Ode3d::changeNotifier :=
proc(obj : plot::Ode3d, newval)
  local slotname, proj, i;
begin
  slotname := op(newval, 1);
  
  if slotname = "Projectors" and
     op(newval, 2) <> NIL and op(newval, 2) <> FAIL then
    newval := op(newval, 2);
    if not testtype(newval, Type::ListOf(DOM_LIST)) then
      return("Projectors must be list of lists");
    end_if;
    
    for proj in newval do
      if proj = [] or not testtype(proj[1], Type::Function) then
        return("Projectors must start with functions");
      end_if;
      
//     //-------------------
//     // test the generator
//     //-------------------
//     if traperror((tmp:= F[j](t[1], Y0))) <> 0 then
//        error("The ".output::ordinal(j)." plot data generator does not handle ".
//              "the ode data. It produced an error when called with ".
//              "(t, Y) = (".expr2text(t[1], Y0).").");
//     end_if;
//     if domtype(tmp) <> DOM_LIST or
//        nops(tmp) <> 3 or 
//        domtype(float(tmp[1])) <> DOM_FLOAT or
//        domtype(float(tmp[2])) <> DOM_FLOAT or
//        domtype(float(tmp[3])) <> DOM_FLOAT then
//        error("The ".output::ordinal(j)." plot data generator does not produce ".
//              "a list of real numerical coordinates of a 3D point. It returned ".
//              expr2text(tmp)." when called with (t, Y) = (".expr2text(t[1], Y0).").");
//     end_if;
      
      if nops(proj) > 3 then
        return("Projectors must have at most three entries");
      end_if;
      
      for i from 2 to 3 do
        if nops(proj) >=i then
          if type(proj[i]) <> "_equal" then
            return("Invalid projector specification: ".expr2text(proj[i]));
          end_if;
          case op(proj[i], 1)
            of Style do
              if not op(proj[i], 2) in {Points, Lines, Splines,
                                        [Lines, Points], [Splines, Points]} then
                return("Invalid style \"".expr2text(op(proj[i],2))."\"");
              end_if;
              break;
            of Color do
              if not testtype(float(op(proj[i], 2)),
                              Type::ListOf(Type::Numeric, 3, 4)) then
                return("Invalid color \"".expr2text(op(proj[i],2))."\"");
              end_if;
              break;
            otherwise
              return("Invalid attribute \"".expr2text(op(proj[i],1))."\"");
          end_case;
        end_if;
      end_for;
    end_for;
  end_if;
  TRUE;
end_proc:

plot::Ode3d::doPlotStatic:=
proc(out, obj, attributes, inheritedAttributes)
  local f, t, Y0, odesolveoptions, odedata, i, j, k, n,
        projectors, proj,
        xmin, xmax, ymin, ymax, zmin, zmax,
        X, Y, Z, color, colors, projected,
        style, submesh, points;
begin
  f := attributes[Function];
  t := attributes[TimeMesh];
  Y0 := attributes[InitialConditions];
  
  // to make the time steps animable,
  // we allow [Automatic, a, a+1, 0.1] etc.
  if t[1] = Automatic then
    if nops(t) <> 4 or
       {op(map(t[2..4], domtype@float))} <> {DOM_FLOAT} then
      error("Invalid TimeMesh specification");
    end_if;
    
    t := [$t[2]..t[3] step t[4]];
  end_if;
  
  odesolveoptions := attributes[ODEMethod];
  if has(attributes, Stepsize) then
    odesolveoptions := odesolveoptions, Stepsize = attributes[Stepsize];
  end_if;
  if has(attributes, RelativeError) then
    odesolveoptions := odesolveoptions,
    RelativeError = attributes[RelativeError];
  end_if;
  if has(attributes, AbsoluteError) then
    odesolveoptions := odesolveoptions,
    AbsoluteError = attributes[AbsoluteError];
  end_if;
  
  n := nops(t);
  odedata := [[t[1], Y0], 0 $ n-1]:  // initialize container for sample points
  userinfo(2, "starting to solve the ode"):
  for i from 2 to n do
    userinfo(3, "computing ".expr2text(i)."-th sample point:");
    odedata[i]:= [t[i], numeric::odesolve(t[i-1]..t[i],
                                          f, odedata[i-1][2],
                                          odesolveoptions)];
    userinfo(3, "--- time: ".expr2text(t[i]) . ", solution: ".
             expr2text(odedata[i][2]));
  end_for:
  userinfo(2, "the ode is solved"):
  
  // ode solved, now plot
  projectors := attributes[Projectors];
  colors := float(attributes[Colors]);
  
  xmin := RD_INF; xmax := RD_NINF;
  ymin := RD_INF; ymax := RD_NINF;
  zmin := RD_INF; zmax := RD_NINF;
  
  for i from 1 to nops(projectors) do
    proj := projectors[i];
    style := [Splines, Points];
    if nops(proj) > 1 and op(proj[2], 1) = Style then
      style := op(proj[2], 2);
    end_if;
    if nops(proj) > 2 and op(proj[3], 1) = Style then
      style := op(proj[3], 2);
    end_if;
    
    color := float(colors[((i-1) mod nops(colors)) + 1]);
    if nops(proj) > 1 and op(proj[2], 1) = Color then
      color := float(op(proj[2], 2));
    end_if;
    if nops(proj) > 2 and op(proj[3], 1) = Color then
      color := float(op(proj[3], 2));
    end_if;
    
    projected := map(odedata, float@proj[1]@op);

    // test if some unaware user returns the projected points
    // as matrix objects. To reduce costs, do this check only
    // for the first point:
    if nops(projected) > 0 and
       domtype(projected[1]) <> DOM_LIST and
       (projected[1])::dom::hasProp(Cat::Matrix) then
         projected:= map(projected, x ->  float([op(x)]));
    end_if:
    
    if not testtype(projected,
                    Type::ListOf(Type::ListOf(DOM_FLOAT, 3, 3))) then
      error("projector ".expr2text(i)." failed to return lists of three reals");
    end_if;
    
    // to allow [Lines, Points] et al., we use "has" here
    if has(style, Lines) then
      out::writePoly3d(attributes, table("LineColor" = color, "PointsVisible" = FALSE, "Closed" = FALSE, "Filled" = FALSE),
        projected, ()->null(), 1..3);
      xmin := min(xmin, op(map(projected, op, 1)));
      xmax := max(xmax, op(map(projected, op, 1)));
      ymin := min(ymin, op(map(projected, op, 2)));
      ymax := max(ymax, op(map(projected, op, 2)));
      zmin := min(zmin, op(map(projected, op, 3)));
      zmax := max(zmax, op(map(projected, op, 3)));
    end_if;
    
    if has(style, Splines) then
      X:= numeric::cubicSpline(op(zip(t, projected, (a, b)->[a, b[1]])));
      Y:= numeric::cubicSpline(op(zip(t, projected, (a, b)->[a, b[2]])));
      Z:= numeric::cubicSpline(op(zip(t, projected, (a, b)->[a, b[3]])));
      
      // can't use plot::MuPlotML with option Raw,
      // since LineColor and PointsVisible would be
      // ignored then.  We don't use AdaptiveMesh anyway,
      // and the splines can't have poles, so plotting
      // straight ahead is feasible:
      points := [];
      submesh := attributes[USubmesh]+1;
      for j from 1 to n-1 do
        for k from 0 to submesh-1 do
          points := points.[[float(X(((submesh-k)*t[j]+k*t[j+1])/submesh)),
               float(Y(((submesh-k)*t[j]+k*t[j+1])/submesh)),
               float(Z(((submesh-k)*t[j]+k*t[j+1])/submesh))]];
        end_for;
      end_for;
      points:=points.[[float(X(t[n])), float(Y(t[n])), float(Z(t[n]))]];
      out::writePoly3d(attributes, table("LineColor" = color, "PointsVisible" = FALSE, "Closed" = FALSE, "Filled" = FALSE),
        points, ()->null(), 1..3);
      xmin := min(xmin, op(map(points, op, 1)));
      xmax := max(xmax, op(map(points, op, 1)));
      ymin := min(ymin, op(map(points, op, 2)));
      ymax := max(ymax, op(map(points, op, 2)));
      zmin := min(zmin, op(map(points, op, 3)));
      zmax := max(zmax, op(map(points, op, 3)));
    end_if;
    
    if has(style, Points) then
      out::writePoints3d(attributes, table("PointColor" = color), projected, ()->null(), 1..3);
      xmin := min(xmin, op(map(projected, op, 1)));
      xmax := max(xmax, op(map(projected, op, 1)));
      ymin := min(ymin, op(map(projected, op, 2)));
      ymax := max(ymax, op(map(projected, op, 2)));
      zmin := min(zmin, op(map(projected, op, 3)));
      zmax := max(zmax, op(map(projected, op, 3)));
    end_if;
    
  end_for;
  
  if xmin <> RD_INF then
    [xmin..xmax, ymin..ymax, zmin..zmax];
  else
    null();
  end_if;
end_proc:

