/* -----------------------------------------------------------
Parallelogram2d -- The graphical primitive for 2D parallelograms

Calls:       plot::Parallelogram2d(center, tangent1, tangent2)

Parameters:  center   - a list [cx, cy]
             tangent1 - a list [t1x, t1y]
             tangent2 - a list [t2x, t2y]

Details:
     The parallelogram with center c, 1st tangent t1, 2nd tangent t2
     encloses the points {c + u*t1 + v*t2, u = -1..1, v = -1..1}.

Example:
>> plot::Parallelogram2d([0,0], [0, a], [2*a, 0], a = 0.. 3)
----------------------------------------------------------------*/ 

plot::createPlotDomain("Parallelogram2d",
                       "graphical primitive for 2D parallelograms",
                       2,
                       [CenterX, CenterY, Center, 
                        LineColor, LineWidth, LineStyle, LinesVisible,
                        AntiAliased, 
                        Filled, FillColor, Color, 
                        FillPattern,
                        [Tangent1X, ["Mandatory", NIL],
                          ["Definition", "Expr", FAIL, "x component of first tangent vector", FALSE]],
                        [Tangent1Y, ["Mandatory", NIL],
                          ["Definition", "Expr", FAIL, "y component of first tangent vector", FALSE]],
                        [Tangent1,  // library interface for [Tangent1X, Tangent1Y]
                         ["Library", NIL,
			  plot::libListOfOptExprXYZ,
			  plot::readListOfOptExprXYZ
                         ],
                         [[[Tangent1X, Tangent1Y]],
                          [[Tangent1X, Tangent1Y, Tangent1Z]]]
                        ],
                        [Tangent2X, ["Mandatory", NIL],
                          ["Definition", "Expr", FAIL, "x component of second tangent vector", FALSE]],
                        [Tangent2Y, ["Mandatory", NIL],
                          ["Definition", "Expr", FAIL, "y component of second tangent vector", FALSE]],
                        [Tangent2,  // library interface for [Tangent2X, Tangent2Y]
                         ["Library", NIL,
			  plot::libListOfOptExprXYZ,
			  plot::readListOfOptExprXYZ
                         ],
                         [[[Tangent2X, Tangent2Y]],
                          [[Tangent2X, Tangent2Y, Tangent2Z]]]
                        ]
                       ]):
//------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//------------------------------------------------------------
plot::Parallelogram2d::styleSheet := table(
      CenterX   = 0,
      CenterY   = 0,
      Tangent1X = 0,
      Tangent1Y = 1,
      Tangent2X = 1,
      Tangent2Y = 0,
      Filled = FALSE
):

//------------------------------------------------------------
plot::Parallelogram2d::setPrimaryColor(LineColor):
//------------------------------------------------------------
plot::Parallelogram2d::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());

    // get arguments which are not yet processed
    other := object::other;

    if nops(other) > 0 then
      if testargs() then
        if nops(other) > 3 then
           error("unexpected argument: ".expr2text(other[4]));
        end_if;

        if not testtype(other[1],
                        Type::ListOf(Type::Arithmetical, 2, 2)) and
          not (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
          error("expecting a list of 2 expressions as 1st argument")
        end_if;        
        if nops(other) > 1 and
          not testtype(other[2],
                       Type::ListOf(Type::Arithmetical, 2, 2)) and
          not (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
          error("expecting a list of 2 expressions as 2nd argument")
        end_if;
        if nops(other) > 2 and
          not testtype(other[3],
                       Type::ListOf(Type::Arithmetical, 2, 2)) and
          not (other[3])::dom::hasProp(Cat::Matrix)=TRUE then
          error("expecting a list of 2 expressions as 3rd argument")
        end_if;
      end_if;
      
      object::Center    := other[1];
      if nops(other) > 1 then
        object::Tangent1  := other[2];
      end_if;
      if nops(other) > 2 then
        object::Tangent2  := other[3];
      end_if;
    end_if;
    // semantically check for validity
    dom::checkObject(object);
  end_proc:
//--------------------------------------------------------------
plot::Parallelogram2d::print :=
  obj -> hold(plot::Parallelogram2d)(obj::Center, obj::Tangent1, obj::Tangent2):
//--------------------------------------------------------------
plot::Parallelogram2d::doPlotStatic:=
  proc(out, obj, attrib, inherited)
  begin
    out::writePoly2d(attrib, table("Closed"=TRUE), [[float(attrib[CenterX] - attrib[Tangent1X] - attrib[Tangent2X]),
          float(attrib[CenterY] - attrib[Tangent1Y] - attrib[Tangent2Y])],
         [float(attrib[CenterX] + attrib[Tangent1X] - attrib[Tangent2X]),
          float(attrib[CenterY] + attrib[Tangent1Y] - attrib[Tangent2Y])],
        [float(attrib[CenterX] + attrib[Tangent1X] + attrib[Tangent2X]),
          float(attrib[CenterY] + attrib[Tangent1Y] + attrib[Tangent2Y])],
         [float(attrib[CenterX] - attrib[Tangent1X] + attrib[Tangent2X]),
          float(attrib[CenterY] - attrib[Tangent1Y] + attrib[Tangent2Y])]]);
  end_proc:
