//      

/* -----------------------------------------------------------
    Piechart2d -- The graphical primitive for 2D pie charts

    Syntax:
    Piechart2d(data<, Titles=t><, Moves=m><, Colors=c><, op1, op2, ...>)

    data                  : list of expressions; the data set
    Titles                : list of equations pos=string; titles for the pies
    Moves                 : list of equations pos=offset; pie positions 
    Colors                : list of colors; pie colors
    op1, op2, ...         : options of the form 'option = value'

    Example:
>> plot::Piechart2d([5,12,38,14,25],
                    Moves=[4=6],
                    Titles=[4="Neue Option: Moves"])

>> plot::Piechart2d([5,12,38],
                    Colors=[RGB::RoyalBlue,RGB::VioletRed,RGB::GreenPale])

>> plot::Piechart2d([5,6,12,4],
                    Header="CICHLIDENZUCHT:",
                    Titles=[1="Aulonocara", 2="Cynotilapia",
                            3="Pseudotropheus",4="Steatocranus"])

>> plot::Piechart2d(matrix([1,5,3,6,5]))

>> plot::Piechart2d(array(1..5, (1)=9, (2)=8, (3)=5, (4)=8, (5)=5 ))

>> plot::Piechart2d([5, 6, 12, 4], Title = "CICHLIDENZUCHT:",
                    Titles = [1 = "Aulonocara", 2 = "Cynotilapia",
                              3 = "Pseudotropheus", 4 = "Steatocranus"], 
                    Moves=[i=exp(-(a-10*i)^2)*10$i=1..4], a=5..45)

// extreme animation
>> cond := (i-1)*PI/2<a and i*PI/2>a:
   piece := piecewise([cond, sin(2*a+(i-1)*PI)/2], [not cond, 0]):
   p2 := plot::Piechart2d([5+a/4, 2, 1+a/2, 4], Title = "CICHLIDENZUCHT:",
                          Center = [sin(a), cos(a)],
                          Radius = 10+sin(2*a),
                          Titles = [1 = "Aulonocara", 2 = "Cynotilapia",
                                    3 = "Pseudotropheus", 4 = "Steatocranus"], 
                          Moves=[i=piece$i=1..4], a=0..2*PI):
  plot(p2):
----------------------------------------------------------------*/ 

plot::createPlotDomain("Piechart2d",
                       "graphical primitive for 2D pie charts",
                       2,
       [[Data, ["Mandatory", NIL],
			 ["Definition", "ExprSeq", FAIL,
			  "Data to plot.", TRUE]],
        CenterX, CenterY, Center, Radius,
        [Moves, ["Optional", NIL],
         ["Definition", "ExprSeq", FAIL, "Pie offsets", FALSE]],
        [Titles,   ["Optional", NIL],
         ["Annotation", "ExprSeq", FAIL, "Pie titles", FALSE]],
        Colors, Color,
        LineColor, LineWidth, LineStyle, LinesVisible, AntiAliased,
        Filled, FillPattern, TextFont]
):

plot::Piechart2d::styleSheet :=
   table(Colors=RGB::ColorList[1..10],
         LineColor=RGB::Black,
         FillPattern=Solid,
         LinesVisible=TRUE, // consistent with Piechart3d
         CenterX = 0,
         CenterY = 0,
         Radius  = 1,
         Data    = [1],
         Moves   = [0],
         Titles  = [""]
   ):

// Hints for parents:
plot::Piechart2d::hints := {Axes = None, Scaling = Constrained}:

// methods yet to be implemented:  convert, convert_to, expr

// here we have to do something
plot::Piechart2d::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;

    if nops(other) > 1 then
      error("unexpected argument: ".expr2text(other[2]))
    end_if;
    
    if nops(other) <> 0 then
      dom::changeNotifier(object, "Data" = other[1]);
    end_if;

    // semantically check for validity
    dom::checkObject(object);
  end_proc:

plot::Piechart2d::print :=
  obj -> hold(plot::Piechart2d)(obj::Data,
                                dom::printAttributes(obj, {Data})):

plot::Piechart2d::changeNotifier :=
  proc(obj, eq)
    local l;
  begin
    if op(eq, 1) = "Data" then
      l := op(eq, 2);
      if not domtype(l)=DOM_LIST
        and not domtype(l) = DOM_ARRAY
        and not l::dom::hasProp(Cat::Matrix) = TRUE then
        error("1st argument: list or array or matrix expected")
      end_if;
      if domtype(l)=DOM_LIST then
        if not testtype(float(l), Type::ListOf(Type::Arithmetical)) then
          error("1st argument: entries of list must be arithmetical expressions")
        end_if:
      elif domtype(l) = DOM_ARRAY then
        if op(l,[0,1]) <> 1
          then error("1st argument: array must be one-dimensional")
        end_if;
      elif l::dom::hasProp( Cat::Matrix ) = TRUE then
        if linalg::nrows(l) <> 1 and linalg::ncols(l) <> 1 then
          error("1st argument: matrix must have one row or one column")
        end_if;
      end_if;
      //--------------------------------------------
      // Matrix, Array --> List :
      if domtype(l) = DOM_ARRAY or l::dom::hasProp( Cat::Matrix ) = TRUE then 
        l := [op(l)];
      end_if;
      // now l is a list
      dom::extslot(obj, "Data", l);
      //-----------------------------------------------
      return(FALSE);
    end_if;
    TRUE
  end_proc:



plot::Piechart2d::doPlotStatic:=
  proc(out, obj, attrib, inherited)
    local l, i, m, c, t, ss, pi, db, a, b, xPos, yPos, hAlign, vAlign,
          xMin, xMax, yMin, yMax, r, center, num, arc1, arc2;
  begin
    //------------  begin:  read values from attrib -------------------------
    if contains(attrib, hold(Data)) then
      l := map(attrib[hold(Data)], float);
      num := nops(l);
    else
      error("data set no longer available; please assign the plot::Piechart2d object to an identifier")
    end:
    
    if nops(l) = 0 then
      // no data given, do nothing
      return([0.0..0.0, 0.0..0.0])
    end_if;
    
    // radius of the pie
    r := float(attrib[Radius]);
    // center of the pie
    center := [attrib[CenterX], attrib[CenterY]];
    c := attrib[Colors];
    if nops(c) < num then
      // cycle trough givn colors
      c := _concat(c $ (num div nops(c)) + 1);
    end_if;
    m := plot::listAttribute2table(Moves, num, Type::NonNegative,
                                   attrib[Moves], 0.0);
    t := plot::listAttribute2table(Titles, num, DOM_STRING,
                                   attrib[Titles], "");
    if  min(op(l)) < 0 then
      error("Data: list entries must be nonnegative")
    end_if;
    //------------  end:  read values from attrib -------------------------
    
    // initial ViewingBox
    xMin := center[1];  xMax := center[1];
    yMin := center[2];  yMax := center[2];

    // sum of data values
    ss := _plus(op(l)); 
    pi := float(PI); b := float(0); 
    
    for i from 1 to num do
      db:= l[i]/ss*2*pi;
      a := b; b := b+db;
      xPos := float(center[1]+m[i]*r*cos((a+b)/2));
      yPos := float(center[2]+m[i]*r*sin((a+b)/2));
      arc1 := xPos+r*cos(a...b);
      arc2 := yPos+r*sin(a...b);
      xMin := min(xMin, op(arc1));
      xMax := max(xMax, op(arc1));
      yMin := min(yMin, op(arc2));
      yMax := max(yMax, op(arc2));
      out::writeArc2d(attrib, table("Closed" = TRUE, "FillColor" = c[i]), xPos, yPos, r, r, float(a), float(b));
    end_for;//i
    // 2. loop:  print titles over (after) pies
    for i from 1 to num do
      db:= l[i]/ss*2*pi;
      a := b; b := b+db;
      if t[i] <> "" then
        xPos := float(center[1]+1.1*(1+m[i])*r*cos((a+b)/2));
        yPos := float(center[2]+1.1*(1+m[i])*r*sin((a+b)/2));
        xMin := min(xMin, xPos);
        xMax := max(xMax, xPos);
        yMin := min(yMin, yPos);
        yMax := max(yMax, yPos);
        if xPos > center[1] then
          hAlign := Left;
        else
          hAlign := Right;
        end_if;
        if yPos > center[2] then
          vAlign := Bottom;
        else
          vAlign := Top;
        end_if;
        out::writeText2d(attrib, table("HorizontalAlignment"=hAlign, "VerticalAlignment"=vAlign), xPos, yPos, t[i]);
      end_if;
    end_for;//i
    
    [xMin..xMax, yMin..yMax];
  end_proc:
