//      

/* -----------------------------------------------------------
    Plane -- The graphical primitive for 3D planes

    Syntax:
    plot::Plane(Point, Normal <, op1, op2, ...>)
    plot::Plane(PN <, op1, op2, ...>)
    plot::Plane(Point1, Point2, Point3 <, op1, op2, ...>)
    plot::Plane(P123 <, op1, op2, ...>)

    Point, Point.i, Normal: lists of 3 expressions or matrices with 3 operands
    PN                    : matrix with 2 columns (6 operands)
    P123                  : matrix with 3 columns (9 operands)
    op1, op2, ...         : options of the form 'option = value'

    Example:
    >> plot::Plane([1, 2, 3], [4, 5, 6])

    >> plot::Plane(matrix([[1,4],[2, 5],[3,6]]))

    >> plot::Plane([1, 2, 3], [4, 5, 6], FillColor = RGB::Pink)
----------------------------------------------------------------*/ 

plot::createPlotDomain("Plane",
                       "graphical primitive for 3D planes",
                       3,
                       [PositionX, PositionY, PositionZ, Position,
                        NormalX, NormalY, NormalZ, Normal, 
                        UMesh, VMesh, Mesh, Filled,
                        LinesVisible, LineColor,
                        FillColor, Color
                       ]):

//-----------------------------------------------------------------------------
plot::Plane::styleSheet:= table(
      PositionX = 0,
      PositionY = 0,
      PositionZ = 0,
      NormalX = 0,
      NormalY = 0,
      NormalZ = 1,
      UMesh = 15, 
      VMesh = 15, 
      LinesVisible = TRUE,
      LineColor = RGB::Black.[0.25], // konsistent with Surface
      FillColor = RGB::LightBlue    // konsistent with Sphere, Parallelogram etc.
):

//-----------------------------------------------------------------------------
// methods yet to be implemented:  convert, convert_to, expr
//-----------------------------------------------------------------------------
// here we have to do something
plot::Plane::new:=
  proc()
    local object, other, i, v21, v31;
  begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());
    
    // get arguments which are not yet processed
    other := object::other;
    
    if testargs() and nops(other) > 0 then
      if not testtype(other[1],
                      Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[1])::dom::hasProp(Cat::Matrix)=TRUE then
        error("expecting a list of 3 expressions as 1st argument")
      end_if;
      
      if nops(other) > 1 and
        not testtype(other[2],
                     Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
        error("expecting a list of 3 expressions as 2nd argument")
      end_if;

      if nops(other) > 2 and
        not testtype(other[3],
                     Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[3])::dom::hasProp(Cat::Matrix)=TRUE then
        error("expecting a list of 3 expressions as 3rd argument")
      end_if;

      if nops(other) > 3 then
       error("unexpected argument: ".expr2text(other[4]));
      end_if;
    end_if;
    
    if nops(other) > 0 then
      for i from 1 to nops(other) do
        if (other[i])::dom::hasProp(Cat::Matrix)=TRUE then
           other[i]:= [op(other[i])];
        end_if;
      end_for;
      if nops(other) = 1 then
         if nops(other[1]) = 3 then
            object::Position:= other[1];
         elif nops(other[1]) = 6 then
            // point and normal given by a matrix
            object::Position:= [other[1][1],other[1][3],other[1][5]];
            object::Normal  := [other[1][2],other[1][4],other[1][6]];
         elif nops(other[1]) = 9 then
            // three points on the plane are given by a matrix
            object::Position := [other[1][1], other[1][4], other[1][7]]:
            v21:= [other[1][2] - other[1][1],
                   other[1][5] - other[1][4],
                   other[1][8] - other[1][7]];
            v31:= [other[1][3] - other[1][1],
                   other[1][6] - other[1][4],
                   other[1][9] - other[1][7]];
            object::Normal := [v21[2]*v31[3] - v21[3]*v31[2],
                               v21[3]*v31[1] - v21[1]*v31[3],
                               v21[1]*v31[2] - v21[2]*v31[1]]:
          else 
            error("expecting a matrix with 2 or 3 columns");
         end_if;
      end_if;
      if nops(other) = 2 then  // point and normal as lists
        object::Position := other[1];
        object::Normal := other[2];
      end_if;
      if nops(other) = 3 then // three points as lists
         object::Position := other[1]:
         v21:= zip(other[2], other[1], _subtract):
         v31:= zip(other[3], other[1], _subtract):
         object::Normal := [v21[2]*v31[3] - v21[3]*v31[2],
                            v21[3]*v31[1] - v21[1]*v31[3],
                            v21[1]*v31[2] - v21[2]*v31[1]]:
      end_if;
    end_if;

    if iszero(object::Normal[1]^2 + object::Normal[2]^2 + object::Normal[3]^2) then
         error("vanishing normal vector");
    end_if;
    // semantically check for validity
    dom::checkObject(object);
  end_proc:

plot::Plane::print :=
  obj -> hold(plot::Plane)(obj::Position, obj::Normal):

plot::Plane::doPlotStatic:=
  proc(out, obj, attrib, inherited)
  begin
    out::writePlane(attrib, table(), float(attrib[PositionX]), float(attrib[PositionY]), float(attrib[PositionZ]),
      float(attrib[NormalX]), float(attrib[NormalY]), float(attrib[NormalZ]));
  end_proc:
