/* ----------------------------------------------------------------------------
plot::Prism -- graphical primitive of a prism

Call(s):
    plot::Prism( <args, ...> )

Parameters:
    -

Options:
    attr, ...   : plot options, i.e., equations Attribute = value, where
                  Attribute is one of the attributes listed below or a hint.

Related Domains:
    plot::Pyramid, plot::Cone, plot::Cylinder, plot::Hexahedron, plot::Box

Example:
    >> plot( plot::Prism() );
    >> plot( plot::Prism( Edges=7) );
    >> plot( plot::Prism( Edges=7, Radius=2) );
    >> plot( plot::Prism( Edges=7, Top=[2,0,2]) )
    >> plot( plot::Prism( Edges=7, Top=[2,0,2], Normal=[0,0,0]) )
    >> plot( plot::Prism( Edges=7, Top=[2,0,2], Normal=[0,0,1]) )
    >> plot( plot::Prism( Edges=7, Base=[10,0,0], Top=[10,0,2]) )
    >> plot( plot::Prism( Edges=7, Base=[10,0,0], Top=[10,0,2], Angle=a, a=0..2*PI) )

Further Reading:
    Refer to plot::Pyramid.mu.

---------------------------------------------------------------------------- */

plot::createPlotDomain("Prism",
                       "graphical primitive for tetrahedrons",
                       3,
   [
    [Edges, ["Mandatory", NIL],
            ["Definition", "Expr", FAIL, "Number of edges.", TRUE]],
    Radius,
    Base, BaseX, BaseY, BaseZ,
    Top, TopX, TopY, TopZ,
    Normal, NormalX, NormalY, NormalZ,
    Angle,
    LinesVisible, LineStyle, LineWidth,
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::Prism::styleSheet:= table(
      LinesVisible  = TRUE,
      LineColor     = RGB::Black.[0.25],
      Edges         = 3,
      Radius        = 1,
      BaseX         = 0,
      BaseY         = 0,
      BaseZ         = 0,
      TopX          = 0,
      TopY          = 0,
      TopZ          = 1,
      NormalX       = 0,
      NormalY       = 0,
      NormalZ       = 0,
      Angle         = 0,
      LineColorDirectionY = 0
):

//-----------------------------------------------------------------------------
plot::Prism::hints:= {
      Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::Prism::new:= proc()
local  object, other;
option escape;
begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());

    // get arguments which are not yet processed
    other := object::other;

    if nops(other) > 0 then
      if testargs() then
        if nops(other) >=1 and not testtype(other[1], Type::Arithmetical) then
          error("1st argument: expecting an expression (the radius)")
        end_if;
        if nops(other) >= 2 and not testtype(other[2],
                       Type::ListOf(Type::Arithmetical, 3, 3)) then
          if not (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
            error("2nd argument: expecting a list of 3 expressions or a matrix (the base center)")
          else
            other[2]:= [op(other[2])];
          end_if;
        end_if;
        if nops(other) >= 3 and not testtype(other[3],
                       Type::ListOf(Type::Arithmetical, 3, 3)) then
          if not (other[3])::dom::hasProp(Cat::Matrix)=TRUE then
            error("3rd argument: expecting a list of 3 expressions or a matrix (the top center)")
          else
            other[3]:= [op(other[3])];
          end_if;
        end_if;
        if nops(other) >= 4 then
          error("unexpected argument: ".expr2text(other[4]))
        end_if;
      end_if;
      object::Radius := other[1];
      if nops(other) > 1 then
        object::Base   := other[2];
      end_if;
      if nops(other) > 2 then
        object::Top    := other[3];
      end_if;
    end_if;
    // semantically check for validity
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::Prism::print:= obj -> hold(plot::Prism)(
                          obj::Radius,
                          obj::Base,
                          obj::Top,
                          hold(Edges) = obj::Edges
                     ):

//-----------------------------------------------------------------------------
plot::Prism::doPlotStatic:= proc(out, object, attributes, inheritedAttributes)
local pyramid;
begin
    attributes[BaseRadius]:= attributes[Radius];
    attributes[TopRadius ]:= attributes[Radius];

    attributes:= out::fixAttributes(attributes, plot::Pyramid);

    pyramid:= plot::Pyramid(op(attributes));
    out(pyramid, inheritedAttributes, Raw)
end_proc:
//------------------------------------------------------------------------