/* ----------------------------------------------------------------------------
plot::Pyramid -- graphical primitive of a Pyramid

Call(s):
    plot::Pyramid( <args, ...> )

Parameters:
    -

Options:
    attr, ...   : plot options, i.e., equations Attribute = value, where
                  Attribute is one of the attributes listed below or a hint.

Related Domains:
    plot::Prism, plot::Cone, plot::Cylinder, plot::Hexahedron, plot::Box

Example:
    >> plot( plot::Pyramid() );
    >> plot( plot::Pyramid( Edges=7) );
    >> plot( plot::Pyramid( Edges=7, BaseRadius=2) );
    >> plot( plot::Pyramid( Edges=7, BaseRadius=2, TopRadius=1) );
    >> plot( plot::Pyramid( Edges=7, Top=[2,0,2]) )
    >> plot( plot::Pyramid( Edges=7, Top=[2,0,2], BaseRadius=2, TopRadius=1) )
    >> plot( plot::Pyramid( Edges=7, Top=[2,0,2], BaseRadius=2, TopRadius=1, Normal=[0,0,0]) )
    >> plot( plot::Pyramid( Edges=7, Top=[2,0,2], BaseRadius=2, TopRadius=1, Normal=[0,0,1]) )
    >> plot( plot::Pyramid( Edges=7, Base=[10,0,0], Top=[10,0,2]) )
    >> plot( plot::Pyramid( Edges=7, Base=[10,0,0], Top=[10,0,2], Angle=a, a=0..2*PI) )

Further Reading:
    Refer to plot::Prism.mu.

---------------------------------------------------------------------------- */

plot::createPlotDomain("Pyramid",
                       "graphical primitive for tetrahedrons",
                       3,
   [
    [Edges, ["Mandatory", NIL],
            ["Definition", "Expr", FAIL, "Number of edges.", TRUE]],
    BaseRadius, TopRadius,
    Base, BaseX, BaseY, BaseZ,
    Top, TopX, TopY, TopZ,
    Normal, NormalX, NormalY, NormalZ,
    Angle,
    LinesVisible, LineStyle, LineWidth,
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::Pyramid::styleSheet:= table(
      LinesVisible  = TRUE,
      LineColor     = RGB::Black.[0.25],
      Edges         = 4,
      BaseRadius    = 1,
      BaseX         = 0,
      BaseY         = 0,
      BaseZ         = 0,
      TopRadius     = 0,
      TopX          = 0,
      TopY          = 0,
      TopZ          = 1,
      NormalX       = 0,
      NormalY       = 0,
      NormalZ       = 0,
      Angle         = 0,
      LineColorDirectionY = 0
):

//-----------------------------------------------------------------------------
plot::Pyramid::hints:= {
      Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::Pyramid::new:= proc()
local  object, other;
option escape;
begin
    // check all known options from the argument list
    object := dom::checkArgs([], args());

    // get arguments which are not yet processed
    other := object::other;

    if testargs() and nops(other) > 0 then
      if nops(other) >=1 and not testtype(other[1], Type::Arithmetical) then
        error("1st argument: expecting an expression (the base radius)")
      end_if;

      if nops(other) >= 2 and not testtype(other[2],
                     Type::ListOf(Type::Arithmetical, 3, 3)) and
        not (other[2])::dom::hasProp(Cat::Matrix)=TRUE then
        error("2nd argument: expecting a list of 3 expressions (the base center)")
      end_if;

      if nops(other) >= 3 and
        (not testtype(other[3], Type::Arithmetical)) and
        (not testtype(other[3], Type::ListOf(Type::Arithmetical, 3, 3))) and
        not (other[3])::dom::hasProp(Cat::Matrix)=TRUE then
        error("3rd argument: expecting an expression (the top radius) ".
              "or a list of 3 expressions (the top center)")
      end_if;
      if nops(other) >= 4 and
        (not testtype(other[4], Type::ListOf(Type::Arithmetical, 3, 3))) and
        not (other[4])::dom::hasProp(Cat::Matrix)=TRUE then
        error("4th argument: expecting a list of 3 expressions (the top center)")
      end_if;
      if nops(other) > 4 then
        error("unexpected argument: ".expr2text(other[5]))
      end_if;
    end_if;

    if nops(other) > 0 then
      object::BaseRadius   := other[1];
      if nops(other) > 1 then
        object::Base       := other[2];
        if nops(other) = 3 then
          object::Top      := other[3];
        elif nops(other) > 2 then
          object::TopRadius:= other[3];
          if nops(other) = 4 then
            object::Top    := other[4];
          end_if;
        end_if;
      end_if;
    end_if;

    // semantically check for validity
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::Pyramid::print:= obj -> hold(plot::Pyramid)(
                          obj::BaseRadius,
                          obj::Base,
                          obj::TopRadius,
                          obj::Top,
                          hold(Edges) = obj::Edges
                         ):

//-----------------------------------------------------------------------------

plot::Pyramid::doPlotStatic := proc(out, object, attributes, inheritedAttributes)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local
    edges, bradius, base, tradius, top, angle,
    p, pi2, i, k, nx, ny, nz,
    n, ra, x, y, z, s, c, t, r,
    fillcolorfunction, points,
    linecolorfunction, haslinecolorfunction;
begin
    // check for fill-color function
    if contains(attributes, FillColorFunction) then
      fillcolorfunction:= float@(attributes[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;

    edges:= round(attributes[Edges]);
    if not testtype(edges, Type::PosInt) then
        error("Number of edges must be positive integer.");
    end_if;

    bradius:= float(attributes[BaseRadius]);
    if not testtype(bradius, DOM_FLOAT) or bradius < 0 then
        error("BaseRadius must be a non-negative real number.");
    end_if;

    tradius:= float(attributes[TopRadius]);
    if not testtype(tradius, DOM_FLOAT) or tradius < 0 then
        error("TopRadius must be a non-negative real number.");
    end_if;

    base:= float([attributes[BaseX], attributes[BaseY], attributes[BaseZ]]);
    top := float([attributes[TopX],  attributes[TopY],  attributes[TopZ]]);

    angle:= float(attributes[Angle]);

    /////////////////////////////////////////////////////////////////////////////////////
    // Align base and top according to the normal vector. This is either specified //////
    // by the user or given by the direct line betwen the base and the top.        //////
    /////////////////////////////////////////////////////////////////////////////////////
    [nx,ny,nz]:= float([attributes[NormalX], attributes[NormalY], attributes[NormalZ]]);

    if iszero(nx) and iszero(ny) and iszero(nz) then
      // The normal vector is not specified by the user. Use vector
      // base-top as normal vector. Afterwards compute the angle of
      // rotation for rotating the edges of the base and top.
      [x,y,z]:= [op(linalg::crossProduct(matrix([0,0,1]), matrix(top)-matrix(base)))]:
      ra:= linalg::angle(matrix([0,0,1]), matrix(top)-matrix(base));
    else
      // The normal vector is given. Compute the axis of rotation
      // and the angle of rotation for rotating the edges of the
      // base and top. The axis of rotation is an orthogonal vector
      // to the normal vector.
      [x,y,z]:= [op(linalg::crossProduct(matrix([0,0,1]),matrix([nx,ny,nz])))]:
      ra:= linalg::angle(matrix([0,0,1]), matrix([nx,ny,nz]));
    end_if;

    n:= specfunc::sqrt(x^2+y^2+z^2);
    if iszero(n) then // no rotation is needed
      r:= 1;
    else
      [x,y,z]:= [x/n,y/n,z/n];
      // rotation matrix, as found in graphics gems, p. 466:
      s:= specfunc::sin(ra);
      c:= specfunc::cos(ra);
      t:= 1-c;
      r:= matrix([[t*x^2+c,   t*x*y-s*z, t*x*z+s*y],
                  [t*x*y+s*z, t*y^2+c,   t*y*z-s*x],
                  [t*x*z-s*y, t*y*z+s*x, t*z^2+c  ]]);
    end_if;
    /////////////////////////////////////////////////////////////////////////////////////

    pi2 := 2.0*float(PI);
    top := float([[op(r*tradius*matrix([sin(k/edges*pi2+PI/edges-angle), cos(k/edges*pi2+PI/edges-angle), 0.0])+top) ] $ k=0..edges]);
    base:= float([[op(r*bradius*matrix([sin(k/edges*pi2+PI/edges-angle), cos(k/edges*pi2+PI/edges-angle), 0.0])+base)] $ k=0..edges]);

    // base: start to write triangle fan to XML stream
    out::writeTriangleFan(attributes, table("MeshVisible"   = FALSE, "PointsVisible" = FALSE, "HasLineColors" = FALSE),
      base, (x,y,z)->fillcolorfunction(1, x, y, z), ()->null(), 1..3);

    // wall: start to write triangle fan to XML stream
    points := map(zip(base,top,DOM_LIST), op);
    out::writeQuadStrip(attributes, table( "MeshVisible"   = FALSE, "PointsVisible" = FALSE ),
      points, ()->null(), (x,y,z)->fillcolorfunction(2,x,y,z), 1..3);

    // top: start to write triangle fan to XML stream
    out::writeTriangleFan(attributes, table("MeshVisible"   = FALSE, "PointsVisible" = FALSE),
      top, ()->null(), (x,y,z)->fillcolorfunction(3, x, y, z), 1..3);

    // base lines
    out::writePoly3d(attributes, table("Filled" = FALSE, "Closed"=TRUE), base,
      (x,y,z)->linecolorfunction(1, x, y, z), 1..3);

    // wall lines
    for p in zip(base,top,DOM_LIST) do
      out::writePoly3d(attributes, table("Filled" = FALSE, "Closed"=TRUE), p,
        (x,y,z)->linecolorfunction(2, x, y, z), 1..3);
    end_for;

    // top lines
    out::writePoly3d(attributes, table("Filled" = FALSE, "Closed"=TRUE), top,
      (x,y,z)->linecolorfunction(3, x, y, z), 1..3);

    //--------------------------------------------------------------------
    // Compute the coordinates of the viewing box as min..max of the base
    // and top coordinates
    //--------------------------------------------------------------------
    return( [min(map(base.top,op,i))..max(map(base.top,op,i)) $ i=1..3] );

  end_proc:
//------------------------------------------------------------------------
