/*===========================================================================
  Quantile-Quantile plots - Plotting the quantiles of two samples against
   each other, to determine a match in type of distribution

  CALL: plot::QQplot(s1, s2, <n>, <DiagonalVisible = TRUE> ))
        plot::QQplot([s1, s2], <n>, <DiagonalVisible = TRUE ))
        plot::QQplot(A, <n>, <DiagonalVisible = TRUE ))
        plot::QQplot(M, <n>, <DiagonalVisible = TRUE ))
        plot::QQplot(S, <c1, c2>,  <n>, <DiagonalVisible = TRUE ))

  PARAMETERS: 
           s1, s2: Lists of numerical data
           n: number of quantile points: a positive integer
           A: a 2-dim array (columns provide the data s1, s2)
           M: a matrix (columns provide the data s1, s2)
           S: a stats-sample (columns provide the data s1, s2)

  INFO:    http://mathworld.wolfram.com/Quantile-QuantilePlot.html
           http://www.itl.nist.gov/div898/handbook/eda/section3/qqplot.htm

  Example:
        n:= 50:
        s1:= [stats::normalRandom(1, 3)() $ i = 1..n]:
        s2:= [stats::uniformRandom(-2, 4)() $ i = 1..n]:
        A:= array(1..n, 1..2, [[s1[i], s2[i]] $ i = 1..n]):
        M:= matrix(A):
        S:= stats::sample([[A[i,j] $ j=1..2] $ i = 1..n]):
        SS:=stats::concatCol(S, S):
        
        plot(
             plot::Scene2d(plot::QQplot(s1, s2, Size = a, a = 10..300)),
             plot::Scene2d(plot::QQplot(s1, s2)),

             plot::Scene2d(plot::QQplot([s1, s2])),
             plot::Scene2d(plot::QQplot(A)), 

             plot::Scene2d(plot::QQplot(M)), 
             plot::Scene2d(plot::QQplot(S)), 

             plot::Scene2d(plot::QQplot(S, 1, 2)), 
             plot::Scene2d(plot::QQplot(S, [1, 2])), 

             plot::Scene2d(plot::QQplot(SS, 3, 4)), 
             plot::Scene2d(plot::QQplot(SS, [2, 3])), 

             Height = 28*unit::cm, Width = 25*unit::cm, 
             Rows = 4, Axes = Frame);

  Also implemented in:  Matlab, Mathematica
  ===========================================================================*/

plot::createPlotDomain("QQplot",
                       "graphical primitive for quantile-quantile plots",
                       2,  // Dimension
		       [ 
                        [Data, ["Mandatory", NIL],
			    ["Definition", "ExprSeq", FAIL,
			     "Data to plot.", TRUE]],
                        [Size, ["Optional", NIL],
                            ["Definition", "Expr", FAIL, "Number of points", TRUE]],
                        AntiAliased,
                        LineStyle,  LineWidth, LineColor,  LinesVisible,
                        PointStyle, PointSize, PointColor, PointsVisible
                       ]):
//----------------------------------------------------------------
plot::QQplot::styleSheet := table(PointsVisible=TRUE,
                                  PointColor = RGB::Black,
                                  LineColor = RGB::Red):
//----------------------------------------------------------------
plot::QQplot::new:=
  proc()
    local object, other, check;
  begin
    object := dom::checkArgs([], args());
    other := object::other;
    if nops(other) > 0 then
      if testtype(other[1], Type::ListOf(Type::Arithmetical)) then
        if nops(other) < 2 then
           error("expecting at least 2 arguments");
        end_if;
        other := [[other[1], other[2]]];
      end_if;
      check := dom::changeNotifier(object, "Data"=other[1]);
      if domtype(check) = DOM_STRING then
        error(check);
      end_if;
    elif object::Data <> FAIL then
      check := dom::changeNotifier(object, "Data"=object::Data);
      if domtype(check) = DOM_STRING then
        error(check);
      end_if;      
    end_if;
    
    dom::checkObject(object);
  end_proc:
//----------------------------------------------------------------
plot::QQplot::print:= b -> "plot::QQplot(...)":

//----------------------------------------------------------------
plot::QQplot::changeNotifier :=
  proc(obj, eq)
    local slotName, newval, l, columnIndices, dim, i, j, numberOfColumns;
  begin
    [slotName, newval] := [op(eq)];
    case slotName
      of "Data" do
        l := newval;
        if not (testtype(l, Type::ListOf(DOM_LIST)) or
                domtype(l) = DOM_ARRAY or
                domtype(l) = stats::sample or
                domtype(l) = DOM_LIST  or
                l::dom::hasProp(Cat::Matrix) = TRUE) then
          return("data expected as a list of lists, an array, ".
                 "a matrix, or a stats::sample");
        end_if;

        //--------------------------------------------
        // process QQplot(List1, options): 
        // convert to QQplot([List1], options)
        //--------------------------------------------
        if domtype(l) = DOM_LIST and
           domtype(expr(l[1]))<>DOM_LIST then 
          l:=[l];
        end_if;

        //--------------------------------------------
        // Matrix, Array --> List of Lists :
        //--------------------------------------------
        if l::dom::hasProp( Cat::Matrix ) = TRUE then
          l:= expr(l): // Matrix -> Array, which will be converted next
        end_if;
        //--------------------------------------------
        // process data array: convert to list of lists
        // Warning: transpose the array (we need a list
        // columns, not a list of rows)!
        //--------------------------------------------
        if domtype(l) = DOM_ARRAY then
          // Array -> List of Lists
          dim := op(l, [0,1]): // Dimension of Array
          if dim = 2 then
            if op(l, [0,3,2]) = op(l, [0,3,1]) then
              error("expecting a matrix/array with 2 columns, ".
                    "got only 1 column"):
            end_if;
            l:=[ [l[i,j] $ i = op(l, [0,2,1])..op(l, [0,2,2])] 
                         $ j = op(l, [0,3,1])..op(l, [0,3,1])+1];
          else
            return("Attribute \"Data\": the data array must be two-dimensional");
          end_if;
        end_if;
   
        // ---------------------------
        // process statistical samples
        // ---------------------------
        if domtype(l) = stats::sample then
          columnIndices:= []:
   
          // check if column indices are provided by the user:
          if nops(obj::other) > 1 and domtype(obj::other[2]) = DOM_LIST 
            then
            // the column indices are specified by a list
            columnIndices:= obj::other[2];
            numberOfColumns:= nops(columnIndices);         
          else
            i:= 1; // initialize: if args(0) = 1, the
                   // following loop will not be entered
            for i from 2 to nops(obj::other) do
              if domtype(obj::other[i]) = DOM_INT then
                columnIndices:= append(columnIndices, obj::other[i]);
              else
                break;
              end_if;
            end_for;
            numberOfColumns:= nops(columnIndices);
          end_if;
          if numberOfColumns = 0 then
            // No columns were specified by the
            // user. Extract *all* columns of l.
            // Check the number of elements in the
            // first row extop(l, 1)[1]:
            numberOfColumns:= nops(extop(l, 1)[1]);
            if numberOfColumns < 2 then
               error("expecting a stats::sample object with at least 2 data columns");
            end_if;
            if numberOfColumns > 2 then
               error("do not know which data columns to extract from the stats::sample object");
            end_if;
            columnIndices:= [i $ i = 1 .. numberOfColumns];
          end_if;
          l:= [stats::getdata(
               TRUE,           // double check the data
               "all_data",     // accept all sorts of numerical data
                               // as well as strings
               numberOfColumns,// number of data columns to be read
               l,              // the sample
               columnIndices   // the indices of the columns to be read
              )];
          // a string returned by stats::getdata indicates an error:
          if domtype(l) = DOM_STRING then 
            error(l);
          end_if;
      
           // check for string columns in l:
          for i from 1 to nops(l) do
            if domtype(l[i][1]) = DOM_STRING then
              l[i]:= NIL;
            end_if;
          end_for:
          if has(l, NIL) then 
            l:= subs(l, NIL = null());
          end_if;
          if nops(l) = 0 then
            error("No numerical data found in attribute \"Data\"");
          end_if;
        end_if; // of domtype(l) = stats::sample
   
        //---------------------------------------------
        // ---> Now: the data l is a list of lists ----
        //---------------------------------------------
        if nops(l) <> 2 then
           error("need 2 data lists, got ".expr2text(nops(l)));
        end_if;
        l := float(l);
        
        // for the inspector:
        dom::extslot(obj, "Data", l);
        
        return(FALSE); // we've done all there is to do
        break;
    of "Size" do
        if not testtype(newval, Type::PosInt) then
           error("Expecting a positive Integer for the attribute \"Size\"");
        end_if;
        // for the inspector:
        dom::extslot(obj, "Size", newval);
        return(FALSE); // we've done all there is to do
        break;
    end_case;
    TRUE;
  end_proc:

//----------------------------------------------------------------------------
plot::QQplot::doPlotStatic := 
  proc(out, object, attributes, inherited)
    local  size, i, j, s, f, qlist, points, xmax, xmin, ymax, ymin;
  begin
    if has(attributes, Size) then
       size:= attributes[Size];
    else
       size:= min(nops(attributes[Data][i]) $ i = 1..2):
    end_if;
    // In animations, Size may come in as a float value:
    if domtype(size) <> DOM_INT then
       size:= round(size);
    end_if;
    for i from 1 to 2 do
      s:= float(attributes[Data][i]):
      f:= stats::empiricalQuantile(s);
      if size = 1 then 
         qlist[i]:= [f(0)];
      else
         qlist[i]:= [f(j/(size - 1))$ j = 0 .. size - 1];
      end_if;
    end_for:
    points:= [ [qlist[1][j], qlist[2][j]] $ j = 1..size];
    attributes := plot::MuPlotML::fixAttributes(attributes, plot::PointList2d);
    xmin := min(op(map(points, op, 1))):
    xmax := max(op(map(points, op, 1))):
    ymin := min(op(map(points, op, 2))):
    ymax := max(op(map(points, op, 2))):

    // print the quantile points
    out::writePoints2d(attributes, table(), points);

    // print the line y = x
    out::writePoly2d(attributes, table("PointsVisible" = FALSE), [[min(xmin, ymin), min(xmin, ymin)],
      [max(xmax, ymax), max(xmax, ymax)]]);

    // make sure the line y = x is visible:
    [xmin, ymin]:= [min(xmin, ymin) $ 2]:
    [xmax, ymax]:= [max(xmax, ymax) $ 2]:

    [xmin..xmax, ymin..ymax];   
end_proc:
