/*  -------------------------------------------------------------------------
 Call: plot::Scatterplot(data, <options>)  

 Parameters:
    data    - sequence of lists [x1, x2, x3, ..], [y1, y2, y3, ...] 
              sequence of lists [x1, y1], [x2, y2], ...
              list of lists [[x1, x2, x3, ...], [y1, y2, y3, ..]]
              list of lists [[x1, y1], [x2, y2], ..]
              a matrix or an  array
    options - LineStyle, LineWidth, LineColor, LinesVisible,
              PointStyle, PointSize, PointColor, PointsVisible

The implementation follows
     http://www.ruf.rice.edu/%7Elane/case_studies/physical_strength/index.html :
     A scatterplot is simply a graph which plots an individuals' score on 
     one variable (e.g. arm strength) against their score on a second
     variable (e.g. supervisory ratings). 

     Scatterplots are used to examine any general trends in the relationship 
     between two variables. If scores on one variable tend to increase with 
     correspondingly high scores of the second variable, a positive relationship 
     is said to exist. If high scores on one variable are associated with low 
     scores on the other, a negative relationship exists. 

     The extent to which the dots in a scatterplot cluster together in the form 
     of a line indicates the strength of the relationship. Scatterplots with dots 
     that are spread apart represent a weak relationship. 

Examples:
    >> L := [[2,1,2,1,2],[1,2,1,2,1]]:
    >> plot(plot::Scatterplot(L)
    >> plot(plot::Scatterplot(L));
    >> plot(plot::Scatterplot(matrix([[5,2],[2,3],[5,8],[3,1],[4,4]])));
    >> plot(plot::Scatterplot(array(1..3, 1..2,
                          (1,1) = sqrt(2), (1,2) = 2, (2,1) = 2, 
                          (2,2) = sin(20), (3,1) = 3, (3,2) = 2)));
    >>  L1:=[float(sin(i/10*PI)*cos(j/400*PI)) $i=1..10$j=1..200]:
        L2:=[sin(i*j/1000) $i=1..10$j=1..200]:
        plot(plot::Scatterplot(L1, L2))

    >>  plot(plot::Scatterplot([op(linalg::hilbert(100),1..100)], 
                               [op(linalg::hilbert(200),101..200)]))
    >>  plot::Scatterplot([1,3,2,3,4,3,4,5,6,5,6,8,7,8,8,10,12,11],
                          [1,3,2,2,3,4,2,3,5,4,6,5,7,6,5,8 ,6 ,9 ])
-------------------------------------------------------------------------*/
plot::createPlotDomain("Scatterplot",
                       "graphical primitive for scatterplots",
                       2,  // Dimension
		       [ [Data, ["Mandatory", NIL],
			    ["Definition", "ExprSeq", FAIL,
			     "Data to plot.", TRUE]],
                       LineStyle,  LineWidth, LineColor,  LinesVisible,
                       AntiAliased,
                       PointStyle, PointSize, PointColor, PointsVisible]):
//----------------------------------------------------------------
plot::Scatterplot::styleSheet := table(PointsVisible=TRUE,
                                       PointColor = RGB::Black,
                                       LineColor = RGB::Red):
//----------------------------------------------------------------
plot::Scatterplot::new:=
  proc()
    local object, other, check;
  begin
    object := dom::checkArgs([], args());
    
    other := object::other;
    if nops(other) > 0 then
      if testtype(other[1], Type::ListOf(Type::Arithmetical)) then
        other := [other];
      end_if;
      check := dom::changeNotifier(object, "Data"=other[1]);
      if domtype(check) = DOM_STRING then
        error(check);
      end_if;
    elif object::Data <> FAIL then
      check := dom::changeNotifier(object, "Data"=object::Data);
      if domtype(check) = DOM_STRING then
        error(check);
      end_if;      
    end_if;
    
    dom::checkObject(object);
  end_proc:
//----------------------------------------------------------------
plot::Scatterplot::print:= 
  proc(b)
  begin
    if _plus(op(map(b::Data, nops))) < 5 then
      hold(plot::Scatterplot)(b::Data);
    else
      "plot::Scatterplot(...)";
    end_if;
  end_proc:
//----------------------------------------------------------------
plot::Scatterplot::changeNotifier :=
  proc(obj, eq)
    local slotName, newval, l, columnIndices, dim, i, j, numberOfColumns;
  begin
    [slotName, newval] := [op(eq)];
    case slotName
      of "Data" do
        l := newval;
        if not (testtype(l, Type::ListOf(DOM_LIST)) or
                domtype(l) = DOM_ARRAY or
                domtype(l) = stats::sample or
                domtype(l) = DOM_LIST  or
                l::dom::hasProp(Cat::Matrix) = TRUE) then
          return("data expected as a list of lists, an array, ".
                 "a matrix, or a stats::sample");
        end_if;

        //--------------------------------------------
        // process scatterplot(List1, options): 
        // convert to scatterplot([List1], options)
        //--------------------------------------------
        if domtype(l) = DOM_LIST and
           domtype(expr(l[1]))<>DOM_LIST then 
          l:=[l];
        end_if;
        //--------------------------------------------
        // process scatterplot([[x1, x2, ..],[y1,y2, ..]], options): 
        // convert to scatterplot([[x1, y1], [x1, y2], ..], options)
        //--------------------------------------------
        if domtype(l) = DOM_LIST and nops(l[1]) <> 2 then 
          l := [ [l[1][i], l[2][i]] $ i = 1..min(nops(l[1]), nops(l[2]))];
        end_if;

        //--------------------------------------------
        // Matrix, Array --> List of Lists :
        //--------------------------------------------
        if l::dom::hasProp( Cat::Matrix ) = TRUE then
          l:= expr(l): // Matrix -> Array, which will be converted next
        end_if;
        //--------------------------------------------
        // process data array: convert to list of lists
        // Warning: transpose the array (we need a list
        // columns, not a list of rows)!
        //--------------------------------------------
        if domtype(l) = DOM_ARRAY then
          // Array -> List of Lists
          dim := op(l, [0,1]): // Dimension of Array
          if dim = 2 then
            if op(l, [0,3,2]) = op(l, [0,3,1]) then
              error("expecting a matrix/array with 2 columns, ".
                    "got only 1 column"):
            end_if;
            l:=[ [l[i,j] $ j = op(l, [0,3,1])..op(l, [0,3,1])+1] 
                         $ i = op(l, [0,2,1])..op(l, [0,2,2])];
          else
            return("Attribute \"Data\": the data array must be two-dimensional");
          end_if;
        end_if;
   
        // ---------------------------
        // process statistical samples
        // ---------------------------
        if domtype(l) = stats::sample then
          columnIndices:= []:
   
          // check if column indices are provided by the user:
          if nops(obj::other) > 1 and domtype(obj::other[2]) = DOM_LIST 
            then
            // the column indices are specified by a list
            columnIndices:= obj::other[2];
            numberOfColumns:= nops(columnIndices);         
          else
            i:= 1; // initialize: if args(0) = 1, the
                   // following loop will not be entered
            for i from 2 to nops(obj::other) do
              if domtype(obj::other[i]) = DOM_INT then
                columnIndices:= append(columnIndices, obj::other[i]);
              else
                break;
              end_if;
            end_for;
            numberOfColumns:= nops(columnIndices);
            
         end_if;
          if numberOfColumns = 0 then
            // No columns were specified by the
            // user. Extract *all* columns of l.
            // Check the number of elements in the
            // first row extop(l, 1)[1]:
            numberOfColumns:= nops(extop(l, 1)[1]);
            columnIndices:= [i $ i = 1 .. numberOfColumns];
          end_if;
          l:= [stats::getdata(
               TRUE,           // double check the data
               "all_data",     // accept all sorts of numerical data
                               // as well as strings
               numberOfColumns,// number of data columns to be read
               l,              // the sample
               columnIndices   // the indices of the columns to be read
              )];
          // a string returned by stats::getdata indicates an error:
          if domtype(l) = DOM_STRING then 
            error(l);
          end_if;
      
           // check for string columns in l:
          for i from 1 to nops(l) do
            if domtype(l[i][1]) = DOM_STRING then
              l[i]:= NIL;
            end_if;
          end_for:
          if has(l, NIL) then 
            l:= subs(l, NIL = null());
          end_if;
          if nops(l) = 0 then
            error("No numerical data found in attribute \"Data\"");
          end_if;

          // rearrange l = [[x1, x2, x3,...], [y1, y2, y3, ...]]
          // to l = [[x1, y1], [x2, y2], ...]
          l := [ [l[1][i], l[2][i]] $ i = 1..nops(l[1])];

        end_if; // of domtype(l) = stats::sample
   
        //---------------------------------------------
        // ---> Now: the data l is a list of lists ----
        //---------------------------------------------
        l := float(l);
        
        // for the inspector:
        dom::extslot(obj, "Data", l);
        
        return(FALSE); // we've done all there is to do
        break;
    end_case;
    TRUE;
  end_proc:

//----------------------------------------------------------------------------
plot::Scatterplot::doPlotStatic := 
  proc(out, ld, attributes, inherited)
    local  points, regLine, xmax, xmin, ymax, ymin;
  begin
    
    points := float(attributes[Data]);
    attributes := plot::MuPlotML::fixAttributes(attributes, plot::PointList2d);

    if nops(points) = 1 then
        // we just plot the point, but no regression line
        return(out::writePoints2d(attributes, table(), points));
    else
        regLine:=op(stats::linReg(points),1);

        xmin := min(op(map(points, op, 1))):
        xmax := max(op(map(points, op, 1))):
        ymin := min(op(map(points, op, 2))):
        ymax := max(op(map(points, op, 2))):

        out::writePoints2d(attributes, table(), points);

        out::writePoly2d(attributes, table("PointsVisible" = FALSE),
          [[xmin, regLine[2]*xmin+regLine[1]], [xmax, regLine[2]*xmax+regLine[1]]]);

        ymin:= min(ymin, regLine[2]*xmin+regLine[1]);
        ymax:= max(ymax, regLine[2]*xmax+regLine[1]);
    end_if;

    [xmin..xmax, ymin..ymax]
end_proc:
