/* ----------------------------------------------------------------------------
plot::SurfaceSet -- The graphical primitive for triangle and quads surface meshes

plot::SurfaceSet(MeshList) creates a 3D graphical object from 
a given triangle or quad mesh.

Call(s):
    plot::SurfaceSet( MeshList
                      <, MeshListType=value>
                      <, MeshListNormals=value>
                      <, UseNormals=value>
                      <, args, ...>
    )

Parameters:
    MeshList        : list of numbers, coordinates of points and normals (optional).
                      points [x,y,z] may be collected in lists with three elements.
					  several points points may be grouped in lists [[x1,y1,z2],...
					  [xn,yn,zn]] as well.

Options:
    MeshListType    : {Triangles, TriangleFan, TriangleStrip, Quads, QuadStrip, ColorQuads}

    MeshListNormals : {None, BeforePoints, BehindPoints, BeforeFacets, BehindFacets}

    UseNormals      : boolean value which specifies whether the normals defined 
                      in the given mesh list are used for plotting the object or 
                      not.

    attr, ...       : plot options, i.e., equations Attribute = value, where 
                      Attribute is one of the attributes listed below or a hint. 

Related Domains:
    plot::Surface, plot::SurfaceSTL, plot::Raster, plot::Rotate3d, plot::Scale3d,
    plot::Translate3d

Details:
    * MeshList contains coordinates of points and normals (optional) of either 
      triangles or quads which define a mesh of a 3D surface.
      The points must be given homogenous: If a normal is given it must be given 
      for all points or facets respectively. Such a surface set represents either
      a set of seperate triangles, a triangle fan, a triangle strip, a set of 
      sepearet quads or a strip of quads.
      MeshList must consist of a homogenous list of list of RGB and RGBa values 
      if MestListType is set to ColorQuads. Each color entry represents a 1 x 1
      rectangle in the x/y plain starting at point (0,0,0).

    * The following formats are accepted for MeshList:

      [ px,pz,pz, ... ]
           MeshListNormals=None, MeshListType=<ANY>
      [ nx,ny,nz,px,py,pz, ... ]
           MeshListNormals=BeforePoints, MeshListType=<ANY>
      [ px,py,pz,nx,ny,nz, ... ]
           MeshListNormals=BehindPoints, MeshListType=<ANY>
      [ nx,ny,nz,px1,py1,pz1,px2,py2,pz2,px3,py3,pz3, ... ]
           MeshListNormals=BeforeFacets, MeshListType=Triangles
      [ px1,py1,pz1,px2,py2,pz2,px3,py3,pz3,nx,ny,nz, ... ]
           MeshListNormals=BehindFacets, MeshListType=Triangles
      [ nx,ny,nz,px1,py1,pz1,px2,py2,pz2,px3,py3,pz3,px4,py4,pz4, ... ]
           MeshListNormals=BeforeFacets, MeshListType=Quads
      [ px1,py1,pz1,px2,py2,pz2,px3,py3,pz3,px4,py4,pz4,nx,ny,nz, ... ]
           MeshListNormals=BehindFacets, MeshListType=Quads

      [ [RGB,...], [RGB,...],... ]
          MeshListNormals=None, MeshListType=ColorQuads, RGB=RGB|RGBa

    * Facet orientation: The facets define the surface of a 3-dimensional object.
      As such, each facet is part of the boundary between the interior and the
      exterior of the object. The orientation of the facets (which way is "out"
      and which way is "in") can be specified in two ways which should be consistent.
      First, the direction of the normal is outward. Second, which is most commonly
      used now-a-day, list the facet vertices in counter-clockwise order when
      looking at the object from the outside (right-hand rule).

    * When setting the attribute UseNormals to FALSE the normals defined
      in MesList are ignored when plotting the object. This reduces the 
      data volume of the graphics object and the computing time. However, 
      it may lead to a less brilliant image.

    * The point list must not contain color values. Use the LineColorFunction
      and/or FillColorFunction instead.

    * The LineColorFunction and FillColorFunction will be called with the
      index of the current vertex as the first parameter, followed by the
      x-, y- and z-coordinate of the current point.

    * For rotated, scaled or translated surface objects use plot::Rotate3d,
      plot::Scale3d or plot::Translate3d, respectively, see example ???.

Examples:
    We create a triangle mesh with normals in front of each triangle and plot
    this object, a tetrahedron, afterwards:

    >> S:= [ 0.0, 0.0, -1.0, -1.5, -1.5, 1.4, 0.0, 1.7, 1.4,
             1.5, -1.5, 1.4, 0.0, 0.88, 0.47, -1.5, -1.5, 1.4,
             1.5, -1.5, 1.4, 0.0, 0.0, -1.4, -0.88, -0.41, 
             0.25, 1.5, -1.5, 1.4, 0.0, 1.7, 1.4, 0.0, 0.0,
             -1.4, 0.88, -0.41, 0.25, 0.0, 1.7, 1.4, -1.5,
             -1.5, 1.4, 0.0, 0.0, -1.4
       ]:

    >> plot(plot::SurfaceSet(S, MeshListNormals=BeforeFacets))

    A FillColorFunction can be given as well. This expects the index
    of the current point as its first parameter 
    followed by the x-, y- and z-coordinate of the current point:

    >> plot(plot::SurfaceSet(S, MeshListNormals=BeforeFacets,
            MeshVisible = TRUE,
            LineColor = RGB::Black,
            FillColorFunction = ((vertex,x,y,z)->
	    [RGB::Red,RGB::Blue,RGB::Green,RGB::Yellow]
	    [vertex+2 div 3])
            ), Scaling=Constrained
       )

Background:
    Notes about the implementation of plot::SurfaceSet:

    * The normal of a facet is used for all its vertices. Due to the
      fact that the triangles and quads share points, these points
      are declared with different normals.

    * Depending on your hardware we recommend to plot objects with no
      more than 50.000 to 150.000 facets. You should activate the option
      'Accelerate OpenGL' in the VCam options menu.

---------------------------------------------------------------------------- */

//-----------------------------------------------------------------------------
plot::createPlotDomain("SurfaceSet", "graphical primitive for 3D surface sets", 3,
   [
    [MeshList,        ["Mandatory", NIL], ["Definition", "ExprSeq",
                       FAIL,
                       "List of vertices.", TRUE]],

    [MeshListType,    ["Optional", Triangles], ["Definition", "Prop",
                       {Triangles, TriangleFan, TriangleStrip, Quads, QuadStrip, ColorQuads},
	               "Type of mesh.", TRUE]],

    [MeshListNormals, ["Optional", None], ["Definition", "Prop",
                       {None, BehindPoints, BeforePoints, BehindFacets, BeforeFacets},
                       "MeshList contains normals.", TRUE]],

    [UseNormals,      ["Optional", TRUE], [["Style","Surface"], "Prop",
                       "Bool",
                       "Use predefined normal vectors.", TRUE]],

    MeshVisible, LineStyle, LineWidth,
    LineColor, LineColorType, LineColor2, LineColorFunction, Color,
    Filled, FillColor, FillColorType, FillColor2, FillColorFunction,
    Shading,
    PointsVisible, PointStyle, PointSize,
    LineColorDirectionX, LineColorDirectionY, LineColorDirectionZ,
    LineColorDirection, FillColorDirection,
    FillColorDirectionX, FillColorDirectionY, FillColorDirectionZ
   ]
):

//-----------------------------------------------------------------------------
plot::SurfaceSet::styleSheet:= table(
      MeshVisible = FALSE,
      LineColor = RGB::Black.[0.25],
      LineColorDirectionY=0
):

//-----------------------------------------------------------------------------
plot::SurfaceSet::hints:= {
      //Scaling = Constrained
}:

//-----------------------------------------------------------------------------
plot::SurfaceSet::new:=
proc()
  local  object, other, i, j;
begin
    // filter out all generic arguments
    object:= dom::checkArgs([],args());

    // process the special attributes in object::other
    other:= object::other;
    if nops(other) > 0 then
      if nops(other) > 1 then
           error("unexpected argument: ".expr2text(other[2]));
      end_if;

      if object::MeshListType = ColorQuads then
        object::FillColorType:= Functional;  // !!!
      end_if;

      if (other[1])::dom::hasProp(Cat::Matrix) = TRUE then
        other[1]:= expr(other[1]);
      end_if;

      case type(other[1])
        of DOM_LIST do
		   // e.g. MeshListType=Triangles, each triangle as [[x1,y1,z1], [x2,y2,z2], [x3,y3,z3]]
		   // e.g. MeshListType=Triangles, each triangle as [x1,y1,z1, x2,y2,z2, x3,y3,z3]
		   // e.g. MeshListType=Quads, each quad as [[x1,y1,z1], [x2,y2,z2], [x3,y3,z3], [x4,y4,z4]]
		   // e.g. MeshListType=Quads, each quad as [x1,y1,z1, x2,y2,z2, x3,y3,z3, x4,y4,z4]
           // e.g. MeshListType=TriangleFan, the fan as [x1,y1,z1, x2,y2,z2, x3,y3,z3, x4,y4,z4]
		   if testtype(other[1],Type::ListOf(Type::ListOf(DOM_LIST))) then
              object::MeshList:= map(map(other[1],op),op);
		   elif testtype(other[1],Type::ListOf(DOM_LIST)) then
              object::MeshList:= map(other[1],op);
           else
              object::MeshList:= other[1];
           end_if:
           break;
        of DOM_ARRAY do
           object::MeshList:= [[(other[1])[i,j] 
                   $ j = op(other[1], [0, 3, 1]) .. op(other[1], [0, 3, 2])]
                   $ i = op(other[1], [0, 2, 1]) .. op(other[1], [0, 2, 2])];
           break;
        otherwise
           error("First argument: expecting list of vertices or color quads");
       end_case;
    end_if;

    // check all attributes
    dom::checkObject(object);
end_proc:

//-----------------------------------------------------------------------------
plot::SurfaceSet::print:= obj -> "plot::SurfaceSet(...)":

//-----------------------------------------------------------------------------
plot::SurfaceSet::doPlotStatic:= proc(out, object, attributes, inheritedAttributes)
    // object:              object that was created by the new method above.
    // attributes:          table of all attributes. hints are not included.
    // inheritedAttributes: just ignore it.
local
    fillcolorfunction, linecolorfunction, haslinecolorfunction,
    l, t, n, u, len, cnt,
    min2, max2,
    i, r, c, eps,
    nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z, p4x, p4y, p4z,
    pointsList;

begin
    // check for fill-color function
    if contains(attributes, FillColorFunction) then
      fillcolorfunction:= float@(attributes[FillColorFunction]); // procedure
    else fillcolorfunction:= () -> null();
    end_if;

    // check for line-color function
    if contains(attributes, LineColorFunction) then
      linecolorfunction:= float@(attributes[LineColorFunction]); // procedure
      haslinecolorfunction:= TRUE;
    else linecolorfunction:= () -> null();
         haslinecolorfunction:= FALSE;
    end_if;

    // initial local variables
    l:= float(attributes[MeshList]);
    t:= attributes[MeshListType];
    n:= attributes[MeshListNormals];
    u:= attributes[UseNormals];
    len:= nops(l);
    cnt:= 0;

    // -------------------------------------------------------------------------
    // Create plain of color quads

    if t = ColorQuads then
      // homogenous list of list of color values expected
      r:= {op(l)};
      if map(r, domtype) <> {DOM_LIST} then
         error("expecting a list of lists of RGB or RGBa values");
      elif domtype(r[1][1]) <> DOM_LIST or not contains({3, 4}, nops(r[1][1])) then
         error("expecting a list of lists of RGB or RGBa values");
      end_if;
      r:= map(r, nops);
      if nops(r) <> 1 then
        error("all sublists (rows of the raster image) must have the same length");
      end_if;

      // no FillColorFunction allowed
      if contains(attributes, FillColorFunction) then
        error("FillColorFunction is not allowed with MeshList = ColorQuads");
      end_if;

      // no normals allowed
      if n <> None then
        error("if MeshListNormals = None expected for MeshList = ColorQuads");
      end_if;

      eps:= 1.0;  // 0 < eps < 1 !!!

      userinfo(1, "Length=".expr2text([0..nops(l[1]), 0..nops(l)]),
                  "Type=".expr2text(t),
                  " [plot::SurfaceSet::doPlotStatic]");

      // start to write mesh to XML stream
      pointsList := [];
      cnt:= 0;
      py:= 0.0;
      for r in l do
        px:= 0.0;
        for c in r do
          cnt:= cnt + 1;
          pointsList := pointsList.[[px    , 0.0, py    , linecolorfunction(cnt, px, py, 0.0), c],
            [px+eps, 0.0, py    , [linecolorfunction(cnt, px, py, 0.0)], [c]],
            [px+eps, 0.0, py+eps, [linecolorfunction(cnt, px, py, 0.0)], [c]],
            [px    , 0.0, py+eps, [linecolorfunction(cnt, px, py, 0.0)], [c]]];
          px:= px + 1.0;
        end_for;
        py:= py + 1.0;
      end_for;

      // close definition of triangle or quads mesh
      return(out::writeQuads(attributes, table(), pointsList,
        ()->op(args(4)), ()->op(args(5)), 1..3));
    end_if;

    // End of plain of color quads
    // -------------------------------------------------------------------------

    // initial bounding box of this object

    // faster versions of min/max for two elements
    min2:= (a,b) -> if a<b then a else b end_if;
    max2:= (a,b) -> if a<b then b else a end_if;

    if len mod 3 <> 0 then
      error("MeshList must contain 3*k, k>0 coordinates");
    end_if;
    if map({op(l)}, domtype) <> {DOM_FLOAT} then
       error("MeshList contains non-floating-point values");
    end_if;

    // start to write mesh to XML stream
    pointsList := [];

    userinfo(1, "Length=".expr2text(len), 
                "Type=".expr2text(t), 
                "Normals=".expr2text(n),
                "UseNormals=".expr2text(u),
                " [plot::SurfaceSet::doPlotStatic]");
    case n
    of None do
       // check length of points list
       if t = Triangles and len mod 9 <> 0 then
         error("3*3*k, k>0 triangle coordinates expected");
       elif t = Quads and len mod 12 <> 0 then
         error("4*3*k, k>0 quad coordinates expected");
       elif contains({TriangleFan,TriangleStrip},t) and (len < 9 or (len-9) mod 3 <> 0) then
         error("3*3+3*k, k>=0 triangle coordinates expected");
       elif t = QuadStrip and (len < 12 or (len-12) mod 6 <> 0) then
         error("4*3+2*3*k, k>=0 quad coordinates expected");
       end_if;

       // write points
       cnt:= 0:
       for i from 1 to len step 3 do
         cnt:= cnt + 1;
         [px, py, pz]:= [l[i], l[i+1], l[i+2]];
         pointsList := pointsList.[[cnt, 0, px, py, pz]];
       end_for;
       break;

    of BehindPoints do
    of BeforePoints do
       // check length of points and normals list
       if t = Triangles and len mod 18 <> 0 then
         error("3*3*2*k, k>0 triangle and normal coordinates expected");
       elif t = Quads   and len mod 24 <> 0 then
         error("4*3*2*k, k>0 quad and normal coordinates expected");
       elif contains({TriangleFan,TriangleStrip},t) and (len < 18 or (len-18) mod 6 <> 0) then
         error("3*3*2+3*2*k, k>=0 triangle and normal coordinates expected");
       elif t = QuadStrip and (len < 24 or (len-24) mod 12 <> 0) then
         error("4*3*2+3*2*2*k, k>=0 quad and normal coordinates expected");
       end_if;

       // write points
       cnt:= 0:
       for i from 1 to len step 6 do
         cnt:= cnt + 1;
         if n = BehindPoints then
           [px, py, pz, nx, ny, nz]:= [l[i], l[i+1], l[i+2], l[i+3], l[i+4], l[i+5]];
         else
           [nx, ny, nz, px, py, pz]:= [l[i], l[i+1], l[i+2], l[i+3], l[i+4], l[i+5]];
         end_if;
         if u then
           pointsList := pointsList.[[cnt, 0, px, py, pz, nx, ny, nz]];
         else // ignore normals
           pointsList := pointsList.[[cnt, 0, px, py, pz]];
         end_if;
       end_for;
       break;

    of BehindFacets do
    of BeforeFacets do
       // check length of points and normals list
       if t = Triangles and len mod 12 <> 0 then
         error("(3*3+3)*k, k>0 triangle and normal coordinates expected");
       elif t = Quads   and len mod 15 <> 0 then
         error("(4*3+3)*k, k>0 quad and normal coordinates expected");
       elif t <> Triangles and t <> Quads then
         error("illegal combination of".
               "MeshListType=".expr2text(t)." and MeshListNormals=".expr2text(n));
       end_if;

       // write points
       if t = Triangles then
	 cnt := 0;
         for i from 1 to len step 12 do
           cnt:= cnt + 1;
           if n = BehindFacets then
             [px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z, nx, ny, nz]:= [l[i],
                l[i+1], l[i+2], l[i+3], l[i+ 4], l[i+ 5], l[i+6],
                l[i+7], l[i+8], l[i+9], l[i+10], l[i+11]
             ];
           else
             [nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z]:= [l[i],
                l[i+1], l[i+2], l[i+3], l[i+ 4], l[i+ 5], l[i+6],
                l[i+7], l[i+8], l[i+9], l[i+10], l[i+11]
             ];
           end_if;
           if u then // ignore normals
            pointsList := pointsList.[[cnt, 0, px, py, pz, nx, ny, nz]];
            pointsList := pointsList.[[cnt, 0, p2x, p2y, p2z, nx, ny, nz]];
            pointsList := pointsList.[[cnt, 0, p3x, p3y, p3z, nx, ny, nz]];
           else
            pointsList := pointsList.[[cnt, 0, px, py, pz]];
            pointsList := pointsList.[[cnt, 0, p2x, p2y, p2z]];
            pointsList := pointsList.[[cnt, 0, p3x, p3y, p3z]];
           end_if;
         end_for;
       else // Quads
	 cnt := 0;
         for i from 1 to len step 15 do
           cnt:= cnt + 1;
           if n = BehindFacets then
             [px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z, p4x, p4y, p4z, nx, ny, nz]:= [l[i],
                l[i+1], l[i+2], l[i+3], l[i+ 4], l[i+ 5], l[i+6],
                l[i+7], l[i+8], l[i+9], l[i+10], l[i+11], l[i+12], l[i+13], l[i+14]
             ];
           else
             [nx, ny, nz, px, py, pz, p2x, p2y, p2z, p3x, p3y, p3z, p4x, p4y, p4z]:= [l[i],
                l[i+1], l[i+2], l[i+3], l[i+ 4], l[i+ 5], l[i+6],
                l[i+7], l[i+8], l[i+9], l[i+10], l[i+11], l[i+12], l[i+13], l[i+14]
             ];
           end_if;
           if u then // ignore normals
            pointsList := pointsList.[[cnt, 0, px, py, pz, nx, ny, nz]];
            pointsList := pointsList.[[cnt, 0, p2x, p2y, p2z, nx, ny, nz]];
            pointsList := pointsList.[[cnt, 0, p3x, p3y, p3z, nx, ny, nz]];
            pointsList := pointsList.[[cnt, 0, p4x, p4y, p4z, nx, ny, nz]];
           else
            pointsList := pointsList.[[cnt, 0, px, py, pz]];
            pointsList := pointsList.[[cnt, 0, p2x, p2y, p2z]];
            pointsList := pointsList.[[cnt, 0, p3x, p3y, p3z]];
            pointsList := pointsList.[[cnt, 0, p4x, p4y, p4z]];
           end_if;
         end_for;
       end_if;
       break;

    otherwise
       error("attribute MeshListNormals=".expr2text(n)." not supported");
    end_case;

    case t
      of Triangles do
        out::writeTriangles(attributes, table(), pointsList,
          (u,v,x,y,z)->linecolorfunction(u,x,y,z),
          (u,v,x,y,z)->fillcolorfunction(u,x,y,z));
        break;
      of TriangleFan do
        out::writeTriangleFan(attributes, table(), pointsList,
          (u,v,x,y,z)->linecolorfunction(u,x,y,z),
          (u,v,x,y,z)->fillcolorfunction(u,x,y,z));
        break;
      of TriangleStrip do
        out::writeTriangleStrip(attributes, table(), pointsList,
          (u,v,x,y,z)->linecolorfunction(u,x,y,z),
          (u,v,x,y,z)->fillcolorfunction(u,x,y,z));
        break;
      of Quads do
        out::writeQuads(attributes, table(), pointsList,
          (u,v,x,y,z)->linecolorfunction(u,x,y,z),
          (u,v,x,y,z)->fillcolorfunction(u,x,y,z), 3..5);
        break;
      of QuadStrip do
        out::writeQuadStrip(attributes, table(), pointsList,
          (u,v,x,y,z)->linecolorfunction(u,x,y,z),
          (u,v,x,y,z)->fillcolorfunction(u,x,y,z), 3..5);
        break;
      otherwise
        error("Unknown Meshtype");
    end_case;

    // close definition of triangle or quads mesh

    // finally, return the viewing box
  end_proc:
//------------------------------------------------------------------------

