// 
// Turtle -- 2D turtle graphic primitives

/*
Example: (a triangle)
>> t:= plot::Turtle():
>> t::line(1):
>> t::right(2*PI/3):
>> t::line(1):
>> t::right(2*PI/3):
>> t::line(1):
>> plot(t)
*/


plot::createPlotDomain("Turtle",
                       "primitive for 2D turtle graphics",
                       2,  // Dimension
                       [LineStyle, LineWidth, LinesVisible, LineColor,
                        AntiAliased, Color,
                        PointsVisible, PointStyle, PointSize, 
                        [CommandList, ["Mandatory", NIL],
                         ["Definition", "ExprSeq", FAIL,
                          "Movement commands of the Turtle.", FALSE]]
                        ]):

plot::Turtle::styleSheet:= table(CommandList = []):
plot::Turtle::hints := {Axes = None, Scaling = Constrained}:

// A new turtle starts at (0,0) heading north,
// empty stack.
plot::Turtle::new :=
  proc()
    option escape;
    local object, other, checkreal, do_append, res;
  begin
    object := dom::checkArgs([], args());
    
    // checkreal: real number, possibly containing animation parameter
    checkreal := proc(r)
                 begin
                   if object::ParameterName <> FAIL and
                      object::ParameterBegin <> FAIL then
                     testtype(float(subs(r,
                                         object::ParameterName
                                         = object::ParameterBegin, 
                                         EvalChanges)),
                              DOM_FLOAT);
                   else
                     testtype(float(r), DOM_FLOAT);
                   end_if;
                 end_proc;
    
//    do_append := (v) -> object::state[nops(object::state)] := v;
    do_append := (v) -> (plot::Turtle::extslot(object, "CommandList",
                                               object::CommandList . [v]));
    
    // setLineColor -- set current turtle color
    dom::extslot(object, "setLineColor",
		 proc(c)
		   name `plot::Turtle(...)::setLineColor`;
		   local cc;
		 begin
		   cc := plot::checkColor("Turtle", `#dummy`, c);
		   
		   if cc::dom = DOM_STRING then
		     error(cc);
		   else
		     do_append(LineColor(rhs(cc)));
		   end_if;
		   object;
		 end_proc);

    // left -- turn left angle d without changing position
    dom::extslot(object, "left",
		 proc(d)
		   name `plot::Turtle(...)::left`;
		 begin
		   if checkreal(d) then
		     do_append(Left(d));
		   else
		     error("angle not real");
		   end_if;
		   object;
		 end_proc);
		 
    // right -- turn right about d without changing position
    dom::extslot(object, "right",
		 proc(d)
		   name `plot::Turtle(...)::right`;
		 begin
		   if checkreal(d) then
		     do_append(Right(d));
		   else
		     error("angle not real");
		   end_if;
		   object;                      
		 end_proc);
    
    // forward -- move about length l
    dom::extslot(object, "forward",
		 proc(l)
		   name `plot::Turtle(...)::forward`;
		 begin
		   if checkreal(l) then
		     do_append(Forward(l));
		   else
		     error("distance not real");
		   end_if;
		   object;
		 end_proc);
    
    // penUp -- next commands won't draw until penDown
    dom::extslot(object, "penUp", () -> (do_append(Up()); object));
    dom::extslot(object, "penDown", () -> (do_append(Down()); object));
    
    // push -- push current state onto stack
    dom::extslot(object, "push", () -> (do_append(Push()); object));
    
    // pop -- pop new turtle state from stack
    dom::extslot(object, "pop", () -> (do_append(Pop()); object));
    
    other := object::other;
    
    if nops(other) > 0 then
      if domtype(other[1])=DOM_LIST then
	if nops(other) > 1 then
	  error("unexpected argument " . expr2text(other[2]));
	end_if;
        other := other[1];
      end_if;
      // changeNotifier performs the syntax checks
      res := plot::Turtle::changeNotifier(object, "CommandList"=other);
      if domtype(res) = DOM_STRING then
        error(res);
      end_if;
    end_if;
    
    dom::checkObject(object);
end_proc:

plot::Turtle::legalIdents := {}:
plot::Turtle::o_slot_ := plot::Turtle::slot:
plot::Turtle::slot :=
proc(o, s)
  local v;
begin
  v := dom::o_slot_(args());
  if v=FAIL and slot(plot::Turtle, s)=FAIL then
    error("Unknown slot ".s);
  end;
  v;
end:

plot::Turtle::changeNotifier :=
  proc(t : plot::Turtle, newval)
    local slot_name, unknown;
  begin
    slot_name := lhs(newval);
    if slot_name = "CommandList" and rhs(newval) <> NIL then
      
      newval := rhs(newval);
      if not testtype(newval, Type::ListOf(Type::Arithmetical)) then
        return("expecting a list of commands");
      end_if;
      
      // transform Push, Pop, Down, Up to Push(), Pop(), Down(), Up():
      newval := subs(newval, [Push = Push(), Pop = Pop(),
                              Down = Down(), Up = Up(), Noop = null()]);
      newval := subs(newval, [Push()() = Push(), Pop()() = Pop(),
                              Down()() = Down(), Up()() = Up()]);
      
      // check for unknown commands
      unknown := map({op(newval)}, op, 0) minus {Push, Pop, Down, Up,
                                                 Left, Right, Forward,
                                                 LineColor};
      
      if unknown <> {} then
        return("Unknown command(s): " . expr2text(op(unknown)));
      end_if;
      
      plot::Turtle::extslot(t, "CommandList",
                            newval);
      FALSE; // everything has been done
    else
      TRUE;
    end_if;
  end_proc:

// Tab completion:
plot::Turtle::complete := () -> (plot::Turtle::knownAttributes union
                                 plot::standardHints union
                                 {hold(setLineColor), hold(left), hold(right),
                                  hold(forward), hold(penUp), hold(penDown),
                                  hold(push), hold(pop)}):

// print -- print turtle
plot::Turtle::print:= proc(t)
                        local cmd;
                      begin
                        cmd := expr2text(t::CommandList);
                        if length(cmd) < 45 then
                          (hold(plot::Turtle)(t::CommandList)):
                        else
                          "plot::Turtle(".cmd[1..25]."...".cmd[-15..-1].")";
                        end_if;
                      end_proc:

// MuPlotML -- return turtle path as polyline
plot::Turtle::doPlotStatic:=
  proc(out, t, attributes, inheritedAttributes)
    local l, minx, maxx, miny, maxy, x, y, direction, penup,
          posStack, color, xs, ys,
          doUp, doDown, doPush, doPop, doLeft, doRight,
          doForward, doLineColor, points;
  begin
    x := 0.0;
    y := 0.0;
    xs := table(0.0=1);
    ys := table(0.0=1);
    direction := 0;
    penup := TRUE;
    posStack := table();
    color := float(attributes[LineColor]);

    points := [];
    doUp := proc()
            begin
              if not penup then
                points := points.[[x, y, [color]]];
                out::writePoly2d(attributes, table("LineColorType"="Functional", "Filled"=FALSE, "Closed"=FALSE),
                  points, ()->op(args(3)));
              end_if;
              penup := TRUE;
            end_proc;
    doDown := proc()
              begin
                if penup then
                  points := [];
                end_if;
                penup := FALSE;
              end_proc;
    doPush := proc()
              begin
                posStack[nops(posStack)] := [x, y, direction, [color]];
              end_proc;
    doPop := proc()
	     begin
	       if iszero(nops(posStack)) then
		 error("empty stack");
	       end_if;
               if not penup then
                points := points.[[x, y, [color]]];
                out::writePoly2d(attributes, table("LineColorType"="Functional", "Filled"=FALSE, "Closed"=FALSE),
                  points, ()->op(args(3)));
               end_if;
               [x, y, direction, color] := posStack[nops(posStack)-1];
               color := op(color);
               delete posStack[nops(posStack)-1];
               if not penup then
                points := [];
               end_if;
             end_proc:
//  doLeft := (a) -> (direction := direction + float(PI/180*a));
//  doRight := (a) -> (direction := direction - float(PI/180*a));
    doLeft := (a) -> (direction := direction + float(a));
    doRight := (a) -> (direction := direction - float(a));
    doForward := proc(l)
                 begin
                   if not penup then
                    points := points.[[x, y, [color]]];
                   end_if;
                   x := x - specfunc::sin(direction)*l;
                   y := y + specfunc::cos(direction)*l;
                   xs[x] := 1;
                   ys[y] := 1;
                 end_proc:
    doLineColor := (c) -> (color := float(c));
    
    l := attributes[CommandList];

    doDown();
    
//    l := array(1..nops(l), l);
    
    map(subs(l, [Up = doUp, Down = doDown, Push = doPush, Pop = doPop,
                 Left = doLeft, Right = doRight, Forward = doForward,
                 LineColor = doLineColor]),
        eval);
    
    doUp();

    xs := map({op(xs)}, op, 1);
    ys := map({op(ys)}, op, 1);
    
    minx := min(op(xs)); maxx := max(op(xs));
    miny := min(op(ys)); maxy := max(op(ys));
    
    [minx..maxx, miny..maxy];
  end_proc:

// end of file
