/* -----------------------------------------------------------
    VectorField2d -- The graphical primitive for 2D vectorfields

    Syntax:
    VectorField2d([v1, v2], x = a..b, y = c..d <, op1, op2, ...>)
    VectorField2d(v1, v2, x = a..b, y = c..d <, op1, op2, ...>)
    v1, v2    : x- and y-component of the vectorfield (functional expressions
                in x and y
    x,y       : identifiers
    a, b, c, d: arithmetical expressions
    options   : plot options for 2D graphical objects 

   Examples:  
   We demonstrate a plot of the vector field 
   v(x,y) = [1, sin(x)+cos(y)].
   field:= plot::VectorField2d(
                [1,sin(x)+cos(y)], x=0..3, y=1..2,
                LineColor = RGB::Red):
   plot(field);
----------------------------------------------------------------*/ 
plot::createPlotDomain("VectorField2d",
                       "graphical primitive for 2D vectorfields"):
// methods yet to be implemented:  convert, convert_to, expr
//----------------------------------------------------------------
plot::VectorField2d::styleSheet:= table(
      XMesh = 11,
      YMesh = 11,
      TipAngle = float(PI/5),
      TipLength = 1.5, 
      TipStyle = Open
):
//----------------------------------------------------------------
plot::VectorField2d::new:=
  proc()
    local object, other;
  begin
    // check all known options from the argument list
    object := dom::checkArgs(["X", "Y"], args()); 
    // get arguments which are not yet processed
    other := object::other;
    
    if nops(other) > 0 then
      if (other[1])::dom::hasProp(Cat::Matrix) = TRUE then
         other[1]:= [op(other[1])];
      end_if;

      if nops(other) = 1 and domtype(other[1]) = DOM_LIST and nops(other[1]) = 2 then
        object::XFunction := other[1][1];
        object::YFunction := other[1][2];
      
      elif nops(other) = 2 then
        object::XFunction := other[1];
        object::YFunction := other[2];
      elif nops(other) > 2 then
        error("unexpected argument: ".expr2text(other[3]));
      end_if;
    end_if;
  
    // semantically check for validity
    dom::checkObject(object);
  end_proc:
//----------------------------------------------------------------
plot::VectorField2d::print :=
  obj -> hold(plot::VectorField2d)([obj::XFunction, obj::YFunction],
                             obj::XName=obj::XRange,
                             obj::YName=obj::YRange):
//----------------------------------------------------------------
plot::VectorField2d::doPlotStatic:=
  proc(out, obj, attributes, inherited)
  local colorfunction, f_u_value, f_v_value,
        u, v, f_u, f_v, indU, indV, stepU, stepV, umesh, umax, umin,
        vmesh, vmax, vmin, points;
  begin

    if contains(attributes, LineColorFunction) then
      colorfunction:=  attributes[LineColorFunction] // 1 procedure,
                                                  // 3 expressions (RGB) or 
                                                  // 4 expressions (RGBa)
    else colorfunction:= () -> null();
    end_if;

    f_u := attributes[XFunction];
    f_v := attributes[YFunction];

    umin  := attributes[XMin]:
    umax  := attributes[XMax]:
    umesh := attributes[XMesh]:
    vmesh := attributes[YMesh]:
    vmin  := attributes[YMin]:
    vmax  := attributes[YMax]:
    if umesh = 1 then
    	umesh :=2 ;
    end_if;
    if vmesh = 1 then
    	vmesh := 2;
    end_if;
    stepU := (umax-umin)/(umesh-1);
    stepV := (vmax-vmin)/(vmesh-1);

    points := [];
    for indV from 0 to vmesh-1 do
      v := (vmesh - 1 - indV)/(vmesh - 1)*vmin + indV/(vmesh - 1)*vmax;
      for indU from 0 to umesh-1 do
        u := (umesh - 1 - indU)/(umesh - 1)*umin + indU/(umesh - 1)*umax;

        // correct Colorfuntion has to be tested!
        // possible: cf(u, v); cf(f_u(..),f_v(..));cf(u, v,f_u(..), f_v(..));
        if traperror((f_u_value:= float(f_u(u, v));
                      f_v_value:= float(f_v(u, v)))) <> 0 or
          map({f_u_value, f_v_value}, domtype) <> {DOM_FLOAT} then
          f_u_value := 0;
          f_v_value := 0;
        end_if;
        points := points.[[u, v, f_u_value, f_v_value]];
      end_for;
    end_for;

    out::writeField2d(attributes, table(), points, colorfunction,
      float(umin), float(vmin), float(stepU), float(stepV), umesh, vmesh, 3..4);
    return ([float(umin)..float(umax), float(vmin)..float(vmax)]);
  end_proc:
