/*++

  RGB colors

  These definitions are taken from:
  Ralf Schaper, Grafik mit Mathematica, Addison-Wesley, 1994, pp. 192-194

  For additional colors see also: X11/lib/X11/rgb.txt

  Example:

  plot( plot::Box(Color=RGB::VioletDark) ):

++*/

RGB := newDomain("RGB"):
RGB::create_dom:=hold(RGB):
RGB::info := "RGB values for color names":

/*++

RGB::ColorNames()      -- returns a list of all (predefined) color names
RGB::ColorNames(color) -- returns a list of all (predefined) color names
                          containing color

Example: RGB::ColorNames(Olive) yields [Olive, OliveGreenDark, OliveDrab]

++*/

RGB::ColorNames := proc(color)
//  option hold;
  local c, eqn, family, l, e;
begin
  family := (if args(0) = 1 then TRUE else FALSE end_if);
  l := [];
  for eqn in RGB do
    e := op(eqn, 2);
    if testtype(e, DOM_LIST) and nops(e) = 3 and
       map({op(e)}, domtype) = {DOM_FLOAT} then
      c := op(eqn, 1);
      // (?i) makes the match case insensitive
      if not family or (family and strmatch(c, "(?i)".color)) then
        l := append(l, c );
      end_if
    end_if
  end_for;
  return(map(sort(l), text2expr));
end_proc:

RGB::plotColorPalette:=
proc( cn = ``, cols = 5, opt = "" )
  local colors, num, header, col, row, rects, i, title, subtitle;
begin
  if testargs() then
    if not testtype(cn, Type::Union(DOM_IDENT, DOM_STRING)) then
      error("identifier or string expected as 1st argument");
    end_if;
    if not testtype(cols, Type::PosInt) then
      error("positive integer number expected as 2dn argument");
    end_if;
  end_if;
  colors:= RGB::ColorNames(cn);
  num := nops(colors);
  if cn = hold(``) then
    header := Header = "All Colors"
  else
    header := Header = "Colors containing '".cn."'"
  end_if;

  row := 0:  col := 0:  rects := [FAIL $ num]:
  for i from 0 to num-1 do
    col := col + 1;
    if col = cols + 1 then
      col := 1;  row := row + 1;
    end_if;
    title := expr2text(colors[num - i]);
    subtitle := "";
    if (opt <> "") then
      subtitle := "\n#" . _concat(op(map(slot(RGB, title), X -> stringlib::format(int2text(ceil(X*255), 16), 2, Right, "0"))));
    end_if;
    rects[num - i] :=
    plot::Rectangle((col-1)*2.5 .. (col-1)*2.5+2,
                    row*1.5 .. row*1.5+1,
                    FillColor = slot(RGB, title),
                    Title=title.subtitle,
                    TitlePosition = [(col-1)*2.5+1.0,
                                     row*1.5-0.25+_if(subtitle <> "", 0.1, 0)],
                    Name = title)
  end_for;

  if nops(rects) > 0 then
    plot(op(rects), Axes=None, plot::Rectangle::TitleFont = ["sans-serif", 8],
         Width = 120*unit::mm, TitleAlignment=Center,
         Filled, FillPattern = Solid, LinesVisible = FALSE,
         Height = round(200/cols*_if(subtitle <> "", 1.8, 1.5)*(row+1)/2.5+0.7)*unit::mm,
         header)
  else
    plot(Axes=None, Header = "No color contains '".cn."'")
  end
end_proc:

RGB::random := () -> [frandom(), frandom(), frandom()]:

// RGB::generator()          => use default color list
// RGB::generator(n)         => use first n colors of default color list
// RGB::generator(clist)     => use given color list
// RGB::generator(c1..c2)    => generate and use color list with 100 colors
// RGB::generator(c1..c2,n)  => generate and use color list with n colors
RGB::generator :=
proc()
  option escape;
  // static vars
  local i, c, colorList;
begin
  colorList:= proc(c,n)
     local c2, c1, i;
  begin
     c1:=op(c,1);
     c2:=op(c,2);
     [ float([(1-i)*c1[1]+i*c2[1], (1-i)*c1[2]+i*c2[2], (1-i)*c1[3]+i*c2[3]])
       $ i=0..1 step 1/(n-1)];
  end_proc:

  if args(0) = 0 then
     // use default color list
     c:= RGB::ColorList;

  elif args(0) = 1 then
     if domtype(args(1)) = DOM_INT then
        // use the first n colors of the default color list
        if args(1) < 1 or args(1) > nops(RGB::ColorList) then
           error("Integer of range 1..".expr2text(nops(RGB::ColorList))." expected");
        end_if;
        c:= RGB::ColorList[1..args(1)];
     elif domtype(args(1)) = DOM_LIST then
        // use given color list
        c:= args(1);
     elif type(args(1)) = "_range" and
          testtype(lhs(args(1)), Type::ListOf(Type::NonNegative,3,4)) and
          testtype(rhs(args(1)), Type::ListOf(Type::NonNegative,3,4)) then
        // create new color list with N colors
        c:= colorList(args(1),100);
     else
        error("Integer, color list or color range expected");
     end_if;
  elif args(0) = 2 then
     if domtype(args(2)) = DOM_INT and args(2) > 0 and
        type(args(1)) = "_range" and
        testtype(lhs(args(1)), Type::ListOf(Type::NonNegative,3,4)) and
        testtype(rhs(args(1)), Type::ListOf(Type::NonNegative,3,4)) then
        // create new color list
        c:= colorList(args(1),args(2));

     else
        error("color range and optional positive integer expected");
     end_if;
  end_if;

  i:= 0;

  // return value
  proc()
  begin
    if i = nops(c) then i:=0 end;
    c[(i:= i+1)]
  end_proc
end_proc:


RGB::ColorList := [

( RGB::Blue             := [0.0,0.0,1.0] ),
( RGB::Red              := [1.0,0.0,0.0] ),
( RGB::Green            := [0.0,1.0,0.0] ),
( RGB::MuPADGold        := [0.7,0.6,0.1] ),
( RGB::Orange           := [1.0,0.4,0.0] ),
( RGB::Cyan             := [0.0,1.0,1.0] ),
( RGB::Magenta          := [1.0,0.0,1.0] ),
( RGB::LimeGreen        := [0.196097,0.803903,0.196097] ),
( RGB::CadmiumYellowLight := [1.0,0.689993,0.059999] ),

( RGB::AlizarinCrimson 	:= [0.889996,0.149998,0.209998] ),
( RGB::Aqua             := [0.2,0.8,0.8] ),
// ( RGB::Lavender 	:= [0.902005,0.902005,0.980407] ),
( RGB::Lavender         := [0.8,0.6,1.0] ),
( RGB::SeaGreen         := [0.2,0.6,0.4] ),
( RGB::AureolineYellow  := [1.0,0.659993,0.140004] ),
( RGB::Banana           := [0.889996,0.810007,0.340007] ),
( RGB::Beige            := [0.640004,0.580004,0.5] ),
( RGB::YellowGreen      := [0.6039,0.803903,0.196097] ),
( RGB::Wheat            := [0.960799,0.870602,0.702002] ),
( RGB::IndianRed        := [0.689993,0.089999,0.119999] ),
( RGB::Black            := [0.0,0.0,0.0] ),


( RGB::AirForceBlue     := [0.36, 0.54, 0.65] ),
( RGB::Amaranth         := [0.89, 0.17, 0.31] ),
( RGB::Amber            := [1.0, 0.74, 0.0] ),
( RGB::Amethyst         := [0.6, 0.4, 0.8] ),
( RGB::Aquamarine       := [0.498001,1.0,0.831401] ),
( RGB::AquamarineMedium := [0.400006,0.803903,0.666707] ),
//( RGB::Asparagus        := [0.48, 0.62, 0.35] ),
( RGB::AzureBlueDark    := [0.0, 0.49, 1.0] ),
( RGB::Bisque           := [1.0,0.898053,0.771844] ),
( RGB::Bistre           := [0.23, 0.16, 0.12] ),
( RGB::Black0           := [1.00, 1.00, 1.00] ),
( RGB::Black5           := [0.95, 0.95, 0.95] ),
( RGB::Black10          := [0.90, 0.90, 0.90] ),
( RGB::Black15          := [0.85, 0.85, 0.85] ),
( RGB::Black20          := [0.80, 0.80, 0.80] ),
( RGB::Black25          := [0.75, 0.75, 0.75] ),
( RGB::Black30          := [0.70, 0.70, 0.70] ),
( RGB::Black35          := [0.65, 0.65, 0.65] ),
( RGB::Black40          := [0.60, 0.60, 0.60] ),
( RGB::Black45          := [0.55, 0.55, 0.55] ),
( RGB::Black50          := [0.50, 0.50, 0.50] ),
( RGB::Black55          := [0.45, 0.45, 0.45] ),
( RGB::Black60          := [0.40, 0.40, 0.40] ),
( RGB::Black65          := [0.35, 0.35, 0.35] ),
( RGB::Black70          := [0.30, 0.30, 0.30] ),
( RGB::Black75          := [0.25, 0.25, 0.25] ),
( RGB::Black80          := [0.20, 0.20, 0.20] ),
( RGB::Black85          := [0.15, 0.15, 0.15] ),
( RGB::Black90          := [0.10, 0.10, 0.10] ),
( RGB::Black95          := [0.05, 0.05, 0.05] ),
( RGB::Black100         := [0.00, 0.00, 0.00] ),
( RGB::BlueGrey         := [0.4,0.4,0.6] ),
//( RGB::BlueLight 	:= [0.678396,0.847102,0.902005] ),
( RGB::BlueLight        :=
  RGB::LightBlue        := [0.2,0.4,1.0] ),
( RGB::BlueMedium       := [0.0,0.0,0.803903] ),
( RGB::BluePale         :=
  RGB::PaleBlue         := [0.6,0.8,1.0] ),
( RGB::BlueViolet       := [0.5412,0.1686,0.886303] ),
( RGB::BondiBlue        := [0.0, 0.58, 0.71] ),
( RGB::Brass            := [0.7, 0.65, 0.25] ),
( RGB::Brick            := [0.610004,0.400006,0.119999] ),
( RGB::BrightGreen      := [0.4, 1.0, 0.0] ),
( RGB::BrightTurquoise  := [0.03, 0.9, 0.87] ),
( RGB::BrightViolet     := [0.8, 0.0, 0.8] ),
( RGB::Bronze           := [0.8, 0.49, 0.19] ),
// ( RGB::Brown 		:= [0.5,0.164693,0.164693] ),
( RGB::Brown            := [0.6,0.2,0.0] ),
( RGB::BrownOadder      := [0.859996,0.159993,0.159993] ),
( RGB::BrownOchre       := [0.53,0.260002,0.119999] ),
( RGB::Buff             := [0.94, 0.86, 0.5] ),
( RGB::Burgundy         := [0.56, 0.0, 0.12] ),
( RGB::BurntOrange      := [0.8, 0.33, 0.0] ),
( RGB::Burlywood        := [0.870602,0.721595,0.529405] ),
( RGB::BurntSienna      := [0.539994,0.209998,0.059999] ),
( RGB::BurntUmber       := [0.539994,0.200003,0.140004] ),
( RGB::Cadet            := [0.372494,0.619602,0.627506] ),
( RGB::CadmiumLemon     := [1.0,0.889996,0.009995] ),
( RGB::CadmiumOrange    := [1.0,0.380001,0.009995] ),
( RGB::CadmiumRedDeep 	:= [0.889996,0.089999,0.050005] ),
( RGB::CadmiumRedLight 	:= [1.0,0.009995,0.050005] ),
( RGB::CadmiumYellow 	  := [1.0,0.599994,0.069994] ),
( RGB::CamoGreen        := [0.47, 0.52, 0.41] ),
( RGB::Cardinal         :=
  RGB::CardinalRed      := [0.76, 0.11, 0.22] ),
( RGB::Carmine          :=
  RGB::CarmineRed       := [0.58, 0.0, 0.09] ),
( RGB::Carrot           := [0.930006,0.569994,0.129994] ),
( RGB::Cerise           := [0.87, 0.19, 0.38] ),
( RGB::Cerulean         := [0.020005,0.719993,0.799997] ),
( RGB::CeruleanDark     := [0.0, 0.48, 0.65] ),
( RGB::CeruleanBlue     := [0.16, 0.32, 0.74] ),
( RGB::Chartreuse       := [0.498001,1.0,0.0] ),
( RGB::Chestnut         := [0.8, 0.36, 0.36] ),
( RGB::Chocolate        := [0.823496,0.411802,0.117603] ),
( RGB::ChromeOxideGreen := [0.400006,0.5,0.080004] ),
( RGB::CinnabarGreen    := [0.380001,0.700003,0.159993] ),
( RGB::Cinnamon         := [0.48, 0.24, 0.0] ),
( RGB::Cobalt           := [0.239998,0.350002,0.670003] ),
( RGB::CobaltGreen      := [0.239998,0.569994,0.250008] ),
( RGB::CobaltVioletDeep := [0.569994,0.129994,0.619999] ),
( RGB::ColdGray         := [0.5,0.539994,0.53] ),
( RGB::ColdGrey         := [0.5,0.539994,0.53] ),
( RGB::Copper           := [0.72, 0.45, 0.2] ),
( RGB::Coral            := [1.0,0.498001,0.3137] ),
( RGB::CoralLight       := [0.941206,0.501999,0.501999] ),
( RGB::Corn             := [0.98, 0.92, 0.32] ),
( RGB::Crimson          := [0.86, 0.07, 0.23] ),
( RGB::CornflowerBlue 	:= [0.392193,0.584307,0.929395] ),
( RGB::DarkBlue         := [0.0,0.0,0.50196] ),
( RGB::DarkBrown        := [0.39, 0.26, 0.12] ),
( RGB::DarkChestnut     := [0.59, 0.41, 0.37] ),
( RGB::DarkGray         := [0.333333,0.333333,0.333333] ),
( RGB::DarkGrey         := [0.333333,0.333333,0.333333] ),
( RGB::DarkIndigo       := [0.19, 0.0, 0.38] ),
( RGB::DarkOrange       := [1.0,0.548997,0.0] ),
( RGB::DarkPink         := [0.9, 0.32, 0.5] ),
( RGB::DarkSalmon       := [0.91, 0.58, 0.47] ),
( RGB::DarkScarlet      := [0.33, 0.01, 0.09] ),
( RGB::DarkSpringGreen  := [0.09, 0.44, 0.27] ),
( RGB::DarkTan          := [0.56, 0.5, 0.31] ),
( RGB::PastelGreen      := [0.01, 0.75, 0.23] ),
( RGB::DarkRed          := [0.50196,0.0,0.0] ),
( RGB::DarkTeal         := [0.0,0.2,0.4]),
( RGB::DarkTurquoise    := [0.06, 0.37, 0.38] ),
( RGB::DarkViolet       := [0.25, 0.19, 0.53] ),
( RGB::DeepPink         := [1.0,0.078402,0.576495] ),
( RGB::DeepOchre        := [0.449995,0.239998,0.099994] ),
( RGB::DimGray          := [0.333333,0.333333,0.333333] ),
( RGB::DimGrey          := [0.333333,0.333333,0.333333] ),
( RGB::DodgerBlue       := [0.117603,0.564699,1.0] ),
( RGB::Eggplant         := [0.6, 0.0, 0.4] ),
( RGB::Eggshell         := [0.990005,0.900006,0.790002] ),
( RGB::EmeraldGreen     := [0.0,0.790002,0.340007] ),
( RGB::EnglishRed       := [0.829997,0.239998,0.099994] ),
( RGB::FernGreen        := [0.3, 0.47, 0.25] ),
( RGB::Firebrick        := [0.698004,0.133305,0.133305] ),
( RGB::Flesh            := [1.0,0.490005,0.250008] ),
( RGB::FleshOchre       := [1.0,0.340007,0.129994] ),
( RGB::ForestGreen      := [0.133305,0.545106,0.133305] ),
( RGB::Fuchsia          := [0.95, 0.0, 0.63] ),
( RGB::Gainsboro        := [0.862697,0.862697,0.862697] ),
( RGB::Gamboge          := [0.89, 0.6, 0.05] ),
( RGB::GeraniumLake     := [0.889996,0.069994,0.189993] ),
( RGB::Gold             := [1.0,0.8,0.0] ),
( RGB::GoldOchre        := [0.780007,0.47,0.149998] ),
( RGB::Goldenrod        := [0.8549,0.647099,0.125507] ),
( RGB::GoldenrodDark    := [0.721595,0.525498,0.043107] ),
// ( RGB::GoldenrodLight 	:= [0.980407,0.980407,0.823496] ),
( RGB::GoldenrodPale    := [0.933302,0.909803,0.666707] ),
//( RGB::GreenDark 	:= [0.0,0.392193,0.0] ),
( RGB::DarkGreen        :=
  RGB::GreenDark        := [0.0,0.2,0.0]),
( RGB::LightGreen       :=
  RGB::GreenLight       := [0.8,1.0,0.8] ),
( RGB::GreenPale        := [0.596103,0.984298,0.596103] ),
( RGB::GreenYellow      := [0.678396,1.0,0.184301] ),
( RGB::GreenishUmber    := [1.0,0.239998,0.050005] ),
( RGB::Gray             := [0.752907,0.752907,0.752907] ),
( RGB::Gray0            := [0.0, 0.0, 0.0] ),
( RGB::Gray5            := [0.05, 0.05, 0.05] ),
( RGB::Gray10           := [0.1, 0.1, 0.1] ),
( RGB::Gray15           := [0.15, 0.15, 0.15] ),
( RGB::Gray20           := [0.2, 0.2, 0.2] ),
( RGB::Gray25           := [0.25, 0.25, 0.25] ),
( RGB::Gray30           := [0.3, 0.3, 0.3] ),
( RGB::Gray35           := [0.35, 0.35, 0.35] ),
( RGB::Gray40           := [0.4, 0.4, 0.4] ),
( RGB::Gray45           := [0.45, 0.45, 0.45] ),
( RGB::Gray50           := [0.5, 0.5, 0.5] ),
( RGB::Gray55           := [0.55, 0.55, 0.55] ),
( RGB::Gray60           := [0.6, 0.6, 0.6] ),
( RGB::Gray65           := [0.65, 0.65, 0.65] ),
( RGB::Gray70           := [0.7, 0.7, 0.7] ),
( RGB::Gray75           := [0.75, 0.75, 0.75] ),
( RGB::Gray80           := [0.8, 0.8, 0.8] ),
( RGB::Gray85           := [0.85, 0.85, 0.85] ),
( RGB::Gray90           := [0.9, 0.9, 0.9] ),
( RGB::Gray95           := [0.95, 0.95, 0.95] ),
( RGB::Gray100          := [1.0, 1.0, 1.0] ),
( RGB::Grey             := [0.752907,0.752907,0.752907] ),
( RGB::Grey0            := [0.0, 0.0, 0.0] ),
( RGB::Grey5            := [0.05, 0.05, 0.05] ),
( RGB::Grey10           := [0.1, 0.1, 0.1] ),
( RGB::Grey15           := [0.15, 0.15, 0.15] ),
( RGB::Grey20           := [0.2, 0.2, 0.2] ),
( RGB::Grey25           := [0.25, 0.25, 0.25] ),
( RGB::Grey30           := [0.3, 0.3, 0.3] ),
( RGB::Grey35           := [0.35, 0.35, 0.35] ),
( RGB::Grey40           := [0.4, 0.4, 0.4] ),
( RGB::Grey45           := [0.45, 0.45, 0.45] ),
( RGB::Grey50           := [0.5, 0.5, 0.5] ),
( RGB::Grey55           := [0.55, 0.55, 0.55] ),
( RGB::Grey60           := [0.6, 0.6, 0.6] ),
( RGB::Grey65           := [0.65, 0.65, 0.65] ),
( RGB::Grey70           := [0.7, 0.7, 0.7] ),
( RGB::Grey75           := [0.75, 0.75, 0.75] ),
( RGB::Grey80           := [0.8, 0.8, 0.8] ),
( RGB::Grey85           := [0.85, 0.85, 0.85] ),
( RGB::Grey90           := [0.9, 0.9, 0.9] ),
( RGB::Grey95           := [0.95, 0.95, 0.95] ),
( RGB::Grey100          := [1.0, 1.0, 1.0] ),
( RGB::Heliotrope       := [0.87, 0.45, 1.0] ),
( RGB::HollywoodCerise  := [0.95, 0.0, 0.63] ),
( RGB::HotPink          := [1.0,0.411802,0.705893] ),
//( RGB::Indigo 		:= [0.03,0.179998,0.329997] ),
( RGB::Indigo           := [0.2,0.2,0.6] ),
( RGB::InternationalKleinBlue := [0.0, 0.18, 0.65] ),
( RGB::IntenrationalOrange := [1.0, 0.3, 0.0] ),
( RGB::IvoryBlack       := [0.159993,0.140004,0.129994] ),
( RGB::Jade             := [0.0, 0.65, 0.41] ),
( RGB::Khaki            := [0.941206,0.902005,0.548997] ),
( RGB::KhakiDark        := [0.741203,0.717597,0.419599] ),
( RGB::LampBlack        := [0.179998,0.280007,0.230003] ),
( RGB::LawnGreen        := [0.486297,0.988205,0.0] ),
( RGB::Lemon            := [0.99, 0.91, 0.06] ),
( RGB::LightGoldenrod 	:= [0.933302,0.866695,0.509796] ),
( RGB::LightGray        := [0.666666,0.666666,0.666666] ),
( RGB::LightGrey        := [0.666666,0.666666,0.666666] ),
( RGB::LightOrange      := [1.0,0.6,0.0] ),
( RGB::LightSalmon      := [1.0,0.627506,0.478393] ),
( RGB::Lilac            := [0.78, 0.63, 0.78] ),
( RGB::Lime             := [0.6,0.8,0.0] ),
( RGB::LightLime        := [0.8, 1.0, 0.0] ),
( RGB::MadderLakeDeep 	:= [0.889996,0.179998,0.189993] ),
( RGB::Malachite        := [0.04, 0.85, 0.31] ),
( RGB::ManganeseBlue    := [0.009995,0.659993,0.619999] ),
( RGB::Maroon           := [0.690207,0.188192,0.376507] ),
( RGB::MarsOrange       := [0.589999,0.269997,0.080004] ),
( RGB::MarsYellow       := [0.889996,0.440001,0.099994] ),
( RGB::Melon            := [0.889996,0.659993,0.410001] ),
( RGB::MidnightBlue     := [0.097995,0.097995,0.439207] ),
( RGB::Mint             := [0.739998,0.990005,0.790002] ),
( RGB::Moccasin         := [1.0,0.894101,0.709799] ),
( RGB::MossGreen        := [0.67, 0.87, 0.67] ),
( RGB::MountbattenPink  := [0.6, 0.47, 0.55] ),
( RGB::Mustard          := [1.0, 0.85, 0.34] ),
( RGB::NaplesYellowDeep := [1.0,0.659993,0.069994] ),
( RGB::Navajo           := [1.0,0.870602,0.678396] ),
( RGB::Navy             := [0.0,0.0,0.501999] ),
( RGB::NavyBlue         := [0.0,0.0,0.501999] ),
( RGB::Ochre            := [0.50196,0.50196,0.0] ),
( RGB::OldGold          := [0.81, 0.7, 0.23] ),
( RGB::Olive            := [0.230003,0.370006,0.170003] ),
( RGB::OliveDrab        := [0.419599,0.556902,0.137303] ),
( RGB::OliveGreen       := [0.2,0.2,0.0]),
( RGB::OliveGreenDark 	:= [0.333293,0.419599,0.184301] ),
( RGB::OrangeRed        := [1.0,0.270608,0.0] ),
( RGB::Orchid           := [0.8549,0.439207,0.839198] ),
( RGB::OrchidDark       := [0.599994,0.196097,0.799997] ),
( RGB::OrchidMedium     := [0.729408,0.333293,0.827494] ),
( RGB::PaleCarmine      := [0.68, 0.25, 0.2] ),
( RGB::PaleChestnut     := [0.86, 0.67, 0.68] ),
( RGB::PaleCornflowerBlue := [0.67, 0.8, 0.93] ),
( RGB::PaleMagenta      := [0.97, 0.51, 0.89] ),
( RGB::PaleMauve        := [0.6, 0.4, 0.4] ),
( RGB::PaleSandyBrown   := [0.85, 0.74, 0.67] ),
( RGB::PastelGreen      := [0.46, 0.86, 0.46] ),
( RGB::Peach            := [0.44, 0.26, 0.26] ),
( RGB::PeachOrange      := [1.0, 0.8, 0.6]), // = RGB::Tan
( RGB::PeachPuff        := [1.0,0.8549,0.725501] ),
( RGB::Peacock          := [0.200003,0.629994,0.790002] ),
( RGB::Pear             := [0.81, 0.88, 0.19] ),
( RGB::PermanentGreen 	:= [0.039995,0.790002,0.170003] ),
( RGB::PermanentRedViolet := [0.859996,0.149998,0.269997] ),
( RGB::PersianBlue      := [0.4, 0.0, 1.0] ),
( RGB::Peru             := [0.803903,0.521607,0.247093] ),
( RGB::PineGreen        := [0.0, 0.47, 0.43] ),
//( RGB::Pink 		:= [1.0,0.752907,0.796106] ),
( RGB::Pink             := [1.0,0.0,1.0] ),
( RGB::PinkLight        := [1.0,0.713706,0.756905] ),
( RGB::PinkOrange       := [1.0, 0.6, 0.4] ),
// ( RGB::Plum 		:= [0.866695,0.627506,0.866695] ),
( RGB::Plum             := [0.6,0.2,0.4] ),
( RGB::PowderBlue       := [0.0, 0.2, 0.6] ),
( RGB::PowderBlueLight  := [0.690207,0.878399,0.902005] ),
( RGB::PrussianBlue     := [0.18, 0.18, 0.31] ),
( RGB::Pumpkin          := [1.0, 0.45, 0.09] ),
( RGB::Purple           := [0.627506,0.125507,0.941206] ),
( RGB::PurpleMedium     := [0.576495,0.439207,0.858806] ),
( RGB::Raspberry        := [0.53,0.149998,0.340007] ),
( RGB::RawSienna        := [0.780007,0.380001,0.080004] ),
( RGB::RawUmber         := [0.449995,0.290002,0.069994] ),
( RGB::RobinEggBlue     := [0.0, 0.8, 0.8] ),
( RGB::Rose             := [1.0,0.6,0.8] ),
( RGB::Russet           := [0.5, 0.27, 0.1] ),
( RGB::Rust             := [0.71, 0.25, 0.05] ),
( RGB::RoseMadder       := [0.889996,0.209998,0.219993] ),
( RGB::RosyBrown 	      := [0.737297,0.560793,0.560793] ),
( RGB::RoyalBlue 	      := [0.254906,0.411802,0.882397] ),
( RGB::SaddleBrown 	    := [0.545106,0.270608,0.074496] ),
( RGB::SafetyOrange     := [1.0, 0.6, 0.0] ),
( RGB::Saffron          := [0.95, 0.76, 0.18] ),
( RGB::Salmon 	        := [0.980407,0.501999,0.447096] ),
( RGB::SandyBrown       := [0.956893,0.643101,0.376507] ),
( RGB::SapGreen         := [0.189993,0.5,0.080004] ),
( RGB::Sapphire         := [0.03, 0.14, 0.4] ),
( RGB::Sangria          := [0.57, 0.0, 0.03] ),
( RGB::Scarlet          := [1.0, 0.14, 0.0] ),
(RGB::SchoolBusYellow   := [1.0, 0.84, 0.0] ),
// ( RGB::SeaGreen 	:= [0.180395,0.545106,0.341197] ),
( RGB::SeaGreenDark     := [0.560793,0.737297,0.560793] ),
( RGB::SeaGreenLight    := [0.125507,0.698004,0.666707] ),
( RGB::SeaGreenMedium 	:= [0.235298,0.702002,0.443098] ),
( RGB::SelectiveYellow  := [1.0, 0.72, 0.0] ),
( RGB::Sepia            := [0.370006,0.149998,0.069994] ),
( RGB::ShockingPink     := [0.98, 0.05, 0.75] ),
( RGB::Sienna           := [0.627506,0.321604,0.176504] ),
( RGB::Silver           := [0.75, 0.75, 0.75] ), // = Grey75
//( RGB::SkyBlue 		:= [0.529405,0.807794,0.921598] ),
( RGB::SkyBlue          := [0.0,0.8,1.0] ),
( RGB::SkyBlueDeep      := [0.0,0.749,1.0] ),
( RGB::SkyBlueLight     := [0.529405,0.807794,0.980407] ),
( RGB::SlateBlue        := [0.415693,0.352901,0.803903] ),
( RGB::SlateBlueDark    := [0.282403,0.239204,0.545106] ),
( RGB::SlateBlueLight 	:= [0.517594,0.439207,1.0] ),
( RGB::SlateBlueMedium 	:= [0.482406,0.407804,0.933302] ),
( RGB::SlateGray        := [0.439207,0.501999,0.564699] ),
( RGB::SlateGrey        := [0.439207,0.501999,0.564699] ),
( RGB::SlateGrayDark    := [0.184301,0.309793,0.309793] ),
( RGB::SlateGreyDark    := [0.184301,0.309793,0.309793] ),
( RGB::SlateGrayLight 	:= [0.466704,0.533296,0.599994] ),
( RGB::SlateGreyLight 	:= [0.466704,0.533296,0.599994] ),
( RGB::Smoke            := [0.960799,0.960799,0.960799] ),
( RGB::SpringGreen      := [0.0,1.0,0.498001] ),
( RGB::SpringGreenMedium:= [0.0,0.980407,0.6039] ),
( RGB::SteelBlue        := [0.274499,0.509796,0.705893] ),
( RGB::SteelBlueLight 	:= [0.690207,0.768593,0.870602] ),
( RGB::SwampGreen       := [0.67, 0.71, 0.55] ),
( RGB::Tan              := [1.0,0.8,0.6] ),
( RGB::Taupe            := [0.73, 0.59, 0.49] ),
( RGB::TeaGreen         := [0.81, 0.94, 0.75] ),
( RGB::Teal             := [0.0,0.50196,0.50196] ),
( RGB::Terracotta       := [0.88, 0.44, 0.35] ),
( RGB::TerreVerte       := [0.219993,0.370006,0.059999] ),
( RGB::Thistle          := [0.847102,0.749,0.847102] ),
( RGB::Tomato           := [1.0,0.388195,0.278405] ),
//( RGB::Turquoise 	:= [0.250999,0.878399,0.815699] ),
( RGB::Turquoise        := [0.0,1.0,1.0] ),
( RGB::TurquoiseLight   :=
  RGB::LightTurquoise   := [0.8,1.0,1.0] ),
// ( RGB::TurquoiseBlue 	:= [0.0,0.780007,0.550005] ),
( RGB::TurquoiseDark 	  := [0.0,0.807794,0.819605] ),
//( RGB::TurquoiseMedium 	:= [0.282403,0.819605,0.799997] ),
( RGB::TurquoiseMedium  := [0.18, 0.83, 0.78] ),
( RGB::TurquoisePale 	  := [0.6863,0.933302,0.933302] ),
( RGB::Ultramarine 	    := [0.069994,0.039995,0.559999] ),
( RGB::UltramarineViolet:= [0.359996,0.140004,0.430006] ),
( RGB::UnitedNationsBlue:= [0.35, 0.57, 0.89] ),
( RGB::VanDykeBrown 	  := [0.370006,0.149998,0.020005] ),
( RGB::VenetianRed 	    := [0.829997,0.099994,0.119999] ),
( RGB::Vermillon        := [1.0, 0.3, 0.0] ),
// ( RGB::Violet 		:= [0.559999,0.370006,0.599994] ),
( RGB::Violet           := [0.50196,0.0,0.50196] ),
( RGB::VioletDark       := [0.580401,0.0,0.827494] ),
( RGB::VioletEggplant   := [0.6, 0.06, 0.6] ),
( RGB::VioletRed        := [0.815699,0.125507,0.564699] ),
( RGB::VioletRedMedium 	:= [0.780404,0.0824,0.521607] ),
( RGB::VioletRedPale    := [0.858806,0.439207,0.576495] ),
( RGB::Viridian         := [0.25, 0.5, 0.42] ),
( RGB::ViridianLight    := [0.430006,1.0,0.440001] ),
( RGB::WarmGray         := [0.5,0.5,0.410001] ),
( RGB::WarmGrey         := [0.5,0.5,0.410001] ),
( RGB::White            := [1.0,1.0,1.0] ),
( RGB::Wisteria         := [0.78, 0.62, 0.86] ),
( RGB::Yellow           := [1.0,1.0,0.0] ),
( RGB::YellowBrown      := [0.86, 0.58, 0.44] ),
( RGB::LightYellow      :=
  RGB::YellowLight      := [1.0,1.0,0.6] ),
( RGB::YellowOchre      := [0.889996,0.509995,0.089999] ),

// very bright colours
( RGB::LemonChiffon 	:= [1.0,0.980407,0.803903] ),
( RGB::Honeydew 	:= [0.941206,1.0,0.941206] ),
( RGB::GoldenrodLight 	:= [0.980407,0.980407,0.823496] ),
( RGB::AliceBlue 	:= [0.941206,0.972503,1.0] ),
( RGB::Titanium 	:= [0.990005,1.0,0.940001] ),
( RGB::Snow 		:= [1.0,0.980407,0.980407] ),
( RGB::BlanchedAlmond 	:= [1.0,0.921598,0.803903] ),
( RGB::MistyRose 	:= [1.0,0.894101,0.882397] ),
( RGB::CyanWhite 	:= [0.878399,1.0,1.0] ),
( RGB::LightBeige 	:= [0.960799,0.960799,0.862697] ),
( RGB::LavenderBlush  := [1.0,0.941206,0.960799] ),
( RGB::LavenderBush  := [1.0,0.941206,0.960799] ),
( RGB::Azure 		:= [0.94174,1.0,1.0] ),
( RGB::Cornsilk 	:= [1.0,0.972503,0.862697] ),
( RGB::OldLace 		:= [0.992203,0.960799,0.902005] ),
( RGB::Zinc 		:= [0.990005,0.97,1.0] ),
( RGB::Seashell 	:= [1.0,0.960799,0.933302] ),
( RGB::PapayaWhip 	:= [1.0,0.9373,0.835307] ),
( RGB::Antique 		:= [0.980575,0.92233,0.844661] ),
( RGB::Ghost 		:= [0.972503,0.972503,1.0] ),
( RGB::MintCream 	:= [0.960799,1.0,0.980407] ),
( RGB::Ivory 		:= [1.0,1.0,0.941206] ),
( RGB::Floral 		:= [1.0,0.980407,0.941206] ),
( RGB::Linen 		:= [0.980407,0.941206,0.902005] ),
( RGB::Celadon         := [0.67, 0.88, 0.68] ),
( RGB::Mauve           := [0.87, 0.69, 1.0] ),
( RGB::PalePink        := [0.98, 0.85, 0.86] ),
( RGB::Periwinkle      := [0.8, 0.8, 1.0] ),

null()
                  ]:

RGB::rgbrev := revert(table([op(RGB)])):
// to avoid having [1.0, 1.0, 1.0] map to
// Grey100 etc, add some values manually:
map(["Magenta", "Cyan", "Black", "White",
     "SlateGreyDark", "ColdGrey", "SlateGreyLight", "WarmGrey",
     "Grey5", "Grey10", "Grey15", "Grey20", "Grey25", "Grey30",
     "Grey35", "Grey40", "Grey45", "Grey50", "Grey55", "Grey60",
     "Grey65", "Grey70", "Grey75", "Grey80", "Grey85", "Grey90", "Grey95",
     "LightYellow",  "PaleBlue", "Grey",
     "LightBlue", "LightGreen", "NavyBlue", "LightGrey", "DarkGrey",
     "LightTurquoise", "DarkGreen", "SlateGrey"],
    x -> (RGB::rgbrev[slot(RGB, x)] := x)):

RGB::colorName :=
proc(rgb)
  local a, rgbname, eps, cands;
begin
  if testtype(rgb, DOM_IDENT) and
    strmatch("".rgb, "^#[A-Fa-f0-9]{6}([A-Fa-f0-9]{2})?$") then
    rgb := "".rgb;
    rgb := [text2int(rgb[2..3], 16)/255.0,
            text2int(rgb[4..5], 16)/255.0,
            text2int(rgb[6..7], 16)/255.0,
            if length(rgb)=9 then
              text2int(rgb[8..9], 16)/255.0
            end];
  end_if;  
  if not testtype(rgb, DOM_LIST) then
    return(procname(args()));
  end_if;
  if nops(rgb)<3 or nops(rgb)>4 then
    error("RGB color expected");
  end_if;
  rgb := float(rgb);
  if map(rgb, domtype) <> [DOM_FLOAT, DOM_FLOAT, DOM_FLOAT] and
    map(rgb, domtype) <> [DOM_FLOAT, DOM_FLOAT, DOM_FLOAT, DOM_FLOAT] then
    return(procname(args()));
  end_if;
  a := null();
  if nops(rgb) = 4 then
    a := rgb[4];
    rgb := rgb[1..3];
  end_if;
  if rgb[1]<0 or rgb[1]>1 or
     rgb[2]<0 or rgb[2]>1 or
     rgb[3]<0 or rgb[3]>1 then
    error("RGB color expected");
  end_if;
  
  rgbname := FAIL;
  if contains(RGB::rgbrev, rgb) then
    rgbname := RGB::rgbrev[rgb];
  elif contains({args()}, Exact) or
    not testtype(rgb, DOM_LIST) or
    nops(rgb)<3 or nops(rgb)>4 then
    rgbname := FAIL
  else
    // find nearest neighbor
    eps := 10.0^(-DIGITS);
    repeat
      cands := select(RGB::ColorList,
        l -> _lazy_and(abs(l[1]-rgb[1]) < eps,
                       abs(l[2]-rgb[2]) < eps,
                       abs(l[3]-rgb[3]) < eps));
      eps := 5.0*eps;
    until nops(cands)>0 end_repeat;
    if nops(cands)>1 then
      cands := prog::sort(cands,
        l -> sqrt(_plus(op(zip(l, rgb, (a, b) -> (a-b)^2)))));
      cands := [cands[1]];
    end_if;
    assert(contains(RGB::rgbrev, cands[1]));
    rgbname := RGB::rgbrev[cands[1]];
  end_if;
  if rgbname = FAIL then
    FAIL
  elif a = null() then
    hold(slot)(hold(RGB), rgbname);
  else
    hold(_concat)(hold(slot)(hold(RGB), rgbname), [a]);
  end_if;
end_proc:

RGB::toHSV :=
  proc(rgb)
    local r,g,b,h,s,v,a,minVal,delta;
  begin
    if domtype(rgb) <> DOM_LIST then
      error("expecting an RGB or RGBA color");
    end_if;
    if nops(rgb) = 4 then
      [r,g,b,a] := rgb;
    elif nops(rgb) = 3 then
      [r,g,b] := rgb;
      a := null();
    else
      error("expecting an RGB or RGBA color");
    end_if;
    // value
    v := max(r,g,b);

    minVal := min(r,g,b);
    delta := v - minVal;
    // saturation
    if iszero(v) then
      s := 0.0;
    else
      s := delta/v;
    end_if;

    // hue
    if iszero(s) then
      // white, black, grey
      h := 0.0;
    elif r = v then
      h := 60.0 * (g-b)/delta;
    elif g = v then
      h := 120 + 60.0 * (b-r)/delta;
    else
      h := 240 + 60.0 * (r-g)/delta;
    end_if;
    if h < 0 then h := h + 360; end_if;
    [h, s, v, a];
  end_proc:

RGB::fromHSV :=
  proc(hsv)
    local h,s,v,r,g,b,a, i, f, p,q,t;
  begin
    if not domtype(hsv)=DOM_LIST then
      error("expecting an HSV or HSVA color");
    end_if;
    hsv := float(hsv);
    if nops(hsv) = 4 then
      [h,s,v,a] := hsv;
    elif nops(hsv) = 3 then
      [h,s,v] := hsv;
      a := null();
    else
      error("expecting an HSV or HSVA color");
    end_if;
    if not map({h,s,v,a}, domtype)={DOM_FLOAT} then
      error("expecting an HSV or HSVA color");
    end_if;

    if iszero(s) then // grey
      r := v;
      g := v;
      b := v;
    else
      while h<0 do h := h+360.0; end_while;
      while h>=360 do h := h - 360.0; end_while;

      h := h/60; // scale to [0,6)
      if floor(h) = 6 then h := h - 6; end_if;

      i := floor(h);
      f := frac(h);
      p := v*(1.0-s);
      q := v*(1.0-(s*f));
      t := v*(1.0-(s*(1-f)));
      case i
	of 0 do
	  [r,g,b] := [v,t,p];
	  break;
	of 1 do
	  [r,g,b] := [q,v,p];
	  break;
	of 2 do
	  [r,g,b] := [p,v,t];
	  break;
	of 3 do
	  [r,g,b] := [p,q,v];
	  break;
	of 4 do
	  [r,g,b] := [t,p,v];
	  break;
	of 5 do
	  [r,g,b] := [v,p,q];
	  break;
	otherwise
	  error("program logic failed");
      end_case;
    end_if;
    [r,g,b,a];
  end_proc:

// Bruton's algorithm
RGB::fromWaveLength :=
proc(l, gamma=0.8)
  local r, g, b, f, ll;
begin
  ll := float(l);
  if domtype(ll) <> DOM_FLOAT then
    ll := unit::convert(ll, unit::nm)/unit::nm;
  end_if;
  if domtype(ll) <> DOM_FLOAT then
    return(procname(args()));
  end_if;

  if ll < 380 or ll > 780 then
    // outside the visible spectrum
    return([0.0$3]);
  end_if;

  [r,g,b] := if ll < 440 then
	       [(440-ll)/60, 0, 1]
	     elif ll < 490 then
	       [0, (ll-440)/50, 1]
	     elif ll < 510 then
	       [0, 1, (510-ll)/20]
	     elif ll < 580 then
	       [(ll-510)/70, 1, 0]
	     elif ll < 645 then
	       [1, (645-ll)/65, 0]
	     else
	       [1, 0, 0]
	     end_if;
  f := if ll < 420 then
	 0.3 + 0.7*(ll-380)/40
       elif ll < 700 then
	 1
       else
	 0.3 + 0.7*(780-ll)/80
       end_if;

  [(f*r)^gamma, (f*g)^gamma, (f*b)^gamma]
end_proc:

RGB::ColorNames_ := {op(RGB::ColorNames())}:
RGB::interface := {plotColorPalette,
                   ColorNames,
                   op(RGB::ColorNames_),
                   fromHSV,
                   toHSV,
                   colorName,
                   fromWaveLength,
                   hold(random),
                   generator}:

// define some methods to avoid creation via make_slot
RGB::evaluate:= FAIL:
RGB::posteval:= FAIL:
RGB::undefinedEntries := FAIL:
RGB::allEntries := FAIL:
RGB::allAutoEntries := FAIL:
RGB::whichEntry := FAIL:
RGB::new_extelement := FAIL:
RGB::create_dom_elem := FAIL:
RGB::Content:=FAIL:
RGB::MMLContent := FAIL:
RGB::Name := "RGB":
RGB::eval := FAIL:
RGB::domtype:= FAIL:
RGB::type:= FAIL:
RGB::isNeg:= FAIL:
RGB::isInverted:= FAIL:
RGB::slot:= FAIL:
RGB::expr2text:= FAIL:
RGB::indets:= FAIL:
RGB::freeIndets:= FAIL:
RGB::bool :=FAIL:
RGB::length := 1:
RGB::rectform:= FAIL:
RGB::contains:= (x,y) -> if x=y then TRUE else FALSE end_if:
RGB::expose := FAIL:
RGB::hastype := FAIL:
RGB::maprec := FAIL:
RGB::_exported := FAIL:
RGB::evaluateIndex := FAIL:
RGB::domain_index := FAIL:
RGB::sortSums := FAIL:
RGB::CF := FAIL: // see GENERATE/CF.mu

RGB::op := extop:
RGB::nops := extnops:

RGB::new := x -> error("RGB should not be used as a function. Please check ?RGB."):

/*--
make_slot -- create default method

make_slot always returns an error.

NOTE: This MUST be the last method defined for RGB.
--*/

RGB::make_slot :=
proc(DOM, Slot)
begin
  if domtype(Slot) = DOM_STRING then
    context(hold(warning)("color ".Slot." undefined, try RGB::ColorNames(\"".
                        Slot."\")")):
  else
    context(hold(warning)("color ".expr2text(Slot).
                        " undefined, try RGB::ColorNames(\"".
                        expr2text(Slot)."\")")):
  end_if;
  FAIL
end_proc:
