/*---------------------------------------------------------------
Numerical evaluation of 2D and 3D curves

plot::curveEval([X, Y, <Z>], 
                t = tmin..tmax, 
                [xmin .. xmax/Automatic, 
                 ymin .. ymax/Automatic,
               < zmin .. zmax/Automatic >
                ], 
                mesh) 
evaluates the procedures X,Y,Z on a mesh t.i in the interval [tmin, tmax]. 
Errors are trapped and  cause the result to split into several 
branches. Also non-real values of X, Y cause a splitting into several 
branches. The mesh is refined whereever is seems appropriate. 
plot::Curve2d/3d call this utility function.

The dimension of the curve is determined by the number
of functions in the first input list.

Calls:  plot::curveEval([X, Y <,Z>], 
                        t = tmin .. tmax,
                        [xmin/Automatic ..xmax/Automatic, 
                         ymin/Automatic ..ymax/Automatic
                        <ymin/Automatic ..ymax/Automatic>
                        ]
                     <, meshpoints> 
                     <, DiscontinuitySearch = TRUE/FALSE>
                     <, AdaptiveMesh = l>
                     <, "NoSingularities">
                       )
Parameters:
       X, Y, <Z>  : procedures accepting one parameter
       t          : the curve parameter: an identifier 
                    (provided by the user, may have properties)
       tmin, tmax : numerical real values 
       xmin, xmax : numerical real values (the viewing box requested
       ymin, ymax : by the user)
       meshpoints : the size of the initial mesh before refinement:
                    an integer >= 2
       l          : the recursive depth of the adaptive refinement:
                    a nonnegative integer
      "NoSingularities": do not enter getAutomaticViewingBox, but accept
                         the numerical min/max values as ViewingBox

Return: A list [ViewingBox, [branch1, branch2, ...], SetOfSingularities]
        with ViewingBox = [Xmin .. Xmax, Ymin .. Ymax],
        each branch is a list of points: branch.i= [[x1, y1], [x2, y2], ..]
        SetOfSingularities = {t1, t2, ...}

Example:
>> plot::curveEval([t -> t*sin(1/t), t -> t*cos(1/t)], 
                    t = 0.001 .. PI, [Automatic, -4 .. 4],
                   1000);
---------------------------------------------------------------*/
plot::curveEval:= proc(f, trange, vbox, meshpoints = 101
                /*,DiscontinuitySearch = TRUE/FALSE, AdaptiveMesh = TRUE/FALSE*/) 
local dim, nosingularities,
      adaptivemesh, adaptive, bisect, branches, branchindex, branch,
      closebranch, currentmeshpoints, cut, eps, findReal, 
      curveEval_rec, getAutomaticViewingBox, 
      getPoint, getPointData, getRange,
      i, lastPointOK, lastx, lastfmin, lastfmax, lock, lowslopes,
      maxrecursionlevel, maxtan, poleOrder, rawbranches, refine,
      s, singularities, _singularities, slopeTable, symbolicBranch, symbolicBranches,
      trackRange, ubbLO, ubbHI, usediscont, width,
      t, dt, lastt, tdata, tmin, tmax, _tmin, _tmax,
      x, fmax, fmin, fdata, 
      F, Fmax, Fmin, FTratio, oldFTratio,
      tmp, MINFLOAT, MAXFLOAT, ST, st
      ;
begin
   dim:= nops(f):     // the dimension is determined from the length of the list f
   MINFLOAT:= 1e-300: // smallest float that can be converted to a C double
   MAXFLOAT:= 1e300:  // largest float that can be converted to a C double
   ST:= time(); // Start time for timinings in userinfos
   //-------------------------------------------------------------------------
   // The following 'max bend angle' of 10 degrees is mentioned explicitly 
   // in the documentation of PLOT/ATTRIBUTES/AdaptiveMesh.tex. Do not change
   // this value without keeping the documentation up to date!
   // Further, keep this synchronized with functionEval!
   //-------------------------------------------------------------------------
   adaptivemesh:= op(select([args()], has, AdaptiveMesh), [1, 2]);
   if adaptivemesh = FAIL then 
      adaptivemesh:= 0;
   end_if;
   nosingularities:= has([args], "NoSingularities"):
   maxrecursionlevel:= adaptivemesh:
     // refine the initial mesh by up to 2^maxrecursionlevel
     // points between each pair of points on the initial mesh
   maxtan:= tan::float(10/180*PI);
                  // stop the refinement when consecutive
                  // line segments are (almost) parallel
                  // within 10 degrees
   // set the options
   adaptive:= bool(adaptivemesh > 0);
   if has([args()], DiscontinuitySearch = FALSE) then
        usediscont:= FALSE;
   else usediscont:= TRUE;
   end_if:
   trange:= op(trange, 2);
   [tmin, tmax, width]:= float([op(trange,1), op(trange,2), 
                                op(trange,2) -op(trange,1)]);

   for i from 1 to dim do
     if vbox[i] = Automatic then
        vbox[i]:= Automatic .. Automatic;
     end_if;
   end_for:

   for i from 1 to dim do
     // initialization for later refinement
     if op(vbox[i], 1) = Automatic then
          ubbLO[i]:= FALSE;    // userboundingbox?
          fmin[i]:=  MAXFLOAT; // The actual viewing box found during the evaluation
          Fmin[i]:= -MAXFLOAT; // The viewing box to be passed to the renderer
     else ubbLO[i]:= TRUE:     // userboundingbox?
          fmin[i]:= op(vbox[i], 1):
          Fmin[i]:= fmin[i];   // The bounding box requested by the user
     end_if;
     if op(vbox[i], 2) = Automatic then
          ubbHI[i]:= FALSE;    // userboundingbox?
          fmax[i]:= -MAXFLOAT; // The actual viewing box found during the evaluation
          Fmax[i]:=  MAXFLOAT; // The viewing box to be passed to the renderer
     else
          ubbHI[i]:= TRUE;     // userboundingbox?
          fmax[i]:= op(vbox[i], 2):
          Fmax[i]:= fmax[i];   // The bounding box requested by the user
     end_if;
   end_for:

   for i from 1 to dim do
     F[i]:= float@(f[i]):
   end_for:

   if meshpoints <= 1 then 
      warning("expecting a number of mesh points >= 2, got: ".
              expr2text(meshpoints).
              " Setting the number of mesh points to 2.");
      meshpoints:= 2:
   end_if:
   dt:= width/(meshpoints - 1):
   //---------------------------------------------------------
   // A heuristic numerical utility to check whether potential
   // singularities found by discont are indeed singularities.
   // Assuming that the singularity is a pole, the order 
   //   s = (g'(x))^2/((g'(x))^2 - g(x)*g''(x)), g(x)=1/f(x)
   // of the pole is estimated. Return values:
   //  0 (no singularity detected) or an approximation of s.
   //---------------------------------------------------------
   poleOrder:= proc(f, t)
   local eps, t1, t2, t3, y1, y2, y3, f0, f1, f2, s;
   save DIGITS;
   begin
       if not usediscont then
          return(0); // don't say anything about the pole order
       end_if;
       eps:= width*10.0^(-10-DIGITS);
       t:= t + width/10^10;  // do disturb the approximation
                            // of the potential singularity
       DIGITS:= 4*(10 + DIGITS);
       [t1, t2, t3]:= [t - eps/2, t + eps/2, t + 3/2*eps]:
       if traperror((
           y1:= 1/f(t1);
           y2:= 1/f(t2);
           y3:= 1/f(t3);
         )) = 0 then
            f0:= y2: 
            f1:= (y3 - y1)/2/eps;
            f2:= ((y1 - y2) + (y3 - y2))/eps^2:
       else return(0);
       end_if;
       // The following s should be the order of the pole
       if traperror((
           s:= float(f1^2/(f0*f2 - f1^2));
           )) <> 0 then
            return(0);  // cannot say anything about the pole order
       elif not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(s)) then
            return(0);  // cannot say anything about the pole order
       else return(Re(s));
       end_if;
   end_proc;
   //-----------------------------------------------------
   // utility getPoint(t): try to evaluate X(t), Y(t), Z(t)
   // If the result consist of real numbers, assign them to the 
   // variables x = [x[1],x[2],x[3]](of curveEval) (SIDE EFFECT!!)
   // and return TRUE.
   // If an error occurs, t is appended to the
   // set of (potential) singularities and FALSE is retuned.
   // If a non-real value X(t) or Y(t) or Z(t) is found, it 
   // is ignored and FALSE is returned.
   // The call getPoint(t) tracks fmin/max[i] ==  xmin, .. , zmax
   // as a SIDE EFFECT.
   // The call getPoint(t, any_second_argument) does not track 
   // xmin, xmax, ymin, ymax, zmin, zmax
   //-----------------------------------------------------
   getPoint:= proc(t) // assigns x:= [X(t),Y(t),Z(t)] as a side effect
   begin
     x:= [0 $ dim]; // this is a variable of plot::curveEval!
     if traperror(((x[i]:= F[i](t)) $ i=1..dim)) <> 0 then
//        singularities:= singularities union {t};
          return(FALSE)
     elif _lazy_and(
             x[i] <> RD_NAN $ i = 1..dim,
             _lazy_or(domtype(x[i]) = DOM_FLOAT,
                      domtype((x[i]:= numeric::complexRound(x[i]))) = DOM_FLOAT)
             $ i = 1..dim,
             x[i] >= -MAXFLOAT $ i = 1..dim, 
             x[i] <=  MAXFLOAT $ i = 1..dim, 
             _lazy_or((iszero(x[i]),
                       x[i] >= MINFLOAT, 
                       x[i] <=-MINFLOAT) $ i = 1..dim)
            ) then
          if args(0) = 1 then
             trackRange(t, x);
          end_if;
          return(TRUE)
     else return(FALSE);
     end_if;
   end_proc:
   //-----------------------------------------------------
   // utility trackRange(t, x): check the global extremal 
   // values fmin/max[i] (== xmin, ... , zmax) and redefine
   // them if necessary.
   //-----------------------------------------------
   trackRange:= proc(t, x) begin
     for i from 1 to dim do
       if x[i] < fmin[i] then fmin[i]:= x[i]; _tmin[i]:= t; end_if;
       if x[i] > fmax[i] then fmax[i]:= x[i]; _tmax[i]:= t; end_if;
     end_for:
   end_proc:
   //--------------------------------------------------------
   // Initialize the variables for the actual bounding box
   //--------------------------------------------------------
   // fmin[i], fmax[i] =  the actual bounding box found during evaluation
   (fmin[i]:= MAXFLOAT; fmax[i]:=-MAXFLOAT;) $ i = 1..dim ;
   //--------------------------------------------------------
   // exception: the t range has length 0
   //--------------------------------------------------------
   singularities:= {}:
   if iszero(width) then
      if getPoint(tmin, 1) then
           return([[x[i] .. x[i] $ i = 1..dim], [[[tmin, op(x)]]], {}]);
      else return([NIL, [[]], singularities]);
      end_if: 
   end_if:
   //--------------------------------------------------------
   // exception: tmin > tmax 
   //--------------------------------------------------------
   if tmin > tmax then
      return([NIL, [[]], {}]);
   end_if:
   //----------------------------------------------------------------
   // pre-processing: search for singularities via discont:
   //----------------------------------------------------------------
   symbolicBranches:= [[tmin, tmax]];
   if usediscont then
     userinfo(1, "Starting search for discontinuities.");
     st:= time():
     eps:= 1e-3*width;
     if traperror((
        singularities:= _union(float(
            numeric::discont(F[i](`#t`), `#t` = tmin .. tmax, eps/100, eps/5)
                                    ) $ i = 1.. dim);
        )) = 0 then
       if type(singularities) = DOM_SET then
         singularities:= select(singularities, t -> (domtype(t) = DOM_FLOAT));
         _singularities:= sort([op(singularities)]);
         s:= nops(_singularities);
         _tmin:= tmin:
         _tmax:= tmax:
         if s > 0 then
           if specfunc::abs(_singularities[1] - tmin) <= eps then 
             _tmin:= tmin + eps; 
           end_if;
           if specfunc::abs(_singularities[s] - tmax) <= eps then 
             _tmax:= tmax - eps; 
           end_if;
         end_if;
         _singularities:= 
            select(_singularities, s -> ((_tmin + 2*eps < s) and 
                                           (s < _tmax - 2*eps)));
         s:= nops(_singularities);
         if s > 0 then symbolicBranches:= [
                [_tmin, _singularities[1] - eps], 
                [_singularities[i] + eps, _singularities[i+1] - eps] $ i=1..s-1,
                [_singularities[s] + eps, _tmax] ];
                // Beware: we do need the left end of the branch
                // to be smaller than the right end:
                symbolicBranches:= select(symbolicBranches,
                                          branch ->  (branch[1] < branch[2]));

         end_if;
       else // we do not want to return a singularity set
            // consisting of symbolical stuff. Pretend
            // that we did not find any singularities
            singularities:= {}:
       end_if;
     end_if;
     userinfo(2, "Number of branches: ".expr2text(nops(symbolicBranches))):
     userinfo(3, "Time for discontinuity search: ".
                 expr2text(time() - st)." msec"):
   end_if;
   //-----------------------------------------------
   // findReal(f, a, b, fb, fmin, fmax) 
   // -- find a value t close to 'a' that produces a real
   //    & finite value f(t). This utility is used to
   //    find points t with real & finite function value 
   //    f(t) close to a problematic point 'a' (either a 
   //    singularity or f(a) = non-real).
   // On input: f(a) = non-real or +/-infinity, 
   //           f(b) = finite & real,
   // findReal(a, b, fb) is *guaranteed* to return an
   // t with finite real f(t).  However, t may coincide 
   // with b.
   //-----------------------------------
   findReal:= proc(a, b, fb)
   local lastt, lastf, lastfmin, lastfmax, t, maxcount, count;
   begin
      maxcount:= 20: // 2^maxcount intermediate points
      (lastfmin[i]:= fmin[i]: lastfmax[i]:= fmax[i]) $ i = 1..dim;
      if has(singularities, a) and
         getPoint(a + (b - a)/10^DIGITS) then
            b:= a + (b - a)/10^DIGITS;
            fb:= x;
            if _lazy_or((fb[i] < lastfmin[i],
                         fb[i] > lastfmax[i]) $ i = 1..dim) then
               branch[b]:= fb;
            end_if;
            return([b, fb]);
      end_if;
      lastt:= b:  // Remember the last OK point t, x, y, z
      lastf:= fb: // with fb = [X(t), Y(t), Z(t)] = real & finite
      t:= (a + b)/2;  // bisect
      if iszero(t - lastt) then
         return([lastt, lastf]); //cancellation
      end_if;
      for count from 1 to maxcount do
        (lastfmin[i]:= fmin[i]: lastfmax[i]:= fmax[i]) $ i = 1..dim;
        if getPoint(t) then  // getPoint(t) assigns x[1],x[2],x[3] as a side effect
             lastt:= t; // This point (x[1], y[1], z[1]) is OK.
             lastf:= x; // Remember x.
             if _lazy_or((x[i] < lastfmin[i],
                          x[i] > lastfmax[i]) $ i = 1..dim) then
                branch[t]:= x;
             end_if;
             b:= t: // Shorten the search interval (a, b] to (a, t]
        else a:= t; // Shorten the search interval (a, b] to (t, b]
        end_if;
        t:= (a + b)/2; // bisect
        if iszero(t - lastt) then
           break;
        end_if;
      end_for:
      return([lastt, lastf]); // return the last real point
   end_proc:
   //-----------------------------------------------------
   // The utility closebranch(branch) closes the current branch
   // by inserting it into the table 'branches' and increments 
   // the 'global' counter 'branchindex' for the next branch.
   //-----------------------------------------------------
   closebranch:= proc(branch) begin 
     if nops(branch) <> 0 then
       branches[branchindex]:= branch: // write branch to branches
       branchindex:= branchindex + 1; // increase branch index for next branch
    end_if;
   end_proc:
   //--------------------------------------------------------
   // initialize the branches table and open the first branch
   //--------------------------------------------------------
   st:= time():
   branches:= table();// table to contain all branches
   branchindex:= 1:   // branch index of the current branch

   //--------------------------------------------------------
   // Start the work: non-adaptive evaluation of the function
   // on the (default or user specified) coarse mesh.
   //--------------------------------------------------------
   for symbolicBranch in symbolicBranches do
     branch:= table():  // open a new branch
     [tmin, tmax]:= symbolicBranch;

     // reset the meshpoints to make sure that the meshpoints are
     // distributed uniformly and symmetrically in the current branch
     currentmeshpoints:= max(2, 1 + ceil((tmax - tmin)/width * (meshpoints - 1))):
     dt:= (tmax - tmin)/(currentmeshpoints - 1);

     // reset singularities for those singularities that
     // will be detected during the numerical evaluation.
     // This set will be filled via getPoint.
//   singularities:= {};
     //-----------------------------------------------------------------
     // Search for a starting point to be inserted into the first branch
     //-----------------------------------------------------------------
     t:= tmin;
     lastt:= t:
     while TRUE do
       if getPoint(t, 1) then // do not track xmin .. zmax
          break;
       end_if;
       lastt:= t;
       t:= t + dt;
       if t > tmax then break; end_if;
     end_while;
     if t > tmax then
        next; // no plot point in the current branch
     end_if;
     if t = tmax and not getPoint(t) then
        next; // no plot point in the current branch
              // There is no need to close this branch,
              // just use it for the next symbolicBranch
     end_if;
     // A starting point (t, [x, y, z]) was found, but not
     // written into the branch table, yet. First,
     // improve this starting point:
     if t <> tmin then
        [t, x]:= findReal(lastt, t, x);
     end_if;
     trackRange(t, x); // just in case findReal returned its
                       // input values t, [x[1],x[2],x[3]] 
                       // ([x[1],y[1],z[1]] were found via 
                       // getPoint(x, 1), i.e., without tracking)
     branch[t]:= x;
     lastt:= t;
     lastx:= x;
     lastPointOK:= TRUE;
     t:= t + dt;
     if t > tmax - dt/100 then
        t:= tmax;
     end_if;
     //------------------------------------------------
     // A starting point was found. Start the iteration
     //------------------------------------------------
     while t <= tmax do
        (lastfmin[i]:= fmin[i]; lastfmax[i]:= fmax[i]) $ i=1..dim;
        if getPoint(t) then
            if _lazy_or((x[i] < lastfmin[i],
                         x[i] > lastfmax[i]) $ i = 1..dim) then
               branch[t]:= x;
             end_if;
            // getPoint(t) produced a real point 
            if not lastPointOK then
               // improve the beginning of the new branch
               [t, x]:= findReal(lastt, t, x);
            end_if;
            branch[t]:= x;
            lastPointOK:= TRUE;
        else
           if lastPointOK then
              [t, x]:= findReal(t, lastt, lastx);
              branch[t]:= x;
              closebranch(branch);
              branch:= table(): // open a new branch
           end_if;
           lastPointOK:= FALSE;
        end_if:
        lastt:=t;
        lastx:=x;
        if t = tmax then 
           break; 
        end_if;
        t:= t + dt;
        if t > tmax - dt/100 then
           t:= tmax;
        end_if;
     end_while;
     closebranch(branch);
  end_for: // for symbolicBranch in symbolicBranches
  userinfo(1, "Evaluation on the initial mesh finished."):
  userinfo(2, "Number of branches: ".expr2text(nops(branches))):
  userinfo(2, "Total number of mesh points: ".
              expr2text(_plus(nops(op(branch, 2)) $ branch in branches))):
  userinfo(3, "Time to compute the initial mesh: ".
               expr2text(time() - st)." msec");
  //------------------------------------------------------------
  // non-adaptive numerical evaluation on the initial mesh
  // finished. The data are stored in the table 'branches',
  // and a copy 'rawbranches'.
  //------------------------------------------------------------

  if _plus(nops(branches[i]) $ i = 1..nops(branches)) = 0 then
      return([NIL, [[]], {}]);
  end_if;

  if not _and((ubbLO[i], ubbHI[i]) $ i=1..dim) then
     rawbranches:= branches:
  end_if;

  //------------------------------------------------------------
  // Determine the scaling ratio between the directions.
  // This aspect ratio is required by the adaptive refinement
  // of the raw mesh further down below. 
  //------------------------------------------------------------
  //----------------------------------------------------------------
  // utility
  //----------------------------------------------------------------
  getRange:= proc(i) // i = 1 = x-direction
                     // i = 2 = y-direction
                     // i = 3 = y-direction
  local branch, fmin, fmax;
  begin 
      fmin:= RD_INF;
      fmax:= RD_NINF;
      for branch in branches do
          // branch = (branchindex= table(t1=[x1,y1,z1], t2=[x2,y2,z2], ...) )
          // op(branch, 2)        = table(t1=[x1,y1,z1], t2=[x2,y2,z2], ...)
          // [op(op(branch, 2))]  =     [ t1=[x1,y1,z1], t2=[x2,y2,z2], ...]
          branch:= map([op(op(branch, 2))], op, [2, i]);
          fmin:= min(fmin, op(branch));
          fmax:= max(fmax, op(branch));
      end_for:
      return([fmin, fmax]):
  end_proc:

  //----------------------------------------------------------------
  // utility: let 'branch' be a table containing the points [x[1],x[2],x[3]] 
  // via branch[t] = x = [x[1],x[2],x[3]]. We need to extract the t values
  // and sort them in ascending order, sorting the [x[1],x[2],x[3]] values 
  // correspondingly.  We sort the indices 't' and use the table to obtain
  // a list of correspondingly ordered x[1],x[2],x[3] values. This way, we
  // can use the kernel sort without library call backs. Instead of passing
  // or returning the data table/data lists, we use the local variables 
  // branch, tdata, fdata of curveEval for speed.
  //----------------------------------------------------------------
  getPointData:= proc(/*<1>, <2>, <3>*/)
  local t, i;
  begin
    // branch = table(t1=[x1,y1,z1], t2=[x2,y2,z2], ..)
    tdata:= sort(map([op(branch)], op, 1)):
    (fdata[i]:=[branch[t][i] $ t in tdata];) $ i in [args()];
  end_proc:
  //----------------------------------------------------------------
  // getAutomaticViewingBox tries to clip the ranges in a smart way.
  // It checks heuristically whether very large/small data should be 
  // interpreted as singularities in the curve.
  // If not, it falls back to the values xmin, .. , zmax found during 
  // the numerical evaluation. These are stord as fmin[i], fmax[i] with
  // i = 1,2,3 Finally, it returns the viewing box Fmin .. Fmax in the
  // i-th direction.
  //----------------------------------------------------------------
  getAutomaticViewingBox:= proc(i) // i = 1 = x-direction
                                   // i = 2 = y-direction
                                   // i = 3 = z-direction
  local n, maxslope, slopes, try, FTratio, newFTratio,
        j, _fdata, minfdata, maxfdata, mindone, maxdone,
        allslopes, alldata,
        eq, eq5, tmp
        ;
  name plot::curveEval::getAutomaticViewingBox;
  begin
    if fmin[i] = MAXFLOAT or fmax[i] = -MAXFLOAT then
       // no real & finite plot point was found: this plot is empty
       return([NIL, NIL]):
    end_if;
    //-----------------------------------------------------
    // The following context construct serves for activating the
    // userinfo commands inside this sub procedure of curveEval
    // by pretending that the userinfo was called in the context
    // of the calling function, i.e., plot::curveEval
    context(hold(userinfo)(10, "Investigating the ".output::ordinal(i)." direction")):
    //-----------------------------------------------------
    if i=1 then
       tmp:= "Testing minimum [u, x(u)] = ":
    elif i=2 then
       tmp:= "Testing minimum [u, y(u)] = ":
    elif i=3 then
       tmp:= "Testing minimum [u, z(u)] = ":
    end_if;
    context(hold(userinfo)(10, tmp.expr2text([_tmin[i], fmin[i]]).
                               ". Pole order = ".expr2text(poleOrder(F[i],_tmin[i]))));
    mindone:= ubbLO[i]:
    if (not mindone) and poleOrder(F[i],_tmin[i]) > 0 then // no negative pole
       Fmin[i]:= fmin[i]:
       mindone:= TRUE;
    end_if;
    //-----------------------------------------------------
    if i=1 then
       tmp:= "Testing maximum [u, x(u)] = ":
    elif i=2 then
       tmp:= "Testing maximum [u, y(u)] = ":
    elif i=3 then
       tmp:= "Testing maximum [u, z(u)] = ":
    end_if;
    context(hold(userinfo)(10, tmp.expr2text([_tmax[i], fmax[i]]).
                               ". Pole order = ".expr2text(poleOrder(F[i],_tmax[i]))));
    maxdone:= ubbHI[i]:
    if (not maxdone) and poleOrder(F[i],_tmax[i]) > 0 then // no positive pole
       Fmax[i]:= fmax[i]:
       maxdone:= TRUE;
    end_if;
    //-----------------------------------------------------
    if mindone and maxdone then
       return([Fmin[i], Fmax[i]]);
    end_if;
    //-----------------------------------------------------
    // Now,  mindone = FALSE or maxdone = FALSE
    //-----------------------------------------------------
    allslopes:= []:
    slopeTable:= table():
    for branch in branches do
       branch:= op(branch, 2):
       n:= nops(branch);
       getPointData(i); // uses branch and sets tdata, fdata[i]
       if n = 0 then next end_if;
       // the right slopes:
       if n = 1 then
          // A single point forming a branch. Make sure that
          // it will be displayed by assigning a 0 slope to it.
          slopes:= [float(0)]
       else
          slopes:= [(fdata[i][j+1] - fdata[i][j])/(tdata[j+1] - tdata[j]) $ 
                    j = 1..n - 1];
          // Beware: the heuristics for eliminating singular points
          // will fail miserably, wenn singular points have small
          // slopes. This may happen, when points close to an even 
          // pole are spaced symmetrically, leading to a 0 slope. 
          // The cure: average between neighbouring slopes.
          // However, if only one point very close to an even 
          // pole is found, the averaged slope may become small, 
          // because the left slope is >> 0 and the right slope is <<0.
          // The cure: use averaging of *absolute* slope values:
          slopes:= map(slopes, specfunc::abs);
          // the averaged slopes:
          slopes:= [slopes[1], (slopes[j-1]+slopes[j])/2 $ j=2..n-1, slopes[n-1]];
       end_if;
       for j from 1 to n do
           if type(slopeTable[slopes[j]]) = "_index" then
                slopeTable[slopes[j]]:= fdata[i][j];
           else slopeTable[slopes[j]]:= slopeTable[slopes[j]], fdata[i][j];
           end_if;
       end_for:
       if allslopes = [] then
            allslopes:= slopes;
       else allslopes:= allslopes . slopes;
       end_if;
    end_for;
    if nops(allslopes) = 0 then
       return([NIL, NIL]);
    end_if;
    assert(getRange(i) = [fmin[i], fmax[i]]);
    if not mindone then
       Fmin[i]:= fmin[i]:
    end_if;
    if not maxdone then
       Fmax[i]:= fmax[i]:
    end_if;
    FTratio:= (Fmax[i] - Fmin[i])/width;
    //-------------------------------------------------------
    // The crucial loop of the heuristics starts: for try ...
    //-------------------------------------------------------
    lowslopes:= allslopes;
    for try from 1 to 6 do
      cut:= min(10.0^(try/3), 100)*FTratio;
      // select from the old lowslopes instead of all data,
      // because we are decreasing FTratio in each 'try' step
      lowslopes:= select(lowslopes, _less, cut);
      if nops(lowslopes) = 0 then 
         // try again with larger FTratio (less points will
         // be removed in the next version of 'lowslopes'):
         FTratio:= 10^3*FTratio;
         // reset lowslopes to *all* the original data,
         // since we are increasing the FTratio
         lowslopes:= allslopes;
         next;
      end_if;
      // use 'sl' in the **set** of lowslopes to avoid
      // redundant multiple entries in the fdata sequence:
      _fdata:= op(map({op(lowslopes)}, sl -> slopeTable[sl]));
      [minfdata, maxfdata]:= [min(_fdata), max(_fdata)];
      if try < 2 then
         maxslope:= max(op(lowslopes));
         if maxslope < (maxfdata - minfdata)/width*10^2 or
            nops(lowslopes) < 10 then
            FTratio:= min((fmax[i] - fmin[i])/width, 5*FTratio);
            // reset lowslopes to *all* the original data, since
            // we are increasing the FTratio
            lowslopes:= allslopes;
            context(hold(userinfo)(10, "Step ".expr2text(try).
                     ". Experimentally increasing the aspect ratio F:U to ".
                     expr2text(FTratio))):
            next;
         end_if;
      end_if; // if try < 2
      if not mindone then 
         Fmin[i]:= minfdata;
      end_if;
      if not maxdone then
         Fmax[i]:= maxfdata;
      end_if;
      newFTratio:= (maxfdata - minfdata)/width;
      if try > 4 and FTratio = newFTratio then
           break;
      else FTratio:= newFTratio;
      end_if;
      context(hold(userinfo)(10, "Step ".expr2text(try).". Modifying the f range to ".
                    expr2text(Fmin[i] .. Fmax[i]).". Trying the new aspect ratio F:U = ".
                    expr2text(FTratio))):
    end_for: // for try from 1 to 6 do

    //----------------------------------------------------------------------
    // A heuristic clipping range Fmin[i] .. Fmax[i] was found.
    //----------------------------------------------------------------------
    // Check, whether fmax[i] is an approximation of infinity
    // or whether fmax[i] is a regular maximum
    if not maxdone then
       if fmax[i] <> Fmax[i] and 
          (fmax[i] <= 0 or 
           Fmax[i] >= fmax[i]/10 or 
           Fmax[i] <= Fmin[i]) then 
           Fmax[i]:= fmax[i]; 
           userinfo(10, "Resetting maximum of f. Choosing f range = ".  expr2text(Fmin[i] .. Fmax[i])):
       end_if;
    end_if;
    // Check, whether fmin[i] is an approximation of -infinity
    // or whether fmin[i] is a regular minimum
    if not mindone then
      if fmin[i] <> Fmin and 
         (fmin[i] >= 0 or 
          Fmin[i] <= fmin[i]/10 or 
          Fmin[i] >= Fmax[i]) then 
          Fmin[i]:= fmin[i]; 
          userinfo(10, "Resetting minimum of f. Choosing f range = ".  expr2text(Fmin[i] .. Fmax[i])):
      end_if;
    end_if;

    userinfo(10, "f range before symmetrization = ".  expr2text(Fmin[i] .. Fmax[i])):
    //----------------------------------------------
    // symmetrize Fmin[i], Fmax[i]
    //----------------------------------------------
    if not (mindone and maxdone) then
      if Fmax[i] < fmax[i] and Fmin[i] > fmin[i] then
        // there is a positive and a negative singularity

        alldata:= []:
        for branch in branches do
            branch:= op(branch, 2):
            getPointData(i); // uses branch and sets fdata
            alldata:= alldata . fdata[i];
        end_for:

        eq:= stats::empiricalQuantile(alldata):
        eq5:= eq(0.5):
        tmp:= sqrt(specfunc::abs(eq5 - Fmin[i])*specfunc::abs(Fmax[i] - eq5));
        if (not iszero(tmp))   and
           fmin[i] < eq5 - tmp and
           eq(0.2) > eq5 - tmp and
           fmax[i] > eq5 + tmp and
           eq(0.8) < eq5 + tmp then
             if not ubbLO[i] then
                Fmin[i]:= eq5 - tmp;
             end_if;
             if not ubbHI[i] then
                Fmax[i]:= eq5 + tmp;
             end_if;
        end_if;
      end_if;
    end_if;
    userinfo(10, "f range after symmetrization = ".  expr2text(Fmin[i] .. Fmax[i])):

    if Fmin[i] > Fmax[i] then 
       Fmin[i]:= (Fmin[i] + Fmax[i])/2:
       Fmax[i]:= Fmin[i]:
    end_if;

    // Heuristics:
    // If 0 < Fmin << Fmax, we probably should set Fmin = 0.0
    // increasing the viewing range up to 10 per cent.
    if (not ubbLO[i]) and Fmin[i] > 0 and Fmin[i] < Fmax[i]/10 then
       Fmin[i]:= float(0);
    end_if;
    // If Fmin << Fmax < 0, we probably should set Fmax = 0.0
    // increasing the viewing range up to 10 per cent.
    if (not ubbHI[i]) and Fmax[i] < 0 and Fmax[i] > Fmin[i]/10 then
       Fmax[i]:= float(0);
    end_if;

    //-----------------------------------------------------
    // automatic viewing box Fmin[i] .. Fmax[i] is computed
    //-----------------------------------------------------

    return([Fmin[i], Fmax[i]]);
  end_proc:

  //-----------------------------
  // Call getAutomaticViewingBox
  //-----------------------------
  st:= time():
  for i from 1 to dim do
    if not (ubbLO[i] and ubbHI[i]) then
       if nosingularities then
          tmp:= [fmin[i], fmax[i]];
       else
          userinfo(1, "Determining the automatic viewing box in the ".
                      output::ordinal(i)." direction"):
          tmp:= getAutomaticViewingBox(i):
       end_if;
       if not has(tmp, NIL) then
          if not ubbLO[i] then
             Fmin[i]:= tmp[1];
          end_if;
          if not ubbHI[i] then
             Fmax[i]:= tmp[2];
          end_if;
       end_if;
    end_if;
  end_for:
     
  if not _and((ubbLO[i], ubbHI[i]) $ i=1..dim) then
     userinfo(2, "Automatic viewing box determined: ".
                 expr2text([Fmin[i]..Fmax[i] $ i=1..dim]));
     userinfo(3, "Time for determining the automatic viewing box: ".
                 expr2text(time() - st)." msec"):
  end_if;
  //--------------------------------------------------------
  // Determine the aspect ratios x:t, y:t, y:t needed
  // for the following refinement.
  //--------------------------------------------------------
  (FTratio[i]:= (Fmax[i] - Fmin[i])/width;) $ i=1..dim;
  //-------------------------------------------------------
  // Next, we go for an adaptive refinement of the branches
  //-------------------------------------------------------
  //------ the bisect utility -----------------------------
  // It just checks whether 2 consecutive line segments are
  // no more than 10 degrees apart (maxtan = tan(10 degrees) 
  // is set at the beginning of this procedure). 
  // bisect requires the FTratios computed above.
  //------------------------------------------------------
  if _and(iszero(FTratio[i]) $ i=1..dim) then
       bisect:= () -> FALSE
  else if dim = 2 then
         bisect:= proc(t1, f1, tm, fm, t2, f2)
         local x1, y1, xm, ym, x2, y2, dx1, dx2, dy1, dy2, tmp1, tmp2;
         begin
             [x1, y1]:= f1;
             [xm, ym]:= fm;
             [x2, y2]:= f2;
             dx1:= (xm - x1)/(tm - t1)*FTratio[2]; 
             dx2:= (x2 - xm)/(t2 - tm)*FTratio[2];
             dy1:= (ym - y1)/(tm - t1)*FTratio[1]; 
             dy2:= (y2 - ym)/(t2 - tm)*FTratio[1];
             // with c = cos(epsilon), t = tan(epsilon), the criterion for
             // bisect is <d1,d2>^2 < c^2*<d1,d1>*<d2,d2)
             // or, equivalently, tmp1 * maxtan < tmp2
             tmp1:= dx1*dx2 + dy1*dy2;
             tmp2:= specfunc::abs(dx1*dy2 - dx2*dy1);
             return(bool( tmp1 * maxtan < tmp2));
         end_proc;
       else // dim = 3
         bisect:= proc(t1, f1, tm, fm, t2, f2)
         local x1, y1, z1, xm, ym, zm, x2, y2, z2,
               dx1, dx2, dy1, dy2, dz1, dz2, tmp1, tmp2, i, s;
         begin
             [x1, y1, z1]:= f1;
             [xm, ym, zm]:= fm;
             [x2, y2, z2]:= f2;
             dx1:= (xm - x1)/(tm - t1); dx2:= (x2 - xm)/(t2 - tm);
             dy1:= (ym - y1)/(tm - t1); dy2:= (y2 - ym)/(t2 - tm);
             dz1:= (zm - z1)/(tm - t1); dz2:= (z2 - zm)/(t2 - tm);
             // with c = cos(epsilon), t = tan(epsilon), the criterion for
             // bisect is <d1,d2>^2 < c^2*<d1,d1>*<d2,d2)
             // or, equivalently, tmp1^2 * maxtan^2 < tmp2
             s:= [FTratio[i] $ i = 1..dim]:
             s:= map(s, r -> if iszero(r) then 1 else r end_if):
             tmp1:= ( dx1*dx2/s[1]*s[2]*s[3]
                     +dy1*dy2*s[1]/s[2]*s[3]
                     +dz1*dz2*s[1]*s[2]/s[3]
                    );
             tmp2:= ( ((dx1*dy2 - dx2*dy1)*s[3])^2
                     +((dx1*dz2 - dx2*dz1)*s[2])^2
                     +((dy1*dz2 - dy2*dz1)*s[1])^2
                    );
             return(bool((tmp1 * maxtan)^2 < tmp2));
         end_proc:
       end_if;
  end_if;
  //----- recursive utility -------------------------
  // Test, whether the mid point c = (a + b)/2 of an
  // interval [a, b] is reasonably represented by the
  // straight line through a and b. If not, split the
  // interval into 2 halves [a, c] and [c, b].
  //-------------------------------------------------
  curveEval_rec:= proc(a, fa, c, fc, b, fb, reclevel)
  local c1, fc1, c2, fc2;
  begin
     // different calls to curveEval_rec communicate
     // with one another via the 'lock' table
     if lock[a, b] = TRUE then return(): end_if:
     if reclevel >= maxrecursionlevel then return(); end_if;
     if a = c or c = b or a = b then return() end_if;
     if bisect(a, fa, c, fc, b, fb) then
        c1:= (a + c)/2: 
        if getPoint(c1) then
            fc1:= x:
            branch[c1]:= fc1;
            curveEval_rec(a, fa, c1, fc1, c, fc, reclevel + 1);
            lock[a, c]:= TRUE;
        end_if;
        c2:= (c + b)/2: 
        if getPoint(c2) then
            fc2:= x:
            branch[c2]:= fc2;
            curveEval_rec(c, fc, c2, fc2, b, fb, reclevel + 1);
            lock[c, b]:= TRUE;
        end_if;
        if lock[a, c] = TRUE and lock[c, b] = TRUE then
            // these conditions garantee that fc1, fc2 were ok.
            curveEval_rec(c1, fc1, c, fc, c2, fc2, reclevel + 1);
      //    lock[c1, c2]:= TRUE;  // no need to lock this interval
        end_if;
     end_if;
     return();
  end_proc:
  //-------------------------------------------------
  // adaptive refinement of the branches. Implement 
  // 'refine' as a procedure (it may be called twice)
  //-------------------------------------------------
  refine:= proc() 
  local tmp, i, j;
  begin
    // do not declare 'branch', 'branchindex', 'lock' as local 
    // to 'refine', because 'curveEval_rec' refers to these
    // values!
    for branch in branches do
        // lock is a table that is used by curveEval_rec.
        // When the entry lock[x2, x3] = TRUE is set by
        // curveEval_rec(x1, x2, x3), then the next call
        // curveEval_rec(x2, x3, x4) will not recurse into
        // the locked interval [x2, x3] (it was handled by the
        // first call curveEval_rec(x1, x2, x3)).
        lock:= table(): // initialize the locks 
        // turn the table 'branch' (with branch[t] = [x,y,z]) into a
        // sorted list of the form branch = [.., [x, z, y] ,..]
        branchindex:= op(branch, 1); // global used by getPoint
        branch:= op(branch, 2): // branch is a table to be enlarged
                                // by the following call to
                                // curveEval_rec
        getPointData(i $ i=1..dim); // uses branch and sets tdata, fdata
        // The following call fills in the 'global' branch
        // as a side effect. The calls for consecutive values
        // of i communicate via the 'lock' table:
        for j from 2 to (nops(tdata) - 1) do
          curveEval_rec(tdata[j-1],[fdata[i][j-1] $ i = 1..dim],
                        tdata[ j ],[fdata[i][ j ] $ i = 1..dim],
                        tdata[j+1],[fdata[i][j+1] $ i = 1..dim],
                        0);
          // Keep the lock table as small as possible. The entry
          // for the first half of the interval above is not 
          // needed anymore, because the next call cannot descend
          // into this interval, anyway.
          delete lock[tdata[j-1], tdata[j]];
        end_for:
        // reinsert the enlarged branch into the branches table:
        branches[branchindex]:= branch;
    end_for:
  end_proc:
  //----------------------------------------------------
  // The curve was evaluated on the initial mesh.
  // If appropriate, the data were split into several
  // branches (stored in the 'branches' table).
  // Next, if AdpativeMesh = TRUE, refine the branches.
  //----------------------------------------------------
  oldFTratio:= FTratio; // remember XTratio for double checking
  if adaptive then
     userinfo(1, "Starting adaptive refinement of the initial mesh."):
     userinfo(3, "Assuming aspect ratio F:U = ".expr2text(FTratio)):
     st:= time():
     refine(); 
     userinfo(2, "New total number of mesh points: ".
                 expr2text(_plus(nops(op(branch,2)) $ branch in branches))):
     userinfo(3, "Time for the adaptive refinement: " .
                 expr2text(time() - st)." msec"):
     //-------------------------------------------------
     // Adaptive refinement of the branches is finished.
     //-------------------------------------------------
     // Double check the automatic ViewingBox
     //-------------------------------------------------
     if not _and((ubbLO[i], ubbHI[i]) $ i=1..dim) then
       oldFTratio:= FTratio; // remember FTratio for double checking
       for i from 1 to dim do
         if nosingularities then
            tmp:= [fmin[i], fmax[i]];
         else
            userinfo(1, "Determining the automatic viewing box in the ".
                        output::ordinal(i)." direction"):
            tmp:= getAutomaticViewingBox(i):
         end_if;
         if not has(tmp, NIL) then
            if not ubbLO[i] then
               Fmin[i]:= tmp[1];
            end_if;
            if not ubbLO[i] then
               Fmax[i]:= tmp[2];
            end_if;
         end_if;
       end_for:
       userinfo(2, "Automatic viewing box determined: ".
                expr2text([Fmin[i]..Fmax[i] $ i = 1..dim])):
       userinfo(3, "Time for determining the automatic viewing box: ".
                expr2text(time() - st)." msec"):
       //--------------------------------------------------------
       // Determine the aspect ratios x:t, y:t, y:t needed
       // for the following refinement.
       //--------------------------------------------------------
       (FTratio[i]:= (Fmax[i] - Fmin[i])/width;) $ i=1..dim;
       for i from 1 to dim do
         if FTratio[i] <> oldFTratio[i] then
            userinfo(3, "The aspect ratio X".expr2text(i).":U has changed from ".
                        expr2text(oldFTratio[i]). " to ".expr2text(FTratio[i])):
         else
            userinfo(3, "Using the aspect ratio X".expr2text(i).":U = ".
                        expr2text(oldFTratio[i]));
         end_if;
       end_for:
       //----------------------------------------------------------------
       // re-refine: the refinement depends crucially on the
       // scaling ratio estimated before the refinement. We
       // double check the scaling ratio. If it has dropped to 
       // less than a quarter of its original value, the maximal
       // bend angle of 6 degrees may have amplified to 24 degrees
       // in the actual plot. In this case, we need to recompute 
       // the refinement.
       //----------------------------------------------------------------
       if _lazy_or((FTratio[i] < oldFTratio[i]/4) $ i=1..dim) then
          userinfo(1, "The aspect ratio changed drastically by ".
                      "the 1st adaptive refinement. Need to recompute.");
          userinfo(1, "Starting 2nd adaptive refinement of the mesh."):
          st:= time():
          // do not use the refined mesh in 'branches', because a second
          // refinement would lead to meshs that are much too fine.
          // Use the initial raw mesh instead. The improved second 
          // refinement relies on the better value of XTratio etc.
          branches:= rawbranches: 
          // Do the work:
          refine();
          userinfo(2, "New total number of meshpoints:".
                      expr2text(_plus(nops(op(branch, 2)) $ branch in branches)));
          userinfo(3, "Time for 2nd adaptive refinement: " .
                      expr2text(time() - st)." msec"):
       end_if; //if FRratio[i] < oldFTration[i]/4
     end_if; // if not ubb
  end_if: // if adaptive
  //----------------------------------------------------------
  // pass the values of the automatic viewing box to the renderer.
  // Enlarge the viewing box by a tiny amount.
  //----------------------------------------------------------
  for i from 1 to dim do
    eps:= (Fmax[i] - Fmin[i])*10^(-DIGITS):
    Fmin[i]:= Fmin[i] - eps:
    Fmax[i]:= Fmax[i] + eps:
  end_for:
  //----------------------------------------------------------
  // Each branch in the 'branches' table is a table with
  // branch[t] = [x(t), y(t), z(t)]. Turn it into a sorted
  // list branch = [[t1,x1,y1,z1], [t2,x2,y2,z2], ...]:
  //----------------------------------------------------------
  userinfo(1, "Computation finished. Preparing the return data."):
  st:= time():
  for branch in branches do
      branchindex:= op(branch, 1); // global used by getPointData
      branch:= op(branch, 2):
      // now, branch = table(t1=[x1,y1,z1], t2=[x2,y2,z2], ..)
      tdata:= sort(map([op(branch)], op, 1)):
      branches[branchindex]:= [ [t, op(branch[t])] $ t in tdata];
  end_for;
  userinfo(3, "Time for preparing the return data: ".
              expr2text(time() - st)." msec"):
  //----------------------------------------------------------
  // return [[[t1,x1,y1,z1],[t2,x2,y2,z2],...], [[t1,x1,y1,z1],  ...  ], ...]
  //         |------------ branches[1] ------|  |---- branches[2] ----|
  //----------------------------------------------------------
  userinfo(3, "Total time needed: ".expr2text(time() - ST)." msec"):
  return([[Fmin[i]..Fmax[i] $ i=1..dim], // the viewing box
      //  map([op(branches)], op, 2),    // the plot data as
                                         // unordered branches
          [branches[i] $ i=1..nops(branches)],// the plot data as ordered
                                              // branches (the Hatch object
                                              // needs ordered branches)
          singularities                  // the set of singularities
         ]);
end_proc:
//--------------------------------------------------------------
