/*
   plot::listAttribute2table:  

   converts a legal list 'values' into a table

   Legal inputs are:
   
   [PosInt=types...]
     
   [v1, v2...] 
   with v1, v2 of type types


   Parameters:
   attr    -- attribute name
   num     -- number of expected arguments
   types   -- a Type of the equations rhs
   value   -- the value to be checked
   default -- fill in value

   Return value:  a table indexed by positive integers

*/

plot::listAttribute2table := 
proc(attr, num, types, value, default)
  local res, i, equations, values, dummy;
begin
  if not testtype(default, types) then
    error("default value '".expr2text(default)."' is not of type '".
          expr2text(types)."'")
  end_if;

  if domtype(value) <> DOM_LIST then
    context(error)("list expected for attribute '".attr."'");
  end_if;

  [equations, values, dummy] := split(value, testtype, "_equal");

  // check equations
  // are all equation right hand sides of the expected type?
  if equations <> [] then
    if map(map(map({op(equations)}, op, 2), float), testtype, types) = {TRUE} then
      // are all equation left hand sides positive integers?
      if map(map({op(equations)}, op, 1), testtype, Type::PosInt) <> {TRUE} then
        context(error)("equations left hand sides must be positive integers for attribute '".attr."'");
      else
        equations := map(equations, x -> op(x, 1) = float(op(x, 2)));
      end_if;
    else
      context(error)("equations right hand sides must be of type '".
                     expr2text(types)."' for attribute '".attr."'");
    end_if;
  end_if;

  // check other values
  // are all values of the expected type?
  if values <> [] then
    if map(map({op(values)}, float), testtype, types) <> {TRUE} then
      context(error)("values must be of type '".
                     expr2text(types)."' for attribute '".attr."'");
    else
      values := map(values, float);
    end_if;
  end_if;

  // if there are other values, the last one is the new fill value
  if nops(values) > 0 then
    default := values[-1];
  end_if;
  
  // start with default values
  res := table(i = default  $ i = 1..num);

  // insert explicit values
  res := table(res, i=values[i] $ i = 1..nops(values));

  // insert equations
  res := table(res, equations);

  res
end:
